package com.digiwin.athena.atdm.util;

import org.apache.commons.lang.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.PropertyResourceBundle;
import java.util.ResourceBundle;
import java.util.stream.Collectors;


public class MessageUtil {
    /**
     * 将国际化信息存放在一个map中
     */
    private static final Map<String, ResourceBundle> MESSAGES = new HashMap<String, ResourceBundle>();

    private static final String BASENAME = "i18n/messages";

    /**
     * 获取国际化信息
     */
    public static String getMessage(String key, Object... params) {
        //获取语言，这个语言是从header中的Accept-Language中获取的，
        //会根据Accept-Language的值生成符合规则的locale，如zh、pt、en等
        String locale1 = null;
        ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (servletRequestAttributes == null) {
            locale1 = "zh_CN";
        } else {
            HttpServletRequest request = servletRequestAttributes.getRequest();
            locale1 = request == null ? "zh_CN" : request.getHeader("locale");
            if (StringUtils.isEmpty(locale1)) {
                locale1 = "zh_CN";
            }
        }

        return getMessageByLocale(key, locale1, params);
    }

    private static ResourceBundle getResourceBundle(Locale locale) {
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        return ResourceBundle.getBundle(BASENAME, locale, classLoader, new ResourceBundle.Control() {
            @Override
            public List<String> getFormats(String baseName) {
                return Arrays.asList("properties");
            }

            @Override
            public ResourceBundle newBundle(String baseName, Locale locale, String format, ClassLoader loader, boolean reload) throws IllegalAccessException, InstantiationException, IOException {

                String bundleName = toBundleName(baseName, locale);
                String resourceName = toResourceName(bundleName, "properties");
                Enumeration<URL> urls = loader.getResources(resourceName);

                PropertyResourceBundle result = null;
                while (urls.hasMoreElements()) {
                    URL url = urls.nextElement();
                    try (InputStream stream = url.openStream()) {
                        PropertyResourceBundle bundle = new PropertyResourceBundle(stream);
                        if (result == null) {
                            result = bundle;
                        } else {
                            result = mergeBundles(result, bundle);
                        }
                    }
                }
                return result;
            }

            private PropertyResourceBundle mergeBundles(PropertyResourceBundle main, PropertyResourceBundle additional) throws IOException {

                Map<String, String> merged = new HashMap<>();
                main.keySet().forEach(k -> merged.put(k, main.getString(k)));
                additional.keySet().forEach(k -> merged.put(k, additional.getString(k)));
                return new PropertyResourceBundle(new ByteArrayInputStream(merged.entrySet().stream()
                        .map(e -> e.getKey() + "=" + e.getValue())
                        .collect(Collectors.joining("\n")).getBytes(StandardCharsets.UTF_8)));
            }
        });
    }

    public static String getMessageByLocale(String key, String locale1, Object... params) {
        String[] s = locale1.split("_");
        Locale locale = new Locale(s[0], s[1]);
        ResourceBundle message = MESSAGES.get(locale.toString());
        if (message == null) {
            synchronized (MESSAGES) {
                //在这里读取配置信息
                message = MESSAGES.get(locale.toString());
                if (message == null) {
                    //注1
                    message = getResourceBundle(locale);
                    MESSAGES.put(locale.toString(), message);
                }
            }
        }
        //此处获取并返回message
        if (params != null) {
            try {
                if (locale1.equals("zh_CN")) {
                    return String.format(new String(message.getString(key).getBytes("UTF-8"), "UTF-8"), params);
                } else {
                    return String.format(message.getString(key), params);
                }
            } catch (UnsupportedEncodingException e) {
                return "";
            }
        }
        try {
            if (locale1.equals("zh_CN")) {
                return new String(message.getString(key).getBytes("UTF-8"), "UTF-8");
            } else {
                return String.format(message.getString(key), params);
            }
        } catch (UnsupportedEncodingException e) {
            return "";
        }
    }

    /**
     * 清除国际化信息
     */
    public static void flushMessage() {
        MESSAGES.clear();
    }
}