package com.digiwin.athena.iam.sdk.manager;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.common.sdk.manager.meta.constants.DwHttpHeaderConstants;
import com.digiwin.athena.common.sdk.manager.util.DwHttpResultParseUtil;
import com.digiwin.athena.common.sdk.manager.util.DwSpringHttpUtil;
import com.digiwin.athena.iam.sdk.meta.dto.IamResultBDTO;
import com.digiwin.athena.iam.sdk.meta.dto.request.UserDataPermissionParamDTO;
import com.digiwin.athena.iam.sdk.meta.dto.request.UserDataPermissionResultDTO;
import com.digiwin.athena.iam.sdk.meta.dto.response.AuthoredUserDTO;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.util.JaStringUtil;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.util.CollectionUtils;

import java.util.*;

import static com.digiwin.athena.iam.sdk.meta.constants.IamErrorCode.P_IAM_500_5001;
import static com.digiwin.athena.iam.sdk.meta.constants.IamErrorCode.P_IAM_500_5002;

/**
 * iam api
 *
 * @author gengwei
 * @since 2024/8/15 11:48
 */
public class IamAuthManager {


    private static String getIamUri() {
        return JaProperty.get("iam.uri");
    }

    private static final ParameterizedTypeReference<IamResultBDTO<AuthoredUserDTO>> AUTHORED_USER_DTO_PARAMETERIZED_TYPE_REFERENCE = new ParameterizedTypeReference<IamResultBDTO<AuthoredUserDTO>>() {
    };

    /**
     * 根据userSid获取租户语言别设置
     */
    public static final String IDENTITY_TOKEN_ANALYZE = getIamUri() + "/api/iam/v2/identity/token/analyze";

    /**
     * 获取用户的数据权限
     */
    public static final String USER_DATA_PERMISSION = getIamUri() + "/api/iam/v2/permission/data";

    /**
     * 根据token获取用户信息
     * 该接口返回结果需要定制化处理
     * <p>
     */
    public static AuthoredUserDTO getUserInfo(String token) {
        if (JaStringUtil.isEmpty(token)) {
            throw BusinessException.buildBizErrorCode(P_IAM_500_5001);
        }
        AuthoredUserDTO result = null;
        try {
            result = DwHttpResultParseUtil.parse(DwSpringHttpUtil.post(IDENTITY_TOKEN_ANALYZE
                    , null
                    , AUTHORED_USER_DTO_PARAMETERIZED_TYPE_REFERENCE
                    , httpHeaders -> httpHeaders.set(DwHttpHeaderConstants.Header_User_Token, token)));
        } catch (BusinessException e) {
            //如果已经封装成立BusinessException   针对token失效的错误码进行统一封装上抛
            e.setBizErrorCode(e.getErrorCode());
            e.setBizErrorMsg(null == e.getErrorMessage() ? "" : e.getErrorMessage().toString());
            throw e;
        } catch (Throwable e) {
            //如果是未知异常进行封装
            throw BusinessException.buildErrorCode(P_IAM_500_5002);
        }
        if (null == result) {
            throw BusinessException.buildErrorCode(P_IAM_500_5002);
        }
        return result;
    }

    /**
     * 获取用户数据权限
     * @param moduleId
     * @param actionId
     * @param sysId
     * @param token
     * @return
     */
    public static UserDataPermissionResultDTO.RowPermission queryUserDataPermission(String moduleId, String actionId, String sysId, String token) {
        // 入参
        UserDataPermissionParamDTO paramDTO = new UserDataPermissionParamDTO();
        paramDTO.setModuleId(moduleId);
        paramDTO.setActionId(actionId);
        paramDTO.setSysId(sysId);
        // 请求
        Map resultMap = DwSpringHttpUtil.post(USER_DATA_PERMISSION
                , paramDTO
                , new ParameterizedTypeReference<Map>() {}
                , httpHeaders -> httpHeaders.set(DwHttpHeaderConstants.Header_User_Token, token));
        // 为了解决BUG#146680，清空数据权限后，rowPermission可能是空数组或者空对象的问题
        if (Objects.isNull(resultMap) || !resultMap.containsKey("rowPermission") || Objects.isNull(resultMap.containsKey("rowPermission"))
                || resultMap.get("rowPermission") instanceof Collection) {
            return null;
        }
        UserDataPermissionResultDTO resultDTO = JSONObject.parseObject(JSONObject.toJSONString(resultMap), UserDataPermissionResultDTO.class);
        // 判断是否有权限
        if (Objects.isNull(resultDTO) || Objects.isNull(resultDTO.getRowPermission())
                || CollectionUtils.isEmpty(resultDTO.getRowPermission().getFilterValue())) {
            return null;
        }
        return resultDTO.getRowPermission();
    }
}
