package com.digiwin.athena.translater.sdk.manager;

import com.digiwin.athena.common.sdk.manager.util.DwHttpResultParseUtil;
import com.digiwin.athena.common.sdk.manager.util.DwSpringHttpUtil;
import com.digiwin.athena.translater.sdk.meta.dto.TranslateResultDTO;
import com.digiwin.athena.translater.sdk.meta.dto.request.TranslateReqDTO;
import com.digiwin.athena.translater.sdk.meta.dto.response.TranslateDataDTO;
import com.digiwin.athena.translater.sdk.meta.dto.response.TranslateRespDTO;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import org.springframework.core.ParameterizedTypeReference;

import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * 繁简体翻译客户端 api
 *
 * @author yinhaiye
 * @since 2024/8/15 11:48
 */
public class TranslateManager {


    private static String getTranslateUri() {
        return JaProperty.get("translate.uri");
    }

    private static final ParameterizedTypeReference<TranslateResultDTO<TranslateRespDTO>> TRANSLATE_RESP_DTO_PARAMETERIZED_TYPE_REFERENCE = new ParameterizedTypeReference<TranslateResultDTO<TranslateRespDTO>>() {
    };

    /**
     * translate 根据内容翻译，传简体翻译繁体，传繁体翻译简体
     */
    private static final String TRANSLATE_API = getTranslateUri() + "/IDWTranslateService/translate";

    /**
     * translateCustom 简繁体都返回
     */
    private static final String TRANSLATE_CUSTOM_API = getTranslateUri() + "/IDWTranslateService/translateCustom";

    /**
     * 翻译简体
     */
    private static final String zh_CN = "zh2Hans";

    /**
     * 翻译繁体
     */
    private static final String zh_TW = "zh2Hant";

    /**
     * 根据字段名称以及内容列表繁简体翻译，简繁体都返回
     * @param reqDTOList  需要翻译的字段以及内容列表
     * @return TranslateDataDTO
     */
    public static TranslateDataDTO translateCustom(List<TranslateReqDTO> reqDTOList) {
        if (JaCollectionUtil.isEmpty(reqDTOList)) {
            JaLog.warn("translateCustom|翻译入参为空");
            return null;
        }
        String[] convertTypes = new String[]{zh_CN, zh_TW};
        // 请求
        TranslateRespDTO respDTO = DwHttpResultParseUtil.parse(DwSpringHttpUtil.post(TRANSLATE_CUSTOM_API, getRequestBody(reqDTOList, convertTypes), TRANSLATE_RESP_DTO_PARAMETERIZED_TYPE_REFERENCE));
        TranslateDataDTO translateDataDTO = respDTO.getData();
        translateDataDTO.getLang().values().forEach(item -> {
            item.setZh_US(item.getZh_CN());
        });
        translateDataDTO.getLanguage().values().forEach(item -> {
            item.setZh_US(item.getZh_CN());
        });
        return translateDataDTO;
    }

    /**
     * 根据字段名称以及内容列表繁简体翻译，传繁体翻译简体
     * @param reqDTOList  需要翻译的字段以及内容列表
     * @return Map<String, String>
     */
    public static Map<String, String> translateCN(List<TranslateReqDTO> reqDTOList) {
        if (JaCollectionUtil.isEmpty(reqDTOList)) {
            JaLog.warn("translateCN|翻译入参为空");
            return null;
        }
        String[] convertTypes = new String[]{zh_CN};
        // 请求
        TranslateRespDTO respDTO = DwHttpResultParseUtil.parse(DwSpringHttpUtil.post(TRANSLATE_API, getRequestBody(reqDTOList, convertTypes), TRANSLATE_RESP_DTO_PARAMETERIZED_TYPE_REFERENCE));
        return respDTO.getData().getZh2Hans();
    }

    /**
     * 根据字段名称以及内容列表繁简体翻译，传简体翻译繁体
     * @param reqDTOList  需要翻译的字段以及内容列表
     * @return Map<String, String>
     */
    public static Map<String, String> translateTW(List<TranslateReqDTO> reqDTOList) {
        if (JaCollectionUtil.isEmpty(reqDTOList)) {
            JaLog.warn("translateTW|翻译入参为空");
            return null;
        }
        String[] convertTypes = new String[]{zh_TW};
        // 请求
        TranslateRespDTO respDTO = DwHttpResultParseUtil.parse(DwSpringHttpUtil.post(TRANSLATE_API, getRequestBody(reqDTOList, convertTypes), TRANSLATE_RESP_DTO_PARAMETERIZED_TYPE_REFERENCE));
        return respDTO.getData().getZh2Hant();
    }

    /**
     * 构造翻译接口请求入参
     * @param reqDTOList    需要翻译的字段以及内容列表
     * @param convertTypes  需要翻译的语言类型
     * @return Map<String, Object>
     */
    private static Map<String, Object> getRequestBody(List<TranslateReqDTO> reqDTOList, String[] convertTypes) {
        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        // 构造请求内容
        Map<String, String> contentMap = new HashMap<>(reqDTOList.size());
        for (TranslateReqDTO translateReqDTO : reqDTOList) {
            contentMap.put(translateReqDTO.getFieldName(), translateReqDTO.getFieldValue());
        }
        requestMap.put("content", contentMap);
        requestMap.put("convertTypes",convertTypes);
        return requestMap;
    }
}
