package com.digiwin.athena.translater.sdk.manager;

import com.digiwin.athena.common.sdk.manager.util.DwHttpResultParseUtil;
import com.digiwin.athena.common.sdk.manager.util.DwSpringHttpUtil;
import com.digiwin.athena.translater.sdk.meta.dto.TranslateResultDTO;
import com.digiwin.athena.translater.sdk.meta.dto.request.TranslateReqDTO;
import com.digiwin.athena.translater.sdk.meta.dto.response.TranslateDataDTO;
import com.digiwin.athena.translater.sdk.meta.dto.response.TranslateRespDTO;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import org.springframework.core.ParameterizedTypeReference;

import java.util.*;
import java.util.regex.Pattern;


/**
 * 繁简体翻译客户端 api
 *
 * @author yinhaiye
 * @since 2024/8/15 11:48
 */
public class TranslateManager {


    private static String getTranslateUri() {
        return JaProperty.get("translate.uri");
    }

    private static final ParameterizedTypeReference<TranslateResultDTO<TranslateRespDTO>> TRANSLATE_RESP_DTO_PARAMETERIZED_TYPE_REFERENCE = new ParameterizedTypeReference<TranslateResultDTO<TranslateRespDTO>>() {
    };

    /**
     * translate 根据内容翻译，传简体翻译繁体，传繁体翻译简体
     */
    private static final String TRANSLATE_API = getTranslateUri() + "/IDWTranslateService/translate";

    /**
     * translateCustom 简繁体都返回
     */
    private static final String TRANSLATE_CUSTOM_API = getTranslateUri() + "/IDWTranslateService/translateCustom";

    /**
     * 翻译简体
     */
    private static final String zh_CN = "zh2Hans";

    /**
     * 翻译繁体
     */
    private static final String zh_TW = "zh2Hant";

    // 正则表达式匹配中文字符
    private static final Pattern CHINESE_PATTERN = Pattern.compile("[\\u4e00-\\u9fa5]");

    /**
     * 根据字段名称以及内容列表繁简体翻译，简繁体都返回
     * @param reqDTOList  需要翻译的字段以及内容列表
     * @return TranslateDataDTO
     */
    public static TranslateDataDTO translateCustom(List<TranslateReqDTO> reqDTOList) {
        if (JaCollectionUtil.isEmpty(reqDTOList)) {
            JaLog.warn("translateCustom|翻译入参为空");
            return null;
        }
        String[] convertTypes = new String[]{zh_CN, zh_TW};
        // 请求
        TranslateRespDTO respDTO = DwHttpResultParseUtil.parse(DwSpringHttpUtil.post(TRANSLATE_CUSTOM_API, getRequestBody(reqDTOList, convertTypes), TRANSLATE_RESP_DTO_PARAMETERIZED_TYPE_REFERENCE));
        TranslateDataDTO translateDataDTO = respDTO.getData();
        if (JaCollectionUtil.isNotEmpty(translateDataDTO.getLang())) {
            translateDataDTO.getLang().values().forEach(item -> {
                // us默认使用简体
                item.setZh_US(item.getZh_CN());
            });
        } else {
            // 结果为空则返回用户输入
            translateDataDTO.setLang(getTranslateLangDataByParam(reqDTOList));
        }
        if (JaCollectionUtil.isNotEmpty(translateDataDTO.getLanguage())) {
            translateDataDTO.getLanguage().values().forEach(item -> {
                // us默认使用简体
                item.setZh_US(item.getZh_CN());
            });
        } else {
            // 结果为空则返回用户输入
            translateDataDTO.setLanguage(getTranslateLangDataByParam(reqDTOList));
        }
        // 过滤出不包含中文的集合
        List<TranslateReqDTO> noChineseList = filterNonChineseList(reqDTOList);
        if (JaCollectionUtil.isNotEmpty(noChineseList)) {
            // 无法翻译的不含汉字的原路返回
            translateDataDTO.getLang().putAll(getTranslateLangDataByParam(noChineseList));
            translateDataDTO.getLanguage().putAll(getTranslateLangDataByParam(noChineseList));
        }
        return translateDataDTO;
    }

    /**
     * 根据字段名称以及内容列表繁简体翻译，传繁体翻译简体
     * @param reqDTOList  需要翻译的字段以及内容列表
     * @return Map<String, String>
     */
    public static Map<String, String> translateCN(List<TranslateReqDTO> reqDTOList) {
        if (JaCollectionUtil.isEmpty(reqDTOList)) {
            JaLog.warn("translateCN|翻译入参为空");
            return null;
        }
        String[] convertTypes = new String[]{zh_CN};
        // 请求
        TranslateRespDTO respDTO = DwHttpResultParseUtil.parse(DwSpringHttpUtil.post(TRANSLATE_API, getRequestBody(reqDTOList, convertTypes), TRANSLATE_RESP_DTO_PARAMETERIZED_TYPE_REFERENCE));
        return respDTO.getData().getZh2Hans();
    }

    /**
     * 根据字段名称以及内容列表繁简体翻译，传简体翻译繁体
     * @param reqDTOList  需要翻译的字段以及内容列表
     * @return Map<String, String>
     */
    public static Map<String, String> translateTW(List<TranslateReqDTO> reqDTOList) {
        if (JaCollectionUtil.isEmpty(reqDTOList)) {
            JaLog.warn("translateTW|翻译入参为空");
            return null;
        }
        String[] convertTypes = new String[]{zh_TW};
        // 请求
        TranslateRespDTO respDTO = DwHttpResultParseUtil.parse(DwSpringHttpUtil.post(TRANSLATE_API, getRequestBody(reqDTOList, convertTypes), TRANSLATE_RESP_DTO_PARAMETERIZED_TYPE_REFERENCE));
        return respDTO.getData().getZh2Hant();
    }

    /**
     * 构造翻译接口请求入参
     * @param reqDTOList    需要翻译的字段以及内容列表
     * @param convertTypes  需要翻译的语言类型
     * @return Map<String, Object>
     */
    private static Map<String, Object> getRequestBody(List<TranslateReqDTO> reqDTOList, String[] convertTypes) {
        // 请求体
        Map<String, Object> requestMap = new HashMap<>();
        // 构造请求内容
        Map<String, String> contentMap = new HashMap<>(reqDTOList.size());
        for (TranslateReqDTO translateReqDTO : reqDTOList) {
            contentMap.put(translateReqDTO.getFieldName(), translateReqDTO.getFieldValue());
        }
        requestMap.put("content", contentMap);
        requestMap.put("convertTypes",convertTypes);
        return requestMap;
    }

    /**
     * 过滤出不包含中文的集合
     * @param reqDTOList 原始集合
     * @return 不包含中文的子集合
     */
    private static List<TranslateReqDTO> filterNonChineseList(List<TranslateReqDTO> reqDTOList) {
        if (JaCollectionUtil.isEmpty(reqDTOList)) {
            return Collections.emptyList();
        }
        List<TranslateReqDTO> result = new ArrayList<>();
        for (TranslateReqDTO reqDTO : reqDTOList) {
            if (!CHINESE_PATTERN.matcher(reqDTO.getFieldValue()).find()) {
                result.add(reqDTO);
            }
        }
        return result;
    }

    /**
     * 将翻译入参转换为多语言
     * @param reqDTOList 翻译入参 不可能为空
     * @return Map
     */
    private static Map<String, TranslateDataDTO.TranslateLangData> getTranslateLangDataByParam(List<TranslateReqDTO> reqDTOList) {
        if (JaCollectionUtil.isEmpty(reqDTOList)) {
            return Collections.emptyMap();
        }
        Map<String, TranslateDataDTO.TranslateLangData> result = new HashMap<>(reqDTOList.size());
        for (TranslateReqDTO translateReqDTO : reqDTOList) {
            TranslateDataDTO.TranslateLangData value = new TranslateDataDTO.TranslateLangData();
            value.setZh_CN(translateReqDTO.getFieldValue());
            value.setZh_US(translateReqDTO.getFieldValue());
            value.setZh_TW(translateReqDTO.getFieldValue());
            result.put(translateReqDTO.getFieldName(), value);
        }
        return result;
    }
}
