package com.digiwin.athena.abt.application.configuration;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.LocalDateTime;
import java.time.ZoneOffset;

/**
 * 配置信息
 *
 * @author chenxsa
 */
@Component
@Data
@Slf4j
public class RemoteProperties {
    @Value("${themeMap.domainUri:}")
    private String webThemeMapDomainUri;

    @Value("${rabbit.mquri:}")
    private String rabbitMqUri;

    @Value("${esp.uri:}")
    private String espUri;

    @Value("${mdc.uri:}")
    private String mdcUri;

    @Value("${eoc.uri:}")
    private String eocUri;

    /**
     * dap中间件-日志中心
     */
    @Value("${dap.lmcUri:}")
    private String dapLmcUri;
    /**
     * 当前进程运行的主机名
     */
    private String host;
    /**
     * 当前进程所在的IP地址
     */
    private String ipAddress;
    /**
     * 空闲内存
     */
    private long freeMemory;
    /**
     * 内存总量
     */
    private long totalMemory;
    /**
     * java虚拟机允许开启的最大的内存
     */
    private long maxMemory;
    /**
     * 操作系统名称
     */
    private String osName;
    /**
     * 进程号
     */
    private long pid;
    /**
     * 程序启动时间
     */
    private LocalDateTime startTime;
    /**
     * 程序运行时间，单位毫秒
     */
    private long runtime;
    /**
     * 线程总量
     */
    private int threadCount;

    public void refresh() {
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        Runtime runtime = Runtime.getRuntime();
        //空闲内存
        long freeMemory = runtime.freeMemory();
        this.setFreeMemory(byteToM(freeMemory));
        //内存总量
        long totalMemory = runtime.totalMemory();
        this.setTotalMemory(byteToM(totalMemory));
        //最大允许使用的内存
        long maxMemory = runtime.maxMemory();
        this.setMaxMemory(byteToM(maxMemory));
        //操作系统
        this.setOsName(System.getProperty("os.name"));
        InetAddress localHost;
        try {
            localHost = InetAddress.getLocalHost();
            String hostName = localHost.getHostName();
            this.setHost(hostName);
            if (ipAddress == null) {
                ipAddress = localHost.getHostAddress();
            }
        } catch (UnknownHostException e) {
            log.error("UnknownHostException:{}",e.getMessage());
            this.setHost("未知");
        }
        //ip
        this.setIpAddress(ipAddress);

        //程序启动时间
        long startTime = runtimeMXBean.getStartTime();

        LocalDateTime startDate = LocalDateTime.ofEpochSecond(startTime / 1000, 0, ZoneOffset.ofHours(0));
        this.setStartTime(startDate);
        //类所在路径
        this.setRuntime(runtimeMXBean.getUptime());
        //线程总数
        this.setThreadCount(ManagementFactory.getThreadMXBean().getThreadCount());
        this.setPid(tryGetPid());
    }

    /**
     * 把byte转换成M
     *
     * @param bytes
     * @return
     */
    static long byteToM(long bytes) {
        long kb = (bytes / 1024 / 1024);
        return kb;
    }

    /**
     * 获取进程号，适用于windows与linux
     *
     * @return
     */
    static long tryGetPid() {
        try {
            String name = ManagementFactory.getRuntimeMXBean().getName();
            String pid = name.split("@")[0];
            return Long.parseLong(pid);
        } catch (NumberFormatException e) {
            return 0;
        }
    }

}
