package com.digiwin.athena.abt.application.dto.migration.atmc.mongodb;

import com.digiwin.athena.abt.application.dto.migration.atmc.backlog.BkItemDTO;
import com.digiwin.athena.abt.core.meta.constants.BkConstant;
import com.digiwin.athena.abt.core.meta.constants.RedisKeyPrefixConstant;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.mongodb.client.model.IndexModel;
import com.mongodb.client.model.IndexOptions;
import lombok.extern.slf4j.Slf4j;
import org.apache.tomcat.util.buf.StringUtils;
import org.bson.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.index.IndexInfo;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;

/**
 * @author：SYQ
 * @date：2021/12/28
 */
@Slf4j
@Service
public class MongodbService {
    @Autowired
    @Qualifier("atmcBkDatabaseMongoTemplate")
    private MongoTemplate atmcBkDatabaseMongoTemplate;

    @Autowired
    private RedisTemplate redisTemplate;

    private static final String LOCK_NAME_SUFFIX = ":createMongoCollection:lock";

    private List<String> existCollectionList = new ArrayList<>();

    /**
     * 创建集合（加锁）
     *
     * @Author：SYQ
     * @Date：2021/12/28 14:52
     */
    public void createCollectionLock(String collectionName, Map<String, Object> indexMap) {
        //判断内存中集合是否存在，存在不需要创建
        if (existCollectionList.contains(collectionName)) {
            return;
        }
        //判断mongoDB中结合是否存在，存在不需要创建
        if (atmcBkDatabaseMongoTemplate.collectionExists(collectionName)) {
            existCollectionList.add(collectionName);
            return;
        }

        String tmpLockName = RedisKeyPrefixConstant.PREFIX + collectionName + LOCK_NAME_SUFFIX;
        DistributeLocker.tryLock30s(redisTemplate, tmpLockName, () -> createCollectionAndIndex(collectionName, indexMap));
    }

    /**
     * 创建集合（加锁）
     *
     * @Author：SYQ
     * @Date：2021/12/28 14:52
     */
    public void createCollectionLock(String collectionName, List<IndexModel> indexList) {
        //判断内存中集合是否存在，存在不需要创建
        if (existCollectionList.contains(collectionName)) {
            return;
        }
        //判断mongoDB中结合是否存在，存在不需要创建
        if (atmcBkDatabaseMongoTemplate.collectionExists(collectionName)) {
            existCollectionList.add(collectionName);
            return;
        }

        String tmpLockName = RedisKeyPrefixConstant.PREFIX + collectionName + LOCK_NAME_SUFFIX;
        DistributeLocker.tryLock30s(redisTemplate, tmpLockName, () -> createCollectionAndIndexMulti(collectionName, indexList));
    }


    /**
     * 创建集合和索引
     *
     * @Author：SYQ
     * @Date：2021/12/28 15:00
     */
    private void createCollectionAndIndex(String collectionName, Map<String, Object> indexMap) {
        //创建集合
        atmcBkDatabaseMongoTemplate.createCollection(collectionName);
        existCollectionList.add(collectionName);

        //创建索引
        if (indexMap != null && indexMap.size() > 0) {
            //索引名称
            String indexName = StringUtils.join(indexMap.keySet(), '_');
            // 该参数为索引的属性配置
            IndexOptions indexOptions = new IndexOptions();
            indexOptions.background(true);
            indexOptions.name(indexName);
            //创建索引
            atmcBkDatabaseMongoTemplate.getCollection(collectionName).createIndex(new Document(indexMap), indexOptions);
        }
    }

    /**
     * 创建单键索引
     *
     * @param collectionName 集合名称
     * @param indexList      单键索引信息
     */
    private void createCollectionAndIndexMulti(String collectionName, List<IndexModel> indexList) {
        //创建集合
        atmcBkDatabaseMongoTemplate.createCollection(collectionName);
        existCollectionList.add(collectionName);

        for (IndexModel indexModel : indexList) {
            atmcBkDatabaseMongoTemplate.getCollection(collectionName).createIndex(indexModel.getKeys(), indexModel.getOptions());

        }
    }


    /**
     * 插入数据
     *
     * @Author：SYQ
     * @Date：2021/12/28 15:48
     */
    public void insert(List<Map> dataList, String collectionName) {
        atmcBkDatabaseMongoTemplate.insert(dataList, collectionName);
    }

    public <T> void insertBatch(Collection<? extends T> batchToSave, String collectionName) {
        atmcBkDatabaseMongoTemplate.insert(batchToSave, collectionName);
    }

    /**
     * 更新
     *
     * @Author：SYQ
     * @Date：2021/12/28 16:26
     */
    public void updateFirst(String collectionName, Query query, Update update) {
        atmcBkDatabaseMongoTemplate.updateFirst(query, update, collectionName);
    }

    /**
     * 执行查询
     *
     * @param collectionName
     * @param query
     * @return
     */
    public List<Map> query(String collectionName, Query query) {
        return atmcBkDatabaseMongoTemplate.find(query, Map.class, collectionName);
    }

    public <T> List<T> find(Query query, Class<T> entityClass, String collectionName) {
        return atmcBkDatabaseMongoTemplate.find(query, entityClass, collectionName);
    }

    /**
     * 执行查询命令
     *
     * @param jsonCommand
     * @return 查询的结果
     */
    public List<Map> executeQueryCommand(String jsonCommand) {
        Map document = atmcBkDatabaseMongoTemplate.executeCommand(jsonCommand);
        Map cursor = (Map) document.get("cursor");
        List<Map> mapList = (List<Map>) cursor.get("firstBatch");
        return mapList;
    }

    /**
     * 执行查询命令
     *
     * @param jsonCommand
     * @return 查询的结果
     */
    public List<Map> executeQueryCommand(Map jsonCommand) {
        return executeQueryCommand(JsonUtils.objectToString(jsonCommand));
    }

    /**
     * 删除索引
     *
     * @Author：SYQ
     * @Date：2022/1/26 13:46
     */
    public void deleteIdexFromCollections(String deleteIndex) {
        Set<String> collectionNameSet = atmcBkDatabaseMongoTemplate.getCollectionNames();
        if (CollectionUtils.isEmpty(collectionNameSet)) {
            return;
        }
        for (String collectionName : collectionNameSet) {
            List<IndexInfo> indexList = atmcBkDatabaseMongoTemplate.indexOps(collectionName).getIndexInfo();
            List<String> indexNameList = new ArrayList<>();
            if (!CollectionUtils.isEmpty(indexList)) {
                for (IndexInfo indexInfo : indexList) {
                    indexNameList.add(indexInfo.getName());
                }
            }
            if (indexNameList.contains(deleteIndex)) {
                atmcBkDatabaseMongoTemplate.indexOps(collectionName).dropIndex(deleteIndex);
            }
        }
    }

    /**
     * 添加索引
     *
     * @Author：SYQ
     * @Date：2022/1/26 14:07
     */
    public void insertIndex(Map<String, Object> indexMap) {
        Set<String> collectionNameSet = atmcBkDatabaseMongoTemplate.getCollectionNames();
        if (CollectionUtils.isEmpty(collectionNameSet)) {
            return;
        }
        for (String collectionName : collectionNameSet) {
            //创建索引
            if (indexMap != null && indexMap.size() > 0) {
                //索引名称
                String indexName = StringUtils.join(indexMap.keySet(), '_');
                // 该参数为索引的属性配置
                IndexOptions indexOptions = new IndexOptions();
                indexOptions.background(true);
                indexOptions.name(indexName);
                //已存在的索引
                List<IndexInfo> indexList = atmcBkDatabaseMongoTemplate.indexOps(collectionName).getIndexInfo();
                List<String> indexNameList = new ArrayList<>();
                if (!CollectionUtils.isEmpty(indexList)) {
                    for (IndexInfo indexInfo : indexList) {
                        indexNameList.add(indexInfo.getName());
                    }
                }
                if (CollectionUtils.isEmpty(indexNameList) || !indexNameList.contains(indexName)) {
                    //创建索引
                    atmcBkDatabaseMongoTemplate.getCollection(collectionName).createIndex(new Document(indexMap), indexOptions);
                }
            }
        }
    }


    /**
     * 更新多条数据
     *
     * @Author：SYQ
     * @Date：2021/12/28 16:26
     */
    public void updateMulti(String collectionName, Query query, Update update) {
        atmcBkDatabaseMongoTemplate.updateMulti(query, update, collectionName);
    }

    /**
     * 删除数据
     *
     * @Author：xusijia
     * @Date：2023/05/22 16:26
     */
    public void deleteData(BkItemDTO bkItemDTO) {
        Query query = Query.query(Criteria.where("cardId").is(bkItemDTO.getCardId()).and("tenantId").is(bkItemDTO.getTenantId()).and("bkContent").is(bkItemDTO.getBkContent()).and("type").is(BkConstant.WORKITEM));
        atmcBkDatabaseMongoTemplate.remove(query, bkItemDTO.getTenantId());
    }

    public void delete(Query query, String collectionName){
        atmcBkDatabaseMongoTemplate.remove(query, collectionName);
    }

}
