package com.digiwin.athena.abt.application.dto.migration.atmc.share.dao;


import com.digiwin.athena.abt.application.dto.migration.atmc.share.entity.ShareCodeMongoEntity;
import com.digiwin.athena.abt.application.utils.RandomUtil;
import com.digiwin.athena.abt.core.meta.constants.SharingTaskConstant;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

@Slf4j
@Service
public class ShareCodeDaoImpl implements ShareCodeDao {

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    @Autowired
    @Qualifier("atmcTempDataMongo")
    private MongoTemplate atmcTempDataMongo;


    /**
     * 分享码集合
     */
    public static final String COLLECTIONNAME_SHARECODE = "atmc_share_code";

    @Override
    public String addShareCodeCache(String shareContent, long sharingCodeTtl, TimeUnit sharingCodeTtlTimeUnit) {
        // 生成分享码
        ShareCodeMongoEntity newShare = null;
        //入库如果shareCode重复，则重试两次
        int retryTimes = 3;
        boolean needRetry = true;
        while (needRetry && retryTimes-- > 0) {
            try {
                newShare = createShareCodeCache(shareContent, sharingCodeTtl, sharingCodeTtlTimeUnit);
                needRetry = false;
            } catch (DuplicateKeyException e2) {
                //主键冲突异常，重试
                log.warn("创建分享码主键冲突，重试[{}]", retryTimes);
            }
        }
        return Objects.isNull(newShare) ? "" : newShare.getShareCode();
    }

    /**
     * 新建分享码
     */
    private ShareCodeMongoEntity createShareCodeCache(String shareContent, long sharingCodeTtl, TimeUnit sharingCodeTtlTimeUnit) {
        String sharingCode = RandomUtil.genRandomStr(SharingTaskConstant.SHARING_CODE_LENGTH);
        return insertShareCodeCache(sharingCode, shareContent, sharingCodeTtl, sharingCodeTtlTimeUnit);
    }

    /**
     * 插入分享码
     *
     * @param sharingCode
     * @param shareContent
     * @param sharingCodeTtl
     * @param sharingCodeTtlTimeUnit
     * @return
     */
    private ShareCodeMongoEntity insertShareCodeCache(String sharingCode, String shareContent, long sharingCodeTtl, TimeUnit sharingCodeTtlTimeUnit) {
        // 生成分享码
        Date createTime = new Date();
        ShareCodeMongoEntity shareCodeMongoEntity = ShareCodeMongoEntity.builder()
                .shareCode(sharingCode)
                .shareContent(shareContent)
                .createTime(createTime)
                .expireTime(DateUtils.addSeconds(createTime, (int) sharingCodeTtlTimeUnit.toSeconds(sharingCodeTtl)))
                .build();
        return atmcTempDataMongo.insert(shareCodeMongoEntity, COLLECTIONNAME_SHARECODE);
    }

    @Override
    public String getShareCodeCache(String sharingCode) {
        String cachedStr = "";
        //mongodb查询分享码
        Criteria cater = new Criteria();
        Query query = Query.query(cater);
        cater.and("shareCode").is(sharingCode);
        ShareCodeMongoEntity cachedShareCodeMongoObj = atmcTempDataMongo.findOne(query, ShareCodeMongoEntity.class, COLLECTIONNAME_SHARECODE);
        if (Objects.nonNull(cachedShareCodeMongoObj)) {
            cachedStr = cachedShareCodeMongoObj.getShareContent();
        }
        return cachedStr;
    }

}
