package com.digiwin.athena.abt.application.service.abt.migration.bk;

import com.digiwin.athena.abt.application.dto.migration.abt.bk.ActivityBusinessKeyDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.excel.CsvFile;
import com.digiwin.athena.abt.application.dto.migration.abt.mq.PtmDataConsistencyMessageDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.mongodb.MongodbService;
import com.digiwin.athena.abt.application.service.abt.migration.event.handler.ActivityBusinessKeyHandler;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.ptm.PtmService;
import com.digiwin.athena.abt.application.utils.CsvUtils;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import com.digiwin.athena.abt.infrastructure.pojo.bo.migration.atmc.PtmActivityRecordDTO;
import com.digiwin.athena.abt.infrastructure.pojo.bo.migration.atmc.PtmProjectRecordBO;
import com.digiwin.athena.abt.infrastructure.pojo.bo.migration.atmc.PtmTaskRecordBO;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 刷新bk
 */
@Service
@Slf4j
public class RefreshBkService {

    @Autowired
    private MongodbService mongodbService;

    @Autowired
    private BusinessKeyMqService businessKeyMqService;

    @Autowired
    private PtmService ptmService;

    @Autowired
    private ActivityBusinessKeyHandler activityBusinessKeyHandler;

    /**
     * 刷新项目的bk
     */
    public List<Long> refreshProjectBk(List<Long> projectIdList, String tenantId, AuthoredUser user){
        if (CollectionUtils.isEmpty(projectIdList) || StringUtils.isEmpty(tenantId)) {
            throw BusinessException.create("projectIdList或tenantId为空！");
        }
        // 查询流程的已经入库的bk
        Query query = Query.query(Criteria.where("type").is("task").and("cardId").in(projectIdList));
        List<Map> existBkList = this.mongodbService.query(tenantId, query);
        // 过滤出缺失bk的projectId
        List<Long> lackProjectIds = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(existBkList)) {
            List<Long> existProjectIdList = existBkList.stream().map(x -> Long.valueOf(String.valueOf(x.get("cardId")))).collect(Collectors.toList());
            lackProjectIds = projectIdList.stream().filter(x -> !existProjectIdList.contains(x)).collect(Collectors.toList());
        } else {
            lackProjectIds = projectIdList;
        }
        log.info("lackProjectIds:{}", JsonUtils.objectToString(lackProjectIds));
        // 修复bk
        if (CollectionUtils.isEmpty(lackProjectIds)) {
            return lackProjectIds;
        }
        for (Long projectId : lackProjectIds) {
            PtmDataConsistencyMessageDTO.Project project = PtmDataConsistencyMessageDTO.Project.builder()
                    .id(projectId)
                    .build();
            businessKeyMqService.processProject(project, user);
        }
        return lackProjectIds;
    }

    /**
     * 刷新项目的bk
     */
    public List<Long> refreshTaskBk(List<PtmDataConsistencyMessageDTO.WorkItem> workItemList, String tenantId, AuthoredUser user){
        if (CollectionUtils.isEmpty(workItemList) || StringUtils.isEmpty(tenantId)) {
            throw BusinessException.create("workItemList或tenantId为空！");
        }
        List<Long> workitemIdList = workItemList.stream().map(PtmDataConsistencyMessageDTO.WorkItem::getId).distinct().collect(Collectors.toList());
        Map<Long, PtmDataConsistencyMessageDTO.WorkItem> workItemRel = workItemList.stream().collect(Collectors.toMap(PtmDataConsistencyMessageDTO.WorkItem::getId, Function.identity()));
        // 查询流程的已经入库的bk
        Query query = Query.query(Criteria.where("type").is("workitem").and("cardId").in(workitemIdList));
        List<Map> existBkList = this.mongodbService.query(tenantId, query);
        // 过滤出缺失bk的projectId
        List<Long> lackworkItemIds = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(existBkList)) {
            List<Long> existWorkItemList =
                    existBkList.stream().map(x -> Long.valueOf(String.valueOf(x.get("cardId")))).collect(Collectors.toList());
            lackworkItemIds = workitemIdList.stream().filter(x -> !existWorkItemList.contains(x)).collect(Collectors.toList());
        } else {
            lackworkItemIds = workitemIdList;
        }
        log.info("lackworkItemIds:{}", JsonUtils.objectToString(lackworkItemIds));
        // 修复bk
        if (CollectionUtils.isEmpty(lackworkItemIds)) {
            return lackworkItemIds;
        }

        log.info("[repairTaskBk start  tenant_id = {}, lackworkItemIds = {}] ", tenantId, lackworkItemIds);
        for (Long workitemId : lackworkItemIds) {
            processWorkItem(workItemRel.get(workitemId), user);
        }
        return workitemIdList;
    }

    /**
     * PTM任务bk
     * @param workItem
     */
    public void processWorkItem(PtmDataConsistencyMessageDTO.WorkItem workItem, AuthoredUser user){
        PtmTaskRecordBO ptmTaskRecordBO = ptmService.getTaskRecord(workItem.getTaskId(),workItem.getId());
        PtmActivityRecordDTO ptmActivityRecordDTO = ptmService.getActivityRecord(workItem.getActivityId());
        PtmProjectRecordBO ptmProjectRecordBO = ptmService.getProjectRecord(workItem.getProjectId());

        ActivityBusinessKeyDTO activityBusinessKeyDTO = new ActivityBusinessKeyDTO();
        activityBusinessKeyDTO.setSubTaskId(ptmProjectRecordBO.getId());
        activityBusinessKeyDTO.setTenantId(ptmProjectRecordBO.getTenantId());
        activityBusinessKeyDTO.setTmTaskId(ptmProjectRecordBO.getProjectDefCode());
        activityBusinessKeyDTO.setState(ptmActivityRecordDTO.getState());
        activityBusinessKeyDTO.setActivityId(workItem.getActivityId());
        activityBusinessKeyDTO.setBusinessUnit(JSONObject.fromObject(ptmTaskRecordBO.getBusinessUnit()));
        activityBusinessKeyDTO.setData(JSONObject.fromObject(ptmTaskRecordBO.getData()));
        activityBusinessKeyDTO.setTmActivityId(ptmTaskRecordBO.getTaskDefCode());
        activityBusinessKeyDTO.setBpmData(JSONObject.fromObject(ptmTaskRecordBO.getData()));
        activityBusinessKeyDTO.setWorkitemId(workItem.getId());
        activityBusinessKeyDTO.setClosed(false);
        activityBusinessKeyDTO.setAuthoredUser(user);
        activityBusinessKeyHandler.processEventForRepair(activityBusinessKeyDTO);
    }


    public void upload(MultipartFile file, AuthoredUser user) {

        if (null == file || file.isEmpty() || org.apache.commons.lang.StringUtils.isEmpty(file.getOriginalFilename())) {
            throw BusinessException.create("500", MessageUtil.getMessage("delivery.fileEmpty"));
        }

        List<CsvFile> csvData = CsvUtils.getCsvDataMethod1(file);

        if (CollectionUtils.isEmpty(csvData)){
            return ;
        }

        // 根据租户id 分批处理
        Map<String, List<CsvFile>> csvDataRel = csvData.stream().collect(Collectors.groupingBy(CsvFile::getTenantId));
        csvDataRel.remove("tenant_id");
        for (Map.Entry<String, List<CsvFile>> entry : csvDataRel.entrySet()) {
            processMap(entry.getKey(), entry.getValue(), user);
        }
    }



    public void processMap(String tenantId, List<CsvFile> csvFiles, AuthoredUser user) {

        List<PtmDataConsistencyMessageDTO.WorkItem> workItemList = new ArrayList<>();
        for (CsvFile csvFile : csvFiles) {
            PtmDataConsistencyMessageDTO.WorkItem workItem = PtmDataConsistencyMessageDTO.WorkItem.builder()
                    .id(Long.valueOf(csvFile.getWorkItemId()))
                    .taskId(Long.valueOf(csvFile.getTaskId()))
                    .activityId(Long.valueOf(csvFile.getActivityId()))
                    .projectId(Long.valueOf(csvFile.getProjectId())).build();
            workItemList.add(workItem);
        }
        refreshTaskBk(workItemList,tenantId,user);
    }


    /**
     * 校验入参projectIds对应的BK是否存在，并返回不存在对应BK的projectId列表
     *
     * @param projectIds
     * @param tenantId
     * @param user
     * @return
     */
    public List<Long> checkProjectBKExist(List<Long> projectIds, String tenantId, AuthoredUser user) {
        if (CollectionUtils.isEmpty(projectIds) || StringUtils.isEmpty(tenantId)) {
            throw BusinessException.create("projectIds或tenantId为空！");
        }
        // 查询流程的已经入库的bk
        Query query = Query.query(Criteria.where("type").is("task")
                .and("cardId").in(projectIds));
        List<Map> existBkList = this.mongodbService.query(tenantId, query);
        // 过滤出缺失bk的projectId
        List<Long> lackProjectIds = null;
        if (CollectionUtils.isNotEmpty(existBkList)) {
            List<Long> existProjectIdList = existBkList.stream()
                    .map(x -> Long.valueOf(String.valueOf(x.get("cardId"))))
                    .collect(Collectors.toList());
            lackProjectIds = projectIds.stream()
                    .filter(x -> !existProjectIdList.contains(x))
                    .collect(Collectors.toList());
        } else {
            lackProjectIds = projectIds;
        }

        return lackProjectIds;
    }

    public List<Map> queryBkByWorkItemId(String tenantId, List<Long> workItemIdList, AuthoredUser user) {
        Query query = Query.query(Criteria.where("type").is("workitem").and("cardId").in(workItemIdList));
        List<Map> existBkList = this.mongodbService.query(tenantId, query);
        return existBkList;
    }

    public List<Long> checkNeedRepair(String tenantId, List<Long> workItemIdList, AuthoredUser user) {

        List<Long> workItemIds = new ArrayList<>();
        if (StringUtils.isBlank(tenantId) || CollectionUtils.isEmpty(workItemIdList)){
            return workItemIds;
        }

        for (Long workItemId : workItemIdList) {
            Query query = Query.query(Criteria.where("type").is("workitem").and("cardId").is(workItemId));
            List<Map> bkMap = mongodbService.query(tenantId, query);
            if (CollectionUtils.isEmpty(bkMap)){
                workItemIds.add(workItemId);
            }
        }
        return workItemIds;
    }
}

