package com.digiwin.athena.abt.application.service.abt.migration.esp.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.digiwin.athena.abt.application.dto.migration.abt.api.DownloadBaseDataParamDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.esp.EspBody;
import com.digiwin.athena.abt.application.dto.migration.abt.esp.EspResponse;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.DataEntryTask;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.ExportFileMsg;
import com.digiwin.athena.abt.application.service.abt.migration.esp.EspService;
import com.digiwin.athena.abt.application.service.abt.migration.inout.MetaDataService;
import com.digiwin.athena.abt.core.meta.constants.ImportAndExportStatisticsConstants;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.ChainInfoUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.esp.sdk.Invoker;
import com.digiwin.athena.esp.sdk.exception.*;
import com.digiwin.athena.esp.sdk.model.RequestModel;
import com.digiwin.athena.esp.sdk.model.ResponseModel;
import com.digiwin.service.permission.DWSecurityTokenGenerator;
import com.digiwin.service.permission.pojo.DWSecurityToken;
import com.digiwin.athena.abt.application.dto.migration.abt.api.ActionInfoDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @ClassName EspServiceImpl
 * @Author zhuangli
 * @Date 2021/12/1 14:23
 * @Version 1.0
 **/
@Slf4j
@Service
public class EspServiceImpl implements EspService {
    @Autowired
    private MetaDataService metaDataService;

    @Override
    public EspResponse sendByBatch(DataEntryTask dataEntryTask, GetActionLocaleResponseDTO metaData,
                                   List<Map> convertedTable, String productName) {
        String targetFieldName = Optional.ofNullable(dataEntryTask.getActionInfo()).map(ActionInfoDTO::getTargetField).orElse(StringUtils.EMPTY);
        final String tableKey = metaDataService.getMainRequestMetadataField(metaData.getRequest(), targetFieldName).getData_name();
        log.info("produceName: {}, tableKey: {}", productName, tableKey);
        Map<String, Object> requestMap = new HashMap<>(1);
        requestMap.put(tableKey, convertedTable);
        if (null != dataEntryTask.getActionInfo() && !CollectionUtils.isEmpty(dataEntryTask.getActionInfo().getActionParas())) {
            requestMap.putAll(dataEntryTask.getActionInfo().getActionParas());
        }
        String dataJson = JSON.toJSONString(requestMap, SerializerFeature.WriteMapNullValue);

        String body = "{\"std_data\":{\"parameter\":" + dataJson + "}}";
        // 建立RequestModel物件並設定內容
        RequestModel requestModel = new RequestModel();
        requestModel.setHostProd("Athena");
        requestModel.setHostVer("1.0");
        requestModel.setHostId("AgileInteraction");
        requestModel.setHostAcct("athena");
        requestModel.setLanguage(dataEntryTask.getLocale());
        requestModel.setTenantId(dataEntryTask.getTenantId());
        requestModel.setServiceProd(productName);

        final String actionId = dataEntryTask.getActionId();
        requestModel.setServiceName(actionId.substring(actionId.contains("esp_") ? 4 : 0));
        requestModel.setBodyJsonString(body);
        //header
        Map<String, String> headerMap = new HashMap<>();
        DWSecurityToken dwSecurityToken;
        try {
            dwSecurityToken = DWSecurityTokenGenerator.parseSecurityToken(dataEntryTask.getUserToken());
        } catch (Exception e) {
            log.error("解析token失败:{}", dataEntryTask.getUserToken());
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0071.getErrCode(), "解析token失败", e);
        }
        headerMap.put("token", dwSecurityToken.getContext().getUserToken());
        headerMap.put("security-token", dataEntryTask.getUserToken());
        headerMap.put("locale", dataEntryTask.getLocale());
        addChainInfoToHeader(headerMap);
        requestModel.setHeaderMap(headerMap);
        // 设置 dataKeyMap 中的 appCode 和作业信息
        Map<String, String> dataKeyMap = new HashMap<>();
        // 有些应用在解析 digi-datakey 的时候，如果里面的值有 null 会报错
        if (StringUtils.isNotBlank(dataEntryTask.getApplication())) {
            dataKeyMap.put("appCode", dataEntryTask.getApplication());
        }
        // 42420 将作业 code 和 type 拿到，透传给 esp，esp 给到业务中台用作运营统计分析
        if (ImportAndExportStatisticsConstants.EXPORT_FILE_TYPE_BUSINESS_DATA.equals(dataEntryTask.getType())) {
            String activityId = dataEntryTask.getActivityId();
            if (StringUtils.isNotBlank(activityId)) {
                dataKeyMap.put("taskType", "DATA_ENTRY");
                dataKeyMap.put("taskCode", activityId);
            }
        }
        if (MapUtils.isNotEmpty(dataKeyMap)) {
            requestModel.setDatakeyMap(dataKeyMap);
        }

        ResponseModel responseModel;
        try {
            responseModel = Invoker.invokeRestSync(requestModel);
            String response = responseModel.getBodyJsonString();
            EspBody espBody = JsonUtils.jsonToObject(response, EspBody.class);
            EspResponse espResponse = new EspResponse();
            if ("0".equals(espBody.getStd_data().getExecution().getCode())) {
                espResponse.setStatus(EspResponse.Status.SUCCESS);
                return espResponse;
            } else {
                throwEspError(requestModel.getServiceName(), responseModel, requestModel);
                espResponse.setData(espBody);
                espResponse.setStatus(EspResponse.Status.FAILED);
                return espResponse;
            }
        } catch (MDCInvocationException | ProductUIDNotFoundException | InvocationException |
                 TargetInvocationException | TenantInfoNotFoundException e) {
            log.error("error request esp", e);
        } catch (Exception e) {
            log.error("error request esp", e);
            EspResponse espResponse = new EspResponse();
            espResponse.setData(null);
            espResponse.setStatus(EspResponse.Status.FAILED);
            return espResponse;
        }
        return null;
    }

    @Override
    public EspResponse getBaseData(Map<String, Object> requestParam, ExportFileMsg exportFileMsg, String productName, String userToken) {
        String body = "{\"std_data\":{\"parameter\":" + JsonUtils.objectToString(requestParam) + "}}";
        // 建立RequestModel物件並設定內容
        RequestModel requestModel = new RequestModel();
        requestModel.setHostProd("Athena");
        requestModel.setHostVer("1.0");
        requestModel.setHostId("AgileInteraction");
        requestModel.setHostAcct("athena");
        requestModel.setLanguage(exportFileMsg.getLocale());
        requestModel.setTenantId(exportFileMsg.getTenantId());
        requestModel.setServiceProd(productName);
        requestModel.setServiceName(exportFileMsg.getActionId().substring(exportFileMsg.getActionId().contains("esp_") ? 4 : 0));
        requestModel.setBodyJsonString(body);
        //header
        Map<String, String> headerMap = new HashMap<>();
        if (StringUtils.isEmpty(userToken)) {
            headerMap.put("token", exportFileMsg.getUserToken());
        } else {
            headerMap.put("security-token", exportFileMsg.getUserToken());
            headerMap.put("token", userToken);
        }
        headerMap.put("locale", exportFileMsg.getLocale());
        addChainInfoToHeader(headerMap);
        requestModel.setHeaderMap(headerMap);
        // 设置 dataKeyMap 中的 appCode 和作业信息
        Map<String, String> dataKeyMap = new HashMap<>();
        // 有些应用在解析 digi-datakey 的时候，如果里面的值有 null 会报错
        if (StringUtils.isNotBlank(exportFileMsg.getApplication())) {
            dataKeyMap.put("appCode", exportFileMsg.getApplication());
        }
        // 42420 将作业 code 和 type 拿到，透传给 esp，esp 给到业务中台用作运营统计分析
        if (ImportAndExportStatisticsConstants.EXPORT_FILE_TYPE_BUSINESS_DATA.equals(exportFileMsg.getType())) {
            String taskCode = Optional.ofNullable(exportFileMsg.getDownloadBaseDataParamDTO())
                    .map(DownloadBaseDataParamDTO::getActionInfo)
                    .map(ActionInfoDTO::getCode)
                    .orElse(null);
            if (StringUtils.isNotBlank(taskCode)) {
                dataKeyMap.put("taskType", "DATA_ENTRY");
                dataKeyMap.put("taskCode", taskCode);
            }
        }
        if (MapUtils.isNotEmpty(dataKeyMap)) {
            requestModel.setDatakeyMap(dataKeyMap);
        }
        ResponseModel responseModel;
        try {
            responseModel = Invoker.invokeRestSync(requestModel);
            String response = responseModel.getBodyJsonString();
            EspBody espBody = JsonUtils.jsonToObject(response, EspBody.class);
            EspResponse espResponse = new EspResponse();
            if ("0".equals(espBody.getStd_data().getExecution().getCode())) {
                espResponse.setStatus(EspResponse.Status.SUCCESS);
                espResponse.setData(espBody);
                return espResponse;
            } else {
                espResponse.setData(espBody);
                espResponse.setStatus(EspResponse.Status.FAILED);
                return espResponse;
            }
        } catch (MDCInvocationException e) {
            log.error("error request esp", e);
        } catch (ProductUIDNotFoundException e) {
            log.error("error request esp", e);
        } catch (InvocationException e) {
            log.error("error request esp", e);
        } catch (TargetInvocationException e) {
            log.error("error request esp", e);
        } catch (TenantInfoNotFoundException e) {
            log.error("error request esp", e);
        } catch (Exception e) {
            log.error("error request esp", e);
            EspResponse espResponse = new EspResponse();
            espResponse.setData(null);
            espResponse.setStatus(EspResponse.Status.FAILED);
            return espResponse;
        }
        return null;
    }

    /**
     * 增加digi-dap-service-chain-info信息
     *
     * @param headers
     */
    private void addChainInfoToHeader(Map<String, String> headers) {
        try {
            if (!headers.containsKey(GlobalConstant.DIGI_DAP_SERVICE_CHAIN_INFO)) {
                headers.put(GlobalConstant.DIGI_DAP_SERVICE_CHAIN_INFO, ChainInfoUtil.genInvokeChainInfo(MDC.get(GlobalConstant.DIGI_DAP_SERVICE_CHAIN_INFO)));
            }
        } catch (Exception e) {

        }
    }

    private void throwEspError(String serviceId, ResponseModel responseModel, RequestModel requestModel) {
        try {
            String serviceProd = requestModel.getServiceProd();
            String apiIdentifier = String.format("调用%s应用，api:%s", serviceProd, serviceId);

            // 1. 检查ESP错误码
            if (isEspError(responseModel)) {
                log.error("{}异常,错误信息: {}", apiIdentifier, responseModel.getEspMessage());
                return; // 发现ESP错误后直接返回
            }

            // 2. 检查响应体是否为空
            String bodyJson = responseModel.getBodyJsonString();
            if (StringUtils.isEmpty(bodyJson)) {
                log.error("{}异常，返回空", apiIdentifier);
                return;
            }

            // 3. 解析JSON并检查错误字段
            Map<String, Object> espResponse = parseAndCheckError(apiIdentifier, bodyJson);
            if (espResponse == null) return;

            // 4. 检查std_data字段
            Map<String, Object> stdDataMap = getAndCheckStdData(apiIdentifier, espResponse, bodyJson);
            if (stdDataMap == null) return;

            // 5. 检查execution字段
            Map<String, Object> execution = getAndCheckExecution(apiIdentifier, stdDataMap, bodyJson);
            if (execution == null) return;

            // 6. 检查业务错误码
            if (!"000".equals(responseModel.getSrvCode())) {
                String description = String.valueOf(execution.getOrDefault("description", ""));
                log.error("业务异常: {}", description);
                log.error("{}异常,返回结果: {}", apiIdentifier, bodyJson);
            }
        } catch (Exception e) {
            log.error("解析调用esp异常报错", e);
        }
    }

    // 辅助方法：检查ESP错误码
    private boolean isEspError(ResponseModel responseModel) {
        return responseModel.getEspCode() != null && !responseModel.getEspCode().startsWith("0");
    }

    // 辅助方法：解析JSON并检查错误字段
    private Map<String, Object> parseAndCheckError(String apiIdentifier, String bodyJson) {
        try {
            Map<String, Object> espResponse = JsonUtils.jsonToObject(bodyJson, HashMap.class);

            if (espResponse.containsKey("error")) {
                log.error("{}异常,错误信息: {}", apiIdentifier, espResponse.get("error"));
                return null;
            }
            return espResponse;
        } catch (Exception e) {
            log.error("{}异常，JSON解析失败: {}", apiIdentifier, bodyJson, e);
            return null;
        }
    }

    // 辅助方法：检查std_data字段
    private Map<String, Object> getAndCheckStdData(String apiIdentifier,
                                                   Map<String, Object> espResponse,
                                                   String bodyJson) {
        Map<String, Object> stdDataMap = safeCast(espResponse.get("std_data"), Map.class);

        if (stdDataMap == null) {
            log.error("{}异常，结果缺失std_data，返回结果: {}", apiIdentifier, bodyJson);
            return null;
        }
        return stdDataMap;
    }

    // 辅助方法：检查execution字段
    private Map<String, Object> getAndCheckExecution(String apiIdentifier,
                                                     Map<String, Object> stdDataMap,
                                                     String bodyJson) {
        Map<String, Object> execution = safeCast(stdDataMap.get("execution"), Map.class);

        if (execution == null) {
            log.error("{}异常，结果缺失execution，返回结果: {}", apiIdentifier, bodyJson);
            return null;
        }
        return execution;
    }

    // 类型安全转换
    @SuppressWarnings("unchecked")
    private <T> T safeCast(Object obj, Class<T> clazz) {
        return clazz.isInstance(obj) ? clazz.cast(obj) : null;
    }
}
