package com.digiwin.athena.abt.application.service.abt.migration.inout;

import cn.hutool.core.convert.Convert;
import com.digiwin.athena.abt.application.configuration.DirectRabbitConfig;
import com.digiwin.athena.abt.application.dto.migration.abt.api.*;
import com.digiwin.athena.abt.application.dto.migration.abt.esp.EspBody;
import com.digiwin.athena.abt.application.dto.migration.abt.esp.EspResponse;
import com.digiwin.athena.abt.application.dto.migration.abt.excel.ExcelParserBean;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.*;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.*;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.DataEntryTask;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.ExportFileMsg;
import com.digiwin.athena.abt.application.dto.response.BatchImportListRespDTO;
import com.digiwin.athena.abt.application.dto.response.ExportBasicDataRespDTO;
import com.digiwin.athena.abt.application.service.abt.migration.assembler.ImportStatisticsAssembler;
import com.digiwin.athena.abt.application.service.abt.migration.esp.EspService;
import com.digiwin.athena.abt.application.service.abt.migration.esp.InitEspProperties;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.AppPermissionUtil;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.ExcelHelper;
import com.digiwin.athena.abt.application.service.abt.migration.lock.LockPool;
import com.digiwin.athena.abt.application.service.abt.migration.worker.DataEntryTaskFactory;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.thememap.ThemeMapService;
import com.digiwin.athena.abt.application.utils.ExcelUtil;
import com.digiwin.athena.abt.application.utils.LockPoolDataEntryHelper;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import com.digiwin.athena.abt.application.utils.MyCollectionUtils;
import com.digiwin.athena.abt.core.meta.constants.EntryConstant;
import com.digiwin.athena.abt.core.meta.constants.ImportAndExportStatisticsConstants;
import com.digiwin.athena.abt.core.meta.dto.CellTypeContainer;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.abt.core.meta.enums.ExcelTypeEnum;
import com.digiwin.athena.abt.core.meta.enums.MetaDataType;
import com.digiwin.athena.abt.infrastructure.mapper.biz.migration.abt.ExportStatisticsMapper;
import com.digiwin.athena.abt.infrastructure.mapper.biz.migration.abt.ImportStatisticsMapper;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.abt.ExportStatisticsPO;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.abt.ImportStatisticsPO;
import com.digiwin.athena.aim.sdk.manager.AudcManager;
import com.digiwin.athena.aim.sdk.meta.dto.request.AccessibleReqDTO;
import com.digiwin.athena.aim.sdk.meta.dto.response.AccessibleRespDTO;
import com.digiwin.athena.aim.sdk.meta.dto.response.ActivityAccessible;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.knowledgegraph.sdk.manager.KnowledgegraphManager;
import com.digiwin.athena.knowledgegraph.sdk.meta.dto.response.thememap.BaseItemRespDTO;
import com.digiwin.athena.knowledgegraph.sdk.meta.dto.response.thememap.ImportAndExportConfigRespDTO;
import com.digiwin.athena.smartdata.sdk.util.JsonUtil;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.meta.adapter.JaI18nAdapter;
import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Channel;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.rabbit.support.DefaultMessagePropertiesConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

/**
 * @ClassName BaseDataEntryServiceImpl
 * @Description 基础数据录入应用服务
 * @Author zhuangli
 * @Date 2021/4/2 16:26
 * @Version 1.0
 **/
@Slf4j
@Service
public class BaseDataEntryApplicationServiceImpl {

    /**
     * 公共基础资料appCode
     */
    public static final String APP_CODE_PUBLIC_BASIC_DATA = "PBD";


    /**
     * 公共基础资料appNameKey，用于i18n
     */
    public static final String APP_NAME_KEY_PUBLIC_BASIC_DATA = "public.basic.data";


    /**
     * 未购买作业
     */
    public static final String ACCESS_NO_BUY = "NOBUY";

    @Autowired
    RabbitTemplate rabbitTemplate;
    @Autowired
    DataEntryTaskFactory dataEntryTaskFactory;
    @Autowired
    ImportStatisticsDomainService importStatisticsDomainService;

    @Autowired
    ExportStatisticsDomainService exportStatisticsDomainService;

    @Autowired
    ImportStatisticsFactory importStatisticsFactory;
    @Autowired
    MetaDataService metaDataService;
    @Autowired
    EspService espService;
    @Autowired
    @Qualifier("rabbitErrorHandlerServiceImpl")
    ErrorHandlerService errorHandlerService;
    @Autowired
    private InitEspProperties espConfig;
    @Autowired
    LockPool lockPool;
    @Value("${data-entry.business.send.retry-times:3}")
    private int maxRetryTimes;
    @Autowired
    private AppPermissionUtil appPermissionUtil;

    @Autowired
    ExcelHelper excelHelper;

    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRED)
    public void upload(UploadParamDTO uploadParam, ExcelParserBean parser) {
        Date now = new Date();
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        GetActionLocaleResponseDTO getActionLocaleResponseDTO = metaDataService.getActionMetaData(uploadParam.getActionId(), athenaUser == null ? uploadParam.getAuthoredUser().getToken() : athenaUser.getToken(), LocaleContextHolder.getLocale().toString());
        ExcelUtil.setDefaultExcelType(parser);
        final Workbook workbook = ExcelUtil.createWorkbook(parser);
        ApiDataFieldLocaleMetadataDTO mainMetadata = metaDataService.getMainMetadata(getActionLocaleResponseDTO, uploadParam.getActionInfo());
        String mainKey = mainMetadata.getData_name();
        Map<String, Set<String>> sheetRequiredFiled = new HashMap<>();//定义每个sheet页必填的字段
        // 添加判空逻辑
        if (null == uploadParam.getActionInfo()) {
            uploadParam.setActionInfo(new ActionInfoDTO());
        }
        uploadParam.getActionInfo().setSheetRequiredFiled(sheetRequiredFiled);
        Map<String, List<ApiDataFieldLocaleMetadataDTO>> sheetMetadataMap = excelHelper.getSheetMetadataMap(workbook, mainMetadata, uploadParam);
        log.info("标题集合处理");
        // 标题集合
        Map<String, List<String>> sheetHeaderMap = excelHelper.readExcelHeaders(workbook, mainKey);
        log.info("数据集合处理");
        // 数据集合
        Map<String, List<Map>> sheetDataMap = excelHelper.readExcelDataForImport(workbook, mainKey, sheetMetadataMap);
        log.info("准备发送数据");
        handleUploadNew(uploadParam, sheetHeaderMap, sheetDataMap, mainMetadata, now);
    }


    /**
     * 处理上传,批量处理上传事件，在原来上传逻辑上层包装一层消息发送，用于批量处理
     * @param uploadParams 单个上传参数
     */
    public void uploadAsync(List<UploadParamDTO> uploadParams) {
        uploadParams.forEach(e->{
            e.setAuthoredUser(AppAuthContextHolder.getContext().getAuthoredUser());
            e.setSecurityToken(AppAuthContextHolder.getContext().getSecurityToken());
            try {
                rabbitTemplate.convertAndSend(DirectRabbitConfig.IMPORT_FILE_EXCHANGE_NAME, DirectRabbitConfig.ROUTING_NAME, JsonUtils.objectToString(e));
                log.info("Message sent: {}", e);
            } catch (Exception exception) {
                log.error("Failed to send message:{}", exception.getMessage());
                throw new RuntimeException(exception);
            }
        });
    }

    @Transactional(rollbackFor = Exception.class)
    public void republish(UploadParamDTO uploadParamDTO) {
        Date now = new Date();
        String masterId = uploadParamDTO.getMasterId();
        importStatisticsDomainService.setRepublishedFlag(masterId);
        ImportStatistics importStatistics = importStatisticsDomainService.getByMasterId(masterId);
        //获取元数据信息
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaData(importStatistics.getActionId(), athenaUser.getToken(), importStatistics.getLocale());
        ApiDataFieldLocaleMetadataDTO mainMetadata = metaDataService.getMainMetadata(metadataDTO, uploadParamDTO.getActionInfo());
        String mainKey = mainMetadata.getData_name();

        TableDTO tableDTO = uploadParamDTO.getTableDTO();
        Table table = ImportStatisticsAssembler.tableToDO(tableDTO);
        // 标题集合
        Map<String, List<String>> sheetHeaderMap = getSheetHeaderMap(mainKey, table.getHeaders());
        // 数据集合
        Map<String, List<Map>> sheetDataMap = getSheetDataMap(mainKey, table.getData());
        handleUploadNew(uploadParamDTO, sheetHeaderMap, sheetDataMap, mainMetadata, now);
    }

    @Transactional(rollbackFor = Exception.class)
    public void republishById(UploadParamDTO uploadParamDTO) {
        Date now = new Date();
        String masterId = uploadParamDTO.getMasterId();
        importStatisticsDomainService.setRepublishedFlag(masterId);
        ImportStatistics importStatistics = importStatisticsDomainService.getByMasterId(masterId);
        buildParam(importStatistics, uploadParamDTO);
        //获取元数据信息
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaData(importStatistics.getActionId(), athenaUser.getToken(), importStatistics.getLocale());
        ApiDataFieldLocaleMetadataDTO mainMetadata = metaDataService.getMainMetadata(metadataDTO, uploadParamDTO.getActionInfo());
        String mainKey = mainMetadata.getData_name();

        TableDTO tableDTO = uploadParamDTO.getTableDTO();
        Table table = ImportStatisticsAssembler.tableToDO(tableDTO);
        // 标题集合
        Map<String, List<String>> sheetHeaderMap = getSheetHeaderMap(mainKey, table.getHeaders());
        // 数据集合
        Map<String, List<Map>> sheetDataMap = getSheetDataMap(mainKey, table.getData());
        handleUploadNew(uploadParamDTO, sheetHeaderMap, sheetDataMap, mainMetadata, now);
    }


    private void buildParam(ImportStatistics importStatistics, UploadParamDTO uploadParamDTO) {
        uploadParamDTO.setActionId(importStatistics.getActionId());
        uploadParamDTO.setActionInfo(JsonUtils.jsonToObject(importStatistics.getActionInfo(), ActionInfoDTO.class));
        uploadParamDTO.setImportModel(importStatistics.getImportModel());
        uploadParamDTO.setActivityId(importStatistics.getActivityId());
    }


    /**
     * 获取sheet页的数据
     *
     * @param sheetName
     * @param dataMapList
     * @return
     */
    private Map<String, List<Map>> getSheetDataMap(String sheetName, List<Map> dataMapList) {
        Map<String, List<Map>> sheetDataMap = new HashMap<>();
        List<Map> datas = new LinkedList<>();
        for (Map dataMap : dataMapList) {
            Map<String, Object> sheetData = new HashMap<>();
            Iterator<Map.Entry<String, Object>> it = dataMap.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, Object> dataEntry = it.next();
                if (dataEntry == null || dataEntry.getValue() == null || dataEntry.getKey().contains(ExcelHelper.ERROR_FIELD_SUFFIX)) {
                    continue;
                }
                // 合并子单身的数据集
                if (dataEntry.getValue() instanceof List) {
                    Map<String, List<Map>> DFieldSheetData = getSheetDataMap(dataEntry.getKey(), (List<Map>) dataEntry.getValue());
                    DFieldSheetData.forEach((DFieldName, DFieldValues) -> {
                        List<Map> allDFieldValues = sheetDataMap.computeIfAbsent(DFieldName, key -> new ArrayList<>());
                        allDFieldValues.addAll(DFieldValues);
                    });
                } else {
                    sheetData.put(dataEntry.getKey(), dataEntry.getValue());
                }
            }
            datas.add(sheetData);
        }
        sheetDataMap.put(sheetName, datas);
        return sheetDataMap;
    }

    /**
     * 获取sheet页的栏位
     *
     * @param sheetName
     * @param headerMapList
     * @return
     */
    private Map<String, List<String>> getSheetHeaderMap(String sheetName, List<Map> headerMapList) {
        Map<String, List<String>> sheetHeaderMap = new HashMap<>();
        List<String> headers = new LinkedList<>();
        for (Map headerMap : headerMapList) {
            if (headerMap.containsKey(ExcelHelper.HEADER_CHILDREN)) {
                sheetHeaderMap.putAll(getSheetHeaderMap((String) headerMap.get(ExcelHelper.HEADER_KEY), (List<Map>) headerMap.get(ExcelHelper.HEADER_CHILDREN)));
            } else {
                headers.add((String) headerMap.get(ExcelHelper.HEADER_KEY));
            }
        }
        sheetHeaderMap.put(sheetName, headers);
        return sheetHeaderMap;
    }

    private void sendBatch(String uuid, List<DataEntryTask> productBatch) {
        log.info("enter send batch");
        ImportStatistics importStatistics = importStatisticsDomainService.getByMasterId(uuid);
        //通过uuid的后n位从锁池获取锁
        int lockId = LockPoolDataEntryHelper.getIdByUUID(uuid, lockPool.getPoolSize());
        ReentrantLock lock = lockPool.getLockById(lockId);
        if (CollectionUtils.isEmpty(productBatch)) {
            return;
        }
        //避免竞争rabbitmq channel资源,先加锁
        lock.lock();
        try (Channel channel = rabbitTemplate.getConnectionFactory().createConnection().createChannel(false)) {
            int batch = 0;
            for (DataEntryTask item : productBatch) {
                batch++;
                int retryTimes = 0;
                do {
                    try {
                        log.info("第{}批次 try send msg {} time", batch, retryTimes + 1);
                        channel.confirmSelect();
                        // 使用Gson解决序列化null值丢失的问题
                        Message message = rabbitTemplate.getMessageConverter().toMessage(JsonUtil.toString(item), new MessageProperties());
                        AMQP.BasicProperties convertedMessageProperties = new DefaultMessagePropertiesConverter().fromMessageProperties(message.getMessageProperties(), "UTF-8");
                        channel.basicPublish(DirectRabbitConfig.EXCHANGE_NAME, DirectRabbitConfig.ROUTING_NAME, true,
                                convertedMessageProperties, message.getBody());
                        log.info("sendBatch before confirm");
                        boolean flag = channel.waitForConfirms();
                        log.info("sendBatch after confirm,flag:{}", flag);
                        if (flag) {
                            break;
                        } else {
                            log.error("ERROR SEND MESSAGE FOR EXCHANGE:[{}] ROUTING_NAME:[{}]", DirectRabbitConfig.EXCHANGE_NAME, DirectRabbitConfig.ROUTING_NAME);
                            throw BusinessException.create(ErrorCodeEnum.NUM_500_0051.getErrCode(), "error send msg");
                        }
                    } catch (Exception e) {
                        log.error("sendBatch has an exception", e);
                        if (retryTimes + 1 > maxRetryTimes) {
                            log.error("重试达到最大次数【{}】", maxRetryTimes);
                            log.error("error Items:{}", item);
                            ImportBatchRecord importBatchRecord = importStatisticsDomainService.getBatchRecordByPIdAndSeq(
                                    item.getMasterId(), item.getBatchNum());
                            importStatisticsFactory.calByTotalAndFailNum(item.getDataList().size(), item.getDataList().size(), importStatistics, importBatchRecord);
                            //更新记录
                            importStatisticsDomainService.updateThisAndBatchRecordsById(importStatistics);
                            log.error("请求退出，超出重试次数");
                        }
                    }
                } while (retryTimes++ <= maxRetryTimes);
            }
        } catch (TimeoutException e) {
            log.error("send msg TimeoutException:{}", e);
        } catch (IOException e) {
            log.error("send msg IOException:{}", e);
        } finally {
            lock.unlock();
        }
    }

    private Boolean sendExportFileMQ(ExportStatistics exportStatistics, ExportFileMsg exportFileMsg, String exchangeName) {
        Boolean result;
        log.info("enter sendExportFileMQ");
        //通过uuid的后n位从锁池获取锁
        int lockId = LockPoolDataEntryHelper.getIdByUUID(exportStatistics.getMasterId(), lockPool.getPoolSize());
        ReentrantLock lock = lockPool.getLockById(lockId);
        if (Objects.isNull(exportFileMsg)) {
            log.error("exportFileMsg is null");
            return false;
        }
        //避免竞争rabbitmq channel资源,先加锁
        lock.lock();
        try (Channel channel = rabbitTemplate.getConnectionFactory().createConnection().createChannel(false)) {
            int retryTimes = 0;
            do {
                try {
                    log.info("try send msg {} time", retryTimes + 1);
                    channel.confirmSelect();
                    Message message = rabbitTemplate.getMessageConverter().toMessage(JsonUtils.objectToString(exportFileMsg), new MessageProperties());
                    AMQP.BasicProperties convertedMessageProperties = new DefaultMessagePropertiesConverter().fromMessageProperties(message.getMessageProperties(), "UTF-8");
                    channel.basicPublish(exchangeName, DirectRabbitConfig.ROUTING_NAME, true,
                            convertedMessageProperties, message.getBody());
                    log.info("waitForConfirms before");
                    boolean flag = channel.waitForConfirms();
                    log.info("waitForConfirms after");
                    if (flag) {
                        break;
                    } else {
                        log.error("ERROR SEND MESSAGE FOR EXCHANGE:[{}] ROUTING_NAME:[{}]", DirectRabbitConfig.EXPORT_FILE_EXCHANGE_NAME, DirectRabbitConfig.ROUTING_NAME);
                        throw BusinessException.create(ErrorCodeEnum.NUM_500_0051.getErrCode(), "error send msg");
                    }
                } catch (Exception e) {
                    if (retryTimes + 1 > maxRetryTimes) {
                        log.error("重试达到最大次数【{}】", maxRetryTimes);
                        log.error("error exportFileMsg:{}", exportFileMsg);
                        //更新记录
                        exportStatistics.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL);
                        exportStatisticsDomainService.updateById(exportStatistics);
                        log.error("请求退出，超出重试次数");
                    }
                }
            } while (retryTimes++ <= maxRetryTimes);
            result = true;
        } catch (Exception e) {
            log.error("send msg TimeoutException:{}", e);
            result = false;
        } finally {
            lock.unlock();
        }
        return result;
    }

    /**
     * 基础资料导出  优化后增加动态绑定队列逻辑 使用rabbitTemplate发送消息
     *
     * @param exportStatistics
     * @param exportFileMsg
     * @param exchangeName
     * @return
     */
    private Boolean sendExportFileMQNew(ExportStatistics exportStatistics, ExportFileMsg exportFileMsg, String exchangeName) {
        Boolean result;
        log.info("enter sendExportFileMQ");
        //通过uuid的后n位从锁池获取锁
        int lockId = LockPoolDataEntryHelper.getIdByUUID(exportStatistics.getMasterId(), lockPool.getPoolSize());
        ReentrantLock lock = lockPool.getLockById(lockId);
        if (Objects.isNull(exportFileMsg)) {
            log.error("exportFileMsg is null");
            return false;
        }
        //避免竞争rabbitmq channel资源,先加锁
        lock.lock();
        try {
            rabbitTemplate.convertAndSend(exchangeName, DirectRabbitConfig.ROUTING_NAME, JsonUtils.objectToString(exportFileMsg));
            result = true;
        } catch (Exception e) {
            log.error("send msg TimeoutException:{}", e);
            result = false;
        } finally {
            lock.unlock();
        }
        return result;
    }

    //旧：上传excel的时候，解析逻辑根据字符串sheet1来处理
    //新：收集所有的sheet页名称，根据metadata来分析哪个是head 哪些是body
    @Deprecated
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRED)
    public void handleUpload(String activityId, String actionId, Map<String, List<List<Object>>> mapData,
                             GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        // 1、数据校验
        if (mapData.size() <= 0) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0052.getErrCode(), MessageUtil.getMessage("delivery.uploadError"));
        }
        Map<String, String> metadataMap = getMetadataMap(getActionLocaleResponseDTO);
//        List<List<Object>> data = mapData.get(importStatisticsDomainService.SHEET_NAME);
        List<List<Object>> data = mapData.get(metadataMap.get(importStatisticsDomainService.SHEET_NAME));

        if (CollectionUtils.isEmpty(data)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0116.getErrCode(), MessageUtil.getMessageByLocale("upload.template.error", LocaleContextHolder.getLocale().toString()));
        }

        if (data.size() <= 2) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0052.getErrCode(), MessageUtil.getMessage("delivery.uploadError"));
        }
        //keyList的顺序和excel中的data顺序一致,处理后的列应当相对应
        List<String> keyList = new LinkedList<>();
        List<List> errorListHolder = new LinkedList<>();
        //通过元数据校验[excel列]是否正确
        if (!metaDataService.validate(getActionLocaleResponseDTO, data, keyList, errorListHolder)) {
            log.error("未匹配的列:errorList:{}", errorListHolder.get(0));
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0053.getErrCode(), MessageUtil.getMessage("delivery.verifyTableError"));
        }

        // 2、解析数据
        //获取所有枚举类型的字段,包含字段名字段类型用用于处理枚举类型数据
        String tableKey = metaDataService.getTableKey(getActionLocaleResponseDTO);
        Map<String, MetaDataType> enumMap = new HashMap<>();
        Map<String, MetaDataType> nonEnumMap = new HashMap<>();
        metaDataService.getMetaDataTypeMap(enumMap, nonEnumMap, getActionLocaleResponseDTO, tableKey);
        //处理枚举类型列
        handleEnumColumn(data, enumMap, keyList);
        handleNonEnumColumn(data, nonEnumMap, keyList);
        //根据activityId获取activityName多语言信息
        String activityName = metaDataService.getActivityNameByActivityId(activityId, null);
        //请求唯一标识,用于处理消息回调
        String masterId = UUID.randomUUID().toString();
        //删除单头表头
        removeHeaders(data);
        //双档有单身情况
        if (mapData.size() > 1) {
            //后续遍历单身
            //todo check sheetName
            mapData.remove(metadataMap.get(importStatisticsDomainService.SHEET_NAME));
            dealBody(getActionLocaleResponseDTO, mapData, data, tableKey, keyList, errorListHolder);
        }

        // 3、更新导入记录
        //更新对应masterId处理消息条数
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        List<DataEntryTask> productBatch = dataEntryTaskFactory.produce(actionId, athenaUser, tableKey, masterId, keyList, data, DirectRabbitConfig.QUEUE_NAME, espConfig.getToken());
        List<ImportBatchRecord> importBatchRecords = importStatisticsFactory.produce(productBatch);
        importStatisticsDomainService.insertByMasterId(activityId, activityName, actionId, athenaUser, masterId, data.size(), importBatchRecords);

        // 4、发送MQ异步处理
        // 先提交事务，再发送MQ，防止MQ消费的时候查询不到
        if (TransactionSynchronizationManager.isActualTransactionActive()) {
            //存在事物
            TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
                @Override
                public void afterCommit() {
                    log.error("transaction commit over ,sendBatch message");
                    sendBatch(productBatch.get(0).getMasterId(), productBatch);
                }
            });
        } else {
            sendBatch(productBatch.get(0).getMasterId(), productBatch);
        }
    }

    //旧：上传excel的时候，解析逻辑根据字符串sheet1来处理
    //新：收集所有的sheet页名称，根据metadata来分析哪个是head 哪些是body
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRED)
    public void handleUploadNew(UploadParamDTO uploadParam, Map<String, List<String>> sheetHeaderMap, Map<String, List<Map>> sheetDataMap,
                                ApiDataFieldLocaleMetadataDTO mainMetadata, Date now) {
        // 1、数据校验
        if (CollectionUtils.isEmpty(sheetDataMap)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0052.getErrCode(), MessageUtil.getMessage("delivery.uploadError"));
        }


        // 获取所有枚举类型的字段,包含字段名字段类型用用于处理枚举类型数据
        Map<String, String> metadataMap = getMetadataMap(mainMetadata);
        String tableKey = metadataMap.get(importStatisticsDomainService.SHEET_NAME);
        // 单头字段集合
        List<String> keyList = mainMetadata.getField().stream().map(item -> item.getData_name()).collect(Collectors.toList());

        // 添加判空
        if (null == uploadParam.getActionInfo()) {
            uploadParam.setActionInfo(new ActionInfoDTO());
        }

        // 2、解析数据
        List<Map> data = dealData(mainMetadata, sheetHeaderMap, sheetDataMap, null, null, uploadParam.getActionInfo().getSheetRequiredFiled());
        if (CollectionUtils.isEmpty(data)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0052.getErrCode(), MessageUtil.getMessage("delivery.uploadError"));
        }

        log.info("{}, 待导入的数据量：{}", uploadParam.getActionId(), data.size());

        // 3、更新导入记录
        //根据activityId获取activityName多语言信息
        String activityName = metaDataService.getActivityNameByActivityId(uploadParam.getActivityId(), uploadParam.getAuthoredUser()==null? null: uploadParam.getAuthoredUser().getToken());
        //请求唯一标识,用于处理消息回调
        String masterId = UUID.randomUUID().toString();
        uploadParam.setMasterId(masterId);
        //更新对应masterId处理消息条数
        List<DataEntryTask> productBatch = dataEntryTaskFactory.produceNew(uploadParam, tableKey, keyList, data);

        log.info("{}, 批次总数：{}", uploadParam.getActionId(), productBatch.size());

        List<ImportBatchRecord> importBatchRecords = importStatisticsFactory.produce(productBatch);
        importStatisticsDomainService.insertByMasterIdNew(uploadParam, activityName, data.size(), importBatchRecords, now);

        // 4、发送MQ异步处理
        // 先提交事务，再发送MQ，防止MQ消费的时候查询不到
        if (TransactionSynchronizationManager.isActualTransactionActive()) {
            //存在事物
            TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
                @Override
                public void afterCommit() {
                    log.error("transaction commit over ,sendBatch message");
                    sendBatch(productBatch.get(0).getMasterId(), productBatch);
                }
            });
        } else {
            sendBatch(productBatch.get(0).getMasterId(), productBatch);
        }
    }

    /**
     * 处理数据
     *
     * @param metadataDTO
     * @param headerMap
     * @param sheetDataMap
     * @param parentBKDataMap 上层BK
     * @return
     */
    private List<Map> dealData(ApiDataFieldLocaleMetadataDTO metadataDTO, Map<String, List<String>> headerMap, Map<String, List<Map>> sheetDataMap, Map parentBKDataMap, Map parentBKMetadataMap, Map<String, Set<String>> sheetRequiredFiled) {
        if (null == metadataDTO) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0091.getErrCode(), MessageUtil.getMessage("basedata.metaDataStructError"));
        }
        List<Map> dataList = new ArrayList<>();
        String sheetName = metadataDTO.getData_name();
        List<Map> sheetDataList = sheetDataMap.get(sheetName);
        Set<String> requiredFields = new HashSet<>();
        if (!CollectionUtils.isEmpty(sheetRequiredFiled)) {
            requiredFields = sheetRequiredFiled.get(sheetName);
        }
        if (CollectionUtils.isEmpty(sheetDataList)) {
            log.error("sheet数据不存在:sheetName:{}", sheetName);
            return dataList;
        }
        List<String> headerList = headerMap.get(sheetName);
        List<String> businessKeys = getBusinessKeys(metadataDTO.getField());
        Map<String, ApiDataFieldLocaleMetadataDTO> bkMetadataMap = getBKMetaDataMap(metadataDTO.getField());
        if (MapUtils.isEmpty(parentBKDataMap)) {
            parentBKDataMap = new HashMap();
        }

        //keyList的顺序和excel中的data顺序一致,处理后的列应当相对应
        List<String> errorListHolder = new LinkedList<>();
        List<String> parentBKList = new ArrayList<>(parentBKDataMap.keySet());
        //通过元数据校验[excel列]是否正确
        if (!metaDataService.validateNew(metadataDTO, headerList, parentBKList, errorListHolder)) {
            log.error("未匹配的列:errorList:{}", errorListHolder);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0053.getErrCode(), MessageUtil.getMessage("delivery.verifyTableError"));
        }
        Map<String, MetaDataType> enumMap = new HashMap<>();
        Map<String, MetaDataType> nonEnumMap = new HashMap<>();
        metaDataService.getMetaDataTypeMapNew(enumMap, nonEnumMap, metadataDTO, parentBKMetadataMap);

        for (Map rowData : sheetDataList) {
            // 空值处理
            handleEnumColumnNew(rowData, enumMap, headerList, requiredFields);
            handleNonEnumColumnNew(rowData, nonEnumMap, headerList, requiredFields);

            // 添加单头数据
            if (MapUtils.isEmpty(parentBKDataMap)) {
                dataList.add(rowData);
                // 添加匹配的BK的单身数据
            } else if (excelHelper.compareBusinessKey(rowData, parentBKDataMap)) {
                dataList.add(rowData);
            } else {
                continue;
            }

            //当前的主键
            Map bkDataMap = new HashMap();
            for (String bk : businessKeys) {
                bkDataMap.put(bk, rowData.get(bk));
            }

            //处理单身
            if (!CollectionUtils.isEmpty(metadataDTO.getField())) {
                for (ApiDataFieldLocaleMetadataDTO child : metadataDTO.getField()) {
                    if (!child.getIs_array()) {
                        continue;
                    }
                    List<Map> childDataList = dealData(child, headerMap, sheetDataMap, bkDataMap, bkMetadataMap, sheetRequiredFiled);
                    // 单身集合添加到单头
                    rowData.put(child.getData_name(), childDataList);
                }
            }
        }
        return dataList;
    }

    /**
     * 获取BK
     *
     * @param cellTypes
     * @return
     */
    private List<String> getBusinessKeys(List<ApiDataFieldLocaleMetadataDTO> cellTypes) {
        List<String> keys = new ArrayList<>();
        if (CollectionUtils.isEmpty(cellTypes)) {
            return keys;
        }
        for (ApiDataFieldLocaleMetadataDTO cellType : cellTypes) {
            if (cellType.getIs_businesskey()) {
                keys.add(cellType.getData_name());
            }
        }
        return keys;
    }

    /**
     * 获取BKMetaDataMap
     *
     * @param cellTypes
     * @return
     */
    private Map<String, ApiDataFieldLocaleMetadataDTO> getBKMetaDataMap(List<ApiDataFieldLocaleMetadataDTO> cellTypes) {
        Map<String, ApiDataFieldLocaleMetadataDTO> bkMetadataMap = new HashMap<>();
        if (CollectionUtils.isEmpty(cellTypes)) {
            return bkMetadataMap;
        }
        for (ApiDataFieldLocaleMetadataDTO cellType : cellTypes) {
            if (cellType.getIs_businesskey()) {
                bkMetadataMap.put(cellType.getData_name(), cellType);
            }
        }
        return bkMetadataMap;
    }


    /**
     * 解析双档单身数据
     *
     * @param getActionLocaleResponseDTO
     * @param mapData
     * @param data
     * @param tableKey
     * @param keyList
     * @param errorListHolder
     */
    private void dealBody(GetActionLocaleResponseDTO getActionLocaleResponseDTO, Map<String, List<List<Object>>> mapData, List<List<Object>> data,
                          String tableKey, List<String> keyList, List<List> errorListHolder) {
        if (CollectionUtils.isEmpty(mapData)) {
            log.info("dealBody - mapData is empty!");
            return;
        }
        //获取表名(最外层的业务key)
        List<String> businessKeyList = getActionLocaleResponseDTO.getRequest().getParameters().get(0).getField()
                .stream().filter(item -> item.getIs_businesskey()).map(item -> item.getData_name())
                .collect(Collectors.toList());
        Set<String> businessKeySet = new HashSet<>(businessKeyList);

        Map<String, Map> groupedSheetMap = new HashMap<>();
        mapData.forEach((sheetName, bodyData) -> {
            //keyList的顺序和excel中的data顺序一致,处理后的列应当相对应
            List<String> bodyKeyList = new LinkedList<>();
            //校验body
            if (!metaDataService.validateBody(sheetName, businessKeySet, getActionLocaleResponseDTO, bodyData, bodyKeyList, errorListHolder)) {
                log.error("未匹配的列:errorList:{}", errorListHolder);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0056.getErrCode(), MessageUtil.getMessage("delivery.verifyTableError"));
            }
            //将excel的二维表转为map并分组
            Map<String, MetaDataType> enumMap = new HashMap<>();
            Map<String, MetaDataType> nonEnumMap = new HashMap<>();
            metaDataService.getBodyMetaDataTypeMap(enumMap, nonEnumMap, getActionLocaleResponseDTO, sheetName, tableKey);
            //处理枚举类型列
            handleEnumColumn(bodyData, enumMap, bodyKeyList);
            handleNonEnumColumn(bodyData, nonEnumMap, bodyKeyList);
            //删除单头表头
            removeHeaders(bodyData);
            List<Map> convertedTable = formTable(bodyKeyList, bodyData);
            //根据businessKey进行分组
            Map<String, List<Map>> groupedMap = MyCollectionUtils
                    .groupingBy(convertedTable, MyCollectionUtils.formMapGroupFunctions(businessKeyList));
            groupedSheetMap.put(sheetName, groupedMap);
        });

        //加入单身key
        keyList.addAll(groupedSheetMap.keySet());
        //获取businessKey下标,便于获取每条单头数据businessKey的组合,且下标顺序应同businessKeyList一致(分组依据)
        List<Integer> businessKeyIndexes = new ArrayList<>();
        for (int i = 0; i < businessKeyList.size(); i++) {
            if (keyList.contains(businessKeyList.get(i))) {
                for (int j = 0; j < keyList.size(); j++) {
                    if (keyList.get(j).equals(businessKeyList.get(i))) {
                        //返回该key位置并加入数组
                        businessKeyIndexes.add(j);
                        break;
                    }
                }
            }
        }
        //遍历每条单头数据,如果有匹配的单身则赋值,否则赋值为null
        data.forEach(item -> groupedSheetMap.forEach((sheetName, groupedData) -> {
            Iterator<Map.Entry<String, List<Map>>> groupedDataIterator = groupedData.entrySet().iterator();
            boolean hitFlag = false;
            while (groupedDataIterator.hasNext()) {
                //如果该条数据的businessKey组合和分组中的组合相匹配则将该条数据加入单身,若找不到,则丢弃单身
                Map.Entry<String, List<Map>> entry = groupedDataIterator.next();
                if (entry.getKey().equals(getBusinessGroupValue(item, businessKeyIndexes))) {
                    List<Map> currentBody = entry.getValue();
                    removeBusinessKey(currentBody, businessKeySet);
                    item.add(currentBody);
                    groupedDataIterator.remove();
                    hitFlag = true;
                    break;
                }
            }
            //保证数据和字段对齐
            if (!hitFlag) {
                item.add(Collections.EMPTY_LIST);
            }
        }));
    }

    private void removeBusinessKey(List<Map> currentBody, Set<String> businessKeySet) {
        if (!CollectionUtils.isEmpty(currentBody)) {
            currentBody.forEach(map -> {
                Set<String> set = map.keySet();
                Iterator<String> it = set.iterator();
                while (it.hasNext()) {
                    String key = it.next();
                    if (businessKeySet.contains(key)) {
                        it.remove();
                    }
                }
            });
        }
    }

    private String getBusinessGroupValue(List<Object> item, List<Integer> businessKeyIndexes) {
        List<Object> selectedItem = new LinkedList<>();
        for (int i = 0; i < businessKeyIndexes.size(); i++) {
            selectedItem.add(item.get(businessKeyIndexes.get(i)));
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < selectedItem.size(); i++) {
            if (i != 0) {

                sb.append(MyCollectionUtils.DEFAULT_DELIMITER).append(selectedItem.get(i));
            } else {
                sb.append(selectedItem.get(i));
            }
        }
        return sb.toString();
    }

    private List<Map> formTable(List<String> tableFields, List<List<Object>> table) {
        List<Map> result = new LinkedList<>();
        table.forEach(item -> {
            Map<String, Object> entity = new HashMap<>();
            AtomicInteger idx = new AtomicInteger();
            item.forEach(inner -> {
                entity.put(tableFields.get(idx.get()), inner);
                idx.getAndIncrement();
            });
            result.add(entity);
        });
        return result;
    }

    private void handleEnumColumn(List<List<Object>> data, Map<String, MetaDataType> enumMap, List<String> keyList) {
        //循环处理列
        for (int i = 0; i < keyList.size(); i++) {
            MetaDataType metaDataType = enumMap.get(keyList.get(i));
            if (null != metaDataType) {
                //说明是枚举类型,循环处理所有枚举列,从第二行开始遍历
                for (int j = 2; j < data.size(); j++) {
                    String cellData = null == data.get(j).get(i) ? "" : (String) data.get(j).get(i);
                    if (StringUtils.isEmpty(cellData)) {
                        //空值处理
                        switch (metaDataType) {
                            case STRING:
                                data.get(j).set(i, EntryConstant.EMPTY_STRING);
                                break;
                            case NUMERIC:
                            case NUMBER:
                                data.get(j).set(i, EntryConstant.ZERO);
                                break;
                            case DATE:
                                data.get(j).set(i, EntryConstant.DATE);
                                break;
                            case DATETIME:
                                data.get(j).set(i, EntryConstant.DATE_TIME);
                                break;
                            case TIME:
                                data.get(j).set(i, EntryConstant.TIME);
                                break;
                            case BOOLEAN:
                                data.get(j).set(i, EntryConstant.FALSE);
                                break;
                            default:
                                data.get(j).set(i, EntryConstant.EMPTY_STRING);
                                break;
                        }
                    } else {
                        int index = cellData.lastIndexOf(".");
                        String[] codeValue = cellData.split("[.]");
                        if (codeValue.length < 2) {
                            log.error("不符合枚举类型row:{},column:{},cellValue:{}", j + 1, i + 1, cellData);
                            throw BusinessException.create(ErrorCodeEnum.NUM_500_0058.getErrCode(), MessageUtil.getMessage("delivery.enumTypeError"));
                        }
                        String code = cellData.substring(0, index);
                        switch (metaDataType) {
                            case NUMBER:
                            case NUMERIC:
                                data.get(j).set(i, Integer.parseInt(code));
                                break;
                            case STRING:
                                data.get(j).set(i, code);
                                break;
                            case BOOLEAN:
                                data.get(j).set(i, Boolean.parseBoolean(code));
                                break;
                            default:
                                throw BusinessException.create(ErrorCodeEnum.NUM_500_0059.getErrCode(), MessageUtil.getMessage("delivery.enumTypeNotSupport"));
                        }
                    }
                }
            }
        }
    }

    private void handleNonEnumColumn(List<List<Object>> data, Map<String, MetaDataType> nonEnumMap, List<String> keyList) {
        //循环处理列
        for (int i = 0; i < keyList.size(); i++) {
            MetaDataType metaDataType = nonEnumMap.get(keyList.get(i));
            if (null != metaDataType) {
                //说明是非枚举类型,循环处理所有枚举列,从第二行开始遍历
                for (int j = 2; j < data.size(); j++) {
                    if (null == data.get(j).get(i) || data.get(j).get(i) instanceof String) {
                        String cellData = null == data.get(j).get(i) ? "" : (String) data.get(j).get(i);
                        if (StringUtils.isEmpty(cellData)) {
                            switch (metaDataType) {
                                case STRING:
                                    data.get(j).set(i, EntryConstant.EMPTY_STRING);
                                    break;
                                case NUMERIC:
                                case NUMBER:
                                    data.get(j).set(i, EntryConstant.ZERO);
                                    break;
                                case DATE:
                                    data.get(j).set(i, EntryConstant.DATE);
                                    break;
                                case DATETIME:
                                    data.get(j).set(i, EntryConstant.DATE_TIME);
                                    break;
                                case TIME:
                                    data.get(j).set(i, EntryConstant.TIME);
                                    break;
                                case BOOLEAN:
                                    data.get(j).set(i, EntryConstant.FALSE);
                                    break;
                                default:
                                    data.get(j).set(i, EntryConstant.EMPTY_STRING);
                                    break;
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * 枚举类型的空值赋默认值
     *
     * @param rowData
     * @param enumMap
     * @param keyList
     * @param requiredFields
     */
    private void handleEnumColumnNew(Map rowData, Map<String, MetaDataType> enumMap, List<String> keyList, Set<String> requiredFields) {
        if (CollectionUtils.isEmpty(enumMap)) {
            return;
        }
        //循环处理列
        for (int i = 0; i < keyList.size(); i++) {
            String keyName = keyList.get(i);
            MetaDataType metaDataType = enumMap.get(keyList.get(i));
            if (null != metaDataType) {
                //说明是枚举类型,循环处理所有枚举列,从第二行开始遍历
                if (null == rowData.get(keyName) || rowData.get(keyName) instanceof String) {  // 添加类型判断，有可能类型为数字等非String类型
                    String cellData = null == rowData.get(keyName) ? "" : (String) rowData.get(keyName);
                    if (StringUtils.isEmpty(cellData)) {
                        // 必填字段为空,不需要赋默认值
                        if (!CollectionUtils.isEmpty(requiredFields) && requiredFields.contains(keyName)) {
                            continue;
                        }
                        //空值处理
                        switch (metaDataType) {
                            case STRING:
                                rowData.put(keyName, EntryConstant.EMPTY_STRING);
                                break;
                            case NUMERIC:
                            case NUMBER:
                                rowData.put(keyName, EntryConstant.ZERO);
                                break;
                            case DATE:
                                rowData.put(keyName, EntryConstant.DATE);
                                break;
                            case DATETIME:
                                rowData.put(keyName, EntryConstant.DATE_TIME);
                                break;
                            case TIME:
                                rowData.put(keyName, EntryConstant.TIME);
                                break;
                            case BOOLEAN:
                                rowData.put(keyName, EntryConstant.FALSE);
                                break;
                            default:
                                rowData.put(keyName, EntryConstant.EMPTY_STRING);
                                break;
                        }
                    } else {
                        int index = cellData.lastIndexOf(".");
                        // 单身有枚举类型的时候，在处理单头的时候已经解析国枚举值了，所以后续无法也不需要根据截取.判断是否为枚举
                    /*String[] codeValue = cellData.split("[.]");
                    if (codeValue.length < 2) {
                        log.error("不符合枚举类型row:{},column:{},cellValue:{}", rowData, i + 1, cellData);
                        throw BusinessException.create(ErrorCodeEnum.NUM_500_0058.getErrCode(),MessageUtil.getMessage("delivery.enumTypeError"));
                    }*/
                        String code;
                        if (index > 0) {
                            code = cellData.substring(0, index);
                        } else {
                            code = cellData;
                        }

                        switch (metaDataType) {
                            case NUMBER:
                            case NUMERIC:
                                rowData.put(keyName, Integer.parseInt(code));
                                break;
                            case STRING:
                                rowData.put(keyName, code);
                                break;
                            case BOOLEAN:
                                rowData.put(keyName, Boolean.parseBoolean(code));
                                break;
                            default:
                                throw BusinessException.create(ErrorCodeEnum.NUM_500_0059.getErrCode(), MessageUtil.getMessage("delivery.enumTypeNotSupport"));
                        }
                    }
                }
            }
        }
    }

    /**
     * 非枚举类型的空值赋默认值
     *
     * @param rowData
     * @param nonEnumMap
     * @param keyList
     * @param requiredFields
     */
    private void handleNonEnumColumnNew(Map rowData, Map<String, MetaDataType> nonEnumMap, List<String> keyList, Set<String> requiredFields) {
        if (CollectionUtils.isEmpty(nonEnumMap)) {
            return;
        }
        //循环处理列
        for (int i = 0; i < keyList.size(); i++) {
            String keyName = keyList.get(i);
            MetaDataType metaDataType = nonEnumMap.get(keyList.get(i));
            if (null != metaDataType) {
                //说明是非枚举类型,循环处理所有枚举列,从第二行开始遍历
                if (null == rowData.get(keyName) || rowData.get(keyName) instanceof String) {
                    String cellData = null == rowData.get(keyName) ? "" : (String) rowData.get(keyName);
                    if (StringUtils.isEmpty(cellData)) {
                        // 必填字段为空,不需要赋默认值
                        if (!CollectionUtils.isEmpty(requiredFields) && requiredFields.contains(keyName)) {
                            continue;
                        }
                        switch (metaDataType) {
                            case STRING:
                                rowData.put(keyName, EntryConstant.EMPTY_STRING);
                                break;
                            case NUMERIC:
                            case NUMBER:
                                rowData.put(keyName, EntryConstant.ZERO);
                                break;
                            case DATE:
                                rowData.put(keyName, EntryConstant.DATE);
                                break;
                            case DATETIME:
                                rowData.put(keyName, EntryConstant.DATE_TIME);
                                break;
                            case TIME:
                                rowData.put(keyName, EntryConstant.TIME);
                                break;
                            case BOOLEAN:
                                rowData.put(keyName, EntryConstant.FALSE);
                                break;
                            default:
                                rowData.put(keyName, EntryConstant.EMPTY_STRING);
                                break;
                        }
                    }
                }
            }
        }
    }

    private void removeHeaders(List<List<Object>> data) {
        data.remove(0);
        data.remove(0);
    }

    public List<ImportStatistics> getRecordsByUserId(AuthoredUser user, String locale, Integer getRecordsNum, String activityName, String startTime, String endTime, String type, Integer state, Integer offset) {
        String userId = String.valueOf(user.getSid());
        List<ImportStatistics> importStatisticsList = importStatisticsDomainService.getRecordsByUserId(userId, user.getTenantId(), locale, activityName, startTime, endTime, type, state, offset, getRecordsNum);
        if (CollectionUtils.isEmpty(importStatisticsList)) {
            return importStatisticsList;
        }

        // 过滤授权应用
//        List<String> activityIds = importStatisticsList.stream().map(ImportStatistics::getActivityId).distinct().collect(Collectors.toList());
//        List<String> hasPermissionActivityCodeList = appPermissionUtil.handleAppPermission(activityIds);
//        importStatisticsList = importStatisticsList.stream().filter(x ->
//                hasPermissionActivityCodeList.contains(x.getActivityId())).collect(Collectors.toList());

        // 截取前5条
        if (Objects.nonNull(getRecordsNum) && importStatisticsList.size() > getRecordsNum) {
            importStatisticsList = importStatisticsList.subList(0, getRecordsNum);
        }

        return importStatisticsList;
    }

    @Autowired
    ImportStatisticsMapper importStatisticsMapper;

    @Autowired
    ImportConfigMapper importConfigMapper;

    @Autowired
    TemplateConfigMapper templateConfigMapper;


    @Autowired
    ExportStatisticsMapper exportStatisticsMapper;

    /**
     * 获取最新的上传时间-批量
     * @param ids 导入记录id
     * @return <导入记录id，最新上传时间> map
     */
    private Map<String, Date> getLatestUploadTime(List<String> ids) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        List<ImportStatisticsPO> latestUploadRecord = importStatisticsMapper.getLatestUploadRecord(ids,String.valueOf(authoredUser.getSid()));
        return latestUploadRecord.stream()
                .collect(Collectors.toMap(ImportStatisticsPO::getActivityId, ImportStatisticsPO::getCreateTime, (oldValue, newValue) -> newValue));
    }

    /**
     * 获取最新的导出时间-批量
     * @param ids 导出记录id
     * @return <导出记录id，最新导出时间> map
     */
    private Map<String, Date> getLatestExportTime(List<String> ids) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        List<ExportStatisticsPO> latestExportRecord = exportStatisticsMapper.getLatestExportRecord(ids,String.valueOf(authoredUser.getSid()),authoredUser.getTenantId());
        return latestExportRecord.stream()
                .collect(Collectors.toMap(ExportStatisticsPO::getActivityId, ExportStatisticsPO::getCreateTime, (oldValue, newValue) -> newValue));
    }


    /**
     * 获取导入基础数据列表
     * @return 导入基础数据列表
     */
    public List<ExportBasicDataRespDTO> getExportBasicDataList() {
        List<ExportBasicDataRespDTO> result = new ArrayList<>();
        //从km获取作业配置
        List<ImportAndExportConfigRespDTO> iAECRespDTO = KnowledgegraphManager.getImportAndExportConfig().data();
        //从km获取当前用户作业信息
        List<BaseItemRespDTO> respList = KnowledgegraphManager.getActivitiesByPattern().data();
        //若没有数据则不做后续操作
        if (org.apache.commons.collections.CollectionUtils.isEmpty(respList)) {
            return result;
        }
        //处理公共作业数据
        respList.forEach(this::resetAppDataForPublic);
        //生成作业map
        Map<String, BaseItemRespDTO> baseItemMap = respList.stream().collect(Collectors.toMap(BaseItemRespDTO::getCode, baseItem -> baseItem, (existing, duplicate) -> existing));
        //生成所有作业id
        List<String> codeList = respList.stream().map(BaseItemRespDTO::getCode).distinct().collect(Collectors.toList());
        //生成最新上传时间map key:作业id value:最新上传时间
        Map<String, Date> latestUploadTime = getLatestExportTime(codeList);
        //生成作业权限map key:作业id value:权限
        Map<String, String> accessMap = getAccessData(codeList);
        //遍历作业配置 生成导出基础数据列表
        for (ImportAndExportConfigRespDTO importAndExportConfigRespDTO : iAECRespDTO) {
            if (ACCESS_NO_BUY.equals(accessMap.get(importAndExportConfigRespDTO.getCode()))) {
                continue;
            }
            // skip not allow
            if (!"allow".equals(accessMap.get(importAndExportConfigRespDTO.getCode()))) {
                continue;
            }
            ExportBasicDataRespDTO exportBasicDataRespDTO = new ExportBasicDataRespDTO();
            exportBasicDataRespDTO.setActivityId(importAndExportConfigRespDTO.getCode());
            exportBasicDataRespDTO.setName(importAndExportConfigRespDTO.getName());
            exportBasicDataRespDTO.setScene(baseItemMap.get(importAndExportConfigRespDTO.getCode()).getGroupName());
            exportBasicDataRespDTO.setActionId(importAndExportConfigRespDTO.getExportConfig().getActionId());
            exportBasicDataRespDTO.setLatestUploadTime(latestUploadTime.get(importAndExportConfigRespDTO.getCode()));
            exportBasicDataRespDTO.setCategory(baseItemMap.get(importAndExportConfigRespDTO.getCode()).getCategory());
            exportBasicDataRespDTO.setType(baseItemMap.get(importAndExportConfigRespDTO.getCode()).getAppName());
            result.add(exportBasicDataRespDTO);
        }
        return result;
    }


    /**
     * 生成逻辑同上面 {@link #getExportBasicDataList()}
     * 获取基础数据列表
     * @param user 当前用户
     * @return 批量导入作业列表
     */
    public List<BatchImportListRespDTO> getBasicDataList(AuthoredUser user) {
        List<BatchImportListRespDTO> result = new ArrayList<>();
        //tenantSid==0 表示没有租户,则返回空数据
        if (user.getTenantSid() == 0L) {
            JaLog.warn("tmDataList-{}:{}:{}:{}: tenantSid is null", user.getTenantId(), user.getUserId(), 0, LocaleContextHolder.getLocale());
            return result;
        }
        List<ImportAndExportConfigRespDTO> iAECRespDTO = KnowledgegraphManager.getImportAndExportConfig().data();
        List<BaseItemRespDTO> respList = KnowledgegraphManager.getActivitiesByPattern().data();
        //若没有数据则不做后续操作
        if (org.apache.commons.collections.CollectionUtils.isEmpty(respList)) {
            return result;
        }
        respList.forEach(this::resetAppDataForPublic);
        Map<String, BaseItemRespDTO> baseItemMap = respList.stream().collect(Collectors.toMap(BaseItemRespDTO::getCode, baseItem -> baseItem, (existing, duplicate) -> existing));
        List<String> codeList = respList.stream().map(BaseItemRespDTO::getCode).distinct().collect(Collectors.toList());
        Map<String, Date> latestUploadTime = getLatestUploadTime(codeList);
        Map<String, String> accessMap = getAccessData(codeList);
        for (ImportAndExportConfigRespDTO importAndExportConfigRespDTO : iAECRespDTO) {
            if (ACCESS_NO_BUY.equals(accessMap.get(importAndExportConfigRespDTO.getCode()))) {
                continue;
            }
            // skip not allow
            if (!"allow".equals(accessMap.get(importAndExportConfigRespDTO.getCode()))) {
                continue;
            }

            BatchImportListRespDTO batchImportListRespDTO = new BatchImportListRespDTO();
            batchImportListRespDTO.setActivityId(importAndExportConfigRespDTO.getCode());
            batchImportListRespDTO.setName(importAndExportConfigRespDTO.getName());
            batchImportListRespDTO.setScene(baseItemMap.get(importAndExportConfigRespDTO.getCode()).getGroupName());
            batchImportListRespDTO.setImportAction(importConfigMapper.toImportActionList(importAndExportConfigRespDTO.getImportConfig()));
            batchImportListRespDTO.setTemplateAction(templateConfigMapper.toImportAction(importAndExportConfigRespDTO.getTemplateConfig()));
            batchImportListRespDTO.setLatestUploadTime(latestUploadTime.get(importAndExportConfigRespDTO.getCode()));
            batchImportListRespDTO.setType(baseItemMap.get(importAndExportConfigRespDTO.getCode()).getAppName());
            result.add(batchImportListRespDTO);
        }

        return result;
    }


    /**
     * 构造作业权限映射关系
     * @param activityId 作业id
     * @return 作业权限映射关系 key:作业id value:权限
     */
    private Map<String, String> getAccessData(List<String> activityId) {
        Map<String, String> resultMap = new HashMap<>();
        if (CollectionUtils.isEmpty(activityId)) {
            return resultMap;
        }
        List<AccessibleReqDTO> typeActivitiesList = new ArrayList<>();
        AccessibleReqDTO accessibleReqDTO = new AccessibleReqDTO();
        accessibleReqDTO.setType("0");
        accessibleReqDTO.setTmActivityIdList(activityId);
        typeActivitiesList.add(accessibleReqDTO);

        List<AccessibleRespDTO> activityPermission = AudcManager.checkAccessible(typeActivitiesList).data();
        if (org.apache.commons.collections.CollectionUtils.isEmpty(activityPermission)) {
            return resultMap;
        }
        // 每个作业的访问权限
        List<ActivityAccessible> activityAccessibleList = activityPermission.get(0).getActivityAccessibleList();
        if (org.apache.commons.collections.CollectionUtils.isEmpty(activityAccessibleList)) {
            return resultMap;
        }
        for (ActivityAccessible activityAccessible : activityAccessibleList) {
            if (Objects.isNull(activityAccessible)) {
                continue;
            }
            resultMap.put(activityAccessible.getTmActivityId(), activityAccessible.getAccess());
        }
        return resultMap;
    }


    /**
     * 若是公共数据，则重置appCode和appName，方便走后续分组逻辑
     *
     * @param baseItemDto 基础资料
     */
    private void resetAppDataForPublic(BaseItemRespDTO baseItemDto) {
        if (baseItemDto.isIfCommon()) {
            baseItemDto.setAppCode(APP_CODE_PUBLIC_BASIC_DATA);
            baseItemDto.setAppName(JaI18nAdapter.getMessage(APP_NAME_KEY_PUBLIC_BASIC_DATA));
        }
    }


    public List<ExportStatistics> getExportRecordsByUserId(AuthoredUser user, String locale, GetRecordsParamDTO param) {
        String userId = String.valueOf(user.getSid());
        List<ExportStatistics> exportStatisticsList = exportStatisticsDomainService.getRecordsByUserId(userId, user.getTenantId(), locale, param);
        if (CollectionUtils.isEmpty(exportStatisticsList)) {
            return exportStatisticsList;
        }
        //type存在为null的场景  但是以下逻辑仅仅只需作用type是0的数据生效
//        if (!"1".equals(param.getType())) {
//            // 过滤授权应用
//            List<String> activityIds = exportStatisticsList.stream()
////                    .filter(s ->"0".equals(s.getType()))
//                    .map(ExportStatistics::getActivityId)
//                    .distinct().collect(Collectors.toList());
////            List<String> hasPermissionActivityCodeList = appPermissionUtil.handleAppPermission(activityIds);
////            exportStatisticsList = exportStatisticsList.stream().filter(x ->
////                    hasPermissionActivityCodeList.contains(x.getActivityId()) || "1".equals(x.getType())).collect(Collectors.toList());
//        }

        // 截取前5条
        if (Objects.nonNull(param.getGetRecordsNum()) && exportStatisticsList.size() > param.getGetRecordsNum()) {
            exportStatisticsList = exportStatisticsList.subList(0, param.getGetRecordsNum());
        }

        return exportStatisticsList;
    }

    public ErrorTable getErrorTableByMasterId(String masterId, String locale) {
        ImportStatistics importStatistics = importStatisticsDomainService.getByMasterId(masterId);
        if (null == importStatistics) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0060.getErrCode(), MessageUtil.getMessage("delivery.dataNotFound"));
        }
        if (0 != importStatistics.getProcessingNum()) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0061.getErrCode(), MessageUtil.getMessage("delivery.existRunningData"));
        }
        if (0 == importStatistics.getFailedNum()) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0062.getErrCode(), MessageUtil.getMessage("delivery.notFoundErrorData"));
        }
        if (StringUtils.isEmpty(importStatistics.getFailedUrl())) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0066.getErrCode(), MessageUtil.getMessage("delivery.notFoundErrorExcel"));
        }
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        //获取元数据信息
        GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaData(importStatistics.getActionId(), athenaUser.getToken(),
                locale);
        List<String> BKs = metaDataService.getResponseBKList(metadataDTO);
        Map<String, CellTypeContainer> cellTypeContainerMap = metaDataService.getResponseCellTypeContainersV2(metadataDTO, athenaUser.getToken(), locale);
        String mainKeyName = metadataDTO.getResponse().getData().getData_name();
        ErrorTable errorTable = errorHandlerService.getErrorTableByMasterId(mainKeyName, importStatistics, cellTypeContainerMap, BKs);
        return errorTable;
    }

    public void downloadErrorTable(String masterId, String locale, Set<String> requiredFields, HttpServletResponse response) {
        ImportStatistics importStatistics = importStatisticsDomainService.getByMasterId(masterId);
        if (null == importStatistics) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0063.getErrCode(), MessageUtil.getMessage("delivery.dataNotFound"));
        }
        if (0 != importStatistics.getProcessingNum()) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0064.getErrCode(), MessageUtil.getMessage("delivery.existRunningData"));
        }
        if (0 == importStatistics.getFailedNum()) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0065.getErrCode(), MessageUtil.getMessage("delivery.notFoundErrorData"));
        }
        if (StringUtils.isEmpty(importStatistics.getFailedUrl())) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0066.getErrCode(), MessageUtil.getMessage("delivery.notFoundErrorExcel"));
        }
        List<Map> headers = metaDataService.getHeaders(importStatistics.getActionId(), AppAuthContextHolder.getContext().getAuthoredUser().getToken(), locale);
        errorHandlerService.downloadErrorTable(importStatistics, headers, requiredFields, response);
    }

    public void downloadErrorTable(String masterId, String locale, HttpServletResponse response) {
        downloadErrorTable(masterId, locale, null, response);
    }

    public byte[] downloadTemplate(DownloadTemplateParamDTO param) {
        return metaDataService.downloadTemplate(param.getActionId(), param.getToken(), param.getLocale(), param.getDownloadTemplateDTO());
    }

    /**
     * 获取表头
     * @param actionId 导出actionId
     * @return ExportHeaderRespDTO 导出筛选表头信息
     */
    public ExportHeaderRespDTO getTableHeaders(String actionId){
        String locale = JaI18nAdapter.getLocale();
        return metaDataService.getTableHeader(actionId, AppAuthContextHolder.getContext().getAuthoredUser().getToken(), locale);
    }

    public UserStatistics getProcessingNum(String userId, String tenantId) {
        return importStatisticsDomainService.getProcessingNum(userId, tenantId);
    }

    public UserStatistics getExportingNum(String userId, String tenantId, String type, List<String> types) {
        return exportStatisticsDomainService.getProcessingNum(userId, tenantId, type, types);
    }

    public ImportStatistics getImportStatistics(String masterId) {
        return importStatisticsDomainService.getByMasterId(masterId);
    }

    private Map<String, List<List<Object>>> transformTableTo2DListMap(String sheetName, Table table, List<String> BKs, List<String> bodyKeys, Map<String, CellTypeContainer> cellTypeContainerMap) {
        final String KEY = "key";
        final String NAME = "name";
        Map<String, List<List<Object>>> mapResult = new HashMap<>();
        List<List<Object>> result = new LinkedList<>();
        List<Map> headers = table.getHeaders();
        List<Object> titles = new LinkedList<>();
        List<Object> keys = new LinkedList<>();
        // 处理中英文标题
        headers.forEach(item -> {
            //单身数据不会添加到第一个sheet
            // 处理单头
            if (!bodyKeys.contains(item.get(KEY))) {
                titles.add(item.get(NAME));
                keys.add(item.get(KEY));
            } else {
                // 处理单身
                //组装新的sheet
                List<List<Object>> body = new LinkedList<>();
                CellTypeContainer cellTypeContainer = cellTypeContainerMap.get(item.get(KEY));
                if (null == cellTypeContainer) {
                    log.error("no cell type container");
                }
                List<CellTypeContainer> bodyCellTypeContainers = cellTypeContainer.getChildren();
                if (CollectionUtils.isEmpty(bodyCellTypeContainers)) {
                    log.error("no container children");
                }
                List<Object> bodyHeaderTitles = new LinkedList<>();
                List<Object> bodyHeaderKeys = new LinkedList<>();
                //单头主键
                BKs.forEach(bk -> {
                    bodyHeaderTitles.add(cellTypeContainerMap.get(bk).getKeyDescription());
                    bodyHeaderKeys.add(cellTypeContainerMap.get(bk).getKeyName());
                });
                //单身key
                bodyCellTypeContainers.forEach(cell -> {
                    // 单身栏位需要剔除_error_msg、和BK重复的栏位
                    if (!cell.getKeyName().contains("_error_msg") && !BKs.contains(cell.getKeyName())) {
                        bodyHeaderTitles.add(cell.getKeyDescription());
                        bodyHeaderKeys.add(cell.getKeyName());
                    }
                });
                body.add(bodyHeaderTitles);
                body.add(bodyHeaderKeys);
                mapResult.put(String.valueOf(item.get(KEY)), body);
            }
        });
        result.add(titles);
        result.add(keys);

        // 处理业务数据
        table.getData().forEach(item -> {
            List<Object> line = new LinkedList<>();
            headers.forEach(header -> {
                // 处理单头
                if (!bodyKeys.contains(header.get(KEY))) {
                    line.add(item.get(header.get(KEY)));
                } else {
                    // 处理单身
                    //获取初始化过得单身数据列表
                    List<List<Object>> body = mapResult.get(header.get(KEY));
                    List<Map> bodyData = JsonUtils.jsonToListObject(JsonUtils.objectToString(item.get(header.get(KEY))), Map.class);
                    if (!CollectionUtils.isEmpty(bodyData)) {
                        List<CellTypeContainer> bodyCellTypeContainers = cellTypeContainerMap.get(header.get(KEY)).getChildren();
                        bodyData.forEach(bodyLine -> {
                            List<Object> bodyLineData = new LinkedList<>();
                            Map bodyLineMap = bodyLine;
                            //添加单头主键数据
                            BKs.forEach(bk -> bodyLineData.add(item.get(bk)));
                            //和单身header一样的顺序添加数据
                            bodyCellTypeContainers.forEach(bodyCellTypeContainer -> {
                                // 单身栏位需要剔除_error_msg、和BK重复的栏位
                                if (!bodyCellTypeContainer.getKeyName().contains("_error_msg") && !BKs.contains(bodyCellTypeContainer.getKeyName())) {
                                    bodyLineData.add(bodyLineMap.get(bodyCellTypeContainer.getKeyName()));
                                }
                            });
                            body.add(bodyLineData);
                        });
                        mapResult.put((String) header.get(KEY), body);
                    }
                }
            });
            result.add(line);
        });
        mapResult.put(sheetName, result);
        return mapResult;
    }

    private List<List<Object>> transformTableTo2DList(Table table) {
        final String KEY = "key";
        final String NAME = "name";
        List<List<Object>> result = new LinkedList<>();

        List<Map> headers = table.getHeaders();
        List<Object> titles = new LinkedList<>();
        List<Object> keys = new LinkedList<>();
        headers.forEach(item -> {
            titles.add(item.get(NAME));
            keys.add(item.get(KEY));
        });
        result.add(titles);
        result.add(keys);
        table.getData().forEach(item -> {
            List<Object> line = new LinkedList<>();
            headers.forEach(header -> line.add(item.get(header.get(KEY))));
            result.add(line);
        });
        return result;
    }

    public ActivityStatistics getActivityStatistics(String activityCode, String userId) {
        return importStatisticsDomainService.getActivityStatistics(activityCode, userId);
    }

    /**
     * @Description 导出基础数据为excel, 支持数据量为60000条.
     * 1.根据actionid调用espapi获取需要导出的数据
     * 2.元数据api获得column名称并和返回数据匹配组装excel数据结构
     * 3.将数据写入excel并导出
     * @Date 13:31 2023/7/26
     * @Param
     **/
    public byte[] downloadBaseData(DownloadBaseDataParamDTO downloadBaseDataParamDTO) {
        byte[] fileBytes = new byte[0];
        String locale = LocaleContextHolder.getLocale().toString();
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String actionId = downloadBaseDataParamDTO.getActionInfo().getActionId();
        String productName = metaDataService.getProductName(athenaUser.getToken(),
                actionId.substring(actionId.contains("esp_") ? 4 : 0));
        // 1、获取元数据信息
        GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaData(actionId, athenaUser.getToken(),
                locale);
        //response主key
        String mainKey = metadataDTO.getResponse().getData().getData_name();
        String mainKeyDesc = metadataDTO.getResponse().getData().getDescription();
        getActionParas(downloadBaseDataParamDTO);

        // 2、调用espapi获取导出数据
        ExportFileMsg exportFileMsg = new ExportFileMsg();
        exportFileMsg.setLocale(locale);
        exportFileMsg.setTenantId(athenaUser.getTenantId());
        exportFileMsg.setUserToken(athenaUser.getToken());
        exportFileMsg.setActionId(actionId);
        exportFileMsg.setApplication(downloadBaseDataParamDTO.getApplication());
        exportFileMsg.setDownloadBaseDataParamDTO(downloadBaseDataParamDTO);
        exportFileMsg.setType(ImportAndExportStatisticsConstants.EXPORT_FILE_TYPE_BUSINESS_DATA);
        EspResponse espResponse = espService.getBaseData(downloadBaseDataParamDTO.getActionInfo().getActionParas(), exportFileMsg, productName, null);
        if (!espResponse.isOK()) {
            log.error("downloadBaseData actionId:{},name:{},locale:{},userToken:{}", actionId, downloadBaseDataParamDTO,
                    locale, athenaUser.getToken());
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0067.getErrCode(), MessageUtil.getMessage("delivery.exportDataError"));
        }
        EspBody espBody = (EspBody) espResponse.getData();
        List<Map<String, Object>> tableList = (List<Map<String, Object>>) (espBody.getStd_data().getParameter()).get(mainKey);
        //获取responseKey类型
        Map<String, CellTypeContainer> cellTypeContainerMap = metaDataService.getResponseCellTypeContainers(metadataDTO, athenaUser.getToken(), locale);
        //根据actionId获取excel列名
        List<String> keyList = new ArrayList<>();
        Map<String, String> headers = metaDataService.getResponseHeaders(metadataDTO, keyList);

        // 3、处理下载基础数据
        try {
            // 双档导出数据最多2000条（按单头数量计算）
            if (!CollectionUtils.isEmpty(tableList) && tableList.size() > ImportAndExportStatisticsConstants.EXPORT_LIMIT && ImportAndExportStatisticsConstants.CATEGORY_DOUBLE_DOCUMENT.equals(downloadBaseDataParamDTO.getActionInfo().getCategory())) {
                tableList = tableList.subList(0, ImportAndExportStatisticsConstants.EXPORT_LIMIT);
            }
            // 过滤导出字段
            getExportFileds(downloadBaseDataParamDTO.getActionInfo().getSelectField(), keyList, cellTypeContainerMap);
            fileBytes = importStatisticsDomainService.handleDownloadBaseData(locale, keyList, headers, cellTypeContainerMap,
                    tableList, ExcelHelper.getSheetName(mainKey, mainKeyDesc, true));
        } catch (UnsupportedEncodingException e) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0068.getErrCode(), MessageUtil.getMessage("delivery.downloadError"));
        }
        return fileBytes;
    }

    public ExportStatisticsDTO queryExportStatistics(String masterId) {
        ExportStatistics exportStatistics = exportStatisticsDomainService.getByMasterId(masterId);
        return ExportStatisticsFactory.exportStatisticsToDTO(exportStatistics);
    }

    @Autowired
    ThemeMapService themeMapService;

    @Autowired
    private ExportBatchRecordService exportBatchRecordService;

    /**
     * @Description 发起异步下载（目前不需要添加事物）
     * @Date 13:31 2023/9/20
     * @Param
     **/
    public Boolean startDownload(DownloadBaseDataParamDTO downloadBaseDataParamDTO) {
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        ExportStatistics exportStatistics = new ExportStatistics();
        exportStatistics.setId(SnowflakeIdWorker.getInstance().newId());
        exportStatistics.setMasterId(UUID.randomUUID().toString());
        exportStatistics.setExportParams(JsonUtils.objectToString(downloadBaseDataParamDTO));
        exportStatistics.setActivityId(downloadBaseDataParamDTO.getActionInfo().getCode());
        //根据activityId获取activityName多语言信息
        String activityName = null;
        String type = Optional.ofNullable(downloadBaseDataParamDTO.getType()).orElse("default");
        switch (type) {
            case "2":
                activityName = themeMapService.getTaskNamesByCodes(exportStatistics.getActivityId());
                exportStatistics.setType(ImportAndExportStatisticsConstants.EXPORT_FILE_TYPE_TASK);
                break;
            case "3":
                activityName = themeMapService.getProjectNamesByCodes(exportStatistics.getActivityId());
                exportStatistics.setType(ImportAndExportStatisticsConstants.EXPORT_FILE_TYPE_PROJECT);
                break;
            default:
                activityName = metaDataService.getActivityNameByActivityId(exportStatistics.getActivityId(), null);
                exportStatistics.setType(ImportAndExportStatisticsConstants.EXPORT_FILE_TYPE_BUSINESS_DATA);
        }
        exportStatistics.setActivityName(activityName);
        exportStatistics.setActionId(downloadBaseDataParamDTO.getActionInfo().getActionId());
        String locale = LocaleContextHolder.getLocale().toString();
        exportStatistics.setFileName(downloadBaseDataParamDTO.getActionInfo().getFileName());
        exportStatistics.setLocale(locale == null ? "zh_CN" : locale);
        exportStatistics.setUserId(String.valueOf(athenaUser.getSid()));
        exportStatistics.setTenantId(athenaUser.getTenantId());
        String securityToken = AppAuthContextHolder.getContext().getSecurityToken();
        exportStatistics.setUserToken(securityToken);
        exportStatistics.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_PROCESSING);
        Date now = new Date();
        exportStatistics.setCreateTime(now);
        exportStatistics.setUpdateTime(now);
        //冗余用户对象
        SimpleUser simpleUser = new SimpleUser();
        simpleUser.setTenantId(athenaUser.getTenantId());
        simpleUser.setTenantSid(athenaUser.getTenantSid());
        simpleUser.setUserId(athenaUser.getUserId());
        simpleUser.setUserSid(athenaUser.getSid());
        exportStatistics.setUserJson(JsonUtils.objectToString(simpleUser));
        // 设置失效时间为5天后
        exportStatistics.setExpireTime(DateUtils.addDays(now, ImportAndExportStatisticsConstants.EXPORT_FILE_EXPIRE_TIME));

        ExportFileMsg exportFileMsg = new ExportFileMsg();
        exportFileMsg.setMasterId(exportStatistics.getMasterId());
        exportFileMsg.setLocale(exportStatistics.getLocale());
        exportFileMsg.setTenantId(exportStatistics.getTenantId());
        exportFileMsg.setActionId(exportStatistics.getActionId());
        exportFileMsg.setUserToken(exportStatistics.getUserToken());
        exportFileMsg.setDownloadBaseDataParamDTO(downloadBaseDataParamDTO);

        //加密密码
        Optional.ofNullable(downloadBaseDataParamDTO).map(DownloadBaseDataParamDTO::getActionInfo).map(ActionInfoDTO::getDefence)
                .filter(StringUtils::isNotBlank).ifPresent(exportStatistics::setDefence);

        // 1、更新导入记录
        //更新对应masterId处理消息条数
        int insertResult = exportStatisticsDomainService.insertByMasterId(exportStatistics);

        // 2、发送MQ异步处理
        if (insertResult == 1) {
            exportBatchRecordService.splitBaseDataEntry(downloadBaseDataParamDTO,exportStatistics);
            return sendExportFileMQNew(exportStatistics, exportFileMsg, DirectRabbitConfig.EXPORT_FILE_EXCHANGE_NAME);
        } else {
            return false;
        }
    }

    /**
     * 获取需要导出的字段
     *
     * @param selectField          用户勾选的导出字段
     * @param keyList              元数据配置的字段
     * @param cellTypeContainerMap
     */
    public void getExportFileds(Map<String, Object> selectField, List<String> keyList, Map<String, CellTypeContainer> cellTypeContainerMap) {
        // 前端传值为空，则不过滤字段
        if (CollectionUtils.isEmpty(selectField)) {
            return;
        }
        Iterator<Map.Entry<String, Object>> selectFieldIterator = selectField.entrySet().iterator();
        while (selectFieldIterator.hasNext()) {
            List<String> exportFileds = (List<String>) selectFieldIterator.next().getValue();
            if (!CollectionUtils.isEmpty(exportFileds)) {
                filterExportFileds(exportFileds, keyList, cellTypeContainerMap);
            }
        }
    }

    /**
     * 从元数据中过滤出用户勾选的导出字段，单身字段不过滤
     *
     * @param exportFileds         用户勾选的导出字段
     * @param keyList              元数据配置的字段
     * @param cellTypeContainerMap
     */
    private void filterExportFileds(List<String> exportFileds, List<String> keyList, Map<String, CellTypeContainer> cellTypeContainerMap) {
        Iterator<String> iterator = keyList.iterator();
        while (iterator.hasNext()) {
            String key = iterator.next();
            CellTypeContainer cellTypeContainer = cellTypeContainerMap.get(key);
            Boolean isArray = false;
            if (Objects.nonNull(cellTypeContainer)) {
                isArray = cellTypeContainer.getArray();
            }
            // 如果用户勾选的导出字段包含元数据字段，且该字段不是单身，则剔除
            if (!exportFileds.contains(key) && Boolean.FALSE.equals(isArray)) {
                iterator.remove();
            }
        }
    }

    /**
     * 创建调用ESP接口入参
     *
     * @param downloadBaseDataParamDTO
     */
    public void getActionParas(DownloadBaseDataParamDTO downloadBaseDataParamDTO) {
        Map<String, Object> paras = downloadBaseDataParamDTO.getActionInfo().getActionParas();
        PageInfo pageInfo = downloadBaseDataParamDTO.getPageInfo();
        List<Map> sortInfo = downloadBaseDataParamDTO.getSortInfo();
        List<Map> searchInfo = downloadBaseDataParamDTO.getSearchInfo();
        if (CollectionUtils.isEmpty(paras)) {
            paras = new HashMap<>();
            downloadBaseDataParamDTO.getActionInfo().setActionParas(paras);
        }
        //处理分页信息
        if (pageInfo != null) {
            if (!pageInfo.isNextAllData()) {
                paras.put("use_has_next", true);
                if (pageInfo.getPageSize() != null) {
                    paras.put("page_size", pageInfo.getPageSize());
                }
                if (pageInfo.getPageNo() != null) {
                    paras.put("page_no", pageInfo.getPageNo());
                }
            }
        }
        //排序
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(sortInfo)) {
            List<Map> newSortInfo = new ArrayList<>();
            for (Map item : sortInfo) {
                if (org.springframework.util.StringUtils.hasText(item.getOrDefault("sortField", "").toString())) {
                    //兼容处理
                    if (item.get("sortType") != null
                            && item.get("sortType").toString().toLowerCase().contains("desc")) {
                        item.put("sortType", "desc");
                    } else {
                        item.put("sortType", "asc");
                    }
                    Map<String, Object> sortItem = new HashMap<>();
                    sortItem.put("sort_field", item.get("sortField"));
                    sortItem.put("sort_type", item.get("sortType"));
                    sortItem.put("sort_seq", item.get("sortSeq"));
                    newSortInfo.add(sortItem);
                }
            }
            paras.put("sort_info", newSortInfo);
        }
        //查询
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(searchInfo)) {
            paras.put("search_value", "");
            paras.put("search_info", searchInfo);
        }
    }

    //处理单档：只要获取单头
    //处理双档：获取head ，遍历获取body
    //调用方式：

    private Map getMetadataMap(GetActionLocaleResponseDTO metadata) {
        HashMap<String, String> map = new HashMap<>();
        String mainKey = metadata.getRequest().getParameters().get(0).getData_name();
        String mainKeyDescription = metadata.getRequest().getParameters().get(0).getDescription();
        map.put(importStatisticsDomainService.SHEET_NAME, mainKey);
        map.put(mainKey, mainKeyDescription);

        return map;
    }

    //处理单档：只要获取单头
    //处理双档：获取head ，遍历获取body
    //调用方式：

    private Map getMetadataMap(ApiDataFieldLocaleMetadataDTO mainMetadata) {
        HashMap<String, String> map = new HashMap<>();
        String mainKey = mainMetadata.getData_name();
        String mainKeyDescription = mainMetadata.getDescription();
        map.put(importStatisticsDomainService.SHEET_NAME, mainKey);
        map.put(mainKey, mainKeyDescription);

        return map;
    }

    /**
     * 发起异步下载 历史项目/任务
     *
     * @param paramMap
     * @return
     */
    public Boolean historyStartDownload(Map<String, Object> paramMap) {
        String fileName = "";
        Map<String, Object> historydDataInfoMap = MapUtils.getMap(paramMap, "historydDataInfo", new HashMap());
        String fileNameParam = MapUtils.getString(historydDataInfoMap, "fileName", "");
        if (StringUtils.isNotEmpty(fileNameParam)) {
            fileName = fileNameParam;
        } else {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
            fileName = "历史项目任务_" + sdf.format(new Date());
        }
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        ExportStatistics exportStatistics = new ExportStatistics();
        exportStatistics.setId(SnowflakeIdWorker.getInstance().newId());
        exportStatistics.setMasterId(UUID.randomUUID().toString());
        exportStatistics.setType(ImportAndExportStatisticsConstants.EXPORT_FILE_TYPE_HISTORY_DATA);
        //由于增加了历史任务项目导出，前端和业务数据导出用的同一个组件  所以后端这里取activityName作为导出的作业名称
        exportStatistics.setActivityName("{\"default\":\"历史项目任务\",\"zh_TW\":\"歷史項目任務\",\"en_US\":\"history project|task\",\"zh_CN\":\"历史项目任务\"}");
        String locale = LocaleContextHolder.getLocale().toString();
        exportStatistics.setFileName(fileName);
        exportStatistics.setLocale(locale == null ? "zh_CN" : locale);
        exportStatistics.setUserId(String.valueOf(athenaUser.getSid()));
        exportStatistics.setTenantId(athenaUser.getTenantId());
        String securityToken = AppAuthContextHolder.getContext().getSecurityToken();
        exportStatistics.setUserToken(securityToken);
        exportStatistics.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_PROCESSING);
        Date now = new Date();
        exportStatistics.setCreateTime(now);
        exportStatistics.setUpdateTime(now);
        //冗余用户对象
        SimpleUser simpleUser = new SimpleUser();
        simpleUser.setTenantId(athenaUser.getTenantId());
        simpleUser.setTenantSid(athenaUser.getTenantSid());
        simpleUser.setUserId(athenaUser.getUserId());
        simpleUser.setUserSid(athenaUser.getSid());
        exportStatistics.setUserJson(JsonUtils.objectToString(simpleUser));
        // 设置失效时间为5天后
        exportStatistics.setExpireTime(DateUtils.addDays(now, ImportAndExportStatisticsConstants.EXPORT_FILE_EXPIRE_TIME));
        ExportFileMsg exportFileMsg = new ExportFileMsg();
        exportFileMsg.setMasterId(exportStatistics.getMasterId());
        exportFileMsg.setLocale(exportStatistics.getLocale());
        exportFileMsg.setTenantId(exportStatistics.getTenantId());
        exportFileMsg.setUserToken(exportStatistics.getUserToken());
        exportFileMsg.setHistoryDownloadBaseDataParam(paramMap);

        //添加压缩密码
        Optional.ofNullable(paramMap).map(t -> t.get("defence")).map(Convert::toStr).filter(StringUtils::isNotBlank)
                .ifPresent(exportStatistics::setDefence);

        // 1、更新导入记录
        //更新对应masterId处理消息条数
        int insertResult = exportStatisticsDomainService.insertByMasterId(exportStatistics);

        // 2、发送MQ异步处理
        if (insertResult == 1) {
            return sendExportFileMQ(exportStatistics, exportFileMsg, DirectRabbitConfig.EXPORT_HISTORY_DATA_FILE_EXCHANGE_NAME);
        } else {
            return false;
        }
    }


    public void updateRetryDownloadState(String masterId) {
        exportStatisticsDomainService.updateRetryDownloadState(masterId);
    }
}

