package com.digiwin.athena.abt.application.service.abt.migration.inout;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.exception.ExcelDataConvertException;
import com.alibaba.excel.metadata.CellExtra;

import com.alibaba.excel.read.metadata.holder.ReadSheetHolder;
import com.digiwin.athena.abt.application.dto.migration.abt.api.ActionInfoDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.api.UploadParamDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.ApiDataFieldLocaleMetadataDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.DataEntryTask;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.ExcelHelperV2;
import com.digiwin.athena.abt.application.utils.ExcelUtil;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import com.digiwin.athena.abt.core.ie.dto.CellPosition;
import com.digiwin.athena.abt.core.ie.excel.convert.ConverterUtils;
import com.digiwin.athena.abt.core.ie.excel.reader.BasetExcelReadListener;

import com.digiwin.athena.abt.core.meta.constants.EntryConstant;
import com.digiwin.athena.abt.core.meta.constants.ImportAndExportStatisticsConstants;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.abt.core.meta.enums.MetaDataType;
import com.digiwin.athena.abt.core.uiils.MongoCacheUtils;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.google.common.collect.Lists;

import com.jugg.agile.framework.core.config.JaProperty;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;


import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;

import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import static com.digiwin.athena.abt.core.meta.constants.ImportAndExportStatisticsConstants.BASIC_DATA_BIZ_TYPE;
import static com.digiwin.athena.abt.core.meta.constants.RedisQueueContant.BASE_IMPORT_BATCH;
import static com.digiwin.athena.abt.core.meta.constants.RedisQueueContant.BASE_IMPORT_LOG_ENABLE;
import static com.digiwin.athena.abt.core.uiils.MongoCacheUtils.BK;
import static com.digiwin.athena.abt.core.uiils.MongoCacheUtils.P_BK;

/**
 * execl 导入解析类
 *
 * @author wzq
 */
@Slf4j
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class BasicDataReadListener extends BasetExcelReadListener<Map<Integer, Object>> {
    // ================ 配置常量 ================
    private static final String MAIN_SHEET_NAME = "sheet1";
    private static final int BATCH_COUNT = JaProperty.getInteger(BASE_IMPORT_BATCH, 300);
    private static final int FIRST_SHEET_INDEX = 0;
    private static final int FIRST_ROW_INDEX = 0;

    // ================ 依赖注入 ================
    @Autowired
    private MetaDataService metaDataService;

    @Autowired
    private ExcelHelperV2 excelHelper;

    // ================ 运行时数据容器 ================
    private final List<Map<String, Object>> cache = new ArrayList<>(BATCH_COUNT);
    @Getter
    private final AtomicLong recordCounter = new AtomicLong(0);
    private final AtomicInteger batchCounter = new AtomicInteger(0);

    // ================ Excel元数据 ================
    /**
     * 实际读取execl表格中的sheet名称,第一个必然就是主表
     */
    private final List<String> originalSheetNames = Lists.newLinkedList();
    /**
     * 根据配置转换后execl表格中的sheet名称,第一个必然就是主表
     */
    private final List<String> normalizedSheetNames = Lists.newLinkedList();
    /**
     * 实际读取execl每个sheet的表头
     */
    private final Map<String, List<String>> sheetHeaders = MapUtil.newHashMap(2, true);
    /**
     * 元数据配置
     */
    private final Map<String, List<ApiDataFieldLocaleMetadataDTO>> metadataConfig = MapUtil.newHashMap(2, true);
    /**
     * 表头名称和对应java类型映射
     */
    private final Map<String, Map<String, String>> columnTypeMapping = MapUtil.newHashMap(2, true);

    // ================ 业务关键字段 ================
    /**
     * 当前sheet业务键
     */
    private List<String> businessKeys = new ArrayList<>();
    /**
     * 父级sheet业务键
     */
    private List<String> parentBusinessKeys = new ArrayList<>();
    private Map<String, ApiDataFieldLocaleMetadataDTO> parentBKMetadataMap = null;
    private final Map<String, MetaDataType> enumMap = MapUtil.newHashMap(2);
    private final Map<String, MetaDataType> nonEnumMap = MapUtil.newHashMap(2);
    // ================ 核心元数据 ================
    private ApiDataFieldLocaleMetadataDTO mainMetadata;
    private String mainTableKey;
    @Getter
    private Date createTime = new Date();
    private String childDataSourceName;

    @Getter
    private String activityName;
    @Getter
    private String masterId = UUID.randomUUID().toString();

    // ================ 构建消息体和每个批次入库数据 ================
    @Getter
    private DataEntryTask dataEntryTask;

    public BasicDataReadListener() {
    }

    @Override
    protected boolean support(String bizType) {
        return BASIC_DATA_BIZ_TYPE.equals(bizType);
    }

    @Override
    public void invokeHeadMap(Map<Integer, String> headMap, AnalysisContext context) {
        ReadSheetHolder sheetContext = context.readSheetHolder();
        int rowIndex = sheetContext.getRowIndex();
        int sheetNo = sheetContext.getSheetNo();
        String sheetName = sheetContext.getSheetName();
        int headRowNumber = sheetContext.getHeadRowNumber();
        log.info("[基础资料导入]sheetName:[{}],sheetNo:{},rowIndex:{},headRowNumber:{}", sheetName, sheetNo, rowIndex, headRowNumber);

        //在读取第一个sheet时候需要初始化一次的数据
        if (isFirstSheetFirstRow(sheetNo, rowIndex)) {
            initializeFirstSheet();
        }

        //根据配置的表头行数进行赋值
        if (isHeaderRow(rowIndex, sheetContext)) {
            processSheetHeader(headMap, sheetName, sheetNo);
        }
    }

    @Override
    protected void processRow(Map<Integer, Object> rowData, AnalysisContext context, int index) {
        int sheetNo = context.readSheetHolder().getReadSheet().getSheetNo();
        String normalizedName = normalizedSheetNames.get(sheetNo);
        List<String> headerNameList = sheetHeaders.get(normalizedName);
        int rowIndex = context.readSheetHolder().getRowIndex() + 1;

        CellPosition cellPosition = CellPosition.builder()
                .execlFileName(((UploadParamDTO) getParam()).getFileName())
                .sheetName(context.readSheetHolder().getSheetName())
                .rowIndex(rowIndex)
                .build();
        Map<String, Object> processedRow = processRowData(cellPosition, rowData, normalizedName, headerNameList);
        processBusinessKeys(processedRow, context);

        cache.add(processedRow);

        // 主表计数
        if (isMainSheet(normalizedName)) {
            recordCounter.incrementAndGet();
        }

        // 批量持久化
        if (cache.size() >= BATCH_COUNT) {
            persistData(normalizedName);
        }
    }

    @Override
    public void processAfterAllAnalysed(AnalysisContext context) {
        //如果导入主表数据为空则抛出异常
        if (recordCounter.get() <= 0) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0052.getErrCode(), MessageUtil.getMessage("delivery.uploadError"));
        }
        Integer sheetNo = context.readSheetHolder().getReadSheet().getSheetNo();
        String sheetName = context.readSheetHolder().getReadSheet().getSheetName();

        if (sheetNo >= normalizedSheetNames.size()) {
            log.error("[基础资料导入]当前sheet数据为空:{}", sheetName);
            return;
        }

        persistData(normalizedSheetNames.get(sheetNo));
    }

    @Override
    public void processAfterException(Exception exception, AnalysisContext context) throws Exception {
        if (exception instanceof ExcelDataConvertException) {
            ExcelDataConvertException convertException = (ExcelDataConvertException) exception;
            int rowIndex = convertException.getRowIndex() + 1;
            int columnIndex = convertException.getColumnIndex() + 1;
            log.error("[基础资料导入]Conversion error at row:{}, column:{}", rowIndex, columnIndex);
        }
        throw exception;
    }

    @Override
    public void processAfterExtra(CellExtra extra, AnalysisContext context) {

    }

    // ================ 状态检查方法 ================
    private boolean isFirstSheetFirstRow(int sheetIndex, int rowIndex) {
        return sheetIndex == FIRST_SHEET_INDEX && rowIndex == FIRST_ROW_INDEX;
    }

    private boolean isHeaderRow(int rowIndex, ReadSheetHolder context) {
        return rowIndex + 1 == context.getHeadRowNumber();
    }

    private boolean isMainSheet(String sheetName) {
        return sheetName.equals(mainMetadata.getData_name());
    }

    private void processSheetHeader(Map<Integer, String> headMap, String sheetName, int sheetNo) {
        // 1. 存储原始表名
        originalSheetNames.add(sheetName);

        // 2. 标准化表名
        String normalizedName = normalizeSheetName(sheetName);
        normalizedSheetNames.add(normalizedName);

        // 3. 提取表头
        List<String> headerKeys = extractHeaders(headMap);
        sheetHeaders.put(normalizedName, headerKeys);

        // 4. 加载元数据
        UploadParamDTO uploadParam = (UploadParamDTO) getParam();
        excelHelper.getSheetMetadataMap(metadataConfig, normalizedName, sheetNo, mainMetadata, uploadParam);

        // 5. 验证并初始化业务键
        List<ApiDataFieldLocaleMetadataDTO> currentMetadata = metadataConfig.get(normalizedName);
        validateCurrentMetadata(currentMetadata, normalizedName);

        // 6. 初始化业务键
        List<ApiDataFieldLocaleMetadataDTO> currentFields = metadataConfig.get(normalizedName);
        businessKeys = getBusinessKeys(currentFields);
        parentBKMetadataMap = findParentWithBusinessKey(mainMetadata, normalizedName);
        parentBusinessKeys = new ArrayList<>(parentBKMetadataMap.keySet());

        // 7. 验证表头
        List<String> errorListHolder = new LinkedList<>();
        if (!validateNew(currentFields, headerKeys, parentBusinessKeys, errorListHolder)) {
            log.error("[基础资料导入]未匹配的列:errorList:{}", errorListHolder);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0053.getErrCode(), MessageUtil.getMessage("delivery.verifyTableError"));
        }

        // 8. 初始化字段类型映射
        initFieldTypeMap(enumMap, nonEnumMap, currentFields, parentBKMetadataMap);

        // 9. 设置上下级数据源
//        parentDataSourceName = findParentWithDataName(mainMetadata, normalizedName);
        childDataSourceName = findChildrenWithDataName(mainMetadata, normalizedName);

        if (JaProperty.getBoolean(BASE_IMPORT_LOG_ENABLE, false)) {
            log.info("[基础资料导入]originalSheetNames     :{}", JsonUtils.objectToString(originalSheetNames));
            log.info("[基础资料导入]normalizedSheetNames   :{}", JsonUtils.objectToString(normalizedSheetNames));
            log.info("[基础资料导入]sheetHeaders           :{}", JsonUtils.objectToString(sheetHeaders));
            log.info("[基础资料导入]metadataConfig         :{}", JsonUtils.objectToString(metadataConfig));
            log.info("[基础资料导入]mainMetadata           :{}", JsonUtils.objectToString(mainMetadata));
            log.info("[基础资料导入]columnTypeMapping      :{}", JsonUtils.objectToString(columnTypeMapping));
            log.info("[基础资料导入]businessKeys           :{}", JsonUtils.objectToString(businessKeys));
            log.info("[基础资料导入]parentBusinessKeys     :{}", JsonUtils.objectToString(parentBusinessKeys));
            log.info("[基础资料导入]recordCounter          :{}", recordCounter);
            log.info("[基础资料导入]mainTableKey           :{}", mainTableKey);
            log.info("[基础资料导入]enumMap                :{}", JsonUtils.objectToString(enumMap));
            log.info("[基础资料导入]nonEnumMap             :{}", JsonUtils.objectToString(nonEnumMap));
            log.info("[基础资料导入]parentBKMetadataMap    :{}", JsonUtils.objectToString(parentBKMetadataMap));
            log.info("[基础资料导入]childDataSourceName    :{}", childDataSourceName);
        }
    }

    private void validateCurrentMetadata(List<ApiDataFieldLocaleMetadataDTO> metadata, String normalizedName) {
        if (CollectionUtils.isEmpty(metadata)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0086.getErrCode(), MessageUtil.getMessage("basedata.apiNoDF"));
        }
        Map<String, String> metaMap = metadata.stream()
                .collect(Collectors.toMap(ApiDataFieldLocaleMetadataDTO::getData_name,
                        ApiDataFieldLocaleMetadataDTO::getData_type, (item1, item2) -> item1));
        columnTypeMapping.put(normalizedName, metaMap);

        List<String> headers = sheetHeaders.get(normalizedName);
        List<String> hasChanges = headers.stream().map(metaMap::get).filter(Objects::nonNull).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(hasChanges)) {
            log.error("[基础资料导入]解析Excel的栏位为空！");
            log.error("[基础资料导入]messages:{},local:{}", MessageUtil.getMessageByLocale("upload.template.error", LocaleContextHolder.getLocale().toString()), LocaleContextHolder.getLocale().toString());
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0116.getErrCode(), MessageUtil.getMessageByLocale("upload.template.error", LocaleContextHolder.getLocale().toString()));
        }
    }

    private static List<String> extractHeaders(Map<Integer, String> headMap) {
        List<String> headerKeys = Lists.newLinkedList();
        headMap.forEach((k, v) -> headerKeys.add(v));
        return headerKeys;
    }

    private String normalizeSheetName(String sheetName) {
        String normalizedName = ExcelUtil.getSheetNameInParens(sheetName);
        if (ExcelUtil.MAIN_SHEET_NAME.equals(normalizedName)) {
            normalizedName = mainMetadata.getData_name();
        }
        return normalizedName;
    }

    private void initializeFirstSheet() {
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        log.info("[基础资料导入]athenaUser:{}", JsonUtils.objectToString(athenaUser));
        UploadParamDTO uploadParam = (UploadParamDTO) getParam();
        uploadParam.setMasterId(masterId);
        uploadParam.setAuthoredUser(uploadParam.getAuthoredUser() == null ? AppAuthContextHolder.getContext().getAuthoredUser() : uploadParam.getAuthoredUser());
        uploadParam.setSecurityToken(uploadParam.getSecurityToken() == null ? AppAuthContextHolder.getContext().getSecurityToken() : uploadParam.getSecurityToken());
        log.info("[基础资料导入]uploadParam:{}", JsonUtils.objectToString(uploadParam));

        GetActionLocaleResponseDTO getActionLocaleResponseDTO = metaDataService.getActionMetaData(uploadParam.getActionId(), athenaUser == null ? uploadParam.getAuthoredUser().getToken() : athenaUser.getToken(), LocaleContextHolder.getLocale().toString());
        mainMetadata = metaDataService.getMainMetadata(getActionLocaleResponseDTO, uploadParam.getActionInfo());
        //对配置进行校验
        if (null == mainMetadata) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0091.getErrCode(), MessageUtil.getMessage("basedata.metaDataStructError"));
        }
        // 添加判空逻辑
        if (null == uploadParam.getActionInfo()) {
            uploadParam.setActionInfo(new ActionInfoDTO());
        }
        uploadParam.getActionInfo().setSheetRequiredFiled(MapUtil.newHashMap(2));

        Map<String, String> metadataMap = getMetadataMap(mainMetadata);
        mainTableKey = metadataMap.get(MAIN_SHEET_NAME);

        activityName = metaDataService.getActivityNameByActivityId(uploadParam.getActivityId(), uploadParam.getAuthoredUser() == null ? null : uploadParam.getAuthoredUser().getToken());

    }

    private Map<String, Object> processRowData(CellPosition cellPosition, Map<Integer, Object> rawRow, String sheetName, List<String> headers) {
        Map<String, Object> result = MapUtil.newHashMap(rawRow.size(), true);

        Map<String, String> typeMap = columnTypeMapping.get(sheetName);

        rawRow.forEach((colIndex, value) -> {
            String header = headers.get(colIndex);
            String dataType = typeMap.get(header);
            cellPosition.setColumnIndex(colIndex + 1);
            Object convertedValue = ConverterUtils.convertCellData(dataType, value, cellPosition);
            result.put(header, convertedValue);
        });

        Set<String> requiredFields = ((UploadParamDTO) getParam()).getActionInfo().getSheetRequiredFiled().get(sheetName);
        handleEnumColumnNew(result, enumMap, headers, requiredFields);
        handleNonEnumColumnNew(result, nonEnumMap, headers, requiredFields);
        return result;
    }

    private void processBusinessKeys(Map<String, Object> row, AnalysisContext context) {
        // 提取业务键
        Map<String, Object> bkMap = extractKeys(row, businessKeys, context);
        Map<String, Object> pBkMap = extractKeys(row, parentBusinessKeys, context);

        // 生成索引字段
        row.put(BK, MongoCacheUtils.convertBkIndex(bkMap));
        row.put(P_BK, MongoCacheUtils.convertBkIndex(pBkMap));
    }

    private Map<String, Object> extractKeys(Map<String, Object> row, List<String> keys, AnalysisContext context) {
        Map<String, Object> result = MapUtil.newHashMap(keys.size(), true);

        //如果业务主键为空则返回
        if (CollectionUtils.isEmpty(keys)) {
            return result;
        }
        keys.forEach(key -> {
            if (row.containsKey(key) && !ObjectUtil.hasEmpty(row.get(key))) {
                result.put(key, row.get(key));
            }
        });
        return result;
    }

    private void persistData(String sheetName) {
        if (!cache.isEmpty()) {
            boolean isMainSheet = isMainSheet(sheetName);
            int batchNo = isMainSheet ? batchCounter.getAndIncrement() : 0;

            // 插入缓存数据
            MongoCacheUtils.insertCache(cache, masterId, batchNo, sheetName, childDataSourceName);

            // 处理主表特殊逻辑
            if (isMainSheet) {
                // 单头字段集合
                produceNew(batchNo, cache.size(), (UploadParamDTO) getParam());
            }

            flushCache();
            flushResult();
        }
    }

    private void produceNew(int batchNo, int totalDataSize, UploadParamDTO uploadParam) {
        if (null == dataEntryTask) {
            dataEntryTask = new DataEntryTask();
            String securityToken = AppAuthContextHolder.getContext().getSecurityToken();
            //异步处理如果当前上下文没有token尝试从参数中获取
            if (securityToken == null) {
                securityToken = uploadParam.getSecurityToken();
            }
            String locale = LocaleContextHolder.getLocale().toString();
            dataEntryTask.setLocale(locale == null ? "zh_CN" : locale);
            dataEntryTask.setKeyList(mainMetadata.getField().stream().map(ApiDataFieldLocaleMetadataDTO::getData_name).collect(Collectors.toList()));
            dataEntryTask.setMasterId(uploadParam.getMasterId());
            dataEntryTask.setActionId(uploadParam.getActionId());
            dataEntryTask.setUserToken(securityToken);
            //异步处理如果当前上下文没有tenantId尝试从参数中获取
            dataEntryTask.setTenantId(AppAuthContextHolder.getContext().getAuthoredUser() == null ? uploadParam.getAuthoredUser().getTenantId() : AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            dataEntryTask.setTableKey(mainTableKey);
            dataEntryTask.setActionInfo(uploadParam.getActionInfo());
            dataEntryTask.setApplication(uploadParam.getApplication());
            dataEntryTask.setActivityId(uploadParam.getActivityId());
            //设置文件url
            dataEntryTask.setFileUrl(uploadParam.getFileUrl());
            dataEntryTask.setType(ImportAndExportStatisticsConstants.EXPORT_FILE_TYPE_BUSINESS_DATA);
            List<DataEntryTask.BatchInfo> batchInfos = new ArrayList<>();
            batchInfos.add(createBatchInfo(batchNo, totalDataSize));
            dataEntryTask.setBatchInfos(batchInfos);
        } else {
            dataEntryTask.getBatchInfos().add(createBatchInfo(batchNo, totalDataSize));
        }
    }

    /**
     * 创建批次信息对象
     */
    private DataEntryTask.BatchInfo createBatchInfo(int batchNo, int totalDataSize) {
        DataEntryTask.BatchInfo batchInfo = new DataEntryTask.BatchInfo();
        batchInfo.setBatchNum(batchNo);
        batchInfo.setTotalDataSize(totalDataSize);
        return batchInfo;
    }

    private void flushResult() {
        if (!results.isEmpty()) {
            results.clear();
        }
    }

    private void flushCache() {
        if (!cache.isEmpty()) {
            cache.clear();
        }
    }

    private Map<String, String> getMetadataMap(ApiDataFieldLocaleMetadataDTO mainMetadata) {
        Map<String, String> map = MapUtil.newHashMap(2);

        String mainKey = mainMetadata.getData_name();
        String mainKeyDescription = mainMetadata.getDescription();
        map.put(MAIN_SHEET_NAME, mainKey);
        map.put(mainKey, mainKeyDescription);
        return map;
    }

    /**
     * 获取bk
     *
     * @param cellTypes
     * @return
     */
    private List<String> getBusinessKeys(List<ApiDataFieldLocaleMetadataDTO> cellTypes) {
        List<String> keys = new ArrayList<>();
        if (CollectionUtils.isEmpty(cellTypes)) {
            return keys;
        }
        for (ApiDataFieldLocaleMetadataDTO cellType : cellTypes) {
            if (cellType.getIs_businesskey()) {
                keys.add(cellType.getData_name());
            }
        }
        return keys;
    }

    /**
     * 获取上层BKMetaDataMap
     */
    private Map<String, ApiDataFieldLocaleMetadataDTO> findParentWithBusinessKey(ApiDataFieldLocaleMetadataDTO root, String targetDataName) {
        ApiDataFieldLocaleMetadataDTO recursive = findRecursive(root, targetDataName);

        Map<String, ApiDataFieldLocaleMetadataDTO> bkMetadataMap = MapUtil.newHashMap(2);
        if (null == recursive || CollectionUtils.isEmpty(recursive.getField())) {
            return bkMetadataMap;
        }
        for (ApiDataFieldLocaleMetadataDTO cellType : recursive.getField()) {
            if (cellType.getIs_businesskey()) {
                bkMetadataMap.put(cellType.getData_name(), cellType);
            }
        }
        return bkMetadataMap;
    }

    /**
     * 获取上层BKMetaDataMap
     */
    private ApiDataFieldLocaleMetadataDTO findRecursive(ApiDataFieldLocaleMetadataDTO current, String targetDataName) {
        if (current == null) {
            return null;
        }
        // 遍历当前对象的字段列表
        if (CollectionUtils.isNotEmpty(current.getField())) {
            for (ApiDataFieldLocaleMetadataDTO child : current.getField()) {
                if (MetaDataType.OBJECT.getName().equals(child.getData_type())) {
                    if (targetDataName.equals(child.getData_name())) {
                        return current;
                    } else {
                        return findRecursive(child, targetDataName);
                    }
                }
            }
        }
        return null;
    }

    /**
     * 获取下层dataname
     */
    private String findChildrenWithDataName(ApiDataFieldLocaleMetadataDTO root, String targetDataName) {
        ApiDataFieldLocaleMetadataDTO targetNode = findChildrenRecursive(root, targetDataName);
        if (targetNode == null || CollectionUtils.isEmpty(targetNode.getField())) {
            return null;
        }
        for (ApiDataFieldLocaleMetadataDTO child : targetNode.getField()) {
            if (MetaDataType.OBJECT.getName().equals(child.getData_type())) {
                return child.getData_name();
            }
        }
        return null;
    }

    /**
     * 获取下层BKMetaDataMap
     */
    private ApiDataFieldLocaleMetadataDTO findChildrenRecursive(ApiDataFieldLocaleMetadataDTO current, String targetDataName) {
        if (current == null) {
            return null;
        }
        // 当前节点匹配，直接返回
        if (MetaDataType.OBJECT.getName().equals(current.getData_type()) && targetDataName.equals(current.getData_name())) {
            return current;
        }
        // 深度优先遍历子节点
        if (CollectionUtils.isNotEmpty(current.getField())) {
            for (ApiDataFieldLocaleMetadataDTO child : current.getField()) {
                // 仅处理OBJECT类型节点以提高效率
                if (MetaDataType.OBJECT.getName().equals(child.getData_type())) {
                    ApiDataFieldLocaleMetadataDTO result = findChildrenRecursive(child, targetDataName);
                    if (result != null) { // 子树中找到目标则逐层返回
                        return result;
                    }
                }
            }
        }
        return null; // 未找到目标节点
    }

    public boolean validateNew(List<ApiDataFieldLocaleMetadataDTO> field, List<String> excelFieldList, List<String> businessKeyList, List<String> errorListHolder) {
        if (CollectionUtils.isEmpty(excelFieldList)) {
            return false;
        }
        //获取元数据导入字段,不需要包含单身字段,单身字段另外组织
        List<String> importFieldListMetaData = field.stream().filter(item -> !item.getIs_array()).map(item -> item.getData_name()).collect(Collectors.toList());
        Iterator<String> it = excelFieldList.iterator();
        while (it.hasNext()) {
            String item = it.next();
            //如果元数据包含excel中的当前列则将当前列加入keyList,并删除该条元数据
            if (!importFieldListMetaData.contains(item) && !businessKeyList.contains(item)) {
                errorListHolder.add(item);
            }
        }
        //如果元数据字段集合为空则说明所有入参都有匹配,否则报错
        if (!CollectionUtils.isEmpty(errorListHolder)) {
            return false;
        }
        return true;
    }

    /**
     * 获取单档、多档（单头）的元数据类型
     *
     * @param enumMap
     * @param nonEnumMap
     * @param field
     * @param parentBKMetadataMap
     */
    public void initFieldTypeMap(Map<String, MetaDataType> enumMap, Map<String, MetaDataType> nonEnumMap,
                                 List<ApiDataFieldLocaleMetadataDTO> field, Map<String, ApiDataFieldLocaleMetadataDTO> parentBKMetadataMap) {
        field.forEach(param -> {
            if (StringUtils.isEmpty(param.getEnum_key())) {
                nonEnumMap.put(param.getData_name(), MetaDataType.valueOf(param.getData_type().toUpperCase()));
            } else {
                // 包含枚举的字段
                enumMap.put(param.getData_name(), MetaDataType.valueOf(param.getData_type().toUpperCase()));
            }
        });

        if (MapUtils.isEmpty(parentBKMetadataMap)) {
            return;
        }
        // 将上层BK也添加到进来，后续也要对BK进行空值处理
        parentBKMetadataMap.forEach((bkName, bkMetadata) -> {
            if (StringUtils.isEmpty(bkMetadata.getEnum_key())) {
                nonEnumMap.put(bkName, MetaDataType.valueOf(bkMetadata.getData_type().toUpperCase()));
            } else {
                // 包含枚举的字段
                enumMap.put(bkName, MetaDataType.valueOf(bkMetadata.getData_type().toUpperCase()));
            }
        });
    }

    /**
     * 枚举类型的空值赋默认值
     *
     * @param rowData
     * @param enumMap
     * @param keyList
     * @param requiredFields
     */
    private void handleEnumColumnNew(Map rowData, Map<String, MetaDataType> enumMap, List<String> keyList, Set<String> requiredFields) {
        if (MapUtils.isEmpty(enumMap)) {
            return;
        }
        //循环处理列
        for (int i = 0; i < keyList.size(); i++) {
            String keyName = keyList.get(i);
            MetaDataType metaDataType = enumMap.get(keyList.get(i));
            if (null != metaDataType) {
                //说明是枚举类型,循环处理所有枚举列,从第二行开始遍历
                if (null == rowData.get(keyName) || rowData.get(keyName) instanceof String) {  // 添加类型判断，有可能类型为数字等非String类型
                    String cellData = null == rowData.get(keyName) ? "" : (String) rowData.get(keyName);
                    if (StringUtils.isEmpty(cellData)) {
                        // 必填字段为空,不需要赋默认值
                        if (!CollectionUtils.isEmpty(requiredFields) && requiredFields.contains(keyName)) {
                            continue;
                        }
                        //空值处理
                        switch (metaDataType) {
                            case STRING:
                                rowData.put(keyName, EntryConstant.EMPTY_STRING);
                                break;
                            case NUMERIC:
                            case NUMBER:
                                rowData.put(keyName, EntryConstant.ZERO);
                                break;
                            case DATE:
                                rowData.put(keyName, EntryConstant.DATE);
                                break;
                            case DATETIME:
                                rowData.put(keyName, EntryConstant.DATE_TIME);
                                break;
                            case TIME:
                                rowData.put(keyName, EntryConstant.TIME);
                                break;
                            case BOOLEAN:
                                rowData.put(keyName, EntryConstant.FALSE);
                                break;
                            default:
                                rowData.put(keyName, EntryConstant.EMPTY_STRING);
                                break;
                        }
                    } else {
                        int index = cellData.lastIndexOf(".");
                        // 单身有枚举类型的时候，在处理单头的时候已经解析国枚举值了，所以后续无法也不需要根据截取.判断是否为枚举
                    /*String[] codeValue = cellData.split("[.]");
                    if (codeValue.length < 2) {
                        log.error("不符合枚举类型row:{},column:{},cellValue:{}", rowData, i + 1, cellData);
                        throw BusinessException.create(ErrorCodeEnum.NUM_500_0058.getErrCode(),MessageUtil.getMessage("delivery.enumTypeError"));
                    }*/
                        String code;
                        if (index > 0) {
                            code = cellData.substring(0, index);
                        } else {
                            code = cellData;
                        }

                        switch (metaDataType) {
                            case NUMBER:
                            case NUMERIC:
                                rowData.put(keyName, Integer.parseInt(code));
                                break;
                            case STRING:
                                rowData.put(keyName, code);
                                break;
                            case BOOLEAN:
                                rowData.put(keyName, Boolean.parseBoolean(code));
                                break;
                            default:
                                throw BusinessException.create(ErrorCodeEnum.NUM_500_0059.getErrCode(), MessageUtil.getMessage("delivery.enumTypeNotSupport"));
                        }
                    }
                }
            }
        }
    }


    /**
     * 非枚举类型的空值赋默认值
     *
     * @param rowData
     * @param nonEnumMap
     * @param keyList
     * @param requiredFields
     */
    private void handleNonEnumColumnNew(Map rowData, Map<String, MetaDataType> nonEnumMap, List<String> keyList, Set<String> requiredFields) {
        if (MapUtils.isEmpty(nonEnumMap)) {
            return;
        }
        //循环处理列
        for (int i = 0; i < keyList.size(); i++) {
            String keyName = keyList.get(i);
            MetaDataType metaDataType = nonEnumMap.get(keyList.get(i));
            if (null != metaDataType) {
                //说明是非枚举类型,循环处理所有枚举列,从第二行开始遍历
                if (null == rowData.get(keyName) || rowData.get(keyName) instanceof String) {
                    String cellData = null == rowData.get(keyName) ? "" : (String) rowData.get(keyName);
                    if (StringUtils.isEmpty(cellData)) {
                        // 必填字段为空,不需要赋默认值
                        if (!CollectionUtils.isEmpty(requiredFields) && requiredFields.contains(keyName)) {
                            continue;
                        }
                        switch (metaDataType) {
                            case STRING:
                                rowData.put(keyName, EntryConstant.EMPTY_STRING);
                                break;
                            case NUMERIC:
                            case NUMBER:
                                rowData.put(keyName, EntryConstant.ZERO);
                                break;
                            case DATE:
                                rowData.put(keyName, EntryConstant.DATE);
                                break;
                            case DATETIME:
                                rowData.put(keyName, EntryConstant.DATE_TIME);
                                break;
                            case TIME:
                                rowData.put(keyName, EntryConstant.TIME);
                                break;
                            case BOOLEAN:
                                rowData.put(keyName, EntryConstant.FALSE);
                                break;
                            default:
                                rowData.put(keyName, EntryConstant.EMPTY_STRING);
                                break;
                        }
                    }
                }
            }
        }
    }
}
