package com.digiwin.athena.abt.application.service.abt.migration.inout;

import com.digiwin.athena.abt.application.dto.migration.abt.api.ExportStatisticsDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ExportBatchRecord;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ExportStatistics;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.aim.TenantService;
import com.digiwin.athena.abt.core.meta.constants.ImportAndExportStatisticsConstants;
import com.digiwin.athena.abt.infrastructure.mapper.biz.migration.abt.ExportBatchRecordMapper;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.abt.ExportBatchRecordPO;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.abt.ExportStatisticsPO;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import net.sf.json.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;

/**
 * @ClassName ExportStatisticsFactory
 * @Description 导出数据工厂类
 * @Author lisheng
 * @Date 2023/9/20 9:35
 * @Version 1.0
 **/
@Service
public class ExportStatisticsFactory {

    @Autowired
    private ExportBatchRecordMapper exportBatchRecordMapper;
    @Autowired
    private TenantService tenantService;

    private static final List<ExportStatisticsDTO> emptyExportStatisticsDTOList = Collections.emptyList();

    private final List<ExportStatistics> emptyImportStatistics = Collections.emptyList();

    public ExportStatistics getExportStatistics(ExportStatisticsPO exportStatisticsPO) {
        if (null == exportStatisticsPO) {
            return null;
        }
        ExportStatistics exportStatistics = new ExportStatistics();
        BeanUtils.copyProperties(exportStatisticsPO, exportStatistics);
        return exportStatistics;
    }

    public ExportStatisticsPO getImportStatisticsPO(ExportStatistics exportStatistics) {
        if (null == exportStatistics) {
            return null;
        }
        ExportStatisticsPO exportStatisticsPO = new ExportStatisticsPO();
        BeanUtils.copyProperties(exportStatistics, exportStatisticsPO);
        return exportStatisticsPO;
    }

    public List<ExportStatistics> getExportStatisticsList(List<ExportStatisticsPO> exportStatisticsPOS, String locale) {
        if (CollectionUtils.isEmpty(exportStatisticsPOS)) {
            return emptyImportStatistics;
        }
        List<ExportStatistics> exportStatisticsList = new LinkedList<>();
        exportStatisticsPOS.forEach(item->{
            ExportStatistics exportStatistics = new ExportStatistics();
            BeanUtils.copyProperties(item, exportStatistics);

            // 查询子表批次的数据
            List<ExportBatchRecordPO> batchRecordPOList = exportBatchRecordMapper.listByMasterId(item.getMasterId(), null);
            List<ExportBatchRecord> exportBatchRecordList = new ArrayList<>();
            // 缓存 tenantId -> token 映射，避免重复查询
            Map<String, String> tenantTokenCache = new HashMap<>();
            for (ExportBatchRecordPO po : batchRecordPOList) {
                ExportBatchRecord record = new ExportBatchRecord();
                BeanUtils.copyProperties(po, record);
                // 处理名称的多语言
                if (StringUtils.isNotBlank(po.getName())) {
                    // 有的名称可能不是多语言别的，直接取前端传的值
                    try {
                        Map<String, String> nameLocaleMap = JsonUtils.jsonToObject(
                                po.getName(),
                                new TypeReference<Map<String, String>>() {
                                }
                        );
                        String activityName = StringUtils.isNotBlank(locale)
                                ? nameLocaleMap.getOrDefault(locale, nameLocaleMap.get("zh_CN"))
                                : nameLocaleMap.get("zh_CN");
                        record.setActivityName(activityName);
                    } catch (Exception e) {
                        record.setActivityName(po.getName());
                    }
                }
                // 处理代理 token，使用缓存避免重复查询
                String proxyToken = po.getProxyToken();
                if (StringUtils.isNotBlank(proxyToken)) {
                    String tenantId = po.getTenantId();
                    String token = tenantTokenCache.get(tenantId);
                    if (token == null) {
                        JSONObject jsonObject = tenantService.queryApiVirtualToken(tenantId);
                        if (!jsonObject.isEmpty() && jsonObject.containsKey("token")) {
                            token = jsonObject.getString("token");
                            tenantTokenCache.put(tenantId, token);
                        }
                    }
                    if (token != null) {
                        record.setProxyToken(token);
                    }
                }
                exportBatchRecordList.add(record);
            }
            exportStatistics.setExportBatchRecords(exportBatchRecordList);
            exportStatisticsList.add(exportStatistics);
        });
        return exportStatisticsList;
    }

    public static List<ExportStatisticsDTO> exportStatisticsListToDTO(List<ExportStatistics> exportStatisticsList) {
        if (CollectionUtils.isEmpty(exportStatisticsList)) {
            return emptyExportStatisticsDTOList;
        }
        List<ExportStatisticsDTO> exportStatisticsDTOList = new LinkedList<>();
        Date now = new Date();
        exportStatisticsList.forEach(item->{
            ExportStatisticsDTO exportStatisticsDTO = new ExportStatisticsDTO();
            BeanUtils.copyProperties(item, exportStatisticsDTO);
            LocalDateTime startTime = exportStatisticsDTO.getCreateTime().toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
            LocalDateTime endTime = exportStatisticsDTO.getUpdateTime().toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
            // 设置文件是否有效
            if(Objects.nonNull(exportStatisticsDTO.getExpireTime()) && now.after(exportStatisticsDTO.getExpireTime())){
                exportStatisticsDTO.setFileState(ImportAndExportStatisticsConstants.FILE_STATE_INVALID);
            }else {
                exportStatisticsDTO.setFileState(ImportAndExportStatisticsConstants.FILE_STATE_VALID);
            }
            if (2 == exportStatisticsDTO.getState()) {
                endTime = LocalDateTime.now();
            }
            // 计算消耗时间，如果如果是0的话 改为1
            String consumeTime = String.valueOf(Duration.between(startTime, endTime).getSeconds());
            if ("0".equals(consumeTime)) {
                consumeTime = "1";
            }
            exportStatisticsDTO.setConsumeTime(consumeTime);

            List<ExportBatchRecord> exportBatchRecords = item.getExportBatchRecords();
            if (!CollectionUtils.isEmpty(exportBatchRecords)) {
                for (ExportBatchRecord batchRecord : exportBatchRecords) {
                    LocalDateTime batchRecordStartTime = batchRecord.getCreateTime()
                            .toInstant()
                            .atZone(ZoneId.systemDefault())
                            .toLocalDateTime();

                    LocalDateTime batchRecordEndTime = (batchRecord.getState() == 2)
                            ? LocalDateTime.now()
                            : batchRecord.getUpdateTime()
                            .toInstant()
                            .atZone(ZoneId.systemDefault())
                            .toLocalDateTime();

                    long seconds = Duration.between(batchRecordStartTime, batchRecordEndTime).getSeconds();
                    int batchRecordConsumeTime = (int) Math.max(seconds, 1);

                    batchRecord.setConsumeTime(String.valueOf(batchRecordConsumeTime));
                    batchRecord.setFileState(exportStatisticsDTO.getFileState());
                }
                exportStatisticsDTO.setExportBatchRecords(exportBatchRecords);
            }
            exportStatisticsDTOList.add(exportStatisticsDTO);
        });
        return exportStatisticsDTOList;

    }

    public static ExportStatisticsDTO exportStatisticsToDTO(ExportStatistics exportStatistics) {
        if (Objects.isNull(exportStatistics)) {
            return new ExportStatisticsDTO();
        }
        ExportStatisticsDTO exportStatisticsDTO = new ExportStatisticsDTO();
        BeanUtils.copyProperties(exportStatistics, exportStatisticsDTO);
        Date now = new Date();
        // 设置文件是否有效
        if(Objects.nonNull(exportStatisticsDTO.getExpireTime()) && now.after(exportStatisticsDTO.getExpireTime())){
            exportStatisticsDTO.setFileState(ImportAndExportStatisticsConstants.FILE_STATE_INVALID);
        }else {
            exportStatisticsDTO.setFileState(ImportAndExportStatisticsConstants.FILE_STATE_VALID);
        }
        return exportStatisticsDTO;
    }
}
