package com.digiwin.athena.abt.application.service.abt.migration.inout;

import cn.hutool.core.map.MapUtil;
import com.digiwin.athena.abt.application.dto.migration.abt.api.ActionInfoDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.api.DapResponse;
import com.digiwin.athena.abt.application.dto.migration.abt.api.DownloadTemplateDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.api.ExportHeaderRespDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.MultiLanguageDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.DataEntryTask;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.validation.DataValidationEnum;
import com.digiwin.athena.abt.application.service.abt.migration.inout.validate.ValidatorLoader;
import com.digiwin.athena.abt.core.meta.dto.CellTypeContainer;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.ApiDataFieldLocaleMetadataDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.GetActionResponseLocaleRequestDTO;
import com.digiwin.athena.abt.core.meta.enums.MetaDataType;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.ExcelHelper;
import com.digiwin.athena.abt.application.utils.ExcelUtil;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import com.digiwin.athena.abt.core.meta.constants.ImportAndExportStatisticsConstants;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.abt.core.meta.dto.Dictionary;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.validation.DataValidation;

import javax.annotation.Resource;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.digiwin.athena.abt.core.meta.constants.ContainerConstant.REQUIRED_STR_FOR_TRUE;

/**
 * @ClassName MetaDataService
 * @Description 元数据service
 * @Author zhuangli
 * @Date 2021/4/8 11:16
 * @Version 1.0
 **/
@Slf4j
@Service
public class MetaDataService {

    @Resource
    RestTemplate restTemplate;

    @Autowired
    ExcelHelper excelHelper;

    @Value("${themeMap.uri}")
    private String knowledgeGraphUrl;

    private final String HEADER_KEY = "key";
    private final String HEADER_NAME = "name";
    private final String METADATA_URL = "/restful/service/knowledgegraph/Action/Metadata?actionId={actionId}";
    private final String DICT_BATCH_URL = "/restful/service/knowledgegraph/Dict/FindDictByKeyList";
    private final String DICT_URL = "/restful/service/knowledgegraph/Dict/DictByKey?key={key}";
    private final String FILE_EXIST_URL = "/restful/service/knowledgegraph/File/FileId?key={key}&version={version}";
    private final String TEMPLATE_DOWNLOAD_URL = "/restful/service/knowledgegraph/File/DownloadTemplate?fileId={fileId}";
    private final String PRODUCT_NAME_URL = "/restful/service/knowledgegraph/Action/ProductName?serviceName={serviceName}";
    private final String ACTIVITY_NAME_URL = "/restful/service/knowledgegraph/task/activityNamesByCode?activityCode={activityCode}";

    public boolean validate(GetActionLocaleResponseDTO getActionLocaleResponseDTO, List<List<Object>> data, List<String> keyList, List<List> errorListHolder) {
        //获取元数据导入字段,不需要包含单身字段,单身字段另外组织
        List<String> importFieldListMetaData = transformToImportFieldList(getActionLocaleResponseDTO);
        //获取excel中的header
        List<String> excelFieldList = getImportFieldListFromExcel(data);
        Iterator<String> it = excelFieldList.iterator();
        while (it.hasNext()) {
            String item = it.next();
            //如果元数据包含excel中的当前列则将当前列加入keyList,并删除该条元数据
            if (importFieldListMetaData.contains(item)) {
                keyList.add(item);
                importFieldListMetaData.remove(item);
                it.remove();
            }
        }
        //如果元数据字段集合为空则说明所有入参都有匹配,否则报错
        if (!importFieldListMetaData.isEmpty()) {
            errorListHolder.add(importFieldListMetaData);
            return false;
        }
        return true;
    }

    public boolean validateBody(String bodyKey, Set<String> businessKeySet, GetActionLocaleResponseDTO getActionLocaleResponseDTO,
                                List<List<Object>> data, List<String> keyList, List<List> errorListHolder) {
        Set businessKeySetCopy = new HashSet(businessKeySet);
        //获取元数据导入字段,不需要包含单身字段,单身字段另外组织
        List<String> importFieldListMetaData = transformBodyToImportFieldList(bodyKey, getActionLocaleResponseDTO);
        //获取excel中的header
        List<String> excelFieldList = getImportFieldListFromExcel(data);
        Iterator<String> it = excelFieldList.iterator();
        while (it.hasNext()) {
            String item = it.next();
            //如果元数据或者businessKey包含excel中的当前列则将当前列加入keyList,并删除该条元数据
            if (importFieldListMetaData.contains(item) || businessKeySetCopy.contains(item)) {
                keyList.add(item);
                importFieldListMetaData.remove(item);
                businessKeySetCopy.remove(item);
                it.remove();
            }
        }
        //如果元数据字段集合为空则说明所有入参都有匹配,否则报错
        if ((importFieldListMetaData.isEmpty() && businessKeySetCopy.isEmpty())) {
            return true;
        } else {
            errorListHolder.add(importFieldListMetaData);
            errorListHolder.add(new ArrayList<>(businessKeySetCopy));
            return false;
        }
    }

    private List<String> getImportFieldListFromExcel(List<List<Object>> data) {
        List<Object> headers = data.get(1);
        List<String> headerList = new LinkedList<>();
        headers.forEach(item -> {
            // 只取字符串类型的栏位,解决bug: 72204
            if (item instanceof String) {
                headerList.add((String) item);
            }
        });
        return headerList;
    }

    public boolean validateNew(ApiDataFieldLocaleMetadataDTO cellTypeContainer, List<String> excelFieldList, List<String> businessKeyList, List<String> errorListHolder) {
        if (CollectionUtils.isEmpty(excelFieldList)) {
            return false;
        }
        //获取元数据导入字段,不需要包含单身字段,单身字段另外组织
        List<String> importFieldListMetaData = cellTypeContainer.getField().stream().filter(item -> !item.getIs_array()).map(item -> item.getData_name()).collect(Collectors.toList());
        Iterator<String> it = excelFieldList.iterator();
        while (it.hasNext()) {
            String item = it.next();
            //如果元数据包含excel中的当前列则将当前列加入keyList,并删除该条元数据
            if (!importFieldListMetaData.contains(item) && !businessKeyList.contains(item)) {
                errorListHolder.add(item);
            }
        }
        //如果元数据字段集合为空则说明所有入参都有匹配,否则报错
        if (!CollectionUtils.isEmpty(errorListHolder)) {
            return false;
        }
        return true;
    }

    private List<String> transformBodyToImportFieldList(String bodyKey, GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getRequest().getParameters().get(0);
        if (null == apiDataFieldLocaleMetadataDTO) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0092.getErrCode(), MessageUtil.getMessage("basedata.metaDataStructError"));
        }
        List<ApiDataFieldLocaleMetadataDTO> bodyApiDataFieldLocaleMetadataDTO = apiDataFieldLocaleMetadataDTO.getField()
                .stream().filter(item -> item.getData_name().equals(bodyKey)).collect(Collectors.toList());
        if (1 != bodyApiDataFieldLocaleMetadataDTO.size()) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0093.getErrCode(), MessageUtil.getMessage("basedata.metaDataStructError"));
        }
        //只返回非单身字段
        return bodyApiDataFieldLocaleMetadataDTO.get(0).getField().stream().filter(item -> !item.getIs_array()).map(item -> item.getData_name()).collect(Collectors.toList());
    }

    private List<String> transformToImportFieldList(GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getRequest().getParameters().get(0);
        if (null == apiDataFieldLocaleMetadataDTO) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0091.getErrCode(), MessageUtil.getMessage("basedata.metaDataStructError"));
        }
        //只返回非单身字段
        return apiDataFieldLocaleMetadataDTO.getField().stream().filter(item -> !item.getIs_array()).map(item -> item.getData_name()).collect(Collectors.toList());
    }

    /**
     * 校验数据，保留校验错误的数据行，删除校验正确的数据行
     *
     * @param dataEntryTask
     * @return 返回校验错误的数据行
     */
    public List<Map> validateByBatch(DataEntryTask dataEntryTask, List<CellTypeContainer> cellTypes) {
        if (null == dataEntryTask.getActionInfo() || MapUtils.isEmpty(dataEntryTask.getActionInfo().getSheetRequiredFiled())) {
            return new LinkedList<>();
        }
        return doValidateData(dataEntryTask.getTableKey(), dataEntryTask.getDataList(), dataEntryTask, cellTypes);
    }

    /**
     * 校验数据行，并返回校验结果
     *
     * @param dataKey
     * @param rowDataList
     * @param dataEntryTask
     * @return
     */
    private List<Map> doValidateData(String dataKey, List<Map> rowDataList, DataEntryTask dataEntryTask, List<CellTypeContainer> cellTypes) {
        Map<String, Set<String>> sheetRequiredFiled = dataEntryTask.getActionInfo().getSheetRequiredFiled();
        Map<String, CellTypeContainer> cellTypeContainerMap = cellTypes.stream().collect(Collectors.toMap(CellTypeContainer::getKeyName, item -> item));
        // 该行以及下层（单身、子单身等）数据中是否存在必填项为空的单元格
        List<Map> errorRowDataList = new ArrayList<>();
        Set<String> requiredFiled = sheetRequiredFiled.get(dataKey);
        Iterator<Map> rowDataListIterator = rowDataList.iterator();
        while (rowDataListIterator.hasNext()) {
            Map rowData = rowDataListIterator.next();
            Map rowDataErrorMsg = new LinkedHashMap<>();
            // 该行数据中是否存在必填项为空的单元格
            Boolean parentHasError = Boolean.FALSE;
            // 下层（单身、子单身等）数据中是否存在必填项为空的单元格
            Boolean sonHasError = Boolean.FALSE;
            Iterator<Map.Entry<String, Object>> rowDataIterator = rowData.entrySet().iterator();
            while (rowDataIterator.hasNext()) {
                Map.Entry<String, Object> cellEntry = rowDataIterator.next();
                CellTypeContainer cellType = cellTypeContainerMap.get(cellEntry.getKey());
                // 1、先递归处理单身、子单身数据
                if (cellEntry.getValue() instanceof List) {
                    List<Map> sonErrorDataList = doValidateData(cellEntry.getKey(), (List<Map>) cellEntry.getValue(), dataEntryTask, cellType.getChildren());
                    if (!CollectionUtils.isEmpty(sonErrorDataList)) {
                        sonHasError = Boolean.TRUE;
                        rowDataErrorMsg.put(cellEntry.getKey(), sonErrorDataList);
                    }
                }

                String errorMsg = null;
                for (DataValidationEnum dataValidationEnum : DataValidationEnum.values()) {
                    DataValidation dataValidation = DataValidationEnum.getDataValidation(dataValidationEnum.getType());
                    // 2、先校验单元格必填字段是否为空
                    if (DataValidationEnum.REQUIRED.getType().equals(dataValidationEnum.getType())) {
                        errorMsg = dataValidation.validate(requiredFiled, cellEntry, dataEntryTask.getLocale());
                        if (StringUtils.isNotEmpty(errorMsg)) {
                            parentHasError = Boolean.TRUE;
                            // 添加单元格错误信息
                            rowDataErrorMsg.put(cellEntry.getKey() + ExcelHelper.ERROR_FIELD_SUFFIX, errorMsg);
                            break;
                        }
                        // 3、校验数据格式
                    } else if (DataValidationEnum.FORMAT.getType().equals(dataValidationEnum.getType())) {
                        errorMsg = dataValidation.validate(cellType, cellEntry, dataEntryTask.getLocale());
                        if (StringUtils.isNotEmpty(errorMsg)) {
                            parentHasError = Boolean.TRUE;
                            // 添加单元格错误信息
                            rowDataErrorMsg.put(cellEntry.getKey() + ExcelHelper.ERROR_FIELD_SUFFIX, errorMsg);
                            break;
                        }
                    } else {
                        log.error("DataValidationEnum is wrong, dataValidationEnum:{}", dataValidationEnum);
                    }
                }
            }
            if (parentHasError || sonHasError) {
                Map errorRow = new LinkedHashMap();
                errorRow.putAll(rowData);
                // 将错误信息添加到数据行
                errorRow.putAll(rowDataErrorMsg);
                errorRowDataList.add(errorRow);
            }
        }
        return errorRowDataList;
    }

    /**
     * 判断单元格是否为空
     *
     * @param requiredFiled
     * @param cellEntry
     * @return
     */
    private Boolean isEmpty(Set<String> requiredFiled, Map.Entry<String, Object> cellEntry) {
        if (CollectionUtils.isEmpty(requiredFiled) || !requiredFiled.contains(cellEntry.getKey())) {
            return false;
        }

        if (null == cellEntry.getValue()) {
            return true;
        }

        if (cellEntry.getValue() instanceof String) {
            return ((String) cellEntry.getValue()).trim().isEmpty();
        }
        return false;
    }

    public String getProductNameBySecurityToken(String token, String serviceName, String securityToken) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        headers.add("security-token", securityToken);
        Map<String, String> parameters = new HashMap<>();
        parameters.put("serviceName", serviceName);
        HttpEntity<Map<String, String>> entity = new HttpEntity<>(headers);
        ResponseEntity<DapResponse> result = restTemplate.exchange(knowledgeGraphUrl + PRODUCT_NAME_URL, HttpMethod.GET, entity, DapResponse.class, parameters);
        if (result.getStatusCodeValue() == 200 && (result.getBody().getStatus() == 200)) {
            String productName = (String) result.getBody().getResponse();
            return productName;
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0104.getErrCode(), "未找到元数据");
        }
    }

    public String getProductName(String token, String serviceName) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        Map<String, String> parameters = new HashMap<>();
        parameters.put("serviceName", serviceName);
        HttpEntity<Map<String, String>> entity = new HttpEntity<>(headers);
        ResponseEntity<DapResponse> result = restTemplate.exchange(knowledgeGraphUrl + PRODUCT_NAME_URL, HttpMethod.GET, entity, DapResponse.class, parameters);
        if (result.getStatusCodeValue() == 200 && (result.getBody().getStatus() == 200)) {
            String productName = (String) result.getBody().getResponse();
            return productName;
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0094.getErrCode(), "未找到元数据");
        }
    }

    public GetActionLocaleResponseDTO getActionMetaDataBySecurityToken(String actionId, String token, String locale, String securityToken) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        headers.add("security-token", securityToken);
        if (!StringUtils.isEmpty(locale)) {
            headers.add("locale", locale);
        }
        Map<String, String> parameters = new HashMap<>();
        parameters.put("actionId", actionId);
        HttpEntity<Map<String, String>> entity = new HttpEntity<>(headers);
        ResponseEntity<DapResponse> result = restTemplate.exchange(knowledgeGraphUrl + METADATA_URL, HttpMethod.GET, entity, DapResponse.class, parameters);
        if (result.getStatusCodeValue() == 200 && (result.getBody().getStatus() == 200)) {
            Map dtoMap = (Map) result.getBody().getResponse();
            String str = JsonUtils.objectToString(dtoMap);
            GetActionLocaleResponseDTO getActionLocaleResponseDTO = JsonUtils.jsonToObject(str, GetActionLocaleResponseDTO.class);
            return getActionLocaleResponseDTO;
        } else {
//            String prodName = espService.getTenantProductList(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId(),actionId);
            String errorCode = ErrorCodeEnum.NUM_500_0105.getErrCode();
//            if (StringUtils.isNotEmpty(prodName)){
//                errorCode = "P." + prodName + ".500.0105";
//            }
            throw BusinessException.create(errorCode, "调用" + knowledgeGraphUrl + "/restful/service/knowledgegraph/Action/Metadata?actionId=" + actionId + "未找到元数据");
        }
    }

    public GetActionLocaleResponseDTO getActionMetaData(String actionId, String token, String locale) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        if (!StringUtils.isEmpty(locale)) {
            headers.add("locale", locale);
        }
        Map<String, String> parameters = new HashMap<>();
        parameters.put("actionId", actionId);
        HttpEntity<Map<String, String>> entity = new HttpEntity<>(headers);
        ResponseEntity<DapResponse> result = restTemplate.exchange(knowledgeGraphUrl + METADATA_URL, HttpMethod.GET, entity, DapResponse.class, parameters);
        if (result.getStatusCodeValue() == 200 && (result.getBody().getStatus() == 200)) {
            Map dtoMap = (Map) result.getBody().getResponse();
            String str = JsonUtils.objectToString(dtoMap);
            GetActionLocaleResponseDTO getActionLocaleResponseDTO = JsonUtils.jsonToObject(str, GetActionLocaleResponseDTO.class);
            return getActionLocaleResponseDTO;
        } else {
//            String prodName = espService.getTenantProductList(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId(),actionId);
            String errorCode = ErrorCodeEnum.NUM_500_0105.getErrCode();
//            if (StringUtils.isNotEmpty(prodName)){
//                errorCode = prodName;
//            }
            throw BusinessException.create(errorCode, "调用" + knowledgeGraphUrl + "/restful/service/knowledgegraph/Action/Metadata?actionId=" + actionId + "未找到元数据");
        }
    }

    public List<Dictionary> getDictByKeyBySecurityToken(String key, String token, String securityToken, String locale) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        if (!StringUtils.isEmpty(locale)) {
            headers.add("locale", locale);
        }
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("key", key);
        HttpEntity<Map<String, Object>> entity = new HttpEntity<>(parameters, headers);
        ResponseEntity<DapResponse> result = restTemplate.exchange(knowledgeGraphUrl + DICT_URL, HttpMethod.GET, entity, DapResponse.class, parameters);
        if (result.getStatusCodeValue() == 200 && (result.getBody().getStatus() == 200)) {
            List dtoMap = (List) result.getBody().getResponse();
            String str = JsonUtils.objectToString(dtoMap);
            List<Dictionary> dictionaries = JsonUtils.jsonToListObject(str, Dictionary.class);
            return dictionaries;
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0097.getErrCode(), "未找到元数据");
        }
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 根据key列表查询字典
     * @Date 9:38 2022/1/6
     * @Param
     **/
    public List<Dictionary> getDictByKey(String key, String token, String locale) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        if (!StringUtils.isEmpty(locale)) {
            headers.add("locale", locale);
        }
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("key", key);
        HttpEntity<Map<String, Object>> entity = new HttpEntity<>(parameters, headers);
        ResponseEntity<DapResponse> result = restTemplate.exchange(knowledgeGraphUrl + DICT_URL, HttpMethod.GET, entity, DapResponse.class, parameters);
        if (result.getStatusCodeValue() == 200 && (result.getBody().getStatus() == 200)) {
            List dtoMap = (List) result.getBody().getResponse();
            String str = JsonUtils.objectToString(dtoMap);
            List<Dictionary> dictionaries = JsonUtils.jsonToListObject(str, Dictionary.class);
            return dictionaries;
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0098.getErrCode(), "未找到元数据");
        }
    }

    public Map<String, List<Dictionary>> getDictByKeyList(List<String> keyList, String token, String securityToken, String locale) {
        if (CollectionUtils.isEmpty(keyList)) {
            return Collections.EMPTY_MAP;
        }
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        if (!StringUtils.isEmpty(securityToken)) {
            headers.add("security-token", securityToken);
        }
        if (!StringUtils.isEmpty(locale)) {
            headers.add("locale", locale);
        }
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("keyList", keyList);
        HttpEntity<Map<String, Object>> entity = new HttpEntity<>(parameters, headers);
        ResponseEntity<DapResponse> result = restTemplate.postForEntity(knowledgeGraphUrl + DICT_BATCH_URL, entity, DapResponse.class);
        if (result.getStatusCodeValue() == 200 && (result.getBody().getStatus() == 200)) {
            Map dtoMap = (Map) result.getBody().getResponse();
            if (CollectionUtils.isEmpty(dtoMap)) {
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0096.getErrCode(), "未找到元数据");
            }
            String str = JsonUtils.objectToString(dtoMap);
            Map<String, List<Dictionary>> dictionaryMaps = JsonUtils.jsonToObject(str, Map.class);
            return dictionaryMaps;
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0096.getErrCode(), "未找到元数据");
        }
    }

    public GetActionLocaleResponseDTO getActionMetaDataBySecurityToken(String actionId, String token, String securityToken) {
        return getActionMetaDataBySecurityToken(actionId, token, "", securityToken);
    }

    public GetActionLocaleResponseDTO getActionMetaData(String actionId, String token) {
        return getActionMetaData(actionId, token, "");
    }

    public String getTableKey(GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        return getActionLocaleResponseDTO.getRequest().getParameters().get(0).getData_name();
    }

    public Boolean useHasNext(GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        if (Objects.isNull(getActionLocaleResponseDTO) || Objects.isNull(getActionLocaleResponseDTO.getRequest())) {
            return false;
        }
        List<ApiDataFieldLocaleMetadataDTO> parameters = getActionLocaleResponseDTO.getRequest().getParameters();
        for (ApiDataFieldLocaleMetadataDTO item : parameters) {
            if (item.getData_name().equals(ImportAndExportStatisticsConstants.USE_HAS_NEXT)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 根据actionId获取excel列名
     * @Date 13:46 2021/12/1
     * @Param
     **/
    public List<Map> getHeaders(String actionId, String userToken, String locale) {
        GetActionLocaleResponseDTO getActionLocaleResponseDTO = getActionMetaData(actionId, userToken, locale);
        return getHeaders(getActionLocaleResponseDTO);
    }

    public List<Map> getHeaders(GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getRequest().getParameters().get(0);
        List<Map> headers = new LinkedList<>();
        apiDataFieldLocaleMetadataDTO.getField().forEach(item -> {
            Map header = new HashMap(2);
            header.put(HEADER_KEY, item.getData_name());
            header.put(HEADER_NAME, item.getDescription());
            headers.add(header);
        });
        return headers;
    }

    public Map<String, String> getHeaderMap(ApiDataFieldLocaleMetadataDTO mainMetadata) {
        Map<String, String> headerMap = new HashMap<>();
        mainMetadata.getField().forEach(
                item -> headerMap.put(item.getData_name(), item.getDescription()));
        return headerMap;
    }

    public Map<String, String> getResponseHeaders(GetActionLocaleResponseDTO getActionLocaleResponseDTO, List<String> keyList) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getResponse().getData();
        Map<String, String> headerMap = new HashMap<>();
        apiDataFieldLocaleMetadataDTO.getField().forEach(item -> {
            // 解决无法导出单身问题 bug:63701
            keyList.add(item.getData_name());
            headerMap.put(item.getData_name(), item.getDescription());
        });
        return headerMap;
    }

    public List<String> getResponseBKList(GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        List<String> keyList = new LinkedList<>();
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getResponse().getData();
        apiDataFieldLocaleMetadataDTO.getField().forEach(item -> {
            if (item.getIs_businesskey()) {
                keyList.add(item.getData_name());
            }
        });
        return keyList;
    }

    public List<String> getResponseBodyKeyList(GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        List<String> keyList = new LinkedList<>();
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getResponse().getData();
        apiDataFieldLocaleMetadataDTO.getField().forEach(item -> {
            if (item.getIs_array()) {
                keyList.add(item.getData_name());
            }
        });
        return keyList;
    }

    public List<Map> getHeaders(String actionId, String userToken) {
        return getHeaders(actionId, userToken, "");
    }

    public String getActivityNameByActivityId(String activityId, String token) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        //如果上下文中token为空，则使用传入的token
        headers.add("token", token == null ? AppAuthContextHolder.getContext().getAuthoredUser().getToken() : token);
        Map<String, String> parameters = new HashMap<>();
        parameters.put("activityCode", activityId);
        HttpEntity<Map<String, String>> entity = new HttpEntity<>(headers);
        ResponseEntity<DapResponse> result = restTemplate.exchange(knowledgeGraphUrl + ACTIVITY_NAME_URL, HttpMethod.GET, entity, DapResponse.class, parameters);
        if (result.getStatusCodeValue() == 200 && (result.getBody().getStatus() == 200)) {
            Map dtoMap = (Map) result.getBody().getResponse();
            return JsonUtils.objectToString(dtoMap);
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0103.getErrCode(), "调用" + knowledgeGraphUrl + "/restful/service/knowledgegraph/task/activityNamesByCode?activityCode=" + activityId + "获取活动名称异常，请检查配置");
        }
    }

    public ExportHeaderRespDTO getTableHeader(String actionId, String token, String locale) {
        // 获取元数据
        GetActionLocaleResponseDTO metadataDTO = getActionMetaDataBySecurityToken(actionId, token, locale, token);
        GetActionLocaleResponseDTO getActionLocaleResponseDTO = getActionMetaData(actionId, token, locale);

        // 获取主元数据
        ApiDataFieldLocaleMetadataDTO mainMetadata = getMainMetadata(getActionLocaleResponseDTO, null);

        // 获取表头信息
        List<String> keyList = new ArrayList<>();
        Map<String, String> headers = getResponseHeaders(metadataDTO, keyList);

        // 构造表头字段列表
        List<ExportHeaderRespDTO.Cell> cells = headers.entrySet().stream()
                .map(entry -> {
                    ExportHeaderRespDTO.Cell cellDTO = new ExportHeaderRespDTO.Cell();
                    cellDTO.setKeyName(entry.getKey());
                    cellDTO.setKeyDescription(entry.getValue());
                    return cellDTO;
                })
                .collect(Collectors.toList());

        // 构造返回对象
        ExportHeaderRespDTO responseDTO = new ExportHeaderRespDTO();
        responseDTO.setMainKey(mainMetadata.getData_name());
        responseDTO.setCell(cells);
        return responseDTO;
    }

    public byte[] downloadTemplate(String actionId, String token, String locale, DownloadTemplateDTO downloadTemplateDTO) {
        byte[] fileBytes = new byte[0];
        //判断文档中心是否存在自定义模板
        String templateFileId = getTemplateFileId(actionId, locale);
        if (!StringUtils.isEmpty(templateFileId)) {
            //文档中心存在自定义模板,从文档中心下载模板
            fileBytes = downloadByTemplateFileId(templateFileId);
            return fileBytes;
        } else {
            //文档中心没有自定义模板,则根据api元数据生成
            GetActionLocaleResponseDTO getActionLocaleResponseDTO = getActionMetaData(actionId, token, locale);
            ApiDataFieldLocaleMetadataDTO mainMetadata = getMainMetadata(getActionLocaleResponseDTO, downloadTemplateDTO.getActionInfo());
            List<CellTypeContainer> cellTypes = getCellTypeContainers(mainMetadata.getField(), token, null, locale, 1);
            List<CellTypeContainer> businessKeyContainer = getBusinessKeyContainer(cellTypes);
            //Workbook wb;
            Sheet sheet;
            XSSFWorkbook wb = new XSSFWorkbook();
            /*创建表单*/
            String mainKey = mainMetadata.getData_name();
            String mainKeyDescription = mainMetadata.getDescription();
            sheet = wb.createSheet(ExcelHelper.getSheetName(mainKey, mainKeyDescription, true));
            Map<String, Object> annotation = getAnnotation(downloadTemplateDTO.getActionInfo());
            createSheetHeader(wb, sheet, businessKeyContainer, cellTypes, locale, downloadTemplateDTO.getRequiredFields(), annotation);
            try {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                wb.write(bos);
                fileBytes = bos.toByteArray();
            } catch (Exception e) {
                log.error("下载导入模板文件异常", e);
            } finally {
                if (null != wb) {
                    try {
                        wb.close();
                    } catch (IOException e) {
                        log.error(e.toString());
                    }
                }
            }
        }
        return fileBytes;
    }

    /**
     * 获取模版批注配置
     *
     * @param actionInfo
     * @return
     */
    private Map<String, Object> getAnnotation(ActionInfoDTO actionInfo) {
        Map<String, Object> annotation = new HashMap<>();
        if (null != actionInfo && MapUtils.isNotEmpty(actionInfo.getAnnotation())) {
            annotation = actionInfo.getAnnotation();
        }
        return annotation;
    }

    /**
     * 获取单头
     *
     * @param getActionLocaleResponseDTO
     * @param actionInfo
     * @return
     */
    public ApiDataFieldLocaleMetadataDTO getMainMetadata(GetActionLocaleResponseDTO getActionLocaleResponseDTO, ActionInfoDTO actionInfo) {
        if (getActionLocaleResponseDTO == null) {
            log.error("getActionLocaleResponseDTO is null!");
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0092.getErrCode(), MessageUtil.getMessage("basedata.metaDataStructError"));
        }

        String targetFieldName = Optional.ofNullable(actionInfo).map(ActionInfoDTO::getTargetField).orElse(StringUtils.EMPTY);
        return getMainRequestMetadataField(getActionLocaleResponseDTO.getRequest(), targetFieldName);
    }

    /**
     * 若targetFieldName在request存在，去则返回该字段；否则返回M类型字段；前面两种都没有，则抛出BusinessException
     *
     * @param request
     * @param targetFieldName
     * @return
     * @throws BusinessException
     */
    public ApiDataFieldLocaleMetadataDTO getMainRequestMetadataField(GetActionResponseLocaleRequestDTO request, String targetFieldName) throws BusinessException {
        if (null == request || CollectionUtils.isEmpty(request.getParameters())) {
            return null;
        }

        List<ApiDataFieldLocaleMetadataDTO> parameters = request.getParameters();
        if (StringUtils.isNotBlank(targetFieldName)) {
            // 取targetField对应的M
            Optional<ApiDataFieldLocaleMetadataDTO> metadataDTOOptional = parameters.stream()
                    .filter(metadataDTO -> metadataDTO.getData_name().equals(targetFieldName.trim()))
                    .findFirst();
            if (metadataDTOOptional.isPresent()) {
                return metadataDTOOptional.get();
            }
        }

        // 默认取第一个M
        Optional<ApiDataFieldLocaleMetadataDTO> mOptional = parameters.stream()
                .filter(metadataDTO -> metadataDTO.getData_type().equals(MetaDataType.OBJECT.getName()))
                .findFirst();
        if (mOptional.isPresent()) {
            return mOptional.get();
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0091.getErrCode(), MessageUtil.getMessage("basedata.metaDataStructError"));
        }
    }

    private void createChildSheet(XSSFWorkbook wb, List<CellTypeContainer> businessKeyContainer, CellTypeContainer cellTypeContainerMain, String locale, Set<String> requiredFields, Map<String, Object> childAnnotation) {
        String mainKey = cellTypeContainerMain.getKeyName();
        String mainKeyDescription = cellTypeContainerMain.getKeyDescription();
        Sheet sheet = wb.createSheet(ExcelHelper.getSheetName(mainKey, mainKeyDescription, false));
        List<CellTypeContainer> cellTypes = cellTypeContainerMain.getChildren();
        Set<String> currentRequiredFields = new HashSet<>();
        currentRequiredFields = reBuildRequiredFields(cellTypes, requiredFields);
        // 支持子单身三层机构
        List<CellTypeContainer> childBusinessKeyContainerList = getBusinessKeyContainer(cellTypes);
        // 兼容DF中也配置了BK的场景:先从DF中剔除调BK，再将BK统一添加到最前面
        excelHelper.addBkContainer(cellTypes, businessKeyContainer);
        createSheetHeader(wb, sheet, childBusinessKeyContainerList, cellTypes, locale, currentRequiredFields, childAnnotation);
    }

    private void createSheetHeader(XSSFWorkbook wb, Sheet sheet, List<CellTypeContainer> businessKeyContainer, List<CellTypeContainer> cellTypes, String locale, Set<String> requiredFields, Map<String, Object> annotation) {
        DataFormat fmt = wb.createDataFormat();
        String langName = LocaleContextHolder.getLocale().toLanguageTag();
        if (StringUtils.isEmpty(langName)) {
            langName = "zh_CN";
        } else {
            langName = StringUtils.replaceChars(langName, '-', '_');
        }
        int index = 0;
        //设置列属性
        for (int i = 0; i < cellTypes.size(); i++) {
            String annotationStr = "";
            MetaDataType metaDataType = cellTypes.get(i).getProtoType();
            CellTypeContainer cellTypeContainer = cellTypes.get(i);
            CellStyle cellStyle = wb.createCellStyle();
            switch (metaDataType) {
                case STRING:
                    annotationStr = getAnnotationStr(annotation, cellTypeContainer, langName);
                    if ((null != cellTypeContainer.getPrecision() && cellTypeContainer.getPrecision().length != 0) || StringUtils.isNotEmpty(annotationStr)) {
                        int max = (null != cellTypeContainer.getPrecision() && cellTypeContainer.getPrecision().length != 0) ? cellTypeContainer.getPrecision()[0] : Integer.MAX_VALUE;
                        ExcelUtil.excelRuleStringLengthWithAnnotation(locale, sheet, 0, max, 2,
                                65536, index, index, annotationStr);
                    }
                    cellStyle.setDataFormat(fmt.getFormat("@"));
                    sheet.setDefaultColumnStyle(index, cellStyle);
                    break;
                case NUMBER:
                    //cellTypeContainer.setPrecision(new int[]{5, 3});
                    annotationStr = getAnnotationStr(annotation, cellTypeContainer, langName);
                    ExcelUtil.excelRuleIsNumberWithAnnotation(locale, sheet, 2, 65536, index, index, annotationStr);
                    break;
                case DATE:
                    annotationStr = getAnnotationStr(annotation, cellTypeContainer, langName);
                    cellStyle.setDataFormat(fmt.getFormat(cellTypeContainer.getDateFormat()));
                    sheet.setDefaultColumnStyle(index, cellStyle);
                    ExcelUtil.excelRuleDateFormatWithAnnotation(locale, sheet, "1720-01-01", "",
                            cellTypeContainer.getDateFormat(), 2, 65536, index, index, annotationStr);
                    /*cellStyle.setDataFormat(fmt.getFormat("@"));
                    sheet.setDefaultColumnStyle(i, cellStyle);*/
                    break;
                case OBJECT:
                    //log.error("不应该出现的类型for actionId:{}", actionId);
                    //判断是否是数组类型,且单头存在businessKey
                    if (cellTypeContainer.getArray()) {
                        if (!CollectionUtils.isEmpty(businessKeyContainer)) {
                            //获取单身表的模版批注
                            Map<String, Object> childAnnotation = MapUtils.getMap(annotation, cellTypeContainer.getKeyName(), new HashMap<>());
                            createChildSheet(wb, businessKeyContainer, cellTypeContainer, locale, requiredFields, childAnnotation);
                        }
                        index--;
                    }
                    if (null != cellTypeContainer.getListEnum() && cellTypeContainer.getListEnum().length != 0) {
                        annotationStr = getAnnotationStr(annotation, cellTypeContainer, langName);
                        ExcelUtil.excelRuleSelectWithAnnotation(locale, sheet, cellTypeContainer.getListEnum(), 2, 65536, index, index, annotationStr);
                    }
                    break;
                case BOOLEAN:
                    annotationStr = getAnnotationStr(annotation, cellTypeContainer, langName);
                    ExcelUtil.excelRuleSelectWithAnnotation(locale, sheet, cellTypeContainer.getListEnum(), 2, 65536, index, index, annotationStr);
                    break;
                case ENUM:
                    if (null != cellTypeContainer.getListEnum() && cellTypeContainer.getListEnum().length != 0) {
                        annotationStr = getAnnotationStr(annotation, cellTypeContainer, langName);
                        ExcelUtil.excelEnumRuleSelect(locale, sheet, cellTypeContainer.getListEnum(), 2, 65536, index, index, annotationStr);
                    }
                    break;
                default:
                    break;
            }
            index++;
        }
        /*创建行Rows及单元格Cells*/
        //第一二行为标题
        Row headRow1 = sheet.createRow(0);
        Row headRow2 = sheet.createRow(1);
        //重置index
        index = 0;
        for (int i = 0; i < cellTypes.size(); i++) {
            //跳过单身字段
            if (!cellTypes.get(i).getArray()) {
                //创建单元格
                Cell cell1 = headRow1.createCell(index);
                Cell cell2 = headRow2.createCell(index);
                //设置值
                if ((!CollectionUtils.isEmpty(requiredFields)) && requiredFields.contains(cellTypes.get(i).getKeyName())) {
                    cell1.setCellValue(ExcelUtil.addRequiredMark(wb, cellTypes.get(i).getKeyDescription(), wb.getFontAt(cell1.getCellStyle().getFontIndex())));
                    cell2.setCellValue(cellTypes.get(i).getKeyName());
                } else {
                    cell1.setCellValue(cellTypes.get(i).getKeyDescription());
                    cell2.setCellValue(cellTypes.get(i).getKeyName());
                }
                index++;
            }
        }
    }

    /**
     * 获取配置的注解字符长  超过150长度 截取
     *
     * @param annotation
     * @param cellTypeContainer
     * @return
     */
    private String getAnnotationStr(Map<String, Object> annotation, CellTypeContainer cellTypeContainer, String langName) {
        Map<String, String> annotationMap = MapUtils.getMap(annotation, cellTypeContainer.getKeyName(), new HashMap());
        String annotationValue = MapUtils.getString(annotationMap, langName, "");
        // 检查annotationStr是否超过150个汉字的长度
        if (StringUtils.isNotEmpty(annotationValue) && annotationValue.length() > 150) {
            // 如果超过150个汉字，则截断至150个汉字
            annotationValue = annotationValue.substring(0, 150);
        }
        return annotationValue;
    }

    private List<CellTypeContainer> getBusinessKeyContainer(List<CellTypeContainer> cellTypes) {
        List<CellTypeContainer> cellTypeContainers = new LinkedList<>();
        for (CellTypeContainer cellType : cellTypes) {
            if (cellType.getBusinessKey()) {
                cellTypeContainers.add(cellType);
            }
        }
        return cellTypeContainers;
    }

    private Set<String> reBuildRequiredFields(List<CellTypeContainer> cellTypes, Set<String> requiredFields) {
        Set<String> resultSet = new HashSet<>(requiredFields);
        if (CollectionUtils.isEmpty(cellTypes)) {
            return resultSet;
        }
        for (CellTypeContainer cellTypeContainer : cellTypes) {
            if (REQUIRED_STR_FOR_TRUE.equals(cellTypeContainer.getRequired())) {
                resultSet.add(cellTypeContainer.getKeyName());
            }

        }
        return resultSet;
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 根据模板fileId下载
     * @Date 15:27 2022/2/10
     * @Param
     **/
    private byte[] downloadByTemplateFileId(String fileId) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        String token = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("token", token);
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("fileId", fileId);
        HttpEntity<Map<String, Object>> entity = new HttpEntity<>(parameters, headers);
        ResponseEntity<byte[]> result = restTemplate.exchange(knowledgeGraphUrl + TEMPLATE_DOWNLOAD_URL, HttpMethod.GET, entity, byte[].class, parameters);
        if (null != result && null != result.getBody()) {
            return result.getBody();
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0118.getErrCode());
        }
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 通过KM获取自定义模板url
     * @Date 14:19 2022/2/10
     * @Param
     **/
    private String getTemplateFileId(String actionId, String locale) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        String token = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("token", token);
        if (!StringUtils.isEmpty(locale)) {
            headers.add("locale", locale);
        } else {
            return null;
        }
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("key", actionId + "&" + locale);
        parameters.put("version", "1.0");
        HttpEntity<Map<String, Object>> entity = new HttpEntity<>(parameters, headers);
        ResponseEntity<DapResponse> result = restTemplate.exchange(knowledgeGraphUrl + FILE_EXIST_URL, HttpMethod.GET, entity, DapResponse.class, parameters);
        return (String) result.getBody().getResponse();
    }

    public List<CellTypeContainer> getCellTypeContainers(List<ApiDataFieldLocaleMetadataDTO> field, String token, String securityToken, String locale, int depth) {
        List<CellTypeContainer> cellTypes = new LinkedList<>();
        List<String> keyList = field.stream()
                .map(ApiDataFieldLocaleMetadataDTO::getEnum_key).collect(Collectors.toList());
        keyList.removeAll(Collections.singleton(null));
        Map<String, List<Dictionary>> dictionaryMap = getDictByKeyList(keyList, token, securityToken, locale);
        field.forEach(item -> {
            CellTypeContainer cellTypeContainer;
            int[] precision;
            String[] listEnum;
            List<Dictionary> dictionaries;
            String dateFormat;
            switch (MetaDataType.valueOf(item.getData_type().toUpperCase())) {
                case NUMERIC:
                case NUMBER:
                    //先判断是否是枚举类型,如果是枚举类型则将cellType转为枚举
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        dictionaries = JsonUtils.jsonToListObject(JsonUtils.objectToString(dictionaryMap.get(item.getEnum_key())), Dictionary.class);
                        listEnum = parseListEnum(dictionaries);
                        cellTypeContainer.setListEnum(listEnum);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.NUMERIC);
                        cellTypeContainer.setCellType(CellType.NUMERIC);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case DATETIME:
                case TIME:
                case DATE:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.DATE);
                    cellTypeContainer.setCellType(CellType.STRING);
                    dateFormat = parseDateFormat(item);
                    cellTypeContainer.setDateFormat(dateFormat);
                    break;
                case STRING:
                    //先判断是否是枚举类型,如果是枚举类型则将cellType转为枚举
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        dictionaries = JsonUtils.jsonToListObject(JsonUtils.objectToString(dictionaryMap.get(item.getEnum_key())), Dictionary.class);
                        listEnum = parseListEnum(dictionaries);
                        cellTypeContainer.setListEnum(listEnum);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer = new CellTypeContainer();
                        cellTypeContainer.setProtoType(MetaDataType.STRING);
                        cellTypeContainer.setCellType(CellType.STRING);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case OBJECT:
                    cellTypeContainer = new CellTypeContainer();
                    //判断是否是单身字段,且深度小于2
                    if (item.getIs_array()) {
                        cellTypeContainer.setChildren(getCellTypeContainers(item.getField(), token, securityToken, locale, depth + 1));
                    }
                    cellTypeContainer.setProtoType(MetaDataType.OBJECT);
                    cellTypeContainer.setCellType(CellType.STRING);
                    listEnum = parseListEnum(item);
                    cellTypeContainer.setListEnum(listEnum);
                    break;
                case BOOLEAN:
                    //先判断是否是枚举类型,如果是枚举类型则将cellType转为枚举
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        dictionaries = JsonUtils.jsonToListObject(JsonUtils.objectToString(dictionaryMap.get(item.getEnum_key())), Dictionary.class);
                        listEnum = parseListEnum(dictionaries);
                        cellTypeContainer.setListEnum(listEnum);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.BOOLEAN);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseBooleanListEnum();
                        cellTypeContainer.setListEnum(listEnum);
                    }
                    break;
                default:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setCellType(CellType.STRING);
                    break;
            }
            cellTypeContainer.setRequired(item.getRequired());
            cellTypeContainer.setBusinessKey(item.getIs_businesskey());
            cellTypeContainer.setArray(item.getIs_array());
            cellTypeContainer.setKeyName(item.getData_name());
            cellTypeContainer.setKeyDescription(item.getDescription());
            cellTypes.add(cellTypeContainer);
        });
        return cellTypes;
    }

    private String[] parseListEnum(List<Dictionary> dictionaries) {
        List<String> select = dictionaries.stream().map(d -> d.getCode() + "." + d.getValue()).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(select)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0100.getErrCode(), "下拉字典不能为空");
        }
        return select.toArray(new String[0]);
    }

    public Map<String, CellTypeContainer> getResponseCellTypeContainersBySecurityToken(GetActionLocaleResponseDTO getActionLocaleResponseDTO, String token, String securityToken, String locale) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getResponse().getData();
        if (Objects.isNull(apiDataFieldLocaleMetadataDTO) || CollectionUtils.isEmpty(apiDataFieldLocaleMetadataDTO.getField())) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0101.getErrCode(), "元数据出参为空");
        }
        List<String> keyList = apiDataFieldLocaleMetadataDTO.getField().stream()
                .map(ApiDataFieldLocaleMetadataDTO::getEnum_key).collect(Collectors.toList());
        keyList.removeAll(Collections.singleton(null));
        //Map<String, List<Dictionary>> dictionaryMap = getDictByKeyList(keyList, token, securityToken, locale);
        List<ApiDataFieldLocaleMetadataDTO> fields = apiDataFieldLocaleMetadataDTO.getField();
        List<CellTypeContainer> cellTypeContainers = getCellTypeContainers(fields, token, securityToken, locale, 1);
        // 修复栏位乱序问题
        Map<String, CellTypeContainer> cellTypeContainerMap = new LinkedHashMap<>();
        cellTypeContainers.forEach(item -> {
            cellTypeContainerMap.put(item.getKeyName(), item);
        });
        /*apiDataFieldLocaleMetadataDTO.getField().forEach(item -> {
            CellTypeContainer cellTypeContainer;
            List<Dictionary> dictionaries;
            int[] precision;
            String[] listEnum;
            String dateFormat;
            switch (MetaDataType.valueOf(item.getData_type().toUpperCase())) {
                case NUMERIC:
                case NUMBER:
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseListEnum(JSON.parseArray(JSON.toJSONString(dictionaryMap.get(item.getEnum_key())), Dictionary.class));
                        cellTypeContainer.setListEnum(listEnum);
                        dictionaries = getDictByKeyBySecurityToken(item.getEnum_key(), token, securityToken, locale);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.NUMBER);
                        cellTypeContainer.setCellType(CellType.NUMERIC);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case DATETIME:
                case TIME:
                case DATE:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.DATE);
                    cellTypeContainer.setCellType(CellType.STRING);
                    dateFormat = parseDateFormat(item);
                    cellTypeContainer.setDateFormat(dateFormat);
                    break;
                case STRING:
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseListEnum(JSON.parseArray(JSON.toJSONString(dictionaryMap.get(item.getEnum_key())), Dictionary.class));
                        cellTypeContainer.setListEnum(listEnum);
                        dictionaries = getDictByKeyBySecurityToken(item.getEnum_key(), token, securityToken, locale);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.STRING);
                        cellTypeContainer.setCellType(CellType.STRING);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case OBJECT:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.OBJECT);
                    cellTypeContainer.setCellType(CellType.STRING);
                    listEnum = parseListEnum(item);
                    cellTypeContainer.setListEnum(listEnum);
                    break;
                case BOOLEAN:
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseListEnum(JSON.parseArray(JSON.toJSONString(dictionaryMap.get(item.getEnum_key())), Dictionary.class));
                        cellTypeContainer.setListEnum(listEnum);
                        dictionaries = getDictByKeyBySecurityToken(item.getEnum_key(), token, securityToken, locale);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.BOOLEAN);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseBooleanListEnum();
                        cellTypeContainer.setListEnum(listEnum);
                    }
                    break;
                default:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setCellType(CellType.STRING);
                    break;
            }
            cellTypeContainerMap.put(item.getData_name(), cellTypeContainer);
        });*/
        parseCellTypeContainerMapDict(cellTypeContainerMap);
        return cellTypeContainerMap;
    }

    public Map<String, CellTypeContainer> getResponseCellTypeContainers(GetActionLocaleResponseDTO getActionLocaleResponseDTO, String token, String locale) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getResponse().getData();
        if (CollectionUtils.isEmpty(apiDataFieldLocaleMetadataDTO.getField())) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0102.getErrCode(), "元数据出参为空");
        }
        List<String> keyList = apiDataFieldLocaleMetadataDTO.getField().stream()
                .map(ApiDataFieldLocaleMetadataDTO::getEnum_key).collect(Collectors.toList());
        keyList.removeAll(Collections.singleton(null));
        //Map<String, List<Dictionary>> dictionaryMap = getDictByKeyList(keyList, token, null, locale);
        List<CellTypeContainer> cellTypeContainers = getCellTypeContainers(apiDataFieldLocaleMetadataDTO.getField(), token, null, locale, 1);
        Map<String, CellTypeContainer> cellTypeContainerMap = cellTypeContainers.stream().collect(Collectors.toMap(CellTypeContainer::getKeyName, Function.identity()));
        /*apiDataFieldLocaleMetadataDTO.getField().forEach(item -> {
            CellTypeContainer cellTypeContainer;
            List<Dictionary> dictionaries;
            int[] precision;
            String[] listEnum;
            switch (MetaDataType.valueOf(item.getData_type().toUpperCase())) {
                case NUMERIC:
                case NUMBER:
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseListEnum(JSON.parseArray(JSON.toJSONString(dictionaryMap.get(item.getEnum_key())), Dictionary.class));
                        cellTypeContainer.setListEnum(listEnum);
                        dictionaries = getDictByKey(item.getEnum_key(), token, locale);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.NUMBER);
                        cellTypeContainer.setCellType(CellType.NUMERIC);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case DATETIME:
                case TIME:
                case DATE:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.DATE);
                    cellTypeContainer.setCellType(CellType.STRING);
                    String dateFormat = parseDateFormat(item);
                    cellTypeContainer.setDateFormat(dateFormat);
                    break;
                case STRING:
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseListEnum(JSON.parseArray(JSON.toJSONString(dictionaryMap.get(item.getEnum_key())), Dictionary.class));
                        cellTypeContainer.setListEnum(listEnum);
                        dictionaries = getDictByKey(item.getEnum_key(), token, locale);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.STRING);
                        cellTypeContainer.setCellType(CellType.STRING);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case OBJECT:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.OBJECT);
                    cellTypeContainer.setCellType(CellType.STRING);
                    listEnum = parseListEnum(item);
                    cellTypeContainer.setListEnum(listEnum);
                    break;
                case BOOLEAN:
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseListEnum(JSON.parseArray(JSON.toJSONString(dictionaryMap.get(item.getEnum_key())), Dictionary.class));
                        cellTypeContainer.setListEnum(listEnum);
                        dictionaries = getDictByKey(item.getEnum_key(), token, locale);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.BOOLEAN);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseBooleanListEnum();
                        cellTypeContainer.setListEnum(listEnum);
                    }
                    break;
                default:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setCellType(CellType.STRING);
                    break;
            }
            cellTypeContainerMap.put(item.getData_name(), cellTypeContainer);
        });*/
        parseCellTypeContainerMapDict(cellTypeContainerMap);
        return cellTypeContainerMap;
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 解析字典为map
     * @Date 17:12 2022/2/18
     * @Param
     **/
    private void parseCellTypeContainerMapDict(Map<String, CellTypeContainer> cellTypeContainerMap) {
        cellTypeContainerMap.forEach((k, v) -> {
            if (!CollectionUtils.isEmpty(v.getDictionaries())) {
                Map<String, String> dictMap = v.getDictionaries().stream().collect(Collectors.toMap(Dictionary::getCode, Dictionary::getValue, (entity1, entity2) -> entity1));
                v.setDictMap(dictMap);
            }
            if (!CollectionUtils.isEmpty(v.getChildren())) {
                v.getChildren().forEach(item -> {
                    if (!CollectionUtils.isEmpty(item.getDictionaries())) {
                        Map<String, String> dictMapInner = item.getDictionaries().stream().collect(Collectors.toMap(Dictionary::getCode, Dictionary::getValue, (entity1, entity2) -> entity1));
                        item.setDictMap(dictMapInner);
                    }
                });
            }
        });
    }

    private String[] parseBooleanListEnum() {
        return new String[]{"TRUE", "FALSE"};
    }

    private String[] parseListEnum(ApiDataFieldLocaleMetadataDTO item) {
        return null;
    }

    private String parseDateFormat(ApiDataFieldLocaleMetadataDTO item) {
        //MultiLanguageDTO multiLanguageDTO = item.getRemark();
        MultiLanguageDTO multiLanguageDTO = null;
        if (null == multiLanguageDTO) {
            switch (MetaDataType.valueOf(item.getData_type().toUpperCase())) {
                case DATETIME:
                    return "yyyy-MM-dd HH:mm:ss";
                case TIME:
                    return "HH:mm:ss";
                case DATE:
                    return "yyyy-MM-dd";
            }
            return "yyyy-MM-dd";
        }
        String locale = LocaleContextHolder.getLocale().toString();
        switch (locale) {
            case "zh_TW":
                return multiLanguageDTO.getZh_TW();
            case "zh_CN":
                return multiLanguageDTO.getZh_CN();
            case "en_US":
                return multiLanguageDTO.getEn();
            default:
                return multiLanguageDTO.getZh_CN();
        }
    }

    private int[] parsePrecision(ApiDataFieldLocaleMetadataDTO metadataDTO) {
        if (!StringUtils.isEmpty(metadataDTO.getPrecision())) {
            String precision = metadataDTO.getPrecision();
            if (StringUtils.isEmpty(precision)) {
                return new int[0];
            }
            if (precision.indexOf(",") != -1) {
                String[] values = precision.substring(precision.indexOf("(") + 1, precision.indexOf(")")).split(",");
                // 解决查询API规格 接口返回"(null,null)"导致解析报错问题
                if (StringUtils.isEmpty(values[0]) || StringUtils.isEmpty(values[1]) || "null".equals(values[0]) || "null".equals(values[1])) {
                    return new int[0];
                }
                int[] precisionValue = new int[]{Integer.valueOf(values[0]), Integer.valueOf(values[1])};
                return precisionValue;
            } else {
                int[] precisionValue = new int[]{Integer.valueOf(precision)};
                return precisionValue;
            }
        }
        return new int[0];
    }

    /**
     * 获取单档、多档（单头）的元数据类型
     *
     * @param enumMap
     * @param nonEnumMap
     * @param getActionLocaleResponseDTO
     * @param tableKey
     */
    public void getMetaDataTypeMap(Map<String, MetaDataType> enumMap, Map<String, MetaDataType> nonEnumMap,
                                   GetActionLocaleResponseDTO getActionLocaleResponseDTO, String tableKey) {
        List<ApiDataFieldLocaleMetadataDTO> objects = getActionLocaleResponseDTO.getRequest().getParameters();
        objects.forEach(item -> {
            if (item.getData_name().equals(tableKey)) {
                List<ApiDataFieldLocaleMetadataDTO> parameters = item.getField();
                parameters.forEach(param -> {
                    if (StringUtils.isEmpty(param.getEnum_key())) {
                        nonEnumMap.put(param.getData_name(), MetaDataType.valueOf(param.getData_type().toUpperCase()));
                    } else {
                        // 包含枚举的字段
                        enumMap.put(param.getData_name(), MetaDataType.valueOf(param.getData_type().toUpperCase()));
                    }
                });
            }
        });
    }

    /**
     * 获取单档、多档（单头）的元数据类型
     *
     * @param enumMap
     * @param nonEnumMap
     * @param cellTypeContainer
     * @param parentBKMetadataMap
     */
    public void getMetaDataTypeMapNew(Map<String, MetaDataType> enumMap, Map<String, MetaDataType> nonEnumMap,
                                      ApiDataFieldLocaleMetadataDTO cellTypeContainer, Map<String, ApiDataFieldLocaleMetadataDTO> parentBKMetadataMap) {
        List<ApiDataFieldLocaleMetadataDTO> parameters = cellTypeContainer.getField();
        parameters.forEach(param -> {
            if (StringUtils.isEmpty(param.getEnum_key())) {
                nonEnumMap.put(param.getData_name(), MetaDataType.valueOf(param.getData_type().toUpperCase()));
            } else {
                // 包含枚举的字段
                enumMap.put(param.getData_name(), MetaDataType.valueOf(param.getData_type().toUpperCase()));
            }
        });

        if (CollectionUtils.isEmpty(parentBKMetadataMap)) {
            return;
        }
        // 将上层BK也添加到进来，后续也要对BK进行空值处理
        parentBKMetadataMap.forEach((bkName, bkMetadata) -> {
            if (StringUtils.isEmpty(bkMetadata.getEnum_key())) {
                nonEnumMap.put(bkName, MetaDataType.valueOf(bkMetadata.getData_type().toUpperCase()));
            } else {
                // 包含枚举的字段
                enumMap.put(bkName, MetaDataType.valueOf(bkMetadata.getData_type().toUpperCase()));
            }
        });
    }

    /**
     * 获取单档、多档（单头）的元数据类型
     *
     * @param enumMap
     * @param nonEnumMap
     * @param getActionLocaleResponseDTO
     * @param tableKey
     */
    public void getMetaDataTypeMapNew(Map<String, MetaDataType> enumMap, Map<String, MetaDataType> nonEnumMap,
                                      GetActionLocaleResponseDTO getActionLocaleResponseDTO, String tableKey) {
        List<ApiDataFieldLocaleMetadataDTO> objects = getActionLocaleResponseDTO.getRequest().getParameters();
        objects.forEach(item -> {
            if (item.getData_name().equals(tableKey)) {
                List<ApiDataFieldLocaleMetadataDTO> parameters = item.getField();
                parameters.forEach(param -> {
                    if (StringUtils.isEmpty(param.getEnum_key())) {
                        nonEnumMap.put(param.getData_name(), MetaDataType.valueOf(param.getData_type().toUpperCase()));
                    } else {
                        // 包含枚举的字段
                        enumMap.put(param.getData_name(), MetaDataType.valueOf(param.getData_type().toUpperCase()));
                    }
                });
            }
        });
    }

    /**
     * 获取多档单身的元数据类型
     *
     * @param enumMap
     * @param nonEnumMap
     * @param getActionLocaleResponseDTO
     * @param tableKey
     */
    public void getBodyMetaDataTypeMap(Map<String, MetaDataType> enumMap, Map<String, MetaDataType> nonEnumMap,
                                       GetActionLocaleResponseDTO getActionLocaleResponseDTO, String bodyKey, String tableKey) {
        List<ApiDataFieldLocaleMetadataDTO> objects = getActionLocaleResponseDTO.getRequest().getParameters();
        for (ApiDataFieldLocaleMetadataDTO item : objects) {
            if (item.getData_name().equals(tableKey)) {
                List<ApiDataFieldLocaleMetadataDTO> parameters = item.getField();
                for (ApiDataFieldLocaleMetadataDTO param : parameters) {
                    if (param.getData_name().equals(bodyKey)) {
                        List<ApiDataFieldLocaleMetadataDTO> bodyParameters = param.getField();
                        bodyParameters.forEach(bodyParameter -> {
                            if (StringUtils.isEmpty(bodyParameter.getEnum_key())) {
                                nonEnumMap.put(bodyParameter.getData_name(), MetaDataType.valueOf(bodyParameter.getData_type().toUpperCase()));
                            } else {
                                // 包含枚举的字段
                                enumMap.put(bodyParameter.getData_name(), MetaDataType.valueOf(bodyParameter.getData_type().toUpperCase()));
                            }
                        });
                        break;
                    }
                }
                break;
            }
        }
    }

    /**
     * 获取多档单身的元数据类型
     *
     * @param enumMap
     * @param nonEnumMap
     * @param bodyFiled
     */
    public void getBodyMetaDataTypeMapCopy(Map<String, MetaDataType> enumMap, Map<String, MetaDataType> nonEnumMap,
                                           ApiDataFieldLocaleMetadataDTO bodyFiled) {
        List<ApiDataFieldLocaleMetadataDTO> bodyParameters = bodyFiled.getField();
        bodyParameters.forEach(bodyParameter -> {
            if (StringUtils.isEmpty(bodyParameter.getEnum_key())) {
                nonEnumMap.put(bodyParameter.getData_name(), MetaDataType.valueOf(bodyParameter.getData_type().toUpperCase()));
            } else {
                // 包含枚举的字段
                enumMap.put(bodyParameter.getData_name(), MetaDataType.valueOf(bodyParameter.getData_type().toUpperCase()));
            }
        });
    }

    public List<Map> validateByBatchV2(DataEntryTask dataEntryTask, List<CellTypeContainer> cellTypes, List<Map> rowDataList) {
        if (null == dataEntryTask.getActionInfo() || MapUtils.isEmpty(dataEntryTask.getActionInfo().getSheetRequiredFiled())) {
            return new LinkedList<>();
        }
        return doValidateDataV2(dataEntryTask.getTableKey(), rowDataList, dataEntryTask, cellTypes);
    }

    public List<CellTypeContainer> getCellTypeContainersV2(List<ApiDataFieldLocaleMetadataDTO> field, String token, String securityToken, String locale, int depth) {
        List<CellTypeContainer> cellTypes = new LinkedList<>();
        List<String> keyList = field.stream()
                .map(ApiDataFieldLocaleMetadataDTO::getEnum_key).collect(Collectors.toList());
        keyList.removeAll(Collections.singleton(null));
        Map<String, List<Dictionary>> dictionaryMap = getDictByKeyList(keyList, token, securityToken, locale);
        field.forEach(item -> {
            CellTypeContainer cellTypeContainer;
            int[] precision;
            String[] listEnum;
            List<Dictionary> dictionaries;
            String dateFormat;
            switch (MetaDataType.valueOf(item.getData_type().toUpperCase())) {
                case NUMERIC:
                case NUMBER:
                    //先判断是否是枚举类型,如果是枚举类型则将cellType转为枚举
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        dictionaries = JsonUtils.jsonToListObject(JsonUtils.objectToString(dictionaryMap.get(item.getEnum_key())), Dictionary.class);
                        listEnum = parseListEnum(dictionaries);
                        cellTypeContainer.setListEnum(listEnum);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.STRING);
                        cellTypeContainer.setCellType(CellType.STRING);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case DATETIME:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.DATETIME);
                    cellTypeContainer.setCellType(CellType.STRING);
                    dateFormat = parseDateFormat(item);
                    cellTypeContainer.setDateFormat(dateFormat);
                    break;
                case TIME:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.TIME);
                    cellTypeContainer.setCellType(CellType.STRING);
                    dateFormat = parseDateFormat(item);
                    cellTypeContainer.setDateFormat(dateFormat);
                    break;
                case DATE:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.DATE);
                    cellTypeContainer.setCellType(CellType.STRING);
                    dateFormat = parseDateFormat(item);
                    cellTypeContainer.setDateFormat(dateFormat);
                    break;
                case STRING:
                    //先判断是否是枚举类型,如果是枚举类型则将cellType转为枚举
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        dictionaries = JsonUtils.jsonToListObject(JsonUtils.objectToString(dictionaryMap.get(item.getEnum_key())), Dictionary.class);
                        listEnum = parseListEnum(dictionaries);
                        cellTypeContainer.setListEnum(listEnum);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer = new CellTypeContainer();
                        cellTypeContainer.setProtoType(MetaDataType.STRING);
                        cellTypeContainer.setCellType(CellType.STRING);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case OBJECT:
                    cellTypeContainer = new CellTypeContainer();
                    //判断是否是单身字段,且深度小于2
                    if (item.getIs_array()) {
                        cellTypeContainer.setChildren(getCellTypeContainers(item.getField(), token, securityToken, locale, depth + 1));
                    }
                    cellTypeContainer.setProtoType(MetaDataType.OBJECT);
                    cellTypeContainer.setCellType(CellType.STRING);
                    listEnum = parseListEnum(item);
                    cellTypeContainer.setListEnum(listEnum);
                    break;
                case BOOLEAN:
                    //先判断是否是枚举类型,如果是枚举类型则将cellType转为枚举
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        dictionaries = JsonUtils.jsonToListObject(JsonUtils.objectToString(dictionaryMap.get(item.getEnum_key())), Dictionary.class);
                        listEnum = parseListEnum(dictionaries);
                        cellTypeContainer.setListEnum(listEnum);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.BOOLEAN);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseBooleanListEnum();
                        cellTypeContainer.setListEnum(listEnum);
                    }
                    break;
                default:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setCellType(CellType.STRING);
                    break;
            }
            cellTypeContainer.setRequired(item.getRequired());
            cellTypeContainer.setBusinessKey(item.getIs_businesskey());
            cellTypeContainer.setArray(item.getIs_array());
            cellTypeContainer.setKeyName(item.getData_name());
            cellTypeContainer.setKeyDescription(item.getDescription());
            cellTypes.add(cellTypeContainer);
        });
        return cellTypes;
    }

    public Map<String, CellTypeContainer> getResponseCellTypeContainersV2(GetActionLocaleResponseDTO getActionLocaleResponseDTO, String token, String locale) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getResponse().getData();
        if (CollectionUtils.isEmpty(apiDataFieldLocaleMetadataDTO.getField())) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0102.getErrCode(), "元数据出参为空");
        }
        List<String> keyList = apiDataFieldLocaleMetadataDTO.getField().stream()
                .map(ApiDataFieldLocaleMetadataDTO::getEnum_key).collect(Collectors.toList());
        keyList.removeAll(Collections.singleton(null));
        //Map<String, List<Dictionary>> dictionaryMap = getDictByKeyList(keyList, token, null, locale);
        List<CellTypeContainer> cellTypeContainers = getCellTypeContainersV2(apiDataFieldLocaleMetadataDTO.getField(), token, null, locale, 1);
        Map<String, CellTypeContainer> cellTypeContainerMap = cellTypeContainers.stream().collect(Collectors.toMap(CellTypeContainer::getKeyName, Function.identity()));

        parseCellTypeContainerMapDict(cellTypeContainerMap);
        return cellTypeContainerMap;
    }

    /**
     * 校验数据行，并返回校验结果
     *
     * @param dataKey
     * @param rowDataList
     * @param dataEntryTask
     * @return
     */
    private List<Map> doValidateDataV2(String dataKey, List<Map> rowDataList, DataEntryTask dataEntryTask, List<CellTypeContainer> cellTypes) {
        // 该行以及下层（单身、子单身等）数据中是否存在必填项为空的单元格
        final List<Map> errorRowDataList = new ArrayList<>();
        final Set<String> requiredFiled = getRequiredFields(dataEntryTask, dataKey);

        final Map<String, CellTypeContainer> cellTypeContainerMap = createCellTypeMap(cellTypes);
        final String locale = dataEntryTask.getLocale();

        rowDataList.forEach(rowData -> {

            Map rowDataErrorMsg = new LinkedHashMap<>();
            // 该行数据中是否存在必填项为空的单元格
            boolean parentHasError = false;
            // 下层（单身、子单身等）数据中是否存在必填项为空的单元格
            boolean sonHasError = false;
            for (Map.Entry<String, Object> cellEntry : (Iterable<Map.Entry<String, Object>>) rowData.entrySet()) {
                CellTypeContainer cellType = cellTypeContainerMap.get(cellEntry.getKey());
                // 1、先递归处理单身、子单身数据
                if (cellEntry.getValue() instanceof List) {
                    List<Map> sonErrorDataList = doValidateDataV2(cellEntry.getKey(), (List<Map>) cellEntry.getValue(), dataEntryTask, cellType.getChildren());
                    if (!CollectionUtils.isEmpty(sonErrorDataList)) {
                        sonHasError = true;
                        rowDataErrorMsg.put(cellEntry.getKey(), sonErrorDataList);
                    }
                }

                String errorMsg = ValidatorLoader.valid(cellEntry, requiredFiled, cellType, locale);
                if (StringUtils.isNotEmpty(errorMsg)) {
                    parentHasError = true;
                    // 添加单元格错误信息
                    rowDataErrorMsg.put(cellEntry.getKey() + ExcelHelper.ERROR_FIELD_SUFFIX, errorMsg);
                }
            }

            if (parentHasError || sonHasError) {
                Map errorRow = MapUtil.newHashMap(rowData.size() + rowDataErrorMsg.size(), true);
                errorRow.putAll(rowData);
                // 将错误信息添加到数据行
                errorRow.putAll(rowDataErrorMsg);
                errorRowDataList.add(errorRow);
            }
        });
        return errorRowDataList;
    }

    private Set<String> getRequiredFields(DataEntryTask task, String dataKey) {
        return Optional.ofNullable(task.getActionInfo().getSheetRequiredFiled())
                .map(map -> map.get(dataKey))
                .orElse(Collections.emptySet());
    }

    private Map<String, CellTypeContainer> createCellTypeMap(List<CellTypeContainer> cellTypes) {
        return cellTypes.stream().collect(Collectors.toMap(
                CellTypeContainer::getKeyName,
                Function.identity(),
                (existing, replacement) -> existing // 处理重复键
        ));
    }
}