package com.digiwin.athena.abt.application.service.abt.migration.inout.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.abt.application.configuration.DMCConfig;
import com.digiwin.athena.abt.application.configuration.DirectRabbitConfig;
import com.digiwin.athena.abt.application.dto.migration.abt.api.MidWareResponse;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.DMCTokenBean;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ErrorTable;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ImportStatistics;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.FileInfo;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.ExcelHelper;
import com.digiwin.athena.abt.application.service.abt.migration.inout.ErrorHandlerService;
import com.digiwin.athena.abt.application.utils.ExcelUtil;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import com.digiwin.athena.abt.core.meta.dto.CellTypeContainer;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.abt.core.meta.enums.ExcelTypeEnum;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.rabbitmq.client.Channel;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.*;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

/**
 * @ClassName RabbitErrorHandlerServiceImpl
 * @Description TODO
 * @Author zhuangli
 * @Date 2021/4/25 16:35
 * @Version 1.0
 **/
@Service
@Slf4j
public class RabbitErrorHandlerServiceImpl implements ErrorHandlerService {

    @Autowired
    RabbitTemplate rabbitTemplate;

    @Resource
    RestTemplate dmcRestTemplate;

    @Autowired
    DMCConfig dmcConfig;

    @Autowired
    DMCTokenBean dmcTokenBean;

    @Autowired
    ExcelHelper excelHelper;

    @Value("${dmc.uri}")
    private String dmcUrl;
    @Value("${athena.auth.appToken}")
    private String appToken;
    private final String EXCEL_SUFFIX = ".xlsx";
    private final String BASE_URL = this.getClass().getClassLoader().getResource("").getFile();
    private final String PACKAGE_NAME = "importError";
    private final String HEADER_KEY = "key";
    private final String HEADER_NAME = "name";
    private final String SHEET_NAME = "sheet1";

    @Override
    public void handleErrorList(ImportStatistics importStatistics, List<Map> errorTable, boolean createQueueFlag) throws IOException {
        final String masterId = importStatistics.getMasterId();
        if (createQueueFlag) {
            try (Channel channel = rabbitTemplate.getConnectionFactory().createConnection().createChannel(false)) {
                Map<String, Object> arguments = Collections.EMPTY_MAP;
                channel.queueDeclare(masterId, false, false, false, arguments);
                channel.queueBind(masterId, DirectRabbitConfig.ERROR_TABLE_EXCHANGE_NAME, masterId);
            } catch (TimeoutException e) {
                log.error("close rabbit channel timeout", e);
            }
        }
        //这里只是将错误数据发送到对应的临时队列
        rabbitTemplate.convertAndSend(DirectRabbitConfig.ERROR_TABLE_EXCHANGE_NAME, masterId, errorTable);
        log.error("sendErrorData for {}", masterId);
    }

    @Override
    public ErrorTable getErrorTableByMasterId(String mainKeyName, ImportStatistics importStatistics, Map<String, CellTypeContainer> cellTypeContainerMap, List<String> mainKeys) {
        final String failedUrl = importStatistics.getFailedUrl();
        final String masterId = importStatistics.getMasterId();
        final String actionId = importStatistics.getActionId();
        final Integer succeededNum = importStatistics.getSucceededNum();
        final Integer processingNum = importStatistics.getProcessingNum();
        final Integer failedNum = importStatistics.getFailedNum();
        final Integer errorNum = importStatistics.getErrorNum();
        final Integer republished = importStatistics.getRepublished();
        if (StringUtils.isEmpty(failedUrl)) {
            log.error("文件不存在,masterId:{}", masterId);
            throw new RuntimeException("文件不存在");
        } else {
            try {
                InputStream inputStream = getErrorTableInputStream(failedUrl);
                Workbook wb = new XSSFWorkbook(inputStream);
                List<String> headerKeys = new LinkedList<>();
                List<Map> headers = excelHelper.readHeaders(wb, cellTypeContainerMap, headerKeys, mainKeys);
                List<Map> data = excelHelper.readRootData(wb, mainKeyName, cellTypeContainerMap);
                ErrorTable errorTable = new ErrorTable();
                errorTable.setHeaders(headers);
                errorTable.setData(data);
                errorTable.setActionId(actionId);
                errorTable.setErrorNum(errorNum);
                errorTable.setSucceededNum(succeededNum);
                errorTable.setFailedNum(failedNum);
                errorTable.setProcessingNum(processingNum);
                errorTable.setRepublished(republished);
                return errorTable;
            } catch (IOException e) {
                log.error("生成错误数据文档IO异常", e);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0073.getErrCode(), "IO异常");
            }
        }
    }

    private InputStream getErrorTableInputStream(String failedUrl) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", dmcTokenBean.getToken());
        headers.add("digi-middleware-auth-app", appToken);
        List list = new ArrayList<>();
        list.add(MediaType.parseMediaType(ExcelTypeEnum.XLSX.value()));
        headers.setAccept(list);
        String url = dmcUrl + "/api/dmc/v2/file/" + dmcConfig.getBucket() + "/download/" + failedUrl;
        ResponseEntity<byte[]> restRes = dmcRestTemplate.exchange(
                url,
                HttpMethod.GET,
                new HttpEntity<byte[]>(headers),
                byte[].class);
        return new ByteArrayInputStream(restRes.getBody());
    }

    @Override
    public void downloadErrorTable(ImportStatistics importStatistics, List<Map> headers, Set<String> requiredFields, HttpServletResponse response) {
        final String failedUrl = importStatistics.getFailedUrl();
        if (StringUtils.isEmpty(failedUrl)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0074.getErrCode(), "没有生成对应的文档");
        }
        response.setContentType(ExcelTypeEnum.XLSX.value());
        //设置文件名
        response.setCharacterEncoding("utf-8");
        String formFileName = getFileName(importStatistics);
        response.addHeader("Content-Disposition", "attachment;filename=" + formFileName + EXCEL_SUFFIX);
        response.addHeader("Access-Control-Expose-Headers", "Content-Disposition");
        InputStream inputStream = getErrorTableInputStream(failedUrl);
        //------------------
        XSSFWorkbook wb;
        try {
            wb = new XSSFWorkbook(inputStream);

        } catch (IOException e) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0075.getErrCode(), "解析excel失败masterid:" + importStatistics.getMasterId());
        }
        Sheet sheet = wb.getSheetAt(0);
        Row headRow1 = sheet.getRow(0);
        Row headRow2 = sheet.getRow(1);
        Map<String, String> headerMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(headers)) {
            headerMap = headers.stream().collect(Collectors.toMap(s -> (String) s.get(HEADER_KEY), s -> (String) s.get(HEADER_NAME)));
        }
        if (!CollectionUtils.isEmpty(headerMap)) {
            for (int i = 0; i < headRow1.getLastCellNum(); i++) {
                //创建单元格
                Cell cell1 = headRow1.getCell(i);
                Cell cell2 = headRow2.getCell(i);
                //设置值
                String langVal = headerMap.get(cell2.getStringCellValue());
                langVal = StringUtils.isEmpty(langVal) ? cell1.getStringCellValue() : langVal;
                if ((!CollectionUtils.isEmpty(requiredFields)) && requiredFields.contains((String) headers.get(i).get(HEADER_KEY))) {
                    cell1.setCellValue(ExcelUtil.addRequiredMark(wb, langVal, wb.getFontAt(cell1.getCellStyle().getFontIndex())));
                } else {
                    cell1.setCellValue(langVal);
                }
            }
        }

        try (OutputStream os = response.getOutputStream()) {
            wb.write(os);
            wb.close();
        } catch (IOException e) {
            if (null != wb) {
                try {
                    wb.close();
                } catch (IOException ioe) {
                    log.error("关闭wb异常", ioe);
                }
            }
            log.error("导出异常数据写入文件异常", e);
        }
        //------------------
        /*byte[] buffer = new byte[1024];
        BufferedInputStream bis = null;
        try {
            bis = new BufferedInputStream(inputStream);
            OutputStream os = response.getOutputStream();
            int i;
            while ((i = bis.read(buffer)) != -1) {
                os.write(buffer, 0, i);
            }
        } catch (Exception e) {
            log.error("Exception occurred", e);
        } finally {
            if (bis != null) {
                try {
                    bis.close();
                } catch (IOException e) {
                    log.error("Exception occurred", e);
                }
            }
        }*/
    }

    /**
     * 获取作业名称规则：根据locale来判断语言环境，如果locale为null则取default
     *
     * @param importStatistics
     * @return
     */
    private String getFileName(ImportStatistics importStatistics) {
        JSONObject jsonObject = JSON.parseObject(importStatistics.getActivityName());
        String fileName = "";
        String locale = importStatistics.getLocale();
        switch (locale) {
            case "zh_CN":
                fileName = jsonObject.getString("default");
                break;
            case "zh_TW":
                fileName = jsonObject.getString("zh_TW");
                break;
            case "en_US":
                fileName = jsonObject.getString("en_US");
                break;
            default:
                fileName = jsonObject.getString("default");
                break;
        }
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH_mm_ss");
        String formatTime = simpleDateFormat.format(importStatistics.getCreateTime());
        fileName = fileName + " " + formatTime;

        try {
            fileName = URLEncoder.encode(fileName, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error("Exception occurred", e);
        }
        // 解决URLEncoder.encode方法会把空格变成加号（+）在前台页面显示的时候会多出加号的问题
        fileName = fileName.replaceAll("\\+", "%20");
        return fileName;
    }

    /**
     * @return java.lang.String
     * @Author zhuangli
     * @Description 全部批次数据处理完, 将所有错误数据存入文档中心
     * @Date 14:15 2021/12/1
     * @Param [cellTypes, headers, importStatistics]
     **/
    @Override
    public String onImportFinish(String sheetName, List<String> keyList, Map<String, CellTypeContainer> cellTypeContainerMap, Map<String, String> headers, ImportStatistics importStatistics) {
        log.error("importFinish for {}", importStatistics.getMasterId());
        final int failedNum = importStatistics.getFailedNum();
        final String masterId = importStatistics.getMasterId();
        if (0 != failedNum) {
            // 从MQ拉取异常数据
            List<Map<String, Object>> errorTable = fetchData(masterId, failedNum);
            /*String filePath = formPath(masterId);
            File file = new File(filePath);*/
            Workbook wb = new XSSFWorkbook();
            /*创建表单*/
            //当前行索引,跳过标题计算
            int rowIndex = importStatistics.getFailedNum() - errorTable.size() + 2;
            String locale = importStatistics.getLocale();
            List<CellTypeContainer> businessKeyContainer = excelHelper.getBusinessKeyContainer(new ArrayList<>(cellTypeContainerMap.values()));
            excelHelper.createDataExcel(sheetName, locale, rowIndex, keyList, cellTypeContainerMap, headers, errorTable, wb, businessKeyContainer, 0, null, null);
            MultipartFile multipartFile;
            try {
                multipartFile = workbookToMultipartFile(wb, importStatistics.getMasterId());
            } catch (IOException e) {
                log.error("转换multipartfile失败:{}", e);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0076.getErrCode(), "转换multipartFile失败:{}", e);
            }
            FileInfo fileInfo = new FileInfo();
            fileInfo.setDisplayName(importStatistics.getMasterId());
            fileInfo.setDirectoryId(dmcConfig.getErrorTableUUID());
            //上传文件至文件服务器
            String failedUrl = upload(multipartFile, fileInfo);
            //删除队列
            clearQueue(masterId);
            return failedUrl;
        } else {
            return "";
        }
    }

    @Override
    public MultipartFile workbookToMultipartFile(Workbook wb, String fileName) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        wb.write(bos);
        byte[] bytes = bos.toByteArray();
        InputStream is = new ByteArrayInputStream(bytes);
        MultipartFile multipartFile = new MockMultipartFile("file", fileName, ExcelTypeEnum.XLSX.value(), is);
        return multipartFile;
    }

    @Override
    public MultipartFile workbookToZipFile(Workbook wb, String fileName) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        wb.write(bos);
        byte[] bytes = bos.toByteArray();
        InputStream is = new ByteArrayInputStream(bytes);
        MultipartFile multipartFile = new MockMultipartFile("file", fileName, ExcelTypeEnum.XLSX.value(), is);
        return multipartFile;
    }

    @Override
    public String upload(MultipartFile uploadFile, FileInfo fileInfo) {
        if (Objects.isNull(uploadFile)) {
            log.error("uploadFile is null!");
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0078.getErrCode());
        }

        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.MULTIPART_FORM_DATA);
        header.add("digi-middleware-auth-user", dmcTokenBean.getToken());
        header.add("digi-middleware-auth-app", appToken);

        HttpHeaders fileHeader = new HttpHeaders();
        fileHeader.setContentType(MediaType.parseMediaType(uploadFile.getContentType()));
        fileHeader.setContentDispositionFormData("file", uploadFile.getOriginalFilename());

        HttpHeaders fileInfoHeader = new HttpHeaders();
        fileInfoHeader.setContentType(MediaType.APPLICATION_JSON);

        try {
            HttpEntity<ByteArrayResource> fileEntity = new HttpEntity<>(new ByteArrayResource(uploadFile.getBytes()),
                    fileHeader);
            HttpEntity<FileInfo> fileInfoEntity = new HttpEntity<>(fileInfo, fileInfoHeader);
            multiValueMap.add("file", fileEntity);
            multiValueMap.add("fileInfo", fileInfoEntity);

            String url = dmcUrl + "/api/dmc/v2/file/Athena/upload";

            HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<>(multiValueMap, header);
            ResponseEntity<MidWareResponse> postForEntity = dmcRestTemplate.postForEntity(url, httpEntity, MidWareResponse.class);
            MidWareResponse<Map> midWareResponse = postForEntity.getBody();
            if (midWareResponse.getSuccess()) {
                return (String) midWareResponse.getData().get("id");
            } else {
                log.error("调用上传api失败");
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0077.getErrCode(), MessageUtil.getMessage("delivery.uploadError2"));
            }
        } catch (Exception e) {
            log.error("", e);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0078.getErrCode(), "", e);
        }

    }

    /**
     * @return
     * @Author zhuangli
     * @Description 从rabbitmq中取得所有失败数据
     * @Date 14:59 2022/2/15
     * @Param
     **/
    private List<Map<String, Object>> fetchData(String masterId, int failedNum) {
        List<Map<String, Object>> errorTable = new ArrayList<>(failedNum);
        while (0 < failedNum) {
            // 拉取消息，存在消息还未入队的情况，若未拉取到则等待1s 解决bug 63555
            Object msgObj = rabbitTemplate.receiveAndConvert(masterId, 1000);
            List<Map<String, Object>> errorBatch = new ArrayList<>();
            if (Objects.nonNull(msgObj)) {
                errorBatch = JsonUtils.jsonToObject(JsonUtils.objectToString(msgObj), List.class);
            } else {
                log.error("fetchData getRabbitMqMsg is null");
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0115.getErrCode(), "拉取导入异常数据失败:{}");
            }

            failedNum -= errorBatch.size();
            errorTable.addAll(errorBatch);
        }
        return errorTable;
    }

    private void clearQueue(String masterId) {
        Channel channel = rabbitTemplate.getConnectionFactory().createConnection().createChannel(false);
        try {
            channel.queueDelete(masterId);
        } catch (IOException e) {
            log.error("删除队列异常:{}", e);
        } finally {
            try {
                channel.close();
            } catch (IOException e) {
                log.error(e.toString());
            } catch (TimeoutException e) {
                log.error(e.toString());
            }
        }

    }

    @Override
    public String uploadFileToHttpServer(File file, FileInfo fileInfo) {
        if (Objects.isNull(file)) {
            log.error("uploadFile is null!");
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0078.getErrCode());
        }

        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.MULTIPART_FORM_DATA);
        header.add("digi-middleware-auth-user", dmcTokenBean.getToken());
        header.add("digi-middleware-auth-app", appToken);

        HttpHeaders fileInfoHeader = new HttpHeaders();
        fileInfoHeader.setContentType(MediaType.APPLICATION_JSON);

        try {
            HttpEntity<FileInfo> fileInfoEntity = new HttpEntity<>(fileInfo, fileInfoHeader);
            multiValueMap.add("file", new FileSystemResource(file));
            multiValueMap.add("fileInfo", fileInfoEntity);

            String url = dmcUrl + "/api/dmc/v2/file/Athena/upload";

            HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<>(multiValueMap, header);
            ResponseEntity<MidWareResponse> postForEntity = dmcRestTemplate.postForEntity(url, httpEntity, MidWareResponse.class);
            MidWareResponse<Map> midWareResponse = postForEntity.getBody();
            if (midWareResponse.getSuccess()) {
                return (String) midWareResponse.getData().get("id");
            } else {
                log.error("调用上传api失败");
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0077.getErrCode(), MessageUtil.getMessage("delivery.uploadError2"));
            }
        } catch (Exception e) {
            log.error("", e);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0078.getErrCode(), "", e);
        }
    }
}


