package com.digiwin.athena.abt.application.service.abt.migration.ptm.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.abt.application.configuration.EnvProperties;
import com.digiwin.athena.abt.application.dto.migration.atmc.backlog.TaskCardDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.ptm.PtmProjectCardRecordDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.ptm.PtmProjectCardRouteDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.thememap.TmActivityResponseDTO;
import com.digiwin.athena.abt.application.service.abt.migration.ptm.CommonPtmTransformService;
import com.digiwin.athena.abt.application.service.abt.migration.ptm.PtmTransformService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.ptm.PtmService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.thememap.ThemeMapService;
import com.digiwin.athena.abt.application.service.atmc.migration.tenantToken.TenantTokenService;
import com.digiwin.athena.abt.application.utils.ManualTaskUtil;
import com.digiwin.athena.abt.application.utils.SummaryMessagesUtils;
import com.digiwin.athena.abt.core.meta.constants.BpmConstant;
import com.digiwin.athena.abt.core.meta.constants.GlobalConstant;
import com.digiwin.athena.abt.core.meta.enums.TmPageName;
import com.digiwin.athena.abt.infrastructure.mapper.biz.migration.atmc.ActivityChangeInfoMapper;
import com.digiwin.athena.abt.infrastructure.pojo.bo.migration.atmc.*;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.ActivityChangeInfo;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.PtmBacklog;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.Route;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.Task;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author lzw
 * @date 2024/4/26
 * @description:
 **/
@Service
@Slf4j
public class PtmTransformServiceImpl implements PtmTransformService {
    public static final String ATHENA ="athena";

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private PtmService ptmService;

    @Autowired
    private ActivityChangeInfoMapper activityChangeInfoMapper;

    @Autowired
    private ThemeMapService themeMapService;

    @Autowired
    private CommonPtmTransformService commonPtmTransformService;

    @Autowired
    private TenantTokenService tenantTokenService;

    @Override
    public Task translateProjectOrProjectCard(Long id) {
        if (!enablePTM()) {
            return null;
        }

        Task task = null;
        PtmProjectRecordBO project = ptmService.getProjectRecord(id);
        if (project != null) {
            PtmProjectCardRecordDTO projectCard = ptmService.getProjectCardRecord(project.getProjectCardId());
            task = Task.builder()
                    .id(project.getId())
                    .tmTaskId(project.getProjectDefCode())
                    .personInCharge(project.getPersonInCharge())
                    .personInChargeName(project.getPersonInChargeName())
                    .dataFrom(BpmConstant.DATA_FROM_PTM)
                    .targetName(projectCard.getTargetName())
                    .importance(project.getImportance() ? 1 : 0)
                    .startTime(project.getStartTime())
                    .endTime(project.getEndTime())
                    .build();
        } else {
            PtmProjectCardRecordDTO projectCard = ptmService.getProjectCardRecord(id);
            task = Task.builder()
                    .id(projectCard.getId())
                    .tmTaskId(projectCard.getProjectDefCode())
                    .personInCharge(projectCard.getPersonInCharge())
                    .personInChargeName(projectCard.getPersonInChargeName())
                    .dataFrom(BpmConstant.DATA_FROM_PTM)
                    .targetName(projectCard.getTargetName())
                    .importance(0)
                    .startTime(projectCard.getStartTime())
                    .endTime(projectCard.getEndTime())
                    .build();
        }

        return task;
    }

    /**
     * 是否启用了PTM
     *
     * @return
     */
    @Override
    public boolean enablePTM() {
        return StringUtils.isNotBlank(envProperties.getPtmUri());
    }

    /**
     * 获取项目中的任意一条路线
     *
     * @param projectId
     * @return
     */
    @Override
    public Route getRouteByProjectId(Long projectId) {
        Route result = null;
        if (!enablePTM()) {
            return result;
        }

        PtmProjectCardRouteDTO ptmRoute = ptmService.getProjectRoute(projectId);
        if (ptmRoute != null && CollectionUtils.isNotEmpty(ptmRoute.getRoutes())) {
            PtmProjectCardRouteDTO.RouteDTO r = ptmRoute.getRoutes().get(0);

            result = Route.builder()
                    .id(r.getId())
                    .routeName(r.getRouteName())
                    .pass(r.getPass())
                    .mainTaskId(r.getProjectCardId())
                    .taskId(r.getProjectId())
                    .routeOrder(r.getRouteOrder())
                    .selected(r.getSelected())
                    .branchIndex(r.getBranchIndex())
                    .build();
        }
        return result;
    }

    /**
     * 任务MQTT专用方法
     *
     * @param backlog
     * @return
     */
    @Override
    public TaskCardDTO translateBacklogForMQTT(PtmBacklog backlog) {
        if (Objects.isNull(backlog) || Objects.equals(backlog.getHide(), 1)) {
            return null;
        }

        List<BacklogBO> result = new ArrayList<>();
        PtmProjectCardRecordDTO ptmProjectCardRecordDTO = ptmService.getProjectCardRecord(backlog.getProjectCardId());
        PtmTaskRecordBO ptmTaskRecordBO = ptmService.getTaskRecord(backlog.getTaskId(),null);
        PtmActivityRecordDTO ptmActivityRecordDTO = ptmService.getActivityRecord(backlog.getActivityId());
        PtmWorkItemRecordBO ptmWorkItemRecordBO = ptmService.getWorkItemRecord(backlog.getWorkItemId());
        PtmTaskCardItemBO ptmTaskCardItemBO = PtmTaskCardItemBO.builder()
                .backlogId(backlog.getBacklogId())
                .workItemId(backlog.getWorkItemId())
                .activityId(backlog.getActivityId())
                .taskId(backlog.getTaskId())
                .projectId(backlog.getProjectId())
                .projectCardId(backlog.getProjectCardId())
                .performerId(backlog.getPerformerId())
                .performerName(backlog.getPerformerName())
                .performerType(ptmWorkItemRecordBO.getPerformerType())
                .merge(backlog.getMerge())
                .createTime(backlog.getCreateTime())
                .tenantId(backlog.getTenantId())
                .targetTenantId(backlog.getTargetTenantId())
                .projectDefCode(ptmProjectCardRecordDTO.getProjectDefCode())
                .projectDefName(ptmProjectCardRecordDTO.getProjectDefName())
                .projectName(ptmProjectCardRecordDTO.getProjectName())
                .taskName(backlog.getTaskName())
                .taskWithName(backlog.getTaskWithName())
                .taskType(ptmTaskRecordBO.getTaskType())
                .taskDefPattern(ptmTaskRecordBO.getTaskDefPattern())
                .taskDefCategory(ptmTaskRecordBO.getTaskDefCategory())
                .taskDefExecuteType(ptmTaskRecordBO.getTaskDefExecuteType())
                .taskDefCode(backlog.getTaskDefCode())
                .taskDefName(backlog.getTaskDefName())
                .taskDefWithName(backlog.getTaskDefWithName())
                .taskCreateTime(backlog.getCreateTime())
                .signReason(ptmActivityRecordDTO.getSignReason())
                .planEndTime(backlog.getPlanEndTime())
                .proxyToken(ptmTaskRecordBO.getProxyToken())
                .activityName(ptmActivityRecordDTO.getActivityName())
                .signType("1")
                .bpmnType(ptmActivityRecordDTO.getBpmnType())
                .bpmActivityId(ptmActivityRecordDTO.getBpmActivityId())
                .bpmActivityName(ptmActivityRecordDTO.getBpmActivityName())
                .build();

        PtmProjectRecordBO ptmProjectRecordDTO = ptmService.getProjectRecord(ptmTaskCardItemBO.getProjectId());
        PtmBacklog ptmBacklog = backlog;

        BacklogBO bl = BacklogBO.builder()
                .id(ptmTaskCardItemBO.getBacklogId())
                .name(ptmTaskCardItemBO.getTaskName())
                // 格式："{"zh_CN":"xxx", "zh_TW": "xxxxx", "en_US": "xxxxxx"}"
                .withName(ptmTaskCardItemBO.getTaskWithName())
                .subName(null)
                .taskName(ptmProjectRecordDTO.getProjectName())
                .taskTargetName(null)       //不需要
                .taskStartTime(ptmProjectRecordDTO.getStartTime())
                .taskEndTime(ptmProjectRecordDTO.getEndTime())
                .taskImportance(ptmProjectRecordDTO.getImportance())
                .importance(ptmBacklog.getFavorite() != null ? ptmBacklog.getFavorite() > 0 : false)      //不用处理
                .content(ptmTaskCardItemBO.getTaskName())
                .importanceReadOnly(false)  //不用处理
                .importanceSource(null)
                .uri(null)
                .startTime(ptmTaskCardItemBO.getCreateTime())
                .endTime(ptmTaskCardItemBO.getPlanEndTime())
                .tmTaskId(ptmProjectRecordDTO.getProjectDefCode())
                .tmActivityId(ptmTaskCardItemBO.getTaskDefCode())
                .type(SummaryMessagesUtils.translateBacklogType(ptmBacklog.getType()))
                .tenantId(ptmTaskCardItemBO.getTenantId())
                .targetTenantId(ptmTaskCardItemBO.getTargetTenantId())
                .actionDefined(true)
                .bpmActivityId(ptmTaskCardItemBO.getTaskId())
                .tmPattern(ptmTaskCardItemBO.getTaskDefPattern())
                .tmCategory(ptmTaskCardItemBO.getTaskDefCategory())
                .checkItems(null).operation(null).summaryLayout(null)
                .summaryLayoutStr(null)
                .error(null)
                .isOwner(true)
                .performerId(ptmTaskCardItemBO.getPerformerId())
                .performerName(ptmTaskCardItemBO.getPerformerName())
                .ownerUserId(ptmTaskCardItemBO.getPerformerId())
                .ownerUserName(ptmTaskCardItemBO.getPerformerName())
                .readCount(ptmBacklog.getReadCount())
                .exception(ptmTaskRecordBO.getHasException())
                .overdue(false)
                .overdueDays(0)
                .overdueHours(0)
                .overdueMinutes(0)
                .reassignFromId(0L)
                .approvalState(ptmBacklog.getApprovalState())
                .todoItems(null)
                .emergency(ptmProjectRecordDTO.getEmergency() == 25)
                .processSerialNumber(ptmTaskRecordBO.getProcessSerialNumber())
                .agentBeginDate(null)
                .agentBeginDate(null)
                .overdueWorkitemId(0L)
                .sourceTenantName(StringUtils.equalsIgnoreCase(ptmBacklog.getTenantId(), ptmBacklog.getTargetTenantId()) ? null : ptmBacklog.getTenantName())
                .workItemId(ptmTaskCardItemBO.getBacklogId())
                .proxyToken(ptmTaskCardItemBO.getProxyToken())
                .historyMessage(null)
                .finishActionId(ptmBacklog.getClosed() ? ptmBacklog.getActionId() : null)
                .tmActivityName(ptmTaskCardItemBO.getTaskDefName())
                .tmActivityWithName(ptmTaskCardItemBO.getTaskDefWithName())
                .dataFrom(BpmConstant.DATA_FROM_PTM)
                .merge(ptmTaskCardItemBO.getMerge())
                .createTime(ptmTaskCardItemBO.getCreateTime())
                .build();

        if (StringUtils.isNotBlank(backlog.getEocCode())) {
            bl.setEocCode(backlog.getEocCode());
            bl.setEocName(backlog.getEocName());
            bl.setEocType(backlog.getEocType());
        } else {
            bl.setEocCode(ptmProjectRecordDTO.getEocCode());
            bl.setEocName(ptmProjectRecordDTO.getEocName());
            bl.setEocType(ptmProjectRecordDTO.getEocType());
        }


        //设置逾期数据
        if (ptmTaskRecordBO.getState() != 3 && ptmTaskRecordBO.getPlanEndTime().isBefore(LocalDateTime.now())) {
            bl.setOverdue(true);
            bl.setOverdueDays((int) Duration.between(ptmTaskRecordBO.getPlanEndTime(), LocalDateTime.now()).toDays());
            bl.setOverdueHours((int) Duration.between(ptmTaskRecordBO.getPlanEndTime(), LocalDateTime.now()).toHours());
            bl.setOverdueMinutes((int) Duration.between(ptmTaskRecordBO.getPlanEndTime(), LocalDateTime.now()).toMinutes());

            //如果任务的逾时间就比当前时间小不足1分钟，会在画面显示逾时0分钟。
            if (bl.getOverdueMinutes() <= 0) {
                bl.setOverdue(false);
            }
        }

        //获取任务数据是否变更
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("workitem_id", backlog.getBacklogId());
        List<ActivityChangeInfo> activityChangeInfoList = activityChangeInfoMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(activityChangeInfoList)) {
            bl.setDataChangeRead(activityChangeInfoList.get(0).getHasRead());
        }
        result.add(bl);

        if (CollectionUtils.isNotEmpty(result)) {
            BacklogBO backlogBO = result.get(0);
            backlogBO.setIsOwner(true);
            backlogBO.setOwnerUserId(backlogBO.getPerformerId());
            backlogBO.setOwnerUserName(backlogBO.getPerformerName());
            ArrayList<BacklogBO> backlogBOS = new ArrayList<>();
            backlogBOS.add(backlogBO);
            setBacklogQueryResultSet(backlogBOS);
            List<TaskCardDTO> taskCardDTOS = getBacklogList(null, true, backlogBOS);
            if (CollectionUtils.isNotEmpty(taskCardDTOS)) {
                return taskCardDTOS.get(0);
            }
            return null;
        }
        return null;
    }

    private void setBacklogQueryResultSet(List<BacklogBO> backlogBOS) {
        backlogBOS.stream().filter(backlogDTO -> backlogDTO.getType() != null && backlogDTO.getTmActivityId() != null && backlogDTO.getTmCategory() != null && !"NONE".equals(backlogDTO.getTmCategory())).forEach(backlogDTO -> {
            if (backlogDTO.isTaskImportance()) {
                backlogDTO.setImportance(true);
                backlogDTO.setImportanceReadOnly(true);
            } else {
                if (backlogDTO.getImportanceSource() != null && Objects.equals(backlogDTO.getImportanceSource().toLowerCase(), ATHENA)) {
                    backlogDTO.setImportanceReadOnly(true);
                } else {
                    backlogDTO.setImportanceReadOnly(false);
                }
            }
            if (!StringUtils.isEmpty(backlogDTO.getApprovalState())) {
                backlogDTO.setApprovalStateCode(backlogDTO.getApprovalState());
                backlogDTO.setApprovalState(GlobalConstant.backlogStatusMap.getOrDefault(backlogDTO.getApprovalState(), ""));
            }
        });

        Map<String, String> approveActivityNameMap = new HashMap<>();
        for (BacklogBO backlogBO : backlogBOS) {
            backlogBO.setActionDefined(!"NONE".equals(backlogBO.getTmCategory()) && backlogBO.getType() != null);
            String key = backlogBO.getTmTaskId() + "-" + backlogBO.getTmActivityId();
            if (approveActivityNameMap.containsKey(key)) {
                backlogBO.setSubName(approveActivityNameMap.get(key));
            } else {
                backlogBO.setSubName(getSubName(backlogBO));
                approveActivityNameMap.put(key, backlogBO.getSubName());
            }
        }
    }

    private List<TaskCardDTO> getBacklogList(String clientId, boolean isGetReassignAble, List<BacklogBO> backlogList) {
        Long lastTime = 0L;
        Long finalLastTime = lastTime;
        List<Map> activities = new ArrayList<>();
        List<TaskCardDTO> list = new ArrayList<>();

        List<Long> backlogIdList = backlogList.stream().map(BacklogBO::getWorkItemId).collect(Collectors.toList());
        Map<Long, PtmMergeBacklogBO> mergeBacklogDTOMap = commonPtmTransformService.dealBacklogData(backlogIdList, false);

        backlogList.forEach(backlog -> {
            TaskCardDTO taskCardDTO = TaskCardDTO.builder()
                    .backlogId(backlog.getWorkItemId())
                    .uri(envProperties.getWebUri() + "task/detail/" + backlog.getWorkItemId() + "?targetTenantId=" + backlog.getTargetTenantId())
                    .name(backlog.getName())
                    .withName(backlog.getWithName())
                    .subName(backlog.getSubName())
                    .projectName(backlog.getTaskName())
                    .projectTargetName(backlog.getTaskTargetName())
                    .projectStartTime(backlog.getTaskStartTime())
                    .projectEndTime(backlog.getTaskEndTime())
                    .projectImportance(backlog.isImportance())
                    .importance(backlog.isImportance())
                    .content(backlog.getContent())
                    .importanceReadOnly(backlog.isImportanceReadOnly())
                    .startTime(backlog.getStartTime())
                    .endTime(backlog.getEndTime())
                    .tmTaskId(backlog.getTmTaskId())
                    .tmActivityId(backlog.getTmActivityId())
                    .type(backlog.getType())
                    .tenantId(backlog.getTenantId())
                    .targetTenantId(backlog.getTargetTenantId())
                    .actionDefined(backlog.isActionDefined())
                    .tmPattern(backlog.getTmPattern())
                    .tmCategory(backlog.getTmCategory())
                    .isOwner(backlog.getIsOwner())
                    .ownerUserId(backlog.getOwnerUserId())
                    .ownerUserName(backlog.getOwnerUserName())
                    .readCount(null != backlog.getReadCount() ? backlog.getReadCount() : 0)
                    .exception(null != backlog.getException() ? backlog.getException() : false)
                    .overdue(null != backlog.getOverdue() ? backlog.getOverdue() : false)
                    .overdueMinutes(null != backlog.getOverdueMinutes() ? backlog.getOverdueMinutes() : 0)
                    .overdueHours(null != backlog.getOverdueHours() ? backlog.getOverdueHours() : 0)
                    .overdueDays(null != backlog.getOverdueDays() ? backlog.getOverdueDays() : 0)
                    .approvalStateCode(backlog.getApprovalStateCode())
                    .approvalState(backlog.getApprovalState())
                    .modifyDate(backlog.getModifyDate())
                    .emergency(backlog.isEmergency())
                    .eocCode(backlog.getEocCode())
                    .eocName(backlog.getEocName())
                    .eocType(backlog.getEocType())
                    .agentBeginDate(backlog.getAgentBeginDate())
                    .agentEndDate(backlog.getAgentEndDate())
                    .sourceTenantName(backlog.getSourceTenantName())
                    .proxyToken(backlog.getProxyToken())
                    .finishActionId(backlog.getFinishActionId())
                    .tmActivityName(backlog.getTmActivityName())
                    .tmActivityWithName(backlog.getTmActivityWithName())
                    .merge(backlog.getMerge())
                    .dataFrom(backlog.getDataFrom())
                    .dataChangeRead((backlog.getDataChangeRead() != null && backlog.getDataChangeRead() == 0))
                    .createTime(backlog.getCreateTime())
                    .build();
            if (backlog.getModifyDate() != null && finalLastTime != null && finalLastTime > 0L && finalLastTime < backlog.getModifyDate().atZone(ZoneId.systemDefault()).toEpochSecond()) {
                taskCardDTO.setHasDataChanged(true);
            }

            if (ManualTaskUtil.isManualTask(backlog.getTmTaskId(), backlog.getTmActivityId())) {
                taskCardDTO.setProjectStartTime(backlog.getStartTime());
                taskCardDTO.setProjectEndTime(backlog.getEndTime());
            }

            //如果存在代理token，则使用租户的虚拟token
            if (!StringUtils.isEmpty(taskCardDTO.getProxyToken())) {
                taskCardDTO.setProxyToken(tenantTokenService.queryVirtualToken(taskCardDTO.getTenantId()));
            }

            if (activities.stream().filter(a -> Objects.equals(taskCardDTO.getTmActivityId(), a.get("activityCode"))).count() == 0) {
                if (!StringUtils.isEmpty(taskCardDTO.getTmTaskId()) && !StringUtils.isEmpty(taskCardDTO.getTmActivityId())) {
                    Map activityMap = new HashMap();
                    activityMap.put("taskCode", taskCardDTO.getTmTaskId());
                    activityMap.put("activityCode", taskCardDTO.getTmActivityId());
                    activities.add(activityMap);
                }
            }
            //合并任务的时间区间
            PtmMergeBacklogBO ptmMergeBacklogBO = mergeBacklogDTOMap.get(taskCardDTO.getBacklogId());
            if (BooleanUtils.isTrue(taskCardDTO.getMerge()) && ptmMergeBacklogBO != null) {
                taskCardDTO.setPlanEndTimeMin(ptmMergeBacklogBO.getPlanEndTimeMin());
                taskCardDTO.setPlanEndTimeMax(ptmMergeBacklogBO.getPlanEndTimeMax());
            }

            log.info(" 【taskCard data】: {}", taskCardDTO);
            list.add(taskCardDTO);
        });

        if (isGetReassignAble) {
            try {
                List<TmActivityResponseDTO> activityDefineCacheList = themeMapService.getBatchActivityAssigns(activities);
                list.forEach(task -> {
                    Optional<BacklogBO> optional = backlogList.stream().filter(bk -> Objects.equals(bk.getWorkItemId(), task.getBacklogId())).findFirst();
                    if (optional.isPresent() && optional.get().getOverdueWorkitemId() != null && optional.get().getOverdueWorkitemId() > 0L) {
                        //逾时任务不能转派
                        task.setReAssignAble(false);
                    } else {
                        Optional<TmActivityResponseDTO> responseDTO = activityDefineCacheList.stream().filter(x -> Objects.equals(x.getActivityId(), task.getTmActivityId())).findFirst();
                        if (responseDTO.isPresent()) {
                            task.setReAssignAble(responseDTO.get().getAssignConfig() != null ? responseDTO.get().getAssignConfig().getAssignAble() : false);
                        } else {
                            //没查到定义，默认不可转派
                            task.setReAssignAble(false);
                        }
                    }
                });
            } catch (Exception e) {
                //查询转派信息应该不影响主要的任务列表返回，所以吃掉异常，记录日志
                log.warn("[任务列表可否转派] 出现异常：{},查询内容：{}", e, activities);
            }
        }
        return list;
    }

    private String getSubName(BacklogBO backlogBO) {
        if ("APPROVAL".equals(backlogBO.getTmCategory())) {
            try {
                TmActivityResponseDTO themeActivity = themeMapService.getActivityAction(backlogBO.getTmTaskId(), backlogBO.getTmActivityId(), TmPageName.TASK_CARD_NAME.getValue());
                if (themeActivity.getLang() != null && themeActivity.getLang().containsKey("startApproveActivityName")) {
                    Map startApproveActivityNameMap = (Map) themeActivity.getLang().get("startApproveActivityName");
                    if (startApproveActivityNameMap != null) {
                        return String.format("{\"zh_TW\":\"%s\",\"en_US\":\"%s\",\"zh_CN\":\"%s\"}", startApproveActivityNameMap.getOrDefault("zh_TW", themeActivity.getStartApproveActivityName()), startApproveActivityNameMap.getOrDefault("en_US", themeActivity.getStartApproveActivityName()), startApproveActivityNameMap.getOrDefault("zh_CN", themeActivity.getStartApproveActivityName()));
                    }
                }
                return themeActivity.getStartApproveActivityName();
            } catch (Exception ex) {
                return "";
            }
        }
        return null;
    }
}
