package com.digiwin.athena.abt.application.service.abt.migration.ptm.impl;

import com.digiwin.athena.abt.application.configuration.EnvProperties;
import com.digiwin.athena.abt.application.dto.migration.abt.sendmsg.WorkItemMessageDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.backlog.CardAbstractDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.eoc.ProxyUserResp;
import com.digiwin.athena.abt.application.dto.migration.atmc.iam.UserDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.message.MessageDO;
import com.digiwin.athena.abt.application.dto.migration.atmc.thememap.TmAppDTO;
import com.digiwin.athena.abt.application.service.abt.migration.ptm.PtmTransformService;
import com.digiwin.athena.abt.application.service.abt.migration.ptm.SendMessageService;
import com.digiwin.athena.abt.application.service.abt.migration.ptm.ShareService;
import com.digiwin.athena.abt.application.service.abt.migration.ptm.WorkItemMessageService;
import com.digiwin.athena.abt.application.service.abt.migration.route.RouteService;
import com.digiwin.athena.abt.application.service.atmc.migration.backlog.CommonBacklogAbstractService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.emc.EmcService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.eoc.EocService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.iam.IamService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.iam.UserService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.ptm.PtmService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.thememap.ThemeMapService;
import com.digiwin.athena.abt.core.meta.enums.TypeEnum;
import com.digiwin.athena.abt.core.meta.enums.message.CategoryEnum;
import com.digiwin.athena.abt.core.meta.enums.message.MsgImportanceEnum;
import com.digiwin.athena.abt.core.meta.enums.message.SubTypeCategoryEnum;
import com.digiwin.athena.abt.infrastructure.mapper.biz.migration.atmc.TaskMapper;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.ReassignTrace;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.Route;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.Task;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.constant.LogConstant;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.TimeUtils;
import com.google.common.collect.ImmutableMap;
import com.jugg.agile.framework.core.config.JaProperty;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.*;

import static com.digiwin.athena.abt.core.meta.constants.MessageTypeConstant.*;

/**
 * @author lzw
 * @date 2024/4/28
 * @description:
 **/
@Service
@Slf4j
public class WorkItemMessageServiceImpl implements WorkItemMessageService {

    private static final Logger logger = LoggerFactory.getLogger(WorkItemMessageServiceImpl.class);

    @Autowired
    private UserService userService;

    @Autowired
    private SendMessageService sendMessageService;

    @Autowired
    protected EocService eocService;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private RouteService routeService;

    @Autowired
    private PtmTransformService ptmTransformService;

    @Autowired
    private CommonBacklogAbstractService backlogHandleService;

    @Autowired
    private PtmService ptmService;

    @Autowired
    private ShareService shareService;

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private TaskMapper taskMapper;

    @Autowired
    private ThemeMapService themeMapService;

    @Autowired
    private IamService iamService;

    @Autowired
    private EmcService emcService;

    // 转派创建
    private final Integer CREATE_TYPE_REASSIGN = 1;
    // 转派创建
    private final Integer CREATE_TYPE_HANDOVER = 10;
    /**
     * 推送钉钉消息pc端url
     */
    public static final String MESSAGE_DINGTALK_URL = "%slogin/dingtalk-login?appId=%s&corpId=%s&dingLog=0&routerLink=";

    /**
     * 推送钉钉消息app端 url
     */
    public static final String MESSAGE_DINGTALK_APP_URL = "%s/#/?appId=%s&appName=%s&corpId=%s&sourceType=ding_talk&dataId=%s&targetTenantId=%s";

    /**
     * 项目卡详情地址，前端地址
     * /project/share/370465551900736/BRV1jMNof6KKu0Y?engineType=TaskEngine
     * 370465551900736 任务id
     * BRV1jMNof6KKu0Y 分享code
     */
    public static final String PROJECT_TASK_DETAIL = "/project/share/%s/%s&engineType=TaskEngine";

    /**
     * 任务卡详情地址，pc前端地址
     * /task/detail/370488915116096?shareCode=rPqQNBCRdE0mCJ2
     * 370488915116096 任务id
     */
    public static final String TASK_DETAIL_SHARE = "task/detail/%s?shareCode=%s";

    /**
     * 待办创建类型
     * 1：转派产生
     * 10：交接创建
     */
    private final List<Integer> workItemCreateTypes = Arrays.asList(CREATE_TYPE_REASSIGN, CREATE_TYPE_HANDOVER);

    private final Map<Integer, Integer> operationTypeMap = ImmutableMap.of(CREATE_TYPE_REASSIGN, 0, CREATE_TYPE_HANDOVER, 2);

    @Override
    public void sendWorkItemMessage(WorkItemMessageDTO message, AuthoredUser authoredUser) {
        String backlogId = String.valueOf(message.getBacklog() == null ? 0L : message.getBacklog().getBacklogId());
        LogDto logDto = new LogDto("发送任务待办消息，待办Id：" + backlogId, authoredUser.getTenantId() + LogConstant.TRACE_SEPARATOR + backlogId);
        log.info(logDto.toString());
        List<WorkItemMessageDTO.WorkItem> workitemList = message.getWorkItemList();
        Boolean isEnabled = JaProperty.getBoolean("athena.messageCenter.enable", Boolean.FALSE);
        log.info("[sendWorkItemMessage] backlogId = {}, athena.messageCenter.enable = {}", backlogId, isEnabled);
        for (WorkItemMessageDTO.WorkItem workItem : workitemList) {
            if (isEnabled) {
                sendWorkitemMessage(message, workItem, authoredUser);
            } else {
                recordMsgWorkitem(message, workItem, authoredUser);
            }
        }
        //如果是任务异常，还要发送项目异常消息
        if (BooleanUtils.isTrue(message.getTask().getHasException())) {
            // 未提供模板，此消息暂不迁移至消息中心
            recordProjectExceptionMsg(message, authoredUser);
        }
    }

    private void sendWorkitemMessage(WorkItemMessageDTO message, WorkItemMessageDTO.WorkItem workItem, AuthoredUser authoredUser) {
        logger.info("sendWorkitemMessage by messageCenter message:{} workItem:{} ,authoredUser:{}", message, workItem, authoredUser.getUserId());
        // 查询用户信息，并发送信息
        UserDTO performer = userService.query(workItem.getPerformerId(), authoredUser.getToken());
        //获取给执行者发送的消息
        MessageDO performerMessage = packageMessageDO(message, performer, authoredUser, workItem);
        logger.info("sendWorkitemMessage by messageCenter performerMessage:{} ", JsonUtils.objectToString(performerMessage));
        // 组装发送消息参数
        Map<String, Object> data = buildMessageData(message, authoredUser, performerMessage);
        logger.info("sendWorkitemMessage by messageCenter data:{} ", JsonUtils.objectToString(data));
        // 根据场景，发送消息  任务新增，交接 ，转派
        if (CREATE_TYPE_REASSIGN.equals(workItem.getCreateType())) {
            logger.info("sendWorkitemMessage by messageCenter ATHENA_TASK_REASSIGN_SCENE start");
            sendMessageService.sendMsg4AllChannel(authoredUser.getToken(), String.valueOf(data.get("tenantId")), String.valueOf(data.get("appCode")), "ATHENA_TASK_REASSIGN_SCENE", data);
        }
        else if (CREATE_TYPE_HANDOVER.equals(workItem.getCreateType())) {
            logger.info("sendWorkitemMessage by messageCenter ATHENA_TASK_HANDOVER_SCENE start");
            sendMessageService.sendMsg4AllChannel(authoredUser.getToken(), String.valueOf(data.get("tenantId")), String.valueOf(data.get("appCode")), "ATHENA_TASK_HANDOVER_SCENE", data);
        }
        else {
            logger.info("sendWorkitemMessage by messageCenter ATHENA_TASK_CREATE_SCENE start");
            sendMessageService.sendMsg4AllChannel(authoredUser.getToken(), String.valueOf(data.get("tenantId")), String.valueOf(data.get("appCode")), "ATHENA_TASK_CREATE_SCENE", data);
        }
        // APP 消息推波
        logger.info("sendWorkitemMessage by messageCenter sendToAppClient start");
        sendMessageService.sendToAppClient(authoredUser, performerMessage);


        //查询代理人信息，发送消息
        ProxyUserResp proxyInfo = eocService.getProxyUser(workItem.getPerformerId(), authoredUser.getToken());
        if (null == proxyInfo || null == proxyInfo.getUserId() || proxyInfo.getUserId().equals(performer.getId())) {
            return;
        }
        UserDTO proxyUser = userService.query(proxyInfo.getUserId(), authoredUser.getToken());
        //获取给代理人发送的消息
        MessageDO proxyPerformerMessage = packageMessageDO(message, proxyUser, authoredUser, workItem);
        proxyPerformerMessage.setSubTypeCategory(SubTypeCategoryEnum.PROXY_REMIND.name());
        // 组装发送消息参数
        Map<String, Object> dataProxy = buildMessageData(message, authoredUser, proxyPerformerMessage);
        logger.info("sendWorkitemMessage by messageCenter data:{} ", JsonUtils.objectToString(dataProxy));
        // 根据场景，发送消息  任务新增，交接 ，转派
        if (CREATE_TYPE_REASSIGN.equals(workItem.getCreateType())) {
            logger.info("sendWorkitemMessage by messageCenter proxy ATHENA_TASK_REASSIGN_SCENE start");
            sendMessageService.sendMsg4AllChannel(authoredUser.getToken(), String.valueOf(dataProxy.get("tenantId")), String.valueOf(dataProxy.get("appCode")), "ATHENA_TASK_REASSIGN_SCENE", dataProxy);
        }
        else if (CREATE_TYPE_HANDOVER.equals(workItem.getCreateType())) {
            logger.info("sendWorkitemMessage by messageCenter proxy ATHENA_TASK_HANDOVER_SCENE start");
            sendMessageService.sendMsg4AllChannel(authoredUser.getToken(), String.valueOf(dataProxy.get("tenantId")), String.valueOf(dataProxy.get("appCode")), "ATHENA_TASK_HANDOVER_SCENE", dataProxy);
        }
        else {
            logger.info("sendWorkitemMessage by messageCenter proxy ATHENA_TASK_CREATE_SCENE start");
            sendMessageService.sendMsg4AllChannel(authoredUser.getToken(), String.valueOf(dataProxy.get("tenantId")), String.valueOf(dataProxy.get("appCode")), "ATHENA_TASK_CREATE_SCENE", dataProxy);
        }
        // APP 消息推波
        logger.info("sendWorkitemMessage by messageCenter proxy sendToAppClient start");
        sendMessageService.sendToAppClient(authoredUser, proxyPerformerMessage);
    }

    private Map<String, Object> buildMessageData(WorkItemMessageDTO message, AuthoredUser authoredUser, MessageDO performerMessage) {

        Map<String, Object> data = new HashMap<>();
        data.put("tenantId", performerMessage.getTenantId());
        // 执行人
        data.put("performer", performerMessage.getUserId());
        data.put("type", performerMessage.getType());
        data.put("subType", performerMessage.getSubType());
        data.put("subTypeCategory", performerMessage.getSubTypeCategory());
        data.put("category", performerMessage.getCategory());
        data.put("importance", performerMessage.getImportance());

        // 获取应用信息
        List<TmAppDTO> tmAppDTOS = themeMapService.getAppInfoByCodeAndType("task", Collections.singletonList(performerMessage.getSubType()));
        String appCode = "";
        String appName = "";
        if (CollectionUtils.isNotEmpty(tmAppDTOS)) {
            appCode = tmAppDTOS.get(0).getAppCode();
            appName = tmAppDTOS.get(0).getAppName();
            data.put("appCode", appCode);
        }

        if (null != performerMessage.getContent()) {
            JSONObject content = (JSONObject) performerMessage.getContent();
            data.put("id", content.get("id"));
            data.put("operationType", content.get("operationType"));
            data.put("operator", content.get("operator"));


            // 发送钉钉参数
            data.put("dataUrl", content.get("emailMsg"));
            //消息详情地址,app点击跳转地址
            String corpid = iamService.getTenantCorpid();
            if (StringUtils.isBlank(corpid)){
                corpid = performerMessage.getTenantId();
            }
            String appUri = getAppShareUrl(String.valueOf(content.get("id")), appName, appCode, performerMessage.getTenantId(), corpid);
            data.put("appUri", appUri);
            //消息详情地址,获取pc分享地址
            String messageUri = getPcShareUrl(authoredUser, String.valueOf(content.get("id")), performerMessage.getType(), appCode, corpid);
            data.put("messageUri", messageUri);

            // 企业微信参数
            List<Map<String, Object>> wecomConfigs = emcService.getWecomConfig();
            if (CollectionUtils.isNotEmpty(wecomConfigs)) {
                try {
                    Map<String, Object> wecomConfig = wecomConfigs.get(0);
                    String agentid = String.valueOf(wecomConfig.get("appId"));
                    String corpId = String.valueOf(wecomConfig.get("corpId"));
                    String redirect_uri = envProperties.getSemcUri() + "/semc/router/routingDistribution?taskId=" + content.get("id") + "&appId=" + agentid + "&tenantId=" + performerMessage.getTenantId();
                    String encode = URLEncoder.encode(redirect_uri, "utf-8");
                    String weChatUrl = "https://open.weixin.qq.com/connect/oauth2/authorize?appid=" + corpId + "&redirect_uri=" + encode + "&response_type=code&scope=snsapi_base&state=STATE&agentid=" + agentid + "#wechat_redirect";
                    String weComMsg = String.format("<a href='%s'>%s</a>", weChatUrl, content.get("emailMsg"));
                    data.put("weComMsg", weComMsg);
                } catch (Exception e) {
                    log.error("发生消息，组装企业微信如参数失败", e);
                }
            }

            if (null != content.get("messageFiled")) {
                JSONObject messageFiled = (JSONObject) content.get("messageFiled");
                data.put("taskName", messageFiled.get("tmActivityName"));
                data.put("taskCode", performerMessage.getSubType());
                if (messageFiled.containsKey("summaryLayout") && null !=  messageFiled.get("summaryLayout")){
                    String summaryLayout = String.valueOf(messageFiled.get("summaryLayout")).replaceAll("<br\\s?/?>", "；");
                    data.put("summaryLayout", summaryLayout);
                }
            }

            // 邮箱参数
            String linkUrl = generateLinkUrl(authoredUser, performerMessage.getType(), String.valueOf(content.get("id")), performerMessage.getTenantId());
            if (StringUtils.isNotBlank(linkUrl)) {
                data.put("linkUrl", linkUrl);
            }
            data.put("content", content.get("emailMsg"));
        }
        data.put("endTime", message.getTask().getPlanEndTime());

        return data;
    }

    /**
     * 获取app分享地址
     *
     * @param taskId
     * @param appName
     * @return https://athena-mobile-inner-test.apps.digiwincloud.com.cn/#/?appId=TBDS
     * &appName=试研北斗&corpId=ding7ab87c867d9170e535c2f4657eb6378f
     * &sourceType=ding_talk&dataId=390488111022272
     * &targetTenantId=ding7ab87c867d9170e535c2f4657eb6378f
     */
    public String getAppShareUrl(String taskId, String appName, String appId, String tenantId, String corpid) {
        String messageUrl = String.format(MESSAGE_DINGTALK_APP_URL, envProperties.getMobileWebUri(), appId, appName, corpid, taskId, tenantId);
        //地址进行URLEncoder 编码
        try {
            messageUrl = URLEncoder.encode(messageUrl, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error("getAppShareUrl URLEncoder报错， e：{}", e);
        }
        return messageUrl;
    }

    /**
     * 获取pc分享地址
     *
     * @param taskId
     * @param msgType
     * @return https://mui-72005592-test.digiwincloud.com.cn/login/dingtalk-login?
     * appId=middleware&corpId=ding7ab87c867d9170e535c2f4657eb6378f
     * &dingLog=2&routerLink=project-task-detail/566521006903360
     */
    public String getPcShareUrl(AuthoredUser user, String taskId, String msgType, String appId, String corpId) {
        String shareUrl = "";
        String messageUrl = String.format(MESSAGE_DINGTALK_URL, envProperties.getWebUri(), appId, corpId);
        //转小写
        msgType = msgType.toLowerCase();
        //转换类型，项目用project，任务卡用task 获取分享code
        String shareMsgType = msgType.equals("activity") ? "task" : "project";
        //请求atmc获取分享code
//        String shareCode = atmcService.getShareCode(taskId, shareMsgType);
        String shareCode = shareService.createSharingCode(user, shareMsgType, Long.valueOf(taskId));
        //项目卡分享
        if (msgType.equals("task")) {
            shareUrl = String.format(PROJECT_TASK_DETAIL, taskId, shareCode);
        }
        //任务卡分享
        if (msgType.equals("activity")) {
            shareUrl = String.format(TASK_DETAIL_SHARE, taskId, shareCode);
        }
        messageUrl = messageUrl + shareUrl;
        //地址进行URLEncoder 编码
        try {
            messageUrl = URLEncoder.encode(messageUrl, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error("senDingDingMessage URLEncoder报错， e：{}", e);
        }
        return messageUrl;
    }

    public String generateLinkUrl(AuthoredUser user, String type, String id, String targetTenantId) {
        //获取消息的类型
        String urlDetail = "";
        if ("activity".equals(type) || "task".equals(type)) {
            String defaultType;
            if ("activity".equals(type)) {
                defaultType = "task";
                urlDetail = "task/detail/" + id + "?targetTenantId=" + targetTenantId + "&shareCode=";
            } else {
                defaultType = "project";
                urlDetail = "project/share/" + id + "/";
            }

            //通过分享任务卡或者项目卡的方式获取分享码
            String shareCode = shareService.createSharingCode(user, defaultType, Long.valueOf(id));
            Task task = taskMapper.selectPartialById(Long.valueOf(id));
            if (task == null) {
                //获取分享的url链接，如果task表里面不存在，就是任务引擎
                urlDetail = envProperties.getWebUri() + urlDetail + shareCode + "?engineType=TaskEngine";
                if (urlDetail.contains("project")) {
                    urlDetail = urlDetail + "&targetTenantId=" + targetTenantId;
                }
            } else {
                urlDetail = envProperties.getWebUri() + urlDetail + shareCode;
                if (urlDetail.contains("project")) {
                    urlDetail = urlDetail + "?targetTenantId=" + targetTenantId;
                }
            }

        }
        return urlDetail;
    }


    private void recordMsgWorkitem(WorkItemMessageDTO message, WorkItemMessageDTO.WorkItem workItem, AuthoredUser authoredUser) {
        logger.info("recordMsgWorkitem message:{} workItem:{} ,authoredUser:{}", message, workItem, authoredUser.getUserId());
        //查询用户信息，并发送信息
        UserDTO performerUser = userService.query(workItem.getPerformerId(), authoredUser.getToken());
        //获取给执行者发送的消息
        MessageDO performerMessage = packageMessageDO(message, performerUser, authoredUser, workItem);
        logger.info("send workitem msg to performer with message:{}", performerMessage);
        String backlogId = String.valueOf(message.getBacklog() == null ? 0L : message.getBacklog().getBacklogId());
        LogDto logDto = new LogDto("给执行者发送消息，待办Id：" + backlogId, authoredUser.getTenantId() + LogConstant.TRACE_SEPARATOR + backlogId);
        log.info(logDto.toString());
        sendMessageService.saveNewMessage(authoredUser, performerMessage);


        //查询代理人信息，发送消息
        ProxyUserResp proxyInfo = eocService.getProxyUser(workItem.getPerformerId(), authoredUser.getToken());
        if (null == proxyInfo || null == proxyInfo.getUserId() || proxyInfo.getUserId().equals(performerUser.getId())) {
            return;
        }
        UserDTO proxyUser = userService.query(proxyInfo.getUserId(),authoredUser.getToken());
        MessageDO proxyMessage = packageMessageDO(message,proxyUser,authoredUser,workItem);
        // 代理人在代理期间收到的新增任务，需要打上代理标签
        proxyMessage.setSubTypeCategory(SubTypeCategoryEnum.PROXY_REMIND.name());
        logger.info("send workitem msg to proxy user with message:{}",performerMessage );
        LogDto logDtoProxy = new LogDto("给代理人发送消息，待办Id：" + backlogId, authoredUser.getTenantId() + LogConstant.TRACE_SEPARATOR + backlogId);
        log.info(logDtoProxy.toString());
        sendMessageService.saveNewMessage(authoredUser, proxyMessage);
    }

    private void recordProjectExceptionMsg(WorkItemMessageDTO message, AuthoredUser authoredUser) {
        UserDTO userDTO = userService.getTargetUser(message.getProject().getPersonInCharge(), authoredUser.getToken());
        if (null == userDTO) {
            return;
        }

        MessageDO toRecordMsg = new MessageDO();

        toRecordMsg.setUserId(userDTO.getId());
        toRecordMsg.setTenantId(authoredUser.getTenantId());
        //发送消息
        toRecordMsg.setSource("athena");

        toRecordMsg.setType(TypeEnum.TASK.getValue());
        toRecordMsg.setSubType(message.getProject().getProjectDefCode());

//        String langName = userService.getUserLangNameByUserId(userDTO.getId(), authoredUser.getTenantId(), authoredUser.getToken());
        String langName = userService.getUserLangMetadataAllTenant(userDTO.getId(), authoredUser.getToken());
        if (org.springframework.util.StringUtils.isEmpty(langName)) {
            langName = "zh_TW";
        }
        String taskName = MessageUtils.getMessageByLanguage(message.getProject().getName(), langName);

        String backlogName = MessageUtils.getMessageByLanguage(message.getTask().getTaskName(), langName);
        // 【09.26-09.30厂内物流】项目进料检验任务出现数据异常，进入异常排除，请关注~
        // 【%s-%s%s】%s任务出现数据异常，进入异常排除，请关注~
        String title = String.format(messageUtils.getMessageByLangName("msg.record.task.exception.title", langName),
                TimeUtils.format(message.getProject().getStartTime(), "MM.dd"),
                TimeUtils.format(message.getProject().getEndTime(), "MM.dd"),
                taskName,
                backlogName);

        toRecordMsg.setSubTypeCategory(SubTypeCategoryEnum.EXCEPTION_OR_OVERDUE.name());
        toRecordMsg.setCategory(CategoryEnum.EXCEPTION.name());
        toRecordMsg.setImportance(MsgImportanceEnum.IMPORTANT.getImportance());

        toRecordMsg.setTitle(title);

        toRecordMsg.setSendDate(LocalDateTime.now());
        toRecordMsg.setStartTime(message.getProject().getStartTime());
        toRecordMsg.setEndTime(message.getProject().getEndTime());

        JSONObject content = new JSONObject();
        content.put("id", String.valueOf(message.getProject().getProjectId()));
        content.put("mainTaskId", String.valueOf(message.getProject().getProjectCardId()));
        content.put("routeId", getRouteIdBySubTaskId(message.getProject().getProjectId()));
        content.put("engineType", message.getProject().getEngineType());

        content.put("title", title);
        content.put("name", taskName);
        content.put("startTime", formatDate(toRecordMsg.getStartTime()));
        content.put("endTime", formatDate(toRecordMsg.getEndTime()));
        content.put("status", 0);

        // 异常
        String msg = "";

        content.put("msg", msg);
        toRecordMsg.setContent(content);

        sendMsgToPerformerAndAgent(authoredUser, userDTO, message.getProject().getPersonInCharge(), toRecordMsg);
    }

    /**
     * 获取【子项目】其中一条路线
     *
     * @param subTaskId
     * @return
     */
    private Long getRouteIdBySubTaskId(Long subTaskId) {
        Route route = this.routeService.getRouteByTaskId(subTaskId);
        if (route == null) {
            route = ptmTransformService.getRouteByProjectId(subTaskId);
        }
        return null != route ? route.getId() : null;
    }

    private String formatDate(LocalDateTime dateTime) {
        return TimeUtils.format(dateTime, TimeUtils.DEFAULT_FORMAT);
    }

    /**
     * 消息发送给执行人（被代理人）和代理人
     *
     * @param authoredUser
     * @param userDTO
     * @param performerId
     * @param newMsg
     */
    protected void sendMsgToPerformerAndAgent(AuthoredUser authoredUser, UserDTO userDTO, String performerId, MessageDO newMsg) {
        log.info("saveNewMessage1: {}", newMsg);
        sendMessageService.saveNewMessage(authoredUser, newMsg);

        // 代理人和执行者（被代理人）不相同，需要再发送消息给被代理人
        if (!StringUtils.equals(performerId, userDTO.getId())) {
            log.info("saveNewMessage2: {}", newMsg);
            newMsg.setUserId(performerId);
            newMsg.setTenantId(authoredUser.getTenantId());
            sendMessageService.saveNewMessage(authoredUser, newMsg);
        }
    }

    /**
     * 封装需要发送的提醒消息
     *
     * @param message      消息原始内容
     * @param authoredUser 用户信息
     * @param workItem     workItem
     * @return 消息内容
     */
    private MessageDO packageMessageDO(WorkItemMessageDTO message, UserDTO userDTO, AuthoredUser authoredUser, WorkItemMessageDTO.WorkItem workItem) {
        MessageDO toRecordMsg = new MessageDO();

        toRecordMsg.setUserId(userDTO.getId());
        toRecordMsg.setState(0);
        if (null != message.getBacklog() && StringUtils.isNotBlank(message.getBacklog().getTenantId())) {
            toRecordMsg.setTenantId(message.getBacklog().getTenantId());
        } else {
            toRecordMsg.setTenantId(authoredUser.getTenantId());
        }
        //发送消息
        toRecordMsg.setSource("athena");
        //查询用户的语言别
        String langName = userService.getUserLangMetadataAllTenant(userDTO.getId(), authoredUser.getToken());
        if (org.springframework.util.StringUtils.isEmpty(langName)) {
            langName = "zh_TW";
        }

        String projectName = MessageUtils.getMessageByLanguage(message.getProject().getName(), langName);

        toRecordMsg.setType(TypeEnum.ACTIVITY.getValue());
        toRecordMsg.setSubType(message.getTask().getTmActivityId());

        String backlogName = MessageUtils.getMessageByLanguage(message.getTask().getTaskName(), langName);
        String tmActivityName = MessageUtils.getMessageByLanguage(message.getTask().getTmActivityName(), langName);

        // 是否有异常
        String title = "";
        String endTime = TimeUtils.format(message.getTask().getPlanEndTime(), "MM/dd");
        JSONObject content = new JSONObject();
        if (null != message.getTask().getHasException() && message.getTask().getHasException()) {
            // 【09.26-09.30工单备料】
//            title = String.format("【%s-%s%s】", startTime, endTime, taskName);
            toRecordMsg.setSubTypeCategory(SubTypeCategoryEnum.EXCEPTION_OR_OVERDUE.name());
            toRecordMsg.setCategory(CategoryEnum.EXCEPTION.name());
            toRecordMsg.setImportance(MsgImportanceEnum.IMPORTANT.getImportance());
        } else if (1 == message.getProject().getEmergency()) {
            // 紧急任务
//            title = String.format("【%s-%s%s】", startTime, endTime, taskName);
            toRecordMsg.setSubTypeCategory(SubTypeCategoryEnum.ATHENA_REMIND.name());
            toRecordMsg.setCategory(CategoryEnum.EMERGENCY.name());
            toRecordMsg.setImportance(MsgImportanceEnum.IMPORTANT.getImportance());
        } else {
//            title = String.format(messageUtils.getMessageByLangName("msg.record.backlog.performer.notice.msg", langName),
//                    TimeUtils.format(message.getTask().getPlanEndTime(), "MM/dd"),
//                    backlogName);
            toRecordMsg.setSubTypeCategory(SubTypeCategoryEnum.ATHENA_REMIND.name());
            toRecordMsg.setCategory(CategoryEnum.NEW.name());
            toRecordMsg.setImportance(MsgImportanceEnum.UNIMPORTANT.getImportance());
            //比较当前消息是新增还是更新
            String backlogId = message.getMessageType();
            if (WORKITEM_MESSAGE_TYPE_UPDATE.equals(backlogId)) {
                toRecordMsg.setCategory(CategoryEnum.UPDATE.name());
            }
            String activityName = "";
            if (tmActivityName.equals(backlogName)) {
                activityName = tmActivityName;
            } else {
                activityName = tmActivityName + "-" + backlogName;
            }
            //组装前端拼接消息所需字段
            JSONObject messageFiled = new JSONObject();
            messageFiled.put("summaryTag", SUMMARY_NO);
            //摘要信息
            String summary = "";
            try {
                CardAbstractDTO cardAbstractDTO = new CardAbstractDTO().setNeedHistoryInfo(false).setCount(30);
                Map summaryMap = backlogHandleService.getBacklogCardAbstractById(message.isMinSplit()?workItem.getId():workItem.getWorkItemId(), authoredUser, cardAbstractDTO,langName);
                log.info("摘要查询结果：{}", summaryMap);
                if (summaryMap != null) {
                    summary = (summaryMap.get("summaryLayout") != null) ? (summaryMap.get("summaryLayout").toString()) : "";
                    if (!StringUtils.isEmpty(summary)) {
                        messageFiled.put("summaryTag", SUMMARY_YES);
                    }
                }
            } catch (Exception e) {
                log.info("摘要查询异常：{}", e.getMessage(),e);
                messageFiled.put("summaryTag", SUMMARY_EXCEPTION);
            }

            messageFiled.put("createTime", TimeUtils.format(message.getTask().getCreateTime(), "MM/dd"));
            messageFiled.put("tmActivityName", activityName);
            messageFiled.put("endTime", endTime);
            messageFiled.put("projectName", projectName);
            messageFiled.put("summaryLayout", summary);


            // 转派和交接产生的任务标注操作人和操作类型
            setReassignOrHandoverInfo(workItem, messageFiled, content);

            content.put("messageFiled", messageFiled);

        }

        toRecordMsg.setTitle(title);

        toRecordMsg.setSendDate(LocalDateTime.now());
        toRecordMsg.setStartTime(message.getTask().getCreateTime());
        toRecordMsg.setEndTime(message.getTask().getPlanEndTime());

//        JSONObject content = new JSONObject();
        content.put("id", String.valueOf(workItem.getId()));
        content.put("title", title);

        content.put("name", backlogName);
        content.put("startTime", formatDate(toRecordMsg.getStartTime()));
        content.put("endTime", formatDate(toRecordMsg.getEndTime()));
        content.put("status", 0);

        // 异常
        String msg = null;
        String emailMsg = null;
        if (null != message.getTask().getHasException() && message.getTask().getHasException()) {
            if (tmActivityName.equals(backlogName)) {
                msg = String.format(messageUtils.getMessageByLangName("msg.record.backlog.exception", langName), tmActivityName, endTime, projectName);
                emailMsg = String.format(messageUtils.getMessageByLangName("msg.record.backlog.exception.mail", langName), tmActivityName, endTime, projectName);
            } else {
                msg = String.format(messageUtils.getMessageByLangName("msg.record.backlog.exception.type.and.name", langName), tmActivityName, endTime, backlogName, projectName);
                emailMsg = String.format(messageUtils.getMessageByLangName("msg.record.backlog.exception.type.and.name.mail", langName), tmActivityName, endTime, backlogName, projectName);
            }
        } else if (1 == message.getProject().getEmergency()) {
            // 状态为紧急，请关注～
            if (tmActivityName.equals(backlogName)) {
                msg = String.format(messageUtils.getMessageByLangName("msg.record.task.emergency.msg", langName), tmActivityName, endTime, projectName);
                emailMsg = String.format(messageUtils.getMessageByLangName("msg.record.task.emergency.msg.mail", langName), tmActivityName, endTime, projectName);
            } else {
                msg = String.format(messageUtils.getMessageByLangName("msg.record.task.emergency.msg.type.and.name", langName), tmActivityName, backlogName, endTime, projectName);
                emailMsg = String.format(messageUtils.getMessageByLangName("msg.record.task.emergency.msg.type.and.name.mail", langName), tmActivityName, backlogName, endTime, projectName);
            }
        } else {
            String messageType = message.getMessageType();
            if (tmActivityName.equals(backlogName)) {
                String msgKey = getLangKey("backlog.add.new.message.project.info", workItem.getCreateType());
                String emailMsgKey = getLangKey("backlog.add.new.message.project.info.mail", workItem.getCreateType());
                msg = String.format(messageUtils.getMessageByLangName(msgKey, langName), TimeUtils.format(message.getTask().getCreateTime(), "MM/dd"), tmActivityName, endTime, projectName, content.get("operator"));
                emailMsg = String.format(messageUtils.getMessageByLangName(emailMsgKey, langName), TimeUtils.format(message.getTask().getCreateTime(), "MM/dd"), tmActivityName, endTime, projectName, content.get("operator"));
                if (WORKITEM_MESSAGE_TYPE_UPDATE.equals(messageType)) {
                    msg = String.format(messageUtils.getMessageByLangName("backlog.update.new.message.project.info", langName), TimeUtils.format(message.getTask().getCreateTime(), "MM/dd"), tmActivityName, endTime, projectName);
                }
            } else {
                String msgKey = getLangKey("backlog.add.new.message.project.info.type.and.name", workItem.getCreateType());
                String emailMsgKey = getLangKey("backlog.add.new.message.project.info.type.and.name.mail", workItem.getCreateType());
                msg = String.format(messageUtils.getMessageByLangName(msgKey, langName), TimeUtils.format(message.getTask().getCreateTime(), "MM/dd"), tmActivityName, backlogName, endTime, projectName, content.get("operator"));
                emailMsg = String.format(messageUtils.getMessageByLangName(emailMsgKey, langName), TimeUtils.format(message.getTask().getCreateTime(), "MM/dd"), tmActivityName, backlogName, endTime, projectName, content.get("operator"));
                if (WORKITEM_MESSAGE_TYPE_UPDATE.equals(messageType)) {
                    msg = String.format(messageUtils.getMessageByLangName("backlog.update.new.message.project.info.type.and.name", langName), TimeUtils.format(message.getTask().getCreateTime(), "MM/dd"), tmActivityName, backlogName, endTime, projectName);
                }
            }
        }
        content.put("msg", msg);
        content.put("emailMsg", emailMsg);
        toRecordMsg.setContent(content);
        return toRecordMsg;
    }

    /**
     * 设置转派或者交接信息
     */
    private void setReassignOrHandoverInfo(WorkItemMessageDTO.WorkItem workItem, JSONObject messageFiled, JSONObject content) {
        if (!workItemCreateTypes.contains(workItem.getCreateType())) {
            return;
        }
        Integer operationType = operationTypeMap.get(workItem.getCreateType());
        messageFiled.put("operationType", operationType);
        content.put("operationType", operationType);
        // 获取转派人/交接人
        List<ReassignTrace> traceList = ptmService.selectReassignOrHandoverTrace(workItem.getFromWorkItemId(), operationType);
        String operator = CollectionUtils.isNotEmpty(traceList) ? traceList.get(0).getOperatorName() : null;
        messageFiled.put("operator", operator);
        content.put("operator", operator);
    }

    /**
     * @param langKey
     * @param createType
     * @return
     */
    private String getLangKey(String langKey, Integer createType) {
        if (Objects.equals(createType, CREATE_TYPE_REASSIGN)) {
            langKey += ".reassign";
        } else if (Objects.equals(createType, CREATE_TYPE_HANDOVER)) {
            langKey += ".handover";
        }
        return langKey;
    }
}
