package com.digiwin.athena.abt.application.service.abt.migration.quartz.impl;

import com.digiwin.athena.abt.application.dto.migration.abt.quartz.CronSchedulerJob;
import com.digiwin.athena.abt.application.service.abt.migration.quartz.JobManager;
import com.digiwin.athena.abt.core.meta.constants.JobConstants;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.quartz.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Map;

/**
 * JobManagerImpl Description
 *
 * @author majianfu
 * @date 2023/3/30
 * @since
 */
@Slf4j
@Service
public class JobManagerImpl implements JobManager {
    @Autowired
    private Scheduler scheduler;



    /**
     * {@inheritDoc}
     */
    @Override
    public void createCronJob(CronSchedulerJob job) {
        this.createCronJob(job, false);
    }

    @Override
    public void createCronJob(CronSchedulerJob job, boolean replace) {
        // 校验cron表达式
        JobDataMap jobDataMap = new JobDataMap();
        jobDataMap.put(JobConstants.JOB_DATA_KEY, job.getJobDataJson());

        // 设置定时任务执行方式
        CronScheduleBuilder scheduleBuilder = CronScheduleBuilder
                .cronSchedule(job.getCron());
        // 构建触发器trigger
        CronTrigger trigger = TriggerBuilder.newTrigger()
                .withIdentity(job.getName(), job.getGroup())
                .withSchedule(scheduleBuilder)
                .build();
        try {
            JobDetail jobDetail = JobBuilder.newJob(job.getJobClass())
                    .withIdentity(job.getName(), job.getGroup())
                    .setJobData(jobDataMap)
                    .build();

            CronExpression.validateExpression(job.getCron());
            if (replace) {
                log.info("update job: {}.{}: job-{}", job.getName(), job.getGroup(), job);
            } else {
                log.info("create job: {}.{}: job-{}", job.getName(), job.getGroup(), job);
            }
            scheduler.scheduleJob(jobDetail, Sets.newHashSet(trigger), replace);
        } catch (Exception ex) {
            log.error("create job: {}.{} error: job-{}, error-{}", job.getName(), job.getGroup(), job, ex.getMessage());
            throw BusinessException.create("add cron job failed.", ex);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean deleteJob(String jobName, String jobGroup) {
        try {
            boolean result = scheduler.deleteJob(JobKey.jobKey(jobName, jobGroup));
            log.info("delete job: {}.{} result: {}", jobName, jobGroup, result);
            return result;
        } catch (Exception ex) {
            log.error("delete job: {}.{} error: {}", jobName, jobGroup, ex.getMessage());
            throw BusinessException.create("delete job failed.", ex);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public <T extends Trigger> T getJobTrigger(String jobName) {
        try {
            return (T) scheduler.getTrigger(TriggerKey.triggerKey(jobName));
        } catch (SchedulerException ex) {
            log.error("get job trigger failed job: {} error: {}", jobName, ex.getMessage());
            throw BusinessException.create("get job trigger failed.", ex);
        }
    }



    /**
     * {@inheritDoc}
     */
    @Override
    public boolean exist(String jobName, String jobGroup) {
        try {
            return scheduler.checkExists(JobKey.jobKey(jobName, jobGroup));
        } catch (SchedulerException ex) {
            log.error("check job exist failed job: {}.{} error: {}", jobName, jobGroup, ex.getMessage());
            throw BusinessException.create("check job exist failed.", ex);
        }
    }

    @Override
    public void pauseScheduleJob(String jobName, String jobGroup) {
        JobKey jobKey = JobKey.jobKey(jobName,jobGroup);
        try {
            scheduler.pauseJob(jobKey);
        } catch (SchedulerException e) {
            log.error("pause job failed: {} {}",jobName,jobGroup,e);
            throw BusinessException.create("定时任务暂停失败:{}", e.getMessage());
        }
    }

    @Override
    public void resumeScheduleJob(String jobName, String jobGroup) {
        JobKey jobKey = JobKey.jobKey(jobName,jobGroup);
        try {
            scheduler.resumeJob(jobKey);
        } catch (SchedulerException e) {
            log.error("resume job failed: {} {}",jobName,jobGroup,e);
            throw BusinessException.create("启动定时任务失败:{}", e.getMessage());
        }
    }

    @Override
    public void runOnce(String jobName, String jobGroup) {
        JobKey jobKey = JobKey.jobKey(jobName,jobGroup);
        try {
            scheduler.triggerJob(jobKey);
        } catch (SchedulerException e) {
            log.error("trigger job failed: {} {}",jobName,jobGroup,e);
            throw BusinessException.create("触发定时任务失败:{}", e.getMessage());
        }
    }

    @Override
    public Map<String, Object> getDetailJobData(String jobName, String jobGroup) {
        JobKey jobKey = JobKey.jobKey(jobName,jobGroup);
        try {
            JobDetail jobDetail = scheduler.getJobDetail(jobKey);
            if(jobDetail != null){
                return jobDetail.getJobDataMap().getWrappedMap();
            }
            return null;
        } catch (SchedulerException e) {
            log.error("get jobDetail failed: {} {}",jobName,jobGroup,e);
            throw BusinessException.create("获取工作详情失败:{}", e.getMessage());
        }
    }
}
