package com.digiwin.athena.abt.application.service.abt.migration.retry.service;

import com.digiwin.athena.abt.application.dto.migration.abt.mq.PtmBacklogMessageDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.ptm.PtmMqMessageDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.ptm.PtmMqType;
import com.digiwin.athena.abt.application.service.abt.migration.retry.constant.MqRetryLogConstant;
import com.digiwin.athena.abt.application.service.abt.migration.retry.constant.MqRetryLogTypeEnum;
import com.digiwin.athena.abt.application.service.abt.migration.retry.dto.MqRetryLogBo;
import com.digiwin.athena.abt.core.meta.enums.PtmMqOperation;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;

import com.digiwin.athena.framework.mq.monitor.util.CommonUtil;
import com.digiwin.athena.framework.mq.retry.context.MQRetryContextHolder;
import com.digiwin.athena.framework.mq.retry.exception.RejectMQException;
import com.jugg.agile.framework.core.config.JaEnvProperty;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.dapper.alarm.adapter.qywx.JaQyWxAlarm;
import com.jugg.agile.framework.core.dapper.log.JaMDC;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.concurrent.TimeUnit;

import static com.digiwin.athena.abt.application.service.abt.migration.retry.constant.MqRetryLogConstant.*;


/**
 * PTM MQ 消息处理类
 */
@Slf4j
@Component
public class MqRetryLogService {
    private static final int MAX_RETRIES = 3;

    @Autowired
    @Qualifier("mongoTemplate")
    private MongoTemplate apiLogMongoTemplate;

    public MqRetryLogBo getRetryLog(Object msgObj) {
        String message = null;
        if (msgObj instanceof String) {
            message = (String) msgObj;
        } else if (msgObj instanceof byte[]) {
            message = new String((byte[]) msgObj);
        } else {
            throw new RejectMQException("该消息类型不支持消费");
        }

        PtmMqMessageDTO messageDTO = JsonUtils.jsonToObject(message, PtmMqMessageDTO.class);

        log.info("根据messageId={},mqQueue={},查询", messageDTO.getMessageId(), MQRetryContextHolder.getContext().getQueueBinding().getQueueName());
        MqRetryLogBo mqRetryLog = queryMqRetryLogByMessageId(messageDTO);
        if (null != mqRetryLog) {
            return mqRetryLog;
        }
        return insertMqRetryLog(messageDTO, message, MqRetryLogTypeEnum.PROCESSING.getType());
    }

    private MqRetryLogBo queryMqRetryLogByMessageId(PtmMqMessageDTO messageDTO) {
        try {
            Query query = Query.query(
                    Criteria.where(MqRetryLogConstant.MESSAGE_ID).is(messageDTO.getMessageId())
                            .and(MqRetryLogConstant.MQ_QUEUE).is(MQRetryContextHolder.getContext().getQueueBinding().getQueueName())
            );
            query.fields().include(MqRetryLogConstant.MQ_RETRY_COUNT)
                    .include(MqRetryLogConstant.LOCAL_RETRY_COUNT)
                    .include(MqRetryLogConstant.STATUS)
                    .include(MqRetryLogConstant.ID)
                    .include(MqRetryLogConstant.ERROR_MSG)
                    .include(MqRetryLogConstant.OPERATION)
                    .include(MqRetryLogConstant.PINPOINT_ID)
                    .include(MqRetryLogConstant.TYPE)
                    .include(MqRetryLogConstant.MESSAGE_ID)
                    .include(MqRetryLogConstant.MQ_QUEUE)
                    .include(MqRetryLogConstant.CREATE_TIME);
            return apiLogMongoTemplate.findOne(query, MqRetryLogBo.class, MqRetryLogConstant.MQ_RETRY_LOG);
        } catch (Exception e) {
            log.error("查询消息日志表失败，messageId:{}, msg:{}", messageDTO.getMessageId(), e.getMessage());
            return null;
        }
    }

    private MqRetryLogBo insertMqRetryLog(PtmMqMessageDTO messageDTO, String data, Integer status) {
        try {
            Date date = new Date();
            MqRetryLogBo mqRetryLog = new MqRetryLogBo();
            mqRetryLog.setMqRetrycount(0);
            mqRetryLog.setLocalRetrycount(0);
            mqRetryLog.setType(messageDTO.getType().getValue());
            mqRetryLog.setMessageId(messageDTO.getMessageId());
            mqRetryLog.setPinpointId(JaMDC.get());
            mqRetryLog.setOperation(messageDTO.getOperation().getValue());
            mqRetryLog.setCreateTime(date);
            mqRetryLog.setUpdateTime(date);
            mqRetryLog.setStatus(status);
            mqRetryLog.setData(data);
            mqRetryLog.setSourceId(MqRetryLogConstant.SOURCE_ID_ABT);
            mqRetryLog.setMqVirtualHost(JaProperty.get(MqRetryLogConstant.V_HOST));
            mqRetryLog.setMqExchange(MQRetryContextHolder.getContext().getQueueBinding().getExchangeName());
            mqRetryLog.setMqRouteKey(MQRetryContextHolder.getContext().getQueueBinding().getRoutingkey());
            mqRetryLog.setMqQueue(MQRetryContextHolder.getContext().getQueueBinding().getQueueName());
            //初始设置失效时间为1小时，消费成功也是1小时后失效
            mqRetryLog.setExpireTime(DateUtils.addSeconds(date, (int) TimeUnit.HOURS.toSeconds(JaProperty.getLong(EXPIRE_SUCCESS, 24L))));
            apiLogMongoTemplate.insert(mqRetryLog, MqRetryLogConstant.MQ_RETRY_LOG);
            return mqRetryLog;
        } catch (Exception e) {
            throw new BusinessException("插入表失败");
        }
    }

    private MqRetryLogBo queryMqRetryLogByBizId(PtmMqMessageDTO messageDTO, Long bizId) {
        try {
            if (null != messageDTO.getType()
                    && null != messageDTO.getOperation()
                    && null != bizId) {
                Query query = Query.query(
                        Criteria.where(MqRetryLogConstant.TYPE).is(messageDTO.getType().getValue())
                                .and(MqRetryLogConstant.OPERATION).is(messageDTO.getOperation().getValue())
                                .and(MqRetryLogConstant.BIZ_ID).is(bizId + "")
                                .and(MqRetryLogConstant.MQ_QUEUE).is(MQRetryContextHolder.getContext().getQueueBinding().getQueueName()));
                query.fields().include(MqRetryLogConstant.MQ_RETRY_COUNT)
                        .include(MqRetryLogConstant.LOCAL_RETRY_COUNT)
                        .include(MqRetryLogConstant.STATUS)
                        .include(MqRetryLogConstant.ID)
                        .include(MqRetryLogConstant.OPERATION)
                        .include(MqRetryLogConstant.PINPOINT_ID)
                        .include(MqRetryLogConstant.TYPE)
                        .include(MqRetryLogConstant.MESSAGE_ID)
                        .include(MqRetryLogConstant.ERROR_MSG)
                        .include(MqRetryLogConstant.MQ_QUEUE)
                        .include(MqRetryLogConstant.CREATE_TIME);
                return apiLogMongoTemplate.findOne(query, MqRetryLogBo.class, MqRetryLogConstant.MQ_RETRY_LOG);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("查询消息日志表失败，messageId:{}, msg:{}", messageDTO.getMessageId(), e.getMessage());
            return null;
        }
    }

    public void fail(MqRetryLogBo mqRetryLog, Throwable ex) {
        int attempt = 0;
        while (true) {
            try {
                attempt++;
                updateMqRetryLog(mqRetryLog, ex, MqRetryLogTypeEnum.FAIL.getType());
                break;
            } catch (Exception e) {
                if (attempt >= MAX_RETRIES) {
                    log.error("更新消息日志表失败，messageId:{}, msg:{}", mqRetryLog.getMessageId(), e.getMessage());
                    break;
                }
                log.error("更新消息日志表失败，messageId:{},重试次数:{},msg:{}", mqRetryLog.getMessageId(), attempt, e.getMessage());
            }
        }
    }

    public void success(MqRetryLogBo mqRetryLog) {
        int attempt = 0;
        while (true) {
            try {
                attempt++;
                updateMqRetryLog(mqRetryLog, null, MqRetryLogTypeEnum.SUCCESS.getType());
                break;
            } catch (Exception e) {
                if (attempt >= MAX_RETRIES) {
                    log.error("更新消息日志表失败，messageId:{},重试次数:{},msg:{}", mqRetryLog.getMessageId(), attempt, e.getMessage());
                    break;
                }
                log.error("更新消息日志表失败，messageId:{},重试次数:{},msg:{}", mqRetryLog.getMessageId(), attempt, e.getMessage());
            }
        }
    }

    private void updateMqRetryLog(MqRetryLogBo mqRetryLog, Throwable ex, Integer status) {

        log.info("根据messageId={},mqQueue={},id={},更新={}", mqRetryLog.getMessageId(), mqRetryLog.getMqQueue(), mqRetryLog.getId(), status);

        Query query = new Query(Criteria.where(MqRetryLogConstant.ID).is(mqRetryLog.getId()));
        Update update = new Update();
        if (null != ex) {
            String errorMsg = ex.getClass().getName() + ":" + ex.getMessage() + ":" + ex.getStackTrace()[0];
            update.set(MqRetryLogConstant.ERROR_MSG, errorMsg);
            update.set(MqRetryLogConstant.ERROR_DETAIL, toString(ex));
            mqRetryLog.setErrorMsg(errorMsg);
        }

        update.set(MqRetryLogConstant.MQ_RETRY_COUNT, MQRetryContextHolder.getContext().getMqRetrycount());
        update.set(MqRetryLogConstant.LOCAL_RETRY_COUNT, MQRetryContextHolder.getContext().getLocalRetrycount());
        update.set(MqRetryLogConstant.UPDATE_TIME, new Date());
        update.set(MqRetryLogConstant.STATUS, status);

        //消费失败设置失效时间为6小时
        if (MqRetryLogTypeEnum.FAIL.getType() == status) {
            update.set(MqRetryLogConstant.EXPIRE_TIME, DateUtils.addSeconds(mqRetryLog.getCreateTime(), (int) TimeUnit.HOURS.toSeconds(JaProperty.getLong(EXPIRE_FAIL, 72L))));
        }
        apiLogMongoTemplate.updateFirst(query, update, MqRetryLogConstant.MQ_RETRY_LOG);
    }

    /**
     * 将Throwable对象转换为字符串输出
     *
     * @param throwable 需要转换的Throwable对象
     * @return 转换后的字符串
     */
    public static String toString(Throwable throwable) {
        if (throwable == null) {
            return "null";
        }

        StringBuilder sb = new StringBuilder(1024);
        sb.append(throwable.toString()).append(System.lineSeparator());
        StackTraceElement[] stackTrace = throwable.getStackTrace();
        for (StackTraceElement element : stackTrace) {
            sb.append("\tat ").append(element.toString()).append(System.lineSeparator());
        }
        return sb.toString();
    }

    public void sendWeixin(MQRetryContextHolder.MQRetryContext context, MqRetryLogBo retryLog, String type) {
        if (!JaProperty.getBoolean(MONITOR, false)) {
            return;
        }
        JaQyWxAlarm jaQyWxAlarm = new JaQyWxAlarm("default");
        jaQyWxAlarm.setUrl("https://qyapi.weixin.qq.com/cgi-bin/webhook/send?key=" + JaProperty.get(WEIXIN_KEY));
        String appEnv = JaEnvProperty.getEnv();

        String content = String.format(WECHAT_FAIL_NOTIFY_TEMPLATE, appEnv, context.getQueueBinding().getQueueName(),
                retryLog.getMessageId(), retryLog.getType(), retryLog.getOperation(),
                type, context.getLocalRetrycount(), context.getMqRetrycount(), null == retryLog.getErrorMsg() ? "" : retryLog.getErrorMsg(),
                retryLog.getPinpointId(), Thread.currentThread().getName(), CommonUtil.now());
        jaQyWxAlarm.alarm(content);
    }
}
