package com.digiwin.athena.abt.application.service.abt.migration.summary;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharPool;
import cn.hutool.core.text.CharSequenceUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.digiwin.athena.abt.application.dto.migration.abt.api.RefreshCardDto;
import com.digiwin.athena.abt.application.dto.migration.abt.api.RefreshCardInfo;
import com.digiwin.athena.abt.application.dto.migration.abt.api.RefreshCardMessageCompensateDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.summary.*;
import com.digiwin.athena.abt.event.job.RefreshCardMessageJob;
import com.digiwin.athena.abt.application.dto.migration.abt.quartz.CronSchedulerJob;
import com.digiwin.athena.abt.application.dto.migration.atmc.agile.CardDataParam;
import com.digiwin.athena.abt.application.dto.migration.atmc.backlog.CardAbstractDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.backlog.CardInfoDto;
import com.digiwin.athena.abt.application.dto.migration.atmc.backlog.CardItemDto;
import com.digiwin.athena.abt.application.dto.migration.atmc.ptm.PtmProjectCardItemDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.thememap.*;
import com.digiwin.athena.abt.event.job.RefreshCardJob;
import com.digiwin.athena.abt.application.service.abt.migration.ptm.CommonPtmTransformService;
import com.digiwin.athena.abt.application.service.abt.migration.quartz.JobManager;
import com.digiwin.athena.abt.application.service.abt.migration.summary.executor.RefreshCardMessageExecutor;
import com.digiwin.athena.abt.application.service.atmc.migration.backlog.BacklogActionCreateService;
import com.digiwin.athena.abt.application.service.atmc.migration.backlog.CommonBacklogAbstractService;
import com.digiwin.athena.abt.application.service.atmc.migration.project.CommonProjectHandleService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.aglie.AglieService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.aim.TenantService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.dataFootPrint.DataFootPrintService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.ptm.PtmService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.thememap.ThemeMapService;
import com.digiwin.athena.abt.application.utils.*;
import com.digiwin.athena.abt.core.meta.constants.ContainerConstant;
import com.digiwin.athena.abt.core.meta.constants.JobConstants;
import com.digiwin.athena.abt.core.meta.constants.ManualTaskConstant;
import com.digiwin.athena.abt.core.meta.constants.RedisKeyPrefixConstant;
import com.digiwin.athena.abt.core.meta.enums.CardTypeEnum;
import com.digiwin.athena.abt.core.meta.enums.TmPageName;
import com.digiwin.athena.abt.infrastructure.mapper.biz.migration.atmc.TaskWorkitemMessageMapper;
import com.digiwin.athena.abt.infrastructure.pojo.bo.migration.atmc.BacklogBO;
import com.digiwin.athena.abt.infrastructure.pojo.bo.migration.atmc.PtmTaskCardItemBO;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.TaskWorkitemMessage;
import com.digiwin.athena.appcore.auth.AppAuthContext;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.jugg.agile.framework.core.util.concurrent.JaRunnable;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.quartz.*;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.util.StopWatch;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * RefreshCardMessageServiceImpl Description
 *
 * @author majianfu
 * @date 2023/3/31
 * @since
 */
@Slf4j
@Service
public class RefreshCardMessageServiceImpl implements RefreshCardMessageService, InitializingBean {

    @Autowired
    private JobManager jobManager;

    @Autowired
    private List<RefreshCardMessageExecutor> refreshCardMessageExecutors;

    @Autowired
    private Scheduler scheduler;
    @Autowired
    private ThreadPoolTaskExecutor threadPoolRefreshCardTaskExecutor;
    @Autowired
    private PtmService ptmService;

    @Autowired
    ThemeMapService themeMapService;

    @Autowired
    private DataFootPrintService dataFootPrintService;

    @Autowired
    BacklogActionCreateService backlogActionCreateService;

    @Autowired
    AglieService aglieService;
    @Autowired
    private BacklogAbstractUtil backlogAbstractUtil;

    @Autowired
    protected CommonProjectHandleService commonProjectHandleService;
    @Autowired
    private StringRedisTemplate redisTemplate;
    @Autowired
    private TaskWorkitemMessageMapper taskWorkItemMessageMapper;
    @Autowired
    private CommonBacklogAbstractService commonBacklogAbstractService;


    @Autowired
    private CommonPtmTransformService commonPtmTransformService;


    @Resource
    private TenantService tenantService;

    @Autowired
    private TaskWorkitemMessageMapper taskWorkitemMessageMapper;

    private static final List<String> REFRESH_CARD_TYPES = Lists.newArrayList(ContainerConstant.REFRESH_CARD_TYPE_SUMMARY, ContainerConstant.REFRESH_CARD_TYPE_BUSINESS);

    private Map<Integer, RefreshCardMessageExecutor> refreshCardMessageExecutorMap = new HashMap<>();


    @Override
    public void afterPropertiesSet() throws Exception {
        for (RefreshCardMessageExecutor refreshCardMessageExecutor : refreshCardMessageExecutors) {
            if (refreshCardMessageExecutorMap.containsKey(refreshCardMessageExecutor.cardType())) {
                throw new IllegalStateException("重复注册 " + refreshCardMessageExecutor.cardType() + " , " + refreshCardMessageExecutor.getClass().getSimpleName());
            }
            refreshCardMessageExecutorMap.put(refreshCardMessageExecutor.cardType(), refreshCardMessageExecutor);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Boolean createRefreshCardMessageJobIfAbsent(RefreshCardMessageDTO refreshCardMessageDTO) {
        // 校验卡片类型是否支持
        CardTypeEnum.checkCardType(refreshCardMessageDTO.getCardType());

        // 检查任务是否存在
        boolean jobExist = jobManager.exist(refreshCardMessageDTO.getId().toString(), JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
        if (jobExist) {
            log.info("refresh card message job:{}.{} already exist, skip create..", refreshCardMessageDTO.getId(), JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
            return Boolean.FALSE;
        }

        CronSchedulerJob job = new CronSchedulerJob();
        // 待办项id作为定时任务的名称
        job.setName(String.valueOf(refreshCardMessageDTO.getId()));
        job.setGroup(JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
        job.setJobClass(RefreshCardMessageJob.class);
        // 生成cron表达式，随机离散在1个小时内执行
        job.setCron(CronExprUtils.genRandomCronExpr());
        job.setDesc(refreshCardMessageDTO.getDesc());
        job.setJobDataJson(JsonUtils.objectToString(refreshCardMessageDTO));
        log.info("create refresh card message job:{}.{}, cron:{}", job.getName(), job.getGroup(), job.getCron());

        jobManager.createCronJob(job);
        return Boolean.TRUE;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void deleteRefreshCardMessageJob(String taskKeyName) {
        log.info("delete refresh card message job:{}.{}", taskKeyName, JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
        jobManager.deleteJob(taskKeyName, JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
    }

    @Override
    public void deleteJob(String jobName, String group) {
        jobManager.deleteJob(jobName, group);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateRefreshCardMessageJobData(RefreshCardMessageDTO refreshCardMessageDTO) {
        CronSchedulerJob job = new CronSchedulerJob();
        // 待办项id作为定时任务的名称
        job.setName(String.valueOf(refreshCardMessageDTO.getId()));
        job.setGroup(JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
        job.setJobClass(RefreshCardMessageJob.class);

        CronTrigger trigger = jobManager.getJobTrigger(String.valueOf(refreshCardMessageDTO.getId()));
        // 使用之前任务的cron表达式
        String cronExpr = null != trigger ? trigger.getCronExpression() : CronExprUtils.genRandomCronExpr();
        job.setCron(cronExpr);
        job.setDesc(refreshCardMessageDTO.getDesc());
        job.setJobDataJson(JsonUtils.objectToString(refreshCardMessageDTO));

        log.info("update refresh card message job:{}.{}", job.getName(), job.getGroup());
        jobManager.createCronJob(job, true);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void refreshCardMessage(RefreshCardMessageDTO refreshCardMessageDTO) {
        CardTypeEnum.checkCardType(refreshCardMessageDTO.getCardType());
        refreshCardMessageExecutorMap.get(refreshCardMessageDTO.getCardType()).execute(refreshCardMessageDTO);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void compensateRefreshCardMessage(List<RefreshCardMessageCompensateDTO> refreshCardMessageCompensateList) {
        refreshCardMessageCompensateList.forEach(refreshCardMessageCompensate -> CardTypeEnum.checkCardType(refreshCardMessageCompensate.getCardType()));

        for (RefreshCardMessageCompensateDTO refreshCardMessageCompensateDTO : refreshCardMessageCompensateList) {
            for (Long cardId : refreshCardMessageCompensateDTO.getCardIds()) {
                RefreshCardMessageDTO refreshCardMessageDTO = new RefreshCardMessageDTO();
                refreshCardMessageDTO.setId(cardId);
                refreshCardMessageDTO.setCardType(refreshCardMessageCompensateDTO.getCardType());
                refreshCardMessageDTO.setTenantId(refreshCardMessageCompensateDTO.getTenantId());
                refreshCardMessageDTO.setRefreshCardType(refreshCardMessageCompensateDTO.getRefreshCardType());
                // 不存在则创建定时任务
                this.createRefreshCardMessageJobIfAbsent(refreshCardMessageDTO);
            }
        }
    }

    @SneakyThrows
    public String triggerManual(String jobName) {
        boolean jobExist = jobManager.exist(jobName, JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
        if (jobExist) {
            JobKey jobKey = JobKey.jobKey(jobName, JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
            scheduler.triggerJob(jobKey);
            return CharSequenceUtil.format("Manual triggering successful, jobName: {}, jobGroup: {}",
                    jobName, JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
        }
        return CharSequenceUtil.format("Job not exist, jobName: {}, jobGroup: {}",
                jobName, JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
    }

    @Override
    public void refreshCard(RefreshCardDto refreshCardDto) {
        String refreshCardType = refreshCardDto.getRefreshCardType();
        if (StringUtils.isBlank(refreshCardType)) {
            refreshCardDto.setRefreshCardType(ContainerConstant.REFRESH_CARD_TYPE_BUSINESS);
        }
        log.info("异步执行任务:{}", JsonUtils.objectToString(refreshCardDto));
        threadPoolRefreshCardTaskExecutor.execute(JaRunnable.get(() -> {
            try {
                int size = getPartition();
                if (size > 0) {
                    // TODO 后期配置中心接入，需删除配置和stopWatch
                    StopWatch stopWatch = new StopWatch();
                    stopWatch.start("refreshCard");
                    AuthoredUser authoredUserThread = AppAuthContextHolder.getContext().getAuthoredUser();
                    if (authoredUserThread == null) {
                        log.warn("当前用户不存在");
                        return;
                    }
                    log.info("租户:{},执行刷新数据:{}", authoredUserThread.getTenantId(), JsonUtils.objectToString(refreshCardDto));
                    // 进行分组
                    List<RefreshCardInfo> cards = refreshCardDto.getCards();
                    // 进行分组
                    Map<String, List<RefreshCardInfo>> collect = cards.stream().collect(Collectors.groupingBy(RefreshCardInfo::getType));
                    collect.forEach((k, v) -> {
                        CardTaskContext cardTaskContext = new CardTaskContext(authoredUserThread, getPartition(), ContainerConstant.CARD_JOB_SINGLE_TASK_SIZE, ContainerConstant.CARD_JOB_BATCH_TASK_SIZE, ContainerConstant.CARD_JOB_MAX_ERROR_SIZE);
                        for (RefreshCardInfo refreshCardInfo : v) {
                            CardJobData jobData = new CardJobData(refreshCardInfo.getCode(), authoredUserThread.getTenantId(), k, refreshCardDto.getRefreshCardType());
                            refreshTaskCard(cardTaskContext, jobData);
                        }
                    });
                    stopWatch.stop();
                    log.info("process time get refreshCard:{}", stopWatch.getTotalTimeSeconds());
                }
            } catch (Exception e) {
                log.error("执行刷新卡片失败:{}", e.getMessage(), e);
            }
        }));
    }

    private int getPartition() {
        String s = redisTemplate.opsForValue().get(RedisKeyPrefixConstant.REFRESH_CARD_PARTITION_SIZE);
        if (StringUtils.isBlank(s)) {
            return ContainerConstant.CARD_JOB_PARTITION_SIZE;
        }
        int size = Integer.parseInt(s);
        return Math.min(size, 100);
    }


    @Override
    public void refreshTaskCard(CardTaskContext cardTaskContext, CardJobData cardJobData) {
        String pageCode = cardJobData.getPagCode();
        String code = cardJobData.getCode();
        String refreshCardType = cardJobData.getRefreshCardType();
        // 根据类型需要判断当前定义是否
        if (TmPageName.TASK_CARD_NAME.getValue().equals(pageCode)) {
            TmActivityResponseDTO activityAction = themeMapService.getActivityAction(null, code, "task-card");
            if (activityAction == null) {
                log.warn("任务定义为空:{}", code);
                return;
            }
            // 查看是否有摘要配置
            TmActivityPageDTO pages = activityAction.getPages();
            if (pages == null) {
                log.warn("任务定义的page为空:{}", code);
                return;
            }
            List<TmDataState> dataStates = pages.getDataStates();
            if (CollectionUtils.isEmpty(dataStates)) {
                log.warn("任务定义的数据状态为空:{}", code);
                return;
            }
            List<String> dataSourceNames = Lists.newArrayListWithExpectedSize(3);
            if (REFRESH_CARD_TYPES.contains(refreshCardType)) {
                setDataSourceNames(dataSourceNames, refreshCardType, dataStates);
            } else {
                for (String cardType : REFRESH_CARD_TYPES) {
                    setDataSourceNames(dataSourceNames, cardType, dataStates);
                }
            }
            if (CollectionUtils.isEmpty(dataSourceNames)) {
                log.warn("根据类型:{},过滤的数据源集合为空", refreshCardType);
                return;
            }
            Set<String> processVariableKeys = getProcessVariableKeys(dataSourceNames, activityAction.getDataSources());

            processTaskCard(cardTaskContext, processVariableKeys, cardJobData);
        } else if (TmPageName.PROJECT_CARD_NAME.getValue().equals(pageCode)) {
            TmTaskDefineResponseDTO task = themeMapService.getTask(code);
            if (task != null) {
                Map pages = task.getPages();
                if (pages == null) {
                    return;
                }
                Object o = pages.get(pageCode);
                if (o == null) {
                    return;
                }
                TmActivityPageDTO tmActivityPageDTO = JsonUtils.jsonToObject(JsonUtils.objectToString(o), TmActivityPageDTO.class);
                boolean flag = false;
                // 判断是否存在摘要和业务字段配置
                if (REFRESH_CARD_TYPES.contains(refreshCardType)) {
                    flag = checkExists(refreshCardType, tmActivityPageDTO);
                } else {
                    for (String cardType : REFRESH_CARD_TYPES) {
                        if (checkExists(cardType, tmActivityPageDTO)) {
                            flag = true;
                            break;
                        }
                    }
                }
                if (!flag) {
                    return;
                }
            }
            processProjectCard(cardTaskContext, cardJobData);
        }
        // 触发一次
        triggerTask(cardTaskContext);
    }

    @Override
    public void updateCardInfo(CardJob cardJob) {
//        JSONObject apiResult = tenantService.queryApiVirtualToken(cardJob.getTenantId());
//        String virtualToken = apiResult.getString("token");
        //将token放入线程内
//        AuthoredUser authoredUser = AppAuthContextUtils.setAuthoredUserContext(virtualToken);
//        authoredUser.setTenantId(cardJob.getTenantId());
        // 这里判断没有定时任务使用，有些场景需要代理token，所以这个地方就用传进来的参数，不使用集成账号了
        AppAuthContext context = AppAuthContextHolder.getContext();
        AuthoredUser authoredUser = context.getAuthoredUser();
        CardJobData jobData = new CardJobData(cardJob.getCode(), cardJob.getTenantId(), cardJob.getPagCode(), cardJob.getRefreshCardType());
        CardTaskContext cardTaskContext = new CardTaskContext(authoredUser, getPartition(), ContainerConstant.CARD_JOB_SINGLE_TASK_SIZE, ContainerConstant.CARD_JOB_BATCH_TASK_SIZE, ContainerConstant.CARD_JOB_MAX_ERROR_SIZE);
        if (TmPageName.PROJECT_CARD_NAME.getValue().equals(cardJob.getPagCode())) {
            processProjectCard(cardJob.getIds(), cardTaskContext, jobData);
        } else {
            processTaskCard(cardJob.getIds(), cardTaskContext, getProcessVariableKeys(cardJob.getCode()), cardJob);
        }
        // 触发一次
        triggerTask(cardTaskContext);
    }

    @Override
    public void updateCardByBk(RefreshCardInfosDTO refreshCardInfos) {
        // 根据bk集合信息查询待处理的活动id
        if (CollectionUtils.isNotEmpty(refreshCardInfos.getRefreshInfos())){
            //组装maps查询数据足迹集合
            List<Map<String, Object>> maps = getBkConcatsData(refreshCardInfos);
            log.info("查询数据足迹入参:{}", JsonUtils.objectToString(maps));
            //查询数据足迹 ---> 查询PTM ---> 铺平更新数据集合map
            List<Map<String, Object>> dataMaps = getUpdateCardData(maps,refreshCardInfos);
            log.info("待更新卡片数据:{}", JsonUtils.objectToString(dataMaps));
            //组装即将更新的业务数据集合
            if (CollectionUtils.isNotEmpty(dataMaps)){
                Map<Long, Map<String, Object>> updateBusinessMap = getUpdataBusinessMap(dataMaps,refreshCardInfos);
                log.info("待更新业务数据:{}", JsonUtils.objectToString(updateBusinessMap));
                backlogAbstractUtil.saveOrUpdateBusinessMessageInner(refreshCardInfos.getTenantId(), refreshCardInfos.getPagCode(), updateBusinessMap);
            }
        }
    }

    @Override
    public void updateTaskWorkItemMessage(UpdateTaskWorkItemMessageDTO updateDTO) {
        Map<String,Object> summaryMap = commonBacklogAbstractService.getBacklogCardAbstractById(updateDTO.getWorkItemId(), AppAuthContextHolder.getContext().getAuthoredUser(), updateDTO.getCardAbstractDTO(), updateDTO.getLangName());
        Long workItemId = updateDTO.getWorkItemId();
        if (MapUtils.isEmpty(summaryMap)) {
            log.info("摘要Map为空不更新：{}", workItemId);
            return;
        }
        String summary = summaryMap.get("summaryLayout") != null ? summaryMap.get("summaryLayout").toString() : "";
        String searchMessage = summaryMap.get("searchMessage") != null ? summaryMap.get("searchMessage").toString() : "";
        if (StringUtils.isBlank(summary)) {
            log.info("摘要为空不更新：{}", workItemId);
            return;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("card_id", workItemId);
        List<TaskWorkitemMessage> taskWorkitemMessageList = taskWorkitemMessageMapper.selectByMap(paramMap);
        if (CollectionUtils.isEmpty(taskWorkitemMessageList)) {
            log.info("TaskWorkitemMessage不存在：{}", workItemId);
            return;
        }
        int updated = taskWorkitemMessageMapper.update(null, new LambdaUpdateWrapper<TaskWorkitemMessage>()
                .eq(TaskWorkitemMessage::getCardId, workItemId)
                .set(TaskWorkitemMessage::getSummary, summary)
                .set(TaskWorkitemMessage::getSearchMessage, searchMessage));
        if (updated>0) {
            log.info("更新摘要成功：{}，{}，{}",workItemId,summary,searchMessage);
        }
    }


    private String mergeMessage(String message, String subMessage) {
        /**
         * 简单合并：直接比较subMessage和message是否相同，相同则返回message；不相同，subMessage追加到message
         */
        if (StringUtils.isBlank(message)) {
            return subMessage;
        }

        if (StringUtils.equals(message, subMessage) || StringUtils.isBlank(subMessage)) {
            return message;
        }

        return message + "<br>" + subMessage;
    }

    private Map<Long, Map<String, Object>> getUpdataBusinessMap(List<Map<String, Object>> dataMaps,RefreshCardInfosDTO refreshCardInfos) {
        Map<Long, Map<String, Object>> updateBusinessMap = Maps.newHashMapWithExpectedSize(dataMaps.size());
        dataMaps.forEach(e ->{
            //获取定义组装businessMessage
            if (TmPageName.TASK_CARD_NAME.getValue().equals(refreshCardInfos.getPagCode())){
                TmActivityResponseDTO activityResponseDTO = (TmActivityResponseDTO) e.get("define");
//                    List<TmDataState> dataStates = activityResponseDTO.getPages().getDataStates();
                TmActivityPageDTO pages = activityResponseDTO.getPages();
                if (pages != null) {
                    if (CollectionUtils.isNotEmpty(pages.getDataStates())) {
                        for (TmDataState dataState : pages.getDataStates()) {
                            if ("all".equals(dataState.getType())) {
                                TmTaskCardFields taskCardFields = dataState.getTaskCardFields();
                                //组装 businessMessage 数据，更新数据
                                Map<String, Object> businessMessageAll = Maps.newHashMap();
                                List<Map<String, Object>> datas = (List<Map<String, Object>>) e.get("data");
                                for (Map<String, Object> pageData : datas) {
                                    Map<String, Object> businessMessage = this.getBusinessMessage(taskCardFields, pageData);
                                    this.mergeBusinessMessage(businessMessage, businessMessageAll);
                                }
                                updateBusinessMap.put(Long.valueOf(e.get("cardId").toString()),businessMessageAll);
                                break;
                            }
                        }
                    }
                }
            }else{
                TmTaskDefineResponseDTO taskDefineResponseDTO = (TmTaskDefineResponseDTO) e.get("define");
                Map pages = taskDefineResponseDTO.getPages();
                if (pages != null) {
                    Object o = pages.get(TmPageName.PROJECT_CARD_NAME);
                    if (o != null) {
                        TmActivityPageDTO tmActivityPageDTO = JsonUtils.jsonToObject(JsonUtils.objectToString(o), TmActivityPageDTO.class);
                        List<TmDataState> dataStates = tmActivityPageDTO.getDataStates();
                        if (CollectionUtils.isNotEmpty(dataStates)) {
                            for (TmDataState dataState : dataStates) {
                                if ("all".equals(dataState.getType())) {
                                    TmTaskCardFields taskCardFields = dataState.getTaskCardFields();
                                    //组装 businessMessage 数据，更新数据
                                    Map<String, Object> businessMessageAll = Maps.newHashMap();
                                    List<Map<String, Object>> datas = (List<Map<String, Object>>) e.get("data");
                                    for (Map<String, Object> pageData : datas) {
                                        Map<String, Object> businessMessage = this.getBusinessMessage(taskCardFields, pageData);
                                        this.mergeBusinessMessage(businessMessage, businessMessageAll);
                                    }
                                    updateBusinessMap.put(Long.valueOf(e.get("cardId").toString()),businessMessageAll);
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        });
        return updateBusinessMap;
    }

    /**
     * 合并mergeBusinessMessage
     *
     * @param source 每一行生成的mergeBusinessMessage
     * @param target 去重汇总后的mergeBusinessMessage
     */
    private void mergeBusinessMessage(Map<String, Object> source, Map<String, Object> target) {
        Set<String> targetKeySet = target.keySet();
        for (String targetKey : targetKeySet) {
            Set<String> sourceSet = (Set<String>) source.remove(targetKey);
            Set<String> targetSet = (Set<String>) target.get(targetKey);
            if(sourceSet != null){
                targetSet.addAll(sourceSet);
            }
        }
        if(source != null){
            target.putAll(source);
        }
    }

    private Map<String, Object> getBusinessMessage(TmTaskCardFields tmTaskCardFields,
                                                   Map<String, Object> data) {
        Map<String, List<Map<String, Object>>> taskCardField = tmTaskCardFields.getTaskCardField();
        Map<String, Object> businessMessage = Maps.newHashMap();
        Map<String, String> fieldPathNameMap = new HashMap<>();
        for (Map.Entry<String, List<Map<String, Object>>> taskCardFieldEntry : taskCardField.entrySet()) {
            // 遍历所有配置内容，例如taskCardField.groups/taskCardField.labels等
            List<Map<String, Object>> value = taskCardFieldEntry.getValue();
            List<TmTaskCardFields.AbstractCardConfig> abstractCardConfigList = JsonUtils.jsonToListObject(
                    JsonUtils.objectToString(value),
                    TmTaskCardFields.AbstractCardConfig.class);
            for (TmTaskCardFields.AbstractCardConfig abstractCardConfig : abstractCardConfigList) {
                String aliasCode = abstractCardConfig.getCode();
                String fieldPath = abstractCardConfig.getValue();
                if (org.apache.commons.lang3.StringUtils.isBlank(aliasCode) || org.apache.commons.lang3.StringUtils.isBlank(fieldPath)) {
                    continue;
                }
                String fieldName = org.apache.commons.lang3.StringUtils.substringAfter(fieldPath, String.valueOf(CharPool.DOT));
                // key=m.mf__code,value=mf
                fieldPathNameMap.putIfAbsent(fieldPath + "__" + aliasCode, fieldName);
            }
        }
        for (Map.Entry<String, String> entry : fieldPathNameMap.entrySet()) {
            String fieldPath = entry.getKey();
            // 获取值
            LinkedHashSet<Object> list = Sets.newLinkedHashSet();
            CommonUtils.addDepData(data,list,entry.getValue().split("\\."),0);
            for (Object value : list) {
                Set<String> valueList = null;
                if (value instanceof Collection) {
                    valueList = Sets.newLinkedHashSet((Collection) value);
                } else if (value != null) {
                    valueList = Sets.newHashSet(String.valueOf(value));
                }
                Set<String> oldValueSet = (Set<String>) businessMessage.get(fieldPath);
                // 值合并去重
                valueList = CollUtil.unionDistinct(oldValueSet, valueList);
                if (CollUtil.isNotEmpty(valueList)) {
                    businessMessage.put(fieldPath, valueList);
                }
            }
        }
        return MapUtils.isEmpty(businessMessage) ? null : businessMessage;
    }

    private List<Map<String, Object>> getBkConcatsData(RefreshCardInfosDTO refreshCardInfos) {
        List<Map<String, Object>> maps = new ArrayList<>();
        refreshCardInfos.getRefreshInfos().forEach(refreshInfo ->{
            Map<String, Object> map = new HashMap();
            if (TmPageName.TASK_CARD_NAME.getValue().equals(refreshCardInfos.getPagCode())){
                map.put("type","task");
                if (StringUtils.isNotEmpty(refreshInfo.getActivityId())){
                    map.put("code",refreshInfo.getActivityId());
                    //查询活动定义获取组装bk
                    TmActivityResponseDTO activityDefine = themeMapService.getActivityAction(ManualTaskConstant.VIRTUAL_PROJECT, refreshInfo.getActivityId(), refreshCardInfos.getPagCode());
                    if (activityDefine != null){
                        //保存定义
                        map.put("define",activityDefine);
                        if (ObjectUtils.isNotEmpty(activityDefine.getInputData())){
                            map.put("bkConcats",getBkConcats(map,activityDefine.getInputData(),refreshInfo.getData()));
                        }
                    }
                }
            }else{
                map.put("type","project");
                if (StringUtils.isNotEmpty(refreshInfo.getTaskId())){
                    map.put("code",refreshInfo.getTaskId());
                    //查询项目定义获取组装bk
                    TmTaskDefineResponseDTO projectDefine = themeMapService.getTask(refreshInfo.getTaskId());
                    if (projectDefine != null){
                        //保存定义
                        map.put("define",projectDefine);
                        if (ObjectUtils.isNotEmpty(projectDefine.getInputData())){
                            map.put("bkConcats",getBkConcats(map,projectDefine.getInputData(),refreshInfo.getData()));
                        }
                    }
                }
            }
            maps.add(map);
        });
        return maps;
    }

    private List<Map<String, Object>> getUpdateCardData(List<Map<String, Object>> maps, RefreshCardInfosDTO refreshCardInfos) {
        List<Map<String, Object>> dataMaps = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(maps)){
            maps.forEach(map -> {
                //查询数据足迹，根据inputData配置信息和应用传递的数据信息构建bk集合和code作为入参查询活动实例id
                List<Map<String,Object>> list =  dataFootPrintService.queryByBk(refreshCardInfos.getTenantId(),JSON.parseObject(JsonUtils.objectToString(map)));
                //组装 actId 对应的业务数据data
                Map<String, List<Object>> collectObj = getActIdToData(map,list);
                //请求 PTM 根据活动实例actId和code查询待办id集合
                List<Map<String,String>> result = ptmService.queryBacklogIds(list.stream().map(e -> Long.valueOf(e.get("actId").toString())).distinct().collect(Collectors.toList()),map.get("type").toString(),map.get("code").toString());
                log.info("PTM返回type:{}, code:{}, 待办id集合:{}",map.get("type"), map.get("code"),result);
                //组装bks-taskId-cardId 铺平关系,用于后续更新mongo
                if (CollectionUtils.isNotEmpty(result)){
                    result.forEach(res ->{
                        Map<String, Object> dataMap = new HashMap<>();
                        //保存当前任务定义，用于后面组装数据使用
                        dataMap.put("define",map.get("define"));
                        dataMap.put("actId",res.get("actId").toString());
                        dataMap.put("cardId",res.get("cardId").toString());
                        //组装业务数据集合 List<Object> 类型
                        dataMap.put("data",collectObj.get(res.get("actId").toString()));
                        dataMaps.add(dataMap);
                    });
                }
            });
        }
        return dataMaps;
    }

    private Map<String, List<Object>> getActIdToData(Map map, List<Map<String, Object>> list) {
        //分组，按活动实例actId分组，actId对应多个bk
        Map<String, List<String>> collect = list.parallelStream().
                collect(Collectors.groupingBy(e -> e.get("actId").toString(),
                        Collectors.mapping(e -> e.get("bkConcat").toString(), Collectors.toList())));

        //获取 bk对应的业务数据 关系
        Map<String, List<Map<String, Object>>> resultMap = (Map<String, List<Map<String, Object>>>) map.get("bkToData");
        Map<String, List<Object>> collectObj = new HashMap<>();
        collect.forEach((k,v)->{
            List<Object> listData = new ArrayList<>();
            //遍历bk集合，获取对应的业务数据
            v.forEach(e ->{
                resultMap.get(e).forEach(data ->{
                    listData.add(data);
                });
            });
            collectObj.put(k,listData);
        });
        return collectObj;
    }

    private List<String> getBkConcats(Map<String,Object> map,Map<String,Object> inputData, List<Map<String, Object>> datas) {
        List<String> uniKeys = new ArrayList<>();
        List<Map<String,Object>> field = (List<Map<String, Object>>)inputData.get("field");
        for (Map<String,Object> fieldMap : field){
            if (fieldMap.containsKey("data_name")){
                uniKeys.add(fieldMap.get("data_name").toString());
            }
        }
        return this.getKeys(map,uniKeys,datas);
    }

    /**
     * 获取业务数据
     * @param pUniKeys unikeys
     * @param datas 数据
     * @return 业务数据
     */
    public static List<String> getKeys(Map<String, Object> map,List<String> pUniKeys, List<Map<String, Object>> datas) {
        log.info("需要更新的业务数据:{}", JsonUtils.objectToString(datas));
        List<String> keys = new ArrayList<>();
        Map<String, List<Map<String, Object>>> resultMap = new HashMap<>();
        for (Map<String, Object> tRow : datas) {
            String key = BKUtils.getKey(pUniKeys, tRow);
            //创建bk与业务数据的对应关系
            if (resultMap.containsKey(key)){
                //若key已经存在，则新增保存到value中
                List<Map<String, Object>> bkDatas = resultMap.get(key);
                bkDatas.add(tRow);
                resultMap.put(key, bkDatas);
            }else{
                //若key不存在，则新增key，并保存value
                List<Map<String, Object>> bkDatas = new ArrayList<>();
                bkDatas.add(tRow);
                resultMap.put(key, bkDatas);
                keys.add(key);
            }
        }
        //创建bk与业务数据的对应关系
        map.put("bkToData", resultMap);
        return keys;
    }

    public void processTaskCard(List<Long> ids, CardTaskContext cardTaskContext, Set<String> processVariableKeys, CardJobData cardJobData) {
        if (CollectionUtils.isNotEmpty(ids)) {
            log.info("执行任务数量:{}", ids.size());
            // 获取bpmdata的入参
            List<List<Long>> partitionSize = CommonUtils.spliceArrays(ids, cardTaskContext.getPartitionSize());
            for (List<Long> idList : partitionSize) {
                List<Map<String, Object>> dataList = new ArrayList<>();
                for (Long id : idList) {
                    Map<String, Object> map = new HashMap<>();
                    map.put("backlogId", id);
                    map.put("keys", processVariableKeys);
                    dataList.add(map);
                }
                Map<String, Object> editActions = backlogActionCreateService.getEditActions(cardTaskContext.getAuthoredUser(), dataList, false, true);
                if (MapUtils.isEmpty(editActions)) {
                    log.warn("获取ActivityData为空,任务id:{}", org.apache.commons.lang3.StringUtils.join(idList, ","));
                    return;
                }
                // 获取bk信息用来判断是否走分批
                Map<String, Object> bkInfo = getBkInfo(cardTaskContext.getAuthoredUser(), cardJobData.getCode());
                if (bkInfo != null && CollectionUtils.isNotEmpty(processVariableKeys)) {
                    log.info("添加分批处理任务:{}", ids.size());
                    CardDataParam cardDataParam = new CardDataParam(ids, editActions, cardJobData.getRefreshCardType(), bkInfo, TmPageName.TASK_CARD_NAME.getValue());
                    addTask(cardTaskContext, cardDataParam, true);
                } else {
                    log.info("添加循环处理任务:{}", ids.size());
                    List<Map<String, Object>> taskList = (List<Map<String, Object>>) editActions.remove("tasks");
                    if (CollectionUtils.isNotEmpty(taskList)) {
                        // 构建根据任务id的分组
                        Map<Long, List<Map<String, Object>>> idToTasksMap = Maps.newHashMapWithExpectedSize(ids.size());
                        for (Map<String, Object> task : taskList) {
                            Long id = MapUtils.getLong(task, "backlogId");
                            if (id != null) {
                                List<Map<String, Object>> list = idToTasksMap.computeIfAbsent(id, e -> new ArrayList<>());
                                list.add(task);
                            }
                        }
                        // 循环执行任务
                        idToTasksMap.forEach((k, v) -> {
                            Map<String, Object> cloneData = Maps.newHashMapWithExpectedSize(editActions.size());
                            cloneData.putAll(editActions);
                            cloneData.put("tasks", v);
                            CardDataParam cardDataParam = new CardDataParam(Lists.newArrayList(k), cloneData, cardJobData.getRefreshCardType(), null, TmPageName.TASK_CARD_NAME.getValue());
                            // 拆分数据
                            addTask(cardTaskContext, cardDataParam, false);
                        });
                    }
                }
            }
            return;
        }
        log.info("根据code:{},待处理任务数为空", cardJobData);
    }


    public void processTaskCard(CardTaskContext cardTaskContext, Set<String> processVariableKeys, CardJobData cardJobData) {
        String code = cardJobData.getCode();
        // 查询待处理的任务
        List<PtmTaskCardItemBO> ptmTaskList = ptmService.queryTaskCardListByCodes(Lists.newArrayList(code));
        if (CollectionUtils.isEmpty(ptmTaskList)) {
            log.info("查询code:{},待处理任务为空", code);
            return;
        }
        // 过滤错误数过大的任务
        List<Long> ids = new ArrayList<>();
        Map<String, Map<Long, Integer>> idErrorCount = cardTaskContext.getIdErrorCount();
        // 项目id调用失败的次数
        Map<Long, Integer> idErrorsMap = idErrorCount.getOrDefault(TmPageName.TASK_CARD_NAME.getValue(), new HashMap<>());
        for (PtmTaskCardItemBO cardItemDTO : ptmTaskList) {
            Integer num = idErrorsMap.getOrDefault(cardItemDTO.getBacklogId(), 0);
            if (num < cardTaskContext.getMaxErrorSize()) {
                ids.add(cardItemDTO.getBacklogId());
            }
        }
        processTaskCard(ids, cardTaskContext, processVariableKeys, cardJobData);

    }

    private void processProjectCard(CardTaskContext cardTaskContext, CardJobData cardJobData) {
        List<PtmProjectCardItemDTO> ptmList = ptmService.queryProjectCardListByCodes(Lists.newArrayList(cardJobData.getCode()));
        if (CollectionUtils.isEmpty(ptmList)) {
            log.info("根据:{},租户:{},没有查询到待处理项目卡", cardJobData.getCode(), cardTaskContext.getAuthoredUser().getTenantId());
            return;
        }
        List<Long> ids = new ArrayList<>();
        Map<String, Map<Long, Integer>> idErrorCount = cardTaskContext.getIdErrorCount();
        // 项目id调用失败的次数
        Map<Long, Integer> idErrorsMap = idErrorCount.getOrDefault(TmPageName.PROJECT_CARD_NAME.getValue(), new HashMap<>());

        for (PtmProjectCardItemDTO ptmProjectCardItemDTO : ptmList) {
            Integer num = idErrorsMap.getOrDefault(ptmProjectCardItemDTO.getId(), 0);
            if (num < cardTaskContext.getMaxErrorSize()) {
                ids.add(ptmProjectCardItemDTO.getId());
            }
        }
        processProjectCard(ids, cardTaskContext, cardJobData);
    }

    private void processProjectCard(List<Long> ids, CardTaskContext cardTaskContext, CardJobData cardJobData) {
        if (ids.size() > 0) {
            log.info("项目卡code:{},执行数量:{}", cardJobData.getCode(), ids.size());
            List<List<Long>> partitionSize = CommonUtils.spliceArrays(ids, cardTaskContext.getPartitionSize());
            for (List<Long> idList : partitionSize) {
                for (Long id : idList) {
                    Map activityData = commonProjectHandleService.getProjectActivityData(id);
                    if (activityData != null) {
                        CardDataParam cardDataParam = new CardDataParam(Lists.newArrayList(id), activityData, cardJobData.getRefreshCardType(), null, TmPageName.PROJECT_CARD_NAME.getValue());
                        addTask(cardTaskContext, cardDataParam, false);
                    }
                }
            }
        }
    }


    /**
     * 获取bk
     *
     * @param authoredUser
     * @param code
     * @return
     */
    private Map<String, Object> getBkInfo(AuthoredUser authoredUser, String code) {
        JSONArray tmTaskAndActivityBk = themeMapService.getTmTaskAndActivityBk("activity", code, authoredUser);
        if (CollectionUtils.isNotEmpty(tmTaskAndActivityBk)) {
            JSONArray bks = ((JSONObject) tmTaskAndActivityBk.get(0)).getJSONArray("bk");
            JSONObject bk = (JSONObject) bks.get(0);
            JSONArray bkFields = bk.getJSONArray("field");
            if (CollectionUtils.isEmpty(bkFields)) {
                // bk不存在
                return null;
            }
            Map<String, Object> bkInfo = new HashMap<>();
            // 设置bk
            bkInfo.put("dataName", bk.getString("dataName"));
            List<String> fields = Lists.newArrayListWithCapacity(bkFields.size());
            for (Object bkField : bkFields) {
                JSONObject json = (JSONObject) bkField;
                fields.add(json.getString("dataName"));
            }
            bkInfo.put("dataName", bk.getString("dataName"));
            bkInfo.put("fields", fields);
            return bkInfo;
        }
        return null;
    }


    private boolean executeTask(CardDataParam cardDataParam, List<CardDataParam> taskData, int maxSize) {
        return taskData != null && (cardDataParam == null || (taskData.size() >= maxSize));
    }


    private boolean checkExists(String refreshCardType, TmActivityPageDTO page) {
        List<TmDataState> dataStates = page.getDataStates();
        if (ContainerConstant.REFRESH_CARD_TYPE_SUMMARY.equals(refreshCardType)) {
            if (CollectionUtils.isEmpty(dataStates)) {
                return CollectionUtils.isNotEmpty(page.getSummaryFields());
            } else {
                boolean flag = false;
                for (TmDataState dataState : dataStates) {
                    if (!"all".equals(dataState.getType())) {
                        flag = CollectionUtils.isNotEmpty(dataState.getSummaryFields());
                        break;
                    }
                }
                return flag;
            }
        } else {
            boolean flag = false;
            if (CollectionUtils.isNotEmpty(dataStates)) {
                for (TmDataState dataState : dataStates) {
                    if ("all".equals(dataState.getType())) {
                        TmTaskCardFields taskCardFields = dataState.getTaskCardFields();
                        flag = taskCardFields != null && MapUtils.isNotEmpty(taskCardFields.getTaskCardField());
                        break;
                    }
                }
            }
            return flag;
        }
    }

    /**
     * 获取对应的数据源名称
     *
     * @param refreshCardType
     * @param dataStates
     */
    private void setDataSourceNames(List<String> dataSourceNames, String refreshCardType, List<TmDataState> dataStates) {
        boolean summary = ContainerConstant.REFRESH_CARD_TYPE_SUMMARY.equals(refreshCardType);
        // 按照以前逻辑摘要累取类型不等于all，业务字段取类型为ALL
        if (CollectionUtils.isEmpty(dataStates)) {
            return;
        }
        Optional<TmDataState> first = dataStates.stream().filter(e -> (summary != "all".equals(e.getType()))).findFirst();
        if (first.isPresent()) {
            TmDataState tmDataState = first.get();
            if (checkExists(summary, tmDataState)) {
                TmDataFilter dataFilter = tmDataState.getDataFilter();
                if (dataFilter != null) {
                    if (summary) {
                        List<String> sourceNames = dataFilter.getDataSourceNames();
                        if (CollectionUtils.isNotEmpty(sourceNames)) {
                            dataSourceNames.addAll(sourceNames);
                        }
                        return;
                    }
                    TmTaskCardFields taskCardFields = tmDataState.getTaskCardFields();
                    String dateSourceName = taskCardFields.getDateSourceName();
                    if (org.apache.commons.lang3.StringUtils.isNoneBlank(dateSourceName)) {
                        dataSourceNames.add(dateSourceName);
                    }
                }
            }
        }
    }


    /**
     * 获取请求流程变量的keys
     *
     * @param code
     * @return
     */
    private Set<String> getProcessVariableKeys(String code) {
        TmActivityResponseDTO activityAction = themeMapService.getActivityAction(null, code, "task-card");
        List<String> dataSourceNames = Lists.newArrayListWithExpectedSize(3);
        for (String cardType : REFRESH_CARD_TYPES) {
            setDataSourceNames(dataSourceNames, cardType, activityAction.getPages().getDataStates());
        }
        return getProcessVariableKeys(dataSourceNames, activityAction.getDataSources());
    }

    /**
     * 获取请求流程变量的keys
     *
     * @param dataSourceNames
     * @param dataSources
     * @return
     */
    private Set<String> getProcessVariableKeys(List<String> dataSourceNames, Map<String, TmActionDTO> dataSources) {
        Set<String> keys = Sets.newHashSet();
        dataSources.forEach((k, v) -> {
            if (dataSourceNames.contains(k)) {
                List<TmActionParameterDTO> actionParams = v.getActionParams();
                if (CollectionUtils.isNotEmpty(actionParams)) {
                    for (TmActionParameterDTO actionParam : actionParams) {
                        if ("PROCESS_VARIABLE".equals(actionParam.getType())) {
                            String value = actionParam.getValue();
                            String[] split = value.split("\\.");
                            keys.add(split[0]);
                        }
                    }
                }
            }
        });
        return keys;
    }

    /**
     * 判断是否有摘要或者业务字段配置
     *
     * @param summary
     * @param tmDataState
     * @return
     */
    private boolean checkExists(boolean summary, TmDataState tmDataState) {
        if (summary) {
            return CollectionUtils.isNotEmpty(tmDataState.getSummaryFields());
        } else {
            TmTaskCardFields taskCardFields = tmDataState.getTaskCardFields();
            return taskCardFields != null && MapUtils.isNotEmpty(taskCardFields.getTaskCardField());
        }
    }

    /**
     * 添加任务
     *
     * @param cardTaskContext
     * @param cardDataParam
     * @param isBatch
     */
    public void addTask(CardTaskContext cardTaskContext, CardDataParam cardDataParam, boolean isBatch) {
        if (cardDataParam != null) {
            // 任务添加到数组上
            cardTaskContext.addTask(cardDataParam, isBatch);
        }
        runTask(cardTaskContext, cardDataParam);
    }

    public void triggerTask(CardTaskContext cardTaskContext) {
        runTask(cardTaskContext, null);
    }

    public void runTask(CardTaskContext cardTaskContext, CardDataParam cardDataParam) {
        int singleTaskSize = cardTaskContext.getSingleTaskSize();
        // 大于最大任务数开始执行循环任务
        if (executeTask(cardDataParam, cardTaskContext.getSingleTasks(), singleTaskSize)) {
            runTask(cardTaskContext, cardTaskContext.getSingleTasks(), singleTaskSize, cardDataParam != null, false);
        }
        int batchTaskSize = cardTaskContext.getBatchTaskSize();
        // 大于最大任务数开始执行任务
        if (executeTask(cardDataParam, cardTaskContext.getBatchTasks(), batchTaskSize)) {
            runTask(cardTaskContext, cardTaskContext.getBatchTasks(), batchTaskSize, cardDataParam != null, true);
        }
    }

    private void runSingleTask(CardTaskContext cardTaskContext, CardDataParam dataParam, List<TaskWorkitemMessage> updateSummaryList) {
        Map<String, Object> activityData = dataParam.getActivityData();
        // 没有task的数据跳过
        List<Map<String, Object>> taskList = (List<Map<String, Object>>) activityData.get("tasks");
        if (CollectionUtils.isEmpty(taskList)) {
            return;
        }
        Map<String, Object> first = taskList.get(0);
        List<Long> ids = dataParam.getIds();
        for (Long id : ids) {
            BacklogBO backlogBO = commonPtmTransformService.selectBacklogForCard(id);
            backlogBO.setTmTaskId(MapUtils.getString(first, "tmTaskId"));
            backlogBO.setTmActivityId(MapUtils.getString(first, "tmActivityId"));
            CardInfoDto cardInfoDto = commonBacklogAbstractService.getSummaryWorkItemLayout(id, backlogBO, activityData, cardTaskContext.getAuthoredUser(), new CardAbstractDTO().setCardId(id).setNeedHistoryInfo(false).setNeedBusinessMessage(true).setBusinessMessageSourceType(CardAbstractDTO.SOURCE_TYPE_API));
            Map<String, Object> cardInfo = cardInfoDto.getCardInfo();
            if (cardInfo != null) {
                String activityMessage = CommonUtils.getActivityMessageStr(cardInfoDto);
                TaskWorkitemMessage taskWorkitemMessage = TaskWorkitemMessage.builder().cardId(id)
                        .activityMessage(activityMessage)
                        .summary(MapUtils.getString(cardInfo, "summaryLayout", ""))
                        .historyMessage(MapUtils.getString(cardInfo, "historyMessage", ""))
                        .searchMessage(MapUtils.getString(cardInfo, "searchMessage", ""))
                        .build();
                updateSummaryList.add(taskWorkitemMessage);
            }
        }
    }

    private void runTask(CardTaskContext cardTaskContext, List<CardDataParam> taskData, int size, boolean sleep, boolean isBatch) {
        if (CollectionUtils.isEmpty(taskData)) {
            return;
        }
        log.info("任务开始执行数量:{}", taskData.size());
        String tenantId = cardTaskContext.getAuthoredUser().getTenantId();
        List<TaskWorkitemMessage> updateSummaryList = Lists.newArrayListWithCapacity(size);
        Map<Long, Map<String, Object>> updateBusinessMap = Maps.newHashMapWithExpectedSize(size);
        StopWatch watch = new StopWatch("TASK-TIME");
        watch.start();
        Map<String, List<CardDataParam>> collect = taskData.stream().collect(Collectors.groupingBy(CardDataParam::getPageCode));
        try {
            collect.forEach((pageCode, v) -> {
                boolean isTaskSingle = TmPageName.TASK_CARD_NAME.getValue().equals(pageCode) && !isBatch;
                for (CardDataParam dataParam : v) {
                    if (isTaskSingle) {
                        runSingleTask(cardTaskContext, dataParam, updateSummaryList);
                        continue;
                    }
                    Map<String, Object> activityData = dataParam.getActivityData();
                    // 没有task的数据跳过
                    List<Map<String, Object>> taskList = (List<Map<String, Object>>) activityData.get("tasks");
                    if (CollectionUtils.isEmpty(taskList)) {
                        continue;
                    }
                    /*summary*/
                    Map<String, Object> first = taskList.get(0);
                    // 获取摘要和业务字段接口
                    try {
                        BaseResultDTO<Map<String, Map<Long, Map<Long, Map<String, Object>>>>> batchCardBusinessMessage = aglieService.getBatchCardBusinessMessage(MapUtils.getString(first, "tmTaskId"), MapUtils.getString(first, "tmActivityId"), dataParam);
                        Map<String, Map<Long, Map<Long, Map<String, Object>>>> response = batchCardBusinessMessage.getResponse();
                        if (MapUtils.isNotEmpty(response)) {
                            Map<Long, Map<Long, Map<String, Object>>> summaryMap = response.get("summary");
                            Map<Long, Map<Long, Map<String, Object>>> businessMap = response.get("business");
                            Map<String, String> cacheCode = Maps.newHashMap();
                            if (summaryMap != null) {
                                summaryMap.forEach((id, activityIdToSummary) -> {
                                    String reApprovalComment = commonBacklogAbstractService.getReApprovalComment(id, cacheCode);
                                    List<CardItemDto> cardItemList = Lists.newArrayList();
                                    Map<String, String> summaryData = Maps.newHashMapWithExpectedSize(3);
                                    activityIdToSummary.forEach((activityId, rowData) -> {
                                        // 需要获取当前任务是否是退回
                                        String summaryLayout = MapUtils.getString(rowData, "summaryLayoutStr", "");
                                        if (StringUtils.isNotBlank(reApprovalComment)) {
                                            if (StringUtils.isNotBlank(summaryLayout)) {
                                                summaryLayout = reApprovalComment + "<br>" + summaryLayout;
                                            } else {
                                                summaryLayout = reApprovalComment;
                                            }
                                        }
                                        rowData.put(CommonUtils.SUMMARY_LAYOUT, summaryLayout);
                                        summaryData.put("summaryLayoutStr", commonBacklogAbstractService.mergeMessage(summaryData.get("summaryLayoutStr"), summaryLayout));
                                        summaryData.put("historyMessage", commonBacklogAbstractService.mergeMessage(summaryData.get("historyMessage"), MapUtils.getString(rowData, "historyMessage", "")));
                                        summaryData.put("searchMessage", commonBacklogAbstractService.mergeMessage(summaryData.get("searchMessage"), MapUtils.getString(rowData, "searchMessage", "")));
                                        if (activityId != -1L) {
                                            cardItemList.add(new CardItemDto(activityId, rowData));
                                        }
                                    });
                                    CardInfoDto cardInfoDto = new CardInfoDto();
                                    cardInfoDto.setCardItemList(cardItemList);
                                    String activityMessage = CommonUtils.getActivityMessageStr(cardInfoDto);
                                    TaskWorkitemMessage taskWorkitemMessage = TaskWorkitemMessage.builder().cardId(id)
                                            .activityMessage(activityMessage)
                                            .summary(MapUtils.getString(summaryData, "summaryLayoutStr", ""))
                                            .historyMessage(MapUtils.getString(summaryData, "historyMessage", ""))
                                            .searchMessage(MapUtils.getString(summaryData, "searchMessage", ""))
                                            .build();
                                    taskWorkitemMessage.setModifyDate(LocalDateTime.now());
                                    updateSummaryList.add(taskWorkitemMessage);
                                });
                            }
                            if (businessMap != null) {
                                Collection<Map<Long, Map<String, Object>>> values = businessMap.values();
                                for (Map<Long, Map<String, Object>> value : values) {
                                    updateBusinessMap.putAll(value);
                                }
                            }
                        }
                    } catch (Exception e) {
                        log.error("调用获取卡片信息错误:{}", e.getMessage(), e);
                        cardTaskContext.recordIdError(pageCode, dataParam.getIds());
                    }
                }
                // 批量入库
                if (CollectionUtils.isNotEmpty(updateSummaryList)) {
                    taskWorkItemMessageMapper.batchUpdate(updateSummaryList);
                    log.info("摘要更新,数量:{}", updateSummaryList.size());
                }
                backlogAbstractUtil.saveOrUpdateBusinessMessageInner(tenantId, pageCode, updateBusinessMap);

            });

        } finally {
            watch.stop();
            log.info("任务结束,耗时:{}", watch.getTotalTimeSeconds());
            // 清空
            taskData.clear();
        }
        if (sleep) {
            try {
                // 休眠20秒
                Thread.sleep(1000 * 20L);
            } catch (InterruptedException e) {
                log.info("线程中断:{}", e.getMessage());
                Thread.currentThread().interrupt();
            }
        }
    }

    @Override
    public void settingSize(Integer size) {
        redisTemplate.opsForValue().set(RedisKeyPrefixConstant.REFRESH_CARD_PARTITION_SIZE, String.valueOf(size));
    }

    /**
     * 创建并更新定时任务
     *
     * @param cardJob
     * @return
     * @throws SchedulerException
     */
    @Override
    public Boolean createOrUpdateCardJob(CardJob cardJob) {
        try {
            // 根据租户查找定时任务
            // 创建jobDetail
            JobKey jobKey = new JobKey(JobConstants.CARD_JOB_NAME, JobConstants.CARD_JOB_NAME);
            if (scheduler.checkExists(jobKey)) {
                // 获取jobDetail
                JobDetail jobDetail = scheduler.getJobDetail(jobKey);
                // 获取jobDataMap
                JobDataMap jobDataMap = jobDetail.getJobDataMap();
                // 获取存的业务数据
                Set<CardJobData> cardData = (Set<CardJobData>) jobDataMap.get(JobConstants.JOB_DATA_KEY);
                // 先删除
                CardJobData cardJobData = new CardJobData();
                BeanUtils.copyProperties(cardJob, cardJobData);
                cardData.remove(cardJobData);
                // 如果开启添加
                if (cardJob.isEnable()) {
                    cardData.add(cardJobData);
                }
                // 更新jobDetail
                scheduler.addJob(jobDetail, true);
                String corn = cardJob.getCorn();
                // 如果存在执行时间更新执行时间
                if (StringUtils.isNotBlank(corn)) {
                    CronTrigger trigger = (CronTrigger) scheduler.getTrigger(TriggerKey.triggerKey(JobConstants.CARD_JOB_NAME, JobConstants.CARD_JOB_NAME));
                    trigger = trigger.getTriggerBuilder().withSchedule(CronScheduleBuilder.cronSchedule(corn)).build();
                    // 重新调度
                    scheduler.rescheduleJob(trigger.getKey(), trigger);
                }
                return false;
            }
            if (cardJob.isEnable()) {
                JobDataMap jobDataMap = new JobDataMap();
                Set<CardJobData> data = new HashSet<>();
                CardJobData cardJobData = new CardJobData();
                BeanUtils.copyProperties(cardJob, cardJobData);
                data.add(cardJobData);
                jobDataMap.put(JobConstants.JOB_DATA_KEY, data);
                jobDataMap.put(ContainerConstant.CARD_JOB_SINGLE_TASK_SIZE_KEY, ContainerConstant.CARD_JOB_SINGLE_TASK_SIZE);
                jobDataMap.put(ContainerConstant.CARD_JOB_BATCH_TASK_SIZE_KEY, ContainerConstant.CARD_JOB_BATCH_TASK_SIZE);
                jobDataMap.put(ContainerConstant.CARD_JOB_PARTITION_SIZE_KEY, ContainerConstant.CARD_JOB_PARTITION_SIZE);
                jobDataMap.put(ContainerConstant.CARD_JOB_MAX_ERROR_SIZE_KEY, ContainerConstant.CARD_JOB_MAX_ERROR_SIZE);
                // 构建JobDetail
                JobDetail jobDetail = JobBuilder.newJob(RefreshCardJob.class)
                        .withIdentity(JobConstants.CARD_JOB_NAME, JobConstants.CARD_JOB_NAME)
                        .usingJobData(jobDataMap)
                        .withDescription("刷新卡片定时任务")
                        .storeDurably() //持久化
                        .build();
                // 按cronExpression表达式构建一个trigger
                CronTrigger trigger = TriggerBuilder.newTrigger()
                        .withIdentity(JobConstants.CARD_JOB_NAME, JobConstants.CARD_JOB_NAME)
                        .startNow()
                        .withSchedule(CronScheduleBuilder.cronSchedule(StringUtils.defaultString(cardJob.getCorn(), JobConstants.CARD_JOB_CRON)))
                        .build();
                //将任务和触发器绑定到调度器
                scheduler.scheduleJob(jobDetail, trigger);
                return true;
            }
        } catch (SchedulerException e) {
            log.error("创建定时任务失败:{}", e.getMessage(), e);
        }
        return false;
    }
}
