package com.digiwin.athena.abt.application.service.atmc.migration.ptm.impl;

import com.digiwin.athena.abt.application.dto.migration.atmc.project.ProjectCardDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.ptm.PtmBacklogRecordDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.ptm.PtmProjectCardItemDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.ptm.PtmProjectCardRecordDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.thememap.TmActivityResponseDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.thememap.TmTaskDefineResponseDTO;
import com.digiwin.athena.abt.application.service.atmc.migration.ptm.PtmProjectTransformService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.ptm.PtmService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.thememap.ThemeMapService;
import com.digiwin.athena.abt.core.meta.constants.BpmConstant;
import com.digiwin.athena.abt.core.meta.enums.BpmEngineCategoryEnum;
import com.digiwin.athena.abt.core.meta.enums.TmPageName;
import com.digiwin.athena.abt.infrastructure.pojo.bo.migration.atmc.PtmProjectRecordBO;
import com.digiwin.athena.abt.infrastructure.pojo.bo.migration.atmc.PtmTaskRecordBO;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.PtmProjectCard;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.framework.mq.retry.exception.RetryMQException;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections4.CollectionUtils;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * PTM 项目卡、项目 与其他转换服务
 * 用于查询PTM 项目卡、项目相关数据，且转换成需要的数据格式
 */
@Service
@Slf4j
public class PtmProjectTransformServiceImpl implements PtmProjectTransformService {


    @Autowired
    private PtmService ptmService;

    @Autowired
    private ThemeMapService themeMapService;


    /**
     * MQTT专用。<br>根据项目卡id 获取 ProjectCardDTO
     *
     * @param projectCard
     * @return 没有返回null
     */
    @Override
    public ProjectCardDTO translateProjectCardForMQTT(PtmProjectCard projectCard, String autherUserId) {
        if (Objects.nonNull(projectCard) && null == projectCard.getHide()) {
            throw new RetryMQException("项目卡未入atmc库,id = " + projectCard.getId() + ",重回队列进行重试");
        }
        if (Objects.isNull(projectCard) || projectCard.getHide() == 1) {
            return null;
        }
        List<PtmProjectCard> projectCardList = new ArrayList<>();
        projectCardList.add(projectCard);
        List<Long> projectCardIds = projectCardList.stream().map(PtmProjectCard::getId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(projectCardIds)) {
            return null;
        }
        List<PtmProjectCardItemDTO> ptmProjectCardList = ptmService.queryProjectCardListByProjectCardId(projectCardIds, autherUserId);
        log.info("translateProjectCard-list:{}", ptmProjectCardList);
        List<ProjectCardDTO> list = setProjectCardList(projectCardList, ptmProjectCardList);
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(0);
        } else {
            //如果list是空，说明项目卡结束了，直接查卡
            PtmProjectCardRecordDTO projectCardRecordDTO = ptmService.getProjectCardRecord(projectCard.getId());
            //region 创建项目卡
            ProjectCardDTO projectCardDTO = ProjectCardDTO.builder()
                    .id(projectCardRecordDTO.getId())
                    .name(projectCardRecordDTO.getProjectName())
                    .withName(projectCardRecordDTO.getProjectWithName())
                    .projectName(projectCardRecordDTO.getProjectDefName())
                    .projectDefWithName(projectCardRecordDTO.getProjectDefWithName())
                    .tmTaskId(projectCardRecordDTO.getProjectDefCode())
                    .tmPattern(projectCardRecordDTO.getProjectDefPattern())
                    .startTime(projectCardRecordDTO.getStartTime())
                    .endTime(projectCardRecordDTO.getEndTime())
                    .importance(false)
                    .createTime(projectCardRecordDTO.getCreateTime())
                    .emergency(0)
                    .changed(false)
                    .changedMainTaskId(0L)
                    .isOwner(true)
                    .ownerUserId(projectCardRecordDTO.getPersonInCharge())
                    .ownerUserName(projectCardRecordDTO.getPersonInChargeName())
                    .readCount(0L)
                    .overdue(LocalDateTime.now().isAfter(projectCardRecordDTO.getEndTime()))
                    .modifyDate(null)
                    .hasDataChanged(false)
                    .tasks(null)
                    .majorData(null)
                    .lastNewSourceIds(null)
                    .engineType(BpmEngineCategoryEnum.TaskEngine.getValue())
                    .reAssignAble(false)
                    .personInCharge(projectCardRecordDTO.getPersonInCharge())
                    .personInChargeName(projectCardRecordDTO.getPersonInChargeName())
                    .dataFrom(BpmConstant.DATA_FROM_PTM)
                    .build();
            return projectCardDTO;
        }
    }

    /**
     * 处理项目卡信息
     */
    @Override
    public List<ProjectCardDTO> setProjectCardList(List<PtmProjectCard> projectCardList, List<PtmProjectCardItemDTO> ptmProjectCardList) {
        List<ProjectCardDTO> list = new ArrayList<>();
        List<String> tmProjectCodes = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(ptmProjectCardList)) {

            List<PtmProjectRecordBO> lastProjectRecordList = ptmService.getLastProjectRecordByProjectCardIds(ptmProjectCardList.stream().map(x -> x.getId()).distinct().collect(Collectors.toList()));

            Map<Long, PtmTaskRecordBO> ptmTaskRecordMap = new HashMap<>();
            List<Long> taskIdList = new ArrayList<>();
            ptmProjectCardList.forEach(ptmProjectCard -> {
                taskIdList.addAll(ptmProjectCard.getBacklogs().stream().map(x -> x.getTaskId()).distinct().collect(Collectors.toList()));
            });
            if (CollectionUtils.isNotEmpty(taskIdList)) {
                List<PtmTaskRecordBO> taskList = ptmService.getTaskRecordByTaskIds(taskIdList.stream().distinct().collect(Collectors.toList()));
                taskList.forEach(t -> {
                    ptmTaskRecordMap.put(t.getId(), t);
                });
            }

            ptmProjectCardList.stream().forEach(ptmProjectCard -> {

                //region 创建项目卡
                ProjectCardDTO card = ProjectCardDTO.builder()
                        .id(ptmProjectCard.getId())
                        .name(ptmProjectCard.getProjectName())
                        .withName(ptmProjectCard.getProjectWithName())
                        .startTime(ptmProjectCard.getStartTime())
                        .endTime(ptmProjectCard.getEndTime())
                        .importance(false)
                        .createTime(ptmProjectCard.getCreateTime())
                        .taskCount(ptmProjectCard.getBacklogCount())
                        .emergency(ptmProjectCard.getEmergencyProjectCount() > 0L ? 1 : 0)
                        .emergencyProjectCount(ptmProjectCard.getEmergencyProjectCount())
                        .subProjectCount(ptmProjectCard.getProjectCount())
                        .subProjectFinishedCount(ptmProjectCard.getProjectFinishedCount())
                        .changed(false)
                        .changedMainTaskId(0L)
                        .isOwner(true)
                        .ownerUserId(ptmProjectCard.getPersonInCharge())
                        .ownerUserName(ptmProjectCard.getPersonInChargeName())
                        .readCount(0L)
                        .overdue(LocalDateTime.now().isAfter(ptmProjectCard.getEndTime()))
                        .modifyDate(null)
                        .hasDataChanged(false)
                        .tasks(null)
                        .projectName(ptmProjectCard.getProjectDefName())
                        .projectDefWithName(ptmProjectCard.getProjectDefWithName())
                        .tmTaskId(ptmProjectCard.getProjectDefCode())
                        .tmPattern(ptmProjectCard.getProjectDefPattern())
                        .majorData(null)
                        .lastNewSourceIds(null)
                        .engineType(ptmProjectCard.getEngineType())
                        .reAssignAble(false)
                        .reassign(ptmProjectCard.getReassign())
                        .handover(ptmProjectCard.getHandover())
                        .personInCharge(ptmProjectCard.getPersonInCharge())
                        .personInChargeName(ptmProjectCard.getPersonInChargeName())
                        .interrupt(ptmProjectCard.getInterrupt())
                        .dataFrom(BpmConstant.DATA_FROM_PTM)
                        .interrupt(ptmProjectCard.getInterrupt())
                        .oldPerformerId(ptmProjectCard.getOldPerformerId())
                        .oldPerformerName(ptmProjectCard.getOldPerformerName())
                        .build();
                //endregion

                if (tmProjectCodes.stream().filter(x -> Objects.equals(x, ptmProjectCard.getProjectDefCode())).count() == 0) {
                    tmProjectCodes.add(ptmProjectCard.getProjectDefCode());
                }

                if (CollectionUtils.isNotEmpty(projectCardList)) {
                    Optional<PtmProjectCard> c = projectCardList.stream().filter(x -> Objects.equals(x.getId(), ptmProjectCard.getId())).findFirst();
                    if (c.isPresent()) {
                        card.setSummary(c.get().getSummary());
                        card.setHistoryMessage(c.get().getHistoryMessage());
                        card.setSearchMessage(c.get().getSearchMessage());
                        card.setReadCount(c.get().getReadCount());
                        card.setImportance(c.get().getFavorite() > 0);
                        card.setSubmitId(c.get().getSubmitId());
                        card.setTenantId(c.get().getTenantId());

                        Optional<PtmProjectRecordBO> lastProjectRecordOptional = lastProjectRecordList.stream().filter(x -> Objects.equals(x.getProjectCardId(), ptmProjectCard.getId())).findFirst();
                        if (lastProjectRecordOptional.isPresent()) {
                            PtmProjectRecordBO lastProjectRecord = lastProjectRecordOptional.get();
                            if (lastProjectRecord.getCreateTime() == null || c.get().getModifyDate() == null || c.get().getReadCount() == null) {
                                System.out.println();
                            }
                            if (lastProjectRecord.getCreateTime().isAfter(c.get().getModifyDate()) || Objects.equals(c.get().getReadCount(), 0)) {
                                card.setLastNewSourceIds(lastProjectRecord.getSourceIds());
                            }
                        }
                    }
                }
                if (!org.springframework.util.StringUtils.isEmpty(ptmProjectCard.getMergeData())) {
                    //region 關鍵業務數據
                    Map mergeData = JsonUtils.jsonToObject(ptmProjectCard.getMergeData(), Map.class);
                    if (MapUtils.isNotEmpty(mergeData) && mergeData.size() == 1) {
                        //mergeData如果只指定了一个字段，才会作业关键业务信息显示
                        card.setMajorData(mergeData.values().stream().findFirst().get().toString());
                    }
                    //endregion
                }

                List<Map<String, Object>> tasks = new ArrayList<>();
                card.setTasks(tasks);
                if (CollectionUtils.isNotEmpty(ptmProjectCard.getBacklogs()) /*&&
                        !Objects.equals(TmConstant.TM_PROJECT_PATTERN_BUSINESS, ptmProjectCard.getProjectDefPattern())*/) {
                    //region 生成任务卡信息

                    //pattern 不为 business 时，才填充任务信息
                    List<PtmBacklogRecordDTO> exceptionBacklogRecords = ptmProjectCard.getBacklogs().stream()
                            .filter(x -> ptmTaskRecordMap.get(x.getTaskId()).getHasException() || LocalDateTime.now().isAfter(x.getPlanEndTime()))
                            .collect(Collectors.toList());
                    if (CollectionUtils.isEmpty(exceptionBacklogRecords)) {
                        //如果没有异常的，则取最选创建的一个任务
                        exceptionBacklogRecords = ptmProjectCard.getBacklogs().stream().sorted(Comparator.comparing(PtmBacklogRecordDTO::getCreateTime)).limit(1).collect(Collectors.toList());
                    }
                    exceptionBacklogRecords.forEach(exceptionBacklogRecord -> {
                        Map<String, Object> task = new HashMap<>();
                        task.put("backlogId", exceptionBacklogRecord.getId());
                        task.put("bpmProcessId", exceptionBacklogRecord.getProjectId());
                        task.put("days", 0);
                        task.put("startTime", exceptionBacklogRecord.getCreateTime());
                        task.put("endTime", exceptionBacklogRecord.getPlanEndTime());
                        task.put("exception", ptmTaskRecordMap.get(exceptionBacklogRecord.getTaskId()).getHasException());
                        task.put("executeType", "manual".equals(ptmTaskRecordMap.get(exceptionBacklogRecord.getTaskId()).getTaskDefExecuteType()));
                        task.put("name", ptmTaskRecordMap.get(exceptionBacklogRecord.getTaskId()).getTaskDefName());
                        if (LocalDateTime.now().isAfter(exceptionBacklogRecord.getPlanEndTime())) {
                            task.put("overdue", true);
                            task.put("overdueDays", Duration.between(exceptionBacklogRecord.getPlanEndTime(), LocalDateTime.now()).toDays());
                            task.put("overdueHours", Duration.between(exceptionBacklogRecord.getPlanEndTime(), LocalDateTime.now()).toHours());
                            task.put("overdueMinutes", Duration.between(exceptionBacklogRecord.getPlanEndTime(), LocalDateTime.now()).toMinutes());
                        } else {
                            task.put("overdue", false);
                            task.put("overdueDays", 0);
                            task.put("overdueHours", 0);
                            task.put("overdueMinutes", 0);
                        }
                        task.put("performerId", exceptionBacklogRecord.getPerformerId());
                        task.put("performerName", exceptionBacklogRecord.getPerformerName());
                        task.put("projectId", exceptionBacklogRecord.getProjectId());
                        task.put("tmTaskId", ptmProjectCard.getProjectDefCode());
                        task.put("tmActivityId", exceptionBacklogRecord.getTaskDefCode());
                        task.put("tmPattern", ptmTaskRecordMap.get(exceptionBacklogRecord.getTaskId()).getTaskDefPattern());
                        task.put("tmCategory", ptmTaskRecordMap.get(exceptionBacklogRecord.getTaskId()).getTaskDefCategory());

                        tasks.add(task);
                    });

                    //endregion
                }

                setProjectCardState(card);

                list.add(card);
            });
        }

        //region 转派数据
        try {
            List<TmTaskDefineResponseDTO> projectCacheMap = themeMapService.getBatchTaskAssigns(tmProjectCodes);
            for (ProjectCardDTO projectCardDTO : list) {
                Optional<TmTaskDefineResponseDTO> tmTaskDefineDTO = projectCacheMap.stream().filter(define -> Objects.equals(define.getTaskId(), projectCardDTO.getTmTaskId())).findFirst();
                if (tmTaskDefineDTO.isPresent()) {
                    projectCardDTO.setReAssignAble(tmTaskDefineDTO.get().getAssignConfig() != null ? tmTaskDefineDTO.get().getAssignConfig().getAssignAble() : false);
                } else {
                    projectCardDTO.setReAssignAble(Boolean.FALSE);
                }
            }
        } catch (Exception e) {
            //查询转派信息应该不影响主要的项目列表返回，所以吃掉异常，记录日志
            log.warn("[PTM项目列表可否转派] 出现异常：{},查询内容：{}", e, tmProjectCodes);
        }
        //endregion

        Map<String, String> approveActivityNameMap = new HashMap<>();
        for (ProjectCardDTO project : list) {
            for (Map<String, Object> activity : project.getTasks()) {
                if ("APPROVAL".equals(activity.get("tmCategory"))) {
                    //查询审核类型活动的子标题
                    String tmTaskId = (String) activity.get("tmTaskId");
                    String tmActivityId = (String) activity.get("tmActivityId");
                    StringBuilder tmBuilder = new StringBuilder();
                    tmBuilder.append(tmTaskId).append("-").append(tmActivityId);
                    String key = tmBuilder.toString();
                    if (approveActivityNameMap.containsKey(key)) {
                        activity.put("subName", approveActivityNameMap.get(key));
                    } else {
                        try {
                            TmActivityResponseDTO themeActivity = themeMapService.getActivityAction(tmTaskId, tmActivityId, TmPageName.PROJECT_CARD_NAME.getValue());
                            activity.put("subName", themeActivity.getStartApproveActivityName());
                            approveActivityNameMap.put(key, themeActivity.getStartApproveActivityName());
                        } catch (Exception e) {
                            approveActivityNameMap.put(key, null);
                            log.warn("[查询kg审核类型活动的子标题] 出现异常：{},查询内容：{}", e, tmProjectCodes);
                        }
                    }
                }
            }
        }
        //性能优化：避免过多重复查询
        Map<String, TmTaskDefineResponseDTO> taskDefineMap = new HashMap<>();
        //region 自定义打开项目卡的方式
        for (ProjectCardDTO projectCardDTO : list) {
            try {
                TmTaskDefineResponseDTO taskDefine;
                if (taskDefineMap.containsKey(projectCardDTO.getTmTaskId())) {
                    taskDefine = taskDefineMap.get(projectCardDTO.getTmTaskId());
                } else {
                    taskDefine = themeMapService.getTask(projectCardDTO.getTmTaskId());
                    taskDefineMap.put(projectCardDTO.getTmTaskId(), taskDefine);
                }
                if (taskDefine != null && taskDefine.getConfig() != null) {
                    if (taskDefine.getConfig().containsKey("action")) {
                        projectCardDTO.setAction((Map) taskDefine.getConfig().get("action"));
                    }
                    //卡片隐藏标记
                    projectCardDTO.setHiddenForUser(MapUtils.getBoolean(taskDefine.getConfig(), "hiddenForUser", Boolean.FALSE));
                }
            } catch (Exception ex) {
                log.warn("[PTM项目列表自定义项目卡打开方式] 出现异常：{},查询内容：{}", ex, projectCardDTO.getTmTaskId());
            }
        }
        taskDefineMap.clear();
        //endregion

        return list;
    }

    /**
     * 设置项目卡状态，是否包含异常和逾期任务
     *
     * @param projectCardDTO
     */
    private void setProjectCardState(ProjectCardDTO projectCardDTO) {

        boolean hasException = false;
        boolean hasOverdue = false;
        for (Map<String, Object> task : projectCardDTO.getTasks()) {
            if (task.containsKey("exception") && (boolean) task.get("exception")) {
                hasException = true;
            }
            if (task.containsKey("overdue") && (boolean) task.get("overdue")) {
                hasOverdue = true;
            }
        }

        projectCardDTO.setException(hasException);
        projectCardDTO.setOverdue(hasOverdue);
    }
}
