package com.digiwin.athena.abt.application.service.atmc.migration.restfull.iam.impl;

import com.digiwin.athena.abt.application.configuration.EnvProperties;
import com.digiwin.athena.abt.application.dto.migration.atmc.eoc.ProxyUserResp;
import com.digiwin.athena.abt.application.dto.migration.atmc.iam.UserDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.iam.UserLangMetadataDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.tenant.WithTenantRespDto;
import com.digiwin.athena.abt.application.dto.migration.atmc.usersetting.BaseUserSettingDto;
import com.digiwin.athena.abt.application.dto.migration.atmc.usersetting.PersonalizedDto;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.eoc.EocService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.iam.UserService;
import com.digiwin.athena.abt.core.meta.constants.IamApiConstant;
import com.digiwin.athena.abt.core.meta.enums.AtmcErrorCodeEnum;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.ExceptionUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.util.*;

@Slf4j
@Service
public class UserServiceImpl implements UserService {

    @Autowired
    EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    @Autowired
    private EocService eocService;

    private static final Logger logger = LoggerFactory.getLogger(UserServiceImpl.class);

    public static final String CATALOG_ID = "basic";

    public static final String MESSAGE_NOTICE_TYPE_KEY = "MessageSendType";

    @Override
    public UserDTO query(String userId, String token) {
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        headers.add(GlobalConstant.IAM_APP_TOKEN, com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("id", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.USER;
        ResponseEntity<UserDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, UserDTO.class);
        return responseEntity.getBody();
    }

    @Override
    public UserDTO query(String userId) {
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_APP_TOKEN, com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("id", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.USER;
        ResponseEntity<UserDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, UserDTO.class);
        UserDTO userDTO = responseEntity.getBody();
        if (userDTO == null || StringUtils.isEmpty(userDTO.getId())) {
            //如果没有获取到人员，则返回null
            log.error("[IAM] 查询 " + userId + " 的用户信息失败,  {}", userDTO);
            userDTO = null;
        }
        return userDTO;
    }

    /**
     * 获取目标用户：优先取当前用户的代理用户，否则返回当前用户
     *
     * @param userId
     * @param token
     * @return
     */
    @Override
    public UserDTO getTargetUser(String userId, String token) {
        UserDTO targetUser = null;
        ProxyUserResp proxyUserResp = eocService.getProxyUser(userId, token);
        if (proxyUserResp != null && StringUtils.hasText(proxyUserResp.getUserId())) {
            targetUser = this.query(proxyUserResp.getUserId(), token);
        }

        if (targetUser == null) {
            targetUser = this.query(userId, token);
        }
        return targetUser;
    }

    @Override
    public String getUserLangMetadataAllTenant(String userId, String token) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);

        log.info("getUserLangMetadataAllTenant-userId:{}", userId);
        Map<String, Object> params = new HashMap<>();
        params.put("catalogId", "basic");
        params.put("key", "dapAcceptLanguage");
        params.put("userId", userId);
        params.put("tenantSid", 0);
        HttpEntity httpEntity = new HttpEntity(params, headers);

        log.info("{}.{}", token, JsonUtils.objectToString(params));

        String url = envProperties.getIamUri() + IamApiConstant.USERMETADATA_USER_GET;

        try {
            ResponseEntity<UserLangMetadataDTO> respEntity = restTemplate.postForEntity(url, httpEntity, UserLangMetadataDTO.class);
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                UserLangMetadataDTO baseResult = respEntity.getBody();
                if (null != baseResult && HttpStatus.OK.value() == baseResult.getCode()) {
                    Optional<Object> langOpt = Optional.ofNullable(baseResult.getData()).map(data -> data.get("value"));
                    //如果用户设置了语言别，则使用用户的语言别，如果没有使用语言别，则使用默认的语言别
                    return langOpt.isPresent() ? String.valueOf(langOpt.get()) : defaultLanguage();
                }
            }
            throw BusinessException.create(respEntity.getStatusCodeValue(), JsonUtils.objectToString(respEntity.getBody()));
        } catch (Exception ex) {
            log.error("{}, 请求body体：{}, 报错信息：", AtmcErrorCodeEnum.IAM_GET_USERMETADATA_FAIL.getErrCode(), JsonUtils.objectToString(params), ex);
            throw ExceptionUtil.wrap(AtmcErrorCodeEnum.IAM_GET_USERMETADATA_FAIL.getErrCode(), ex);
        }
    }


    @Override
    public boolean getMessageDoNotDisturbUserId(String userId, String tenantId, String token) {
        try {
            //获取员工ID
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            headers.add(GlobalConstant.IAM_USER_TOKEN, token);
            //post body
            Map<String, String> param = new HashMap<>();
            param.put("tenantId", tenantId);
            param.put("userId", userId);
            param.put("catalogId", "basic");
            param.put("key", "doNotDisturb");
            HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
            String url = envProperties.getIamUri() + IamApiConstant.USER_METADATA_VALUE;
            ResponseEntity<Map> responseEntity = restTemplate.postForEntity(url, httpEntity, Map.class);
            Map result = responseEntity.getBody();
            if (result.containsKey("data")) {
                return "1".equals(result.get("data").toString());
            }
            return false;
        } catch (Exception ex) {
            logger.error(ex.getMessage(), ex);
            return false;
        }
    }

    /**
     * 根据用户id查询出用户的email
     * update: iam接口安全调整给了替换方案，替换为接口/api/iam/v2/user/withtenant
     *
     * @param userId 用户id
     * @return email
     */
    @Override
    public String queryUserEmail(String userId) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, Object> param = new HashMap<>();
        param.put("id", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.TENANT_INFO;

        try {
            ResponseEntity<WithTenantRespDto> dataEntity = restTemplate.postForEntity(url, httpEntity, WithTenantRespDto.class);

            if (dataEntity.getBody() != null) {
                WithTenantRespDto baseDto = dataEntity.getBody();
                if (null == baseDto || null == baseDto.getEmail()) {
                    logger.error("Query user email with no data return:{}", userId);
                    return null;
                }
                return baseDto.getEmail();
            }
        } catch (Exception e) {
            logger.error("Query User Email info Error error with param:{}", userId, e);
            return null;
        }
        return null;
    }

    /**
     * 查询用户的个性化设置数据
     *
     * @param tenantSid 租户SID，注意非租户id
     * @param userId    用户id
     * @param key       需要查询的key
     * @param catalogId 类目id
     * @return PersonalizedDto
     */
    @Override
    public PersonalizedDto queryPersonalizedInfo(Long tenantSid, String userId, String key, String catalogId) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, Object> param = new HashMap<>();
        param.put("tenantSid", tenantSid);
        param.put("userId", userId);
        param.put("catalogId", catalogId);
        param.put("key", key);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        log.info("[UserServiceImpl.queryPersonalizedInfo],param :{},headers :{}", param, headers);

        String url = envProperties.getIamUri() + IamApiConstant.USERMETADATA_USER_GET;
        ResponseEntity<BaseUserSettingDto> dataEntity = restTemplate.postForEntity(url, httpEntity, BaseUserSettingDto.class);
        if (dataEntity.getBody() == null) {
            logger.error("can not find user meta response body  with:{}", param);
            return null;
        } else if (dataEntity.getBody().getData() == null) {
            logger.error("can not find user meta response data  with:{}", param);
            return null;
        } else {
            BaseUserSettingDto baseDto = dataEntity.getBody();
            return JsonUtils.jsonToObject(JsonUtils.objectToString(baseDto.getData()), PersonalizedDto.class);
        }
    }

    /**
     * 获取内部员工
     *
     * @Author：SYQ
     * @Date：2021/6/15 14:42
     */
    @Override
    public List<String> getEmployee(List<String> userList, AuthoredUser user) {
        List<String> memberList = new ArrayList<>();
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, user.getToken());
        //地址
        String url = envProperties.getIamUri() + IamApiConstant.IAM_V_2_USER_LIST;
        //参数
        HttpEntity<?> httpEntity = new HttpEntity<>(userList, headers);
        //获取用户信息
        ResponseEntity<Map> responseEntity = restTemplate.postForEntity(url, httpEntity, Map.class);
        //判断调用API是否成功
        if (responseEntity.getStatusCode() == HttpStatus.OK && responseEntity.getBody().get("data") != null) {
            //获取data数据
            JSONArray jsonArray = JSONArray.fromObject(responseEntity.getBody().get("data"));
            //循环判断用户是否为内部用户
            for (Object object : jsonArray) {
                //用户数据
                Map map = JsonUtils.jsonToObject(JsonUtils.objectToString(object), Map.class);
                //type为0的是内部用户
                if ("0".equals(map.get("type").toString())) {
                    String id = eocService.getEocEmpInfo(map.get("id").toString(), user);
                    memberList.add(id);
                }
            }
        }
        return memberList;
    }

    /**
     * 通过用户ID 获取员工的多语言
     *
     * @param userId
     * @param token
     * @return
     */
    @Override
    public String getUserLangNameByUserId(String userId, String tenantId, String token) {
        return getUserLangMetadataAllTenant(userId, token);
    }


    /**
     * 获取用户默认的语言别
     *
     * @return 语言别信息
     */
    private String defaultLanguage() {
        return org.apache.commons.lang.StringUtils.equals("SIMPLIFIED_CHINESE", envProperties.getDefaultLanguage()) ? Locale.SIMPLIFIED_CHINESE.toString() : Locale.TRADITIONAL_CHINESE.toString();
    }
}
