package com.digiwin.athena.abt.application.service.atmc.migration.tenantToken.impl;

import com.digiwin.athena.abt.application.service.atmc.migration.restfull.aim.TenantService;
import com.digiwin.athena.abt.application.service.atmc.migration.tenantToken.TenantTokenService;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.concurrent.TimeUnit;

@Service
@Slf4j
public class TenantTokenServiceImpl implements TenantTokenService {

    /**
     * 租户虚拟token前缀
     */
    private final String TENANT_VIRTUAL_TOKEN_PREFIX = "atmc:cache:tenant:virtualToken:";

    @Autowired
    private TenantService tenantService;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    /**
     * 获取用指定租户的虚拟token
     *
     * @param tenantId 租户id
     * @return 租户信息
     */
    @Override
    public String queryVirtualToken(String tenantId) {
        try {
            //从redis缓存中获取虚拟的token
            String virtualToken = getCacheVirtualToken(tenantId);
            //redis中不存在，则调用api获取
            if (StringUtils.isEmpty(virtualToken)) {
                //调用api查询租户的虚拟token
                JSONObject apiResult = tenantService.queryApiVirtualToken(tenantId);
                //如果api没有返回数据，则返回null对象
                if (apiResult.isEmpty() || !apiResult.containsKey("token")) {
                    return null;
                }
                //api中返回的失效时间
                long virtualExpireInitial = apiResult.getLong("tokenExpiresIn");
                long cacheExpire = (virtualExpireInitial < 10000L) ? virtualExpireInitial : (virtualExpireInitial / 10L);
                virtualToken = apiResult.getString("token");
                //将数据放入redis缓存中
                stringRedisTemplate.opsForValue().set(TENANT_VIRTUAL_TOKEN_PREFIX + tenantId, virtualToken, cacheExpire/1000, TimeUnit.SECONDS);
            }
            return virtualToken;
        } catch (Exception e) {
            log.warn("QueryVirtualToken error", e);
            return null;
        }

    }

    /**
     * 检查租户的缓存token是否存在
     *
     * @param tenantId 租户id
     * @return 虚拟token
     */
    private String getCacheVirtualToken(String tenantId) {
        String virtualTokenKey = TENANT_VIRTUAL_TOKEN_PREFIX + tenantId;
        if (Boolean.FALSE.equals(stringRedisTemplate.hasKey(virtualTokenKey))) {
            return null;
        }
        return stringRedisTemplate.opsForValue().get(virtualTokenKey);
    }

}
