package com.digiwin.athena.abt.application.utils;


import net.sf.json.JSONObject;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

/**
 * bk处理的工具类
 */
public class BKUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(BKUtils.class);


    /**
     * bk索引值的分隔符
     */
    private static final String KEY_VALUE_SPLIT = ":";


    /**
     * INDEX字符串
     */
    private static final String INDEX_STR = "index";

    /**
     * 数据足迹中 Mongodb中BK自定义连接分隔符
     */
    private static final String CUSTOM_SEPARATOR = "$&";

    private BKUtils() {

    }

    /**
     * 单个排序方法
     *
     * @param map
     * @return
     */
    public static TreeMap sortMap(Map map) {
        TreeMap treeMap = new TreeMap();
        map.forEach((k, v) -> {
            treeMap.put(k, v);
        });
        return treeMap;
    }

    /**
     * 递归将对象数据转为treemap实现排序
     *
     * @param map 需要进行排序的对象
     * @return 排序后的集合
     */
    public static TreeMap recursionToTreeMap(Map<String, Object> map) {
        TreeMap tempTreeMap = new TreeMap();
        for (Map.Entry entry : map.entrySet()) {
            if (entry.getValue() instanceof Map) {
                tempTreeMap.put(entry.getKey(), sortMap((Map) entry.getValue()));
            } else {
                tempTreeMap.put(entry.getKey(), entry.getValue());
            }
        }
        return tempTreeMap;
    }


    /**
     * 目前BK生成，仅支持单层key-value结构，不支持复杂对象的key-Collection类型
     * add by gonghongxing for 2023-06bk专项
     *
     * @return list
     */
    public static List<Map<String, Object>> convertBkIndex(Map<String, Object> bkObject) {
        List<Map<String, Object>> list = new ArrayList<>();
        //如果map不为空,遍历map集合
        if (!MapUtils.isEmpty(bkObject)) {
            for (Map.Entry<String, Object> entry : bkObject.entrySet()) {
                //如果是对象或者集合类型或者是字符串类型，但值为空，则忽略，暂不支持搜索
                if (entry.getValue() instanceof Collection || entry.getValue() instanceof Map) {
                    continue;
                }
                Map<String, Object> tempMap = new HashMap<>(1);
                tempMap.put(INDEX_STR, entry.getKey() + KEY_VALUE_SPLIT + entry.getValue());
                list.add(tempMap);
            }
        }
        return list;
    }

    /**
     * 简单类型的bkSearch.index生成
     * 目前BK生成，仅支持单层key-value结构，不支持复杂对象的key-map，key-list类型
     * add by gonghongxing for 2023-06bk专项
     *
     * @return list
     */
    public static List<Map<String, Object>> convertBkIndexSimple(Object object) {
        List<Map<String, Object>> list = new ArrayList<>();
        if (null != object) {
            Map<String, Object> tempMap = new HashMap<>(1);
            tempMap.put(INDEX_STR, String.valueOf(object));
            list.add(tempMap);
        }
        return list;
    }


    /**
     * 判断字符串是否是json字符串
     *
     * @param jsonStr jsonStr
     * @return boolean
     */
    public static boolean checkJsonStr(String jsonStr) {
        if (jsonStr.startsWith("{") && jsonStr.endsWith("}")) {
            try {
                JSONObject.fromObject(jsonStr);
                return true;
            } catch (Exception e) {
                LOGGER.warn("string :{}is not json", jsonStr, e);
                return false;
            }
        }
        return false;
    }

    /**
     * 将bk对象转为字符串类型，提供查询
     *
     * @param param bk对象
     * @return 查询内容
     */
    public static List<String> bkObjectToIndex(Map<String, Object> param) {
        //对象不存在，则直接返回空的集合
        if (param.isEmpty()) {
            //warm日志记录
            LOGGER.warn("bk object is empty with initial param:{}", param);
            return Collections.emptyList();
        }
        List<Map.Entry<String, Object>> list = new ArrayList<>(param.entrySet());

        //perf-优化查询条件的bk排序，根据值长度排序，减少空数据查询，导致的索引使用率低，引擎的效率问题
        list.sort((o1, o2) -> {
            int length1 = String.valueOf(o1.getValue()).length();
            int length2 = String.valueOf(o2.getValue()).length();
            return length2 - length1;
        });
        //以":"分隔键值，组成bkindex需要的value信息
        List<String> indexList = new ArrayList<>();
        list.forEach(x -> indexList.add(x.getKey() + KEY_VALUE_SPLIT + x.getValue()));
        return indexList;
    }

    /**
     * 将bk对象转为字符串类型，提供查询
     *
     * @param bkList bkList数组
     * @return 查询内容
     */
    public static List<List<String>> bkObjectToIndexList(List<Map> bkList) {
        if (null == bkList || bkList.isEmpty()) {
            //warm日志记录
            LOGGER.warn("bk object list is empty with initial param:{}", bkList);
            return Collections.emptyList();
        }
        List<List<String>> bkIndexList = new ArrayList<>();
        for (Map map : bkList) {
            List<String> tempList = bkObjectToIndex(map);
            if (tempList.isEmpty()) continue;
            bkIndexList.add(tempList);
        }
        return bkIndexList;
    }

    /**
     * 获取业务数据
     * @param pUniKeys unikeys
     * @param pRow 数据
     * @return 业务数据
     */
    public static String getKey(List<String> pUniKeys, Map<String, Object> pRow) {
        StringBuilder keySb = new StringBuilder();
        for (String tUniKey : pUniKeys) {
            String tTmpKey = (pRow.get(tUniKey) == null) ? "" : pRow.get(tUniKey) + "";
            keySb.append(tTmpKey);
            keySb.append(CUSTOM_SEPARATOR);
        }
        return keySb.toString();
    }
}
