package com.digiwin.athena.abt.application.utils;

import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.digiwin.athena.abt.application.dto.migration.atmc.backlog.BacklogCardField;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.aglie.AglieService;
import com.digiwin.athena.abt.core.meta.constants.BkConstant;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 卡面工具类
 *
 * @Author hejy
 * @Version 1.0
 * @Date 2024/4/28 10:07
 */
@Slf4j
@Component
public class BacklogAbstractUtil {

    @Resource
    private AglieService aglieService;

    @Resource(name = "atmcCardBusinessMessageMongoTemplate")
    private MongoTemplate atmcCardBusinessMessageMongoTemplate;

    public static final String BACKLOG_CARD_FIELD = "backlogCardField";

    /**
     * 生成BusinessMessage并入库
     *
     * @param activityDataDTO
     */
    public void getAndSaveBusinessMessage(Map activityDataDTO) {
        try {
            String tmTaskId = MapUtils.getString(activityDataDTO, "tmTaskId");
            String tmActivityId = MapUtils.getString(activityDataDTO, "tmActivityId");
            BaseResultDTO<Map<String, Object>> cardBusinessMessageResp = aglieService.getCardBusinessMessage(tmTaskId, tmActivityId, activityDataDTO);
            Map<String, Object> cardBusinessMessage = cardBusinessMessageResp.getResponse();
            if (MapUtils.isNotEmpty(cardBusinessMessage)) {
                Long cardId = MapUtils.getLong(activityDataDTO, "cardId");
                String pageCode = MapUtils.getString(activityDataDTO, "pageCode");
                String tenantId = MapUtils.getString(activityDataDTO, "tenantId");
                this.saveBusinessMessageInner(cardId, tenantId, pageCode, cardBusinessMessage);
            } else {
                log.warn("Get businessMessage empty, tmTaskId: {}, tmActivityId: {}", tmTaskId, tmActivityId);
            }
        } catch (Exception e) {
            log.warn("Get and Save businessMessage failed: {}", ExceptionUtil.getSimpleMessage(e));
        }
    }

    private void saveBusinessMessageInner(Long cardId, String tenantId, String pageCode, Map<String, Object> businessMessage) {
        // 先删除再新增
        this.removeBusinessMessage(Lists.newArrayList(cardId), tenantId, pageCode);
        BacklogCardField backlogCardField = buildBacklogCardField(cardId,tenantId,pageCode,businessMessage);
        if (CollectionUtils.isEmpty(backlogCardField.getValueList())){
            return;
        }
        atmcCardBusinessMessageMongoTemplate.insert(backlogCardField, BACKLOG_CARD_FIELD);
    }

    /**
     * 删除BusinessMessage
     *
     * @param cardIds
     * @param tenantId
     * @param pageCode
     */
    public void removeBusinessMessage(Collection<Long> cardIds, String tenantId, String pageCode) {
        if (CollectionUtils.isEmpty(cardIds) || StringUtils.isBlank(tenantId) || StringUtils.isBlank(pageCode)) {
            log.debug("RemoveBusinessMessage fail, cardId: {}, tenantId: {}, pageCode: {}", null, tenantId, pageCode);
            return;
        }
        Criteria criteria = Criteria.where(BkConstant.CARD_ID).in(cardIds)
                .and(BkConstant.PAGE_CODE).is(pageCode)
                .and(BkConstant.TENANT_ID).is(tenantId);
        atmcCardBusinessMessageMongoTemplate.remove(Query.query(criteria), BACKLOG_CARD_FIELD);
    }

    public BacklogCardField buildBacklogCardField(Long cardId, String tenantId, String pageCode,Map<String, Object> businessMessage){
        Map<String, LinkedHashSet<Object>> valueMap = Maps.newHashMap();
        // 新增；按代理编码聚合去重
        if (MapUtils.isEmpty(businessMessage)) {
            return new BacklogCardField();
        }
        for (Map.Entry<String, Object> entry : businessMessage.entrySet()) {
            String fieldPathCombine = entry.getKey();
            List<String> fieldPathList = CharSequenceUtil.split(fieldPathCombine, "__");
            String aliasField = fieldPathList.get(1);
            // 单身的拼接用点存入库会报错切前台对象的key用包含点会报错
            Set<Object> aliasFieldValueSet = valueMap.computeIfAbsent(aliasField.replaceAll("\\.","#"), v -> Sets.newLinkedHashSet());
            Object fieldValueObj = entry.getValue();
            if (fieldValueObj instanceof Collection) {
                aliasFieldValueSet.addAll((Collection) fieldValueObj);
            } else {
                aliasFieldValueSet.add(fieldValueObj);
            }
        }
        BacklogCardField backlogCardField = new BacklogCardField()
                .setCardId(cardId)
                .setPageCode(pageCode)
                .setTenantId(tenantId)
                .setValueList(Lists.newArrayList(valueMap));
        return backlogCardField;
    }

    /**
     *  存入标签分组筛选排序业务字段值
     * @param tenantId
     * @param pageCode
     * @param cardToData
     */
    public void saveOrUpdateBusinessMessageInner(String tenantId, String pageCode, Map<Long, Map<String, Object>> cardToData) {
        if(MapUtils.isEmpty(cardToData)){
            return;
        }
        Set<Long> set = cardToData.keySet();
        this.removeBusinessMessage(set, tenantId, pageCode);
        List<BacklogCardField> rows = new ArrayList<>(cardToData.size());
        cardToData.forEach((k,v)->{
            if(!MapUtils.isEmpty(v)){
                BacklogCardField backlogCardField = buildBacklogCardField(k, tenantId, pageCode, v);
                if (CollectionUtils.isNotEmpty(backlogCardField.getValueList())){
                    rows.add(backlogCardField);
                }
            }
        });
        if(rows.size()>0){
            atmcCardBusinessMessageMongoTemplate.insert(rows, BACKLOG_CARD_FIELD);
        }
    }

    /**
     * 获取卡面（任务、项目卡）所需业务数据
     *
     * @param cardIdList 待办卡id集合
     * @param tenantId   租户id
     * @return null 或 key: cardId，key：代理字段code，value：业务值数组
     */
    public Map<Long, Map<String, List<String>>> getAndBuildBusinessMessage(List<Long> cardIdList, String pageCode, String tenantId) {
        Map<Long, Map<String, List<String>>> businessMessage = null;
        try {
            long start = System.currentTimeMillis();
            businessMessage = this.getAndBuildBusinessMessageInner(cardIdList, pageCode, tenantId);
            log.debug("Get batch businessMessage cost: {}ms, batchSize: {}", System.currentTimeMillis() - start, cardIdList.size());
        } catch (Exception e) {
            log.warn("Get batch businessMessage failed: {}", ExceptionUtil.getSimpleMessage(e));
        }
        return businessMessage;
    }

    public Map<Long, Map<String, List<String>>> getAndBuildBusinessMessageInner(List<Long> cardIdList, String pageCode, String tenantId) {
        if (StringUtils.isBlank(tenantId)) {
            tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        }
        Criteria criteria = Criteria.where(BkConstant.CARD_ID).in(cardIdList)
                .and(BkConstant.PAGE_CODE).is(pageCode)
                .and(BkConstant.TENANT_ID).is(tenantId);
        List<BacklogCardField> backlogCardFieldList = atmcCardBusinessMessageMongoTemplate.find(new Query(criteria),
                BacklogCardField.class, BACKLOG_CARD_FIELD);
        Map<Long,Map<String, List<String>>> map = Maps.newHashMapWithExpectedSize(backlogCardFieldList.size());
        for (BacklogCardField backlogCardField : backlogCardFieldList) {
            List<Map<String, LinkedHashSet<Object>>> valueList = backlogCardField.getValueList();
            if(CollectionUtils.isNotEmpty(valueList)){
                Map<String, LinkedHashSet<Object>> row = valueList.get(0);
                Map<String, List<String>> firstRow = Maps.newHashMapWithExpectedSize(row.size());
                row.forEach((k,v)->{
                    if(CollectionUtils.isNotEmpty(v)){
                        firstRow.put(k,v.stream().map(String::valueOf).collect(Collectors.toList()));
                    }
                });
                map.put(backlogCardField.getCardId(),firstRow);
            }
        }
        return map;
    }

    public Map<String, Object> getBusinessMessage(Map activityDataDTO) {
        try {
            String tmTaskId = MapUtils.getString(activityDataDTO, "tmTaskId");
            String tmActivityId = MapUtils.getString(activityDataDTO, "tmActivityId");
            BaseResultDTO<Map<String, Object>> cardBusinessMessageResp = aglieService.getCardBusinessMessage(tmTaskId, tmActivityId, activityDataDTO);
            Map<String, Object> cardBusinessMessage = cardBusinessMessageResp.getResponse();
            return cardBusinessMessage;
        } catch (Exception e) {
            log.warn("Get and Save businessMessage failed: {}", ExceptionUtil.getSimpleMessage(e));
        }
        return null;
    }
}
