/*
 * 文 件 名:  ExcelParser.java
 * 版    权:
 * 描    述:   <描述>
 * 版    本：       <版本号>
 * 创 建 人:  <创建人>
 * 创建时间: 2018年3月15日
 *
 */
package com.digiwin.athena.abt.application.utils;

import com.digiwin.athena.abt.application.dto.migration.abt.excel.CellTypeHolder;
import com.digiwin.athena.abt.application.dto.migration.abt.excel.ExcelParserBean;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.ApiDataFieldLocaleMetadataDTO;
import com.digiwin.athena.abt.core.meta.dto.CellTypeContainer;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.abt.core.meta.enums.ExcelTypeEnum;
import com.digiwin.athena.abt.core.meta.enums.MetaDataType;
import com.digiwin.athena.appcore.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.xssf.usermodel.XSSFDataValidation;
import org.apache.poi.xssf.usermodel.XSSFFont;
import org.apache.poi.xssf.usermodel.XSSFRichTextString;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.context.i18n.LocaleContextHolder;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Slf4j
public class ExcelUtil {

    public static final int ENUM_PROMPT_LENGTH_LIMIT = 255;

    public static final String MAIN_SHEET_NAME = "sheet1";

    private static String handleLocale(String locale) {
        if (StringUtils.isEmpty(locale)) {
            return "zh_CN";
        } else if (locale.equals("en")) {
            return "en_US";
        }
        return locale;
    }

    public static XSSFRichTextString addRequiredMark(XSSFWorkbook wb, String s, Font cellFont) {
        XSSFFont font = wb.createFont();
        font.setColor(XSSFFont.COLOR_RED);
        font.setBold(true);
        XSSFRichTextString str = new XSSFRichTextString("*" + s);
        // "*" 红色加粗
        str.applyFont(0, 1, font);
        // 内容保持原有的字体样式
        if (1 < str.getString().length()) {
            str.applyFont(1, str.length(), cellFont);
        }
        return str;
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 约束单元格数据长度
     * @Date 10:53 2021/11/15
     * @Param [locale] 多语言类别 [sheet]excel工作簿 [min]长度最小值 [max]长度最大值 [firstRow]起始行 [lastRow]结束行65536
     * [firstCol]起始列 [lastCol]结束列
     **/
    public static void excelRuleStringLength(String locale, Sheet sheet, int min, int max, int firstRow, int lastRow, int firstCol, int lastCol) {
        excelRuleStringLengthWithAnnotation(locale, sheet, min, max, firstRow, lastRow, firstCol, lastCol, "");
    }

    /**
     *
     * @param locale
     * @param sheet
     * @param min
     * @param max
     * @param firstRow
     * @param lastRow
     * @param firstCol
     * @param lastCol
     * @param annotationStr
     * @Description 约束单元格数据长度,批注    优先使用配置的模版批注
     */
    public static void excelRuleStringLengthWithAnnotation(String locale, Sheet sheet, int min, int max, int firstRow, int lastRow, int firstCol, int lastCol,String annotationStr) {
        locale = handleLocale(locale);
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        //唯一
        DataValidationConstraint constraint = helper.createNumericConstraint(
                DataValidationConstraint.ValidationType.TEXT_LENGTH,
                DataValidationConstraint.OperatorType.BETWEEN, String.valueOf(min), String.valueOf(max));
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        String title = MessageUtil.getMessageByLocale("excel.string.title.length", locale, max);
        String content = MessageUtil.getMessageByLocale("excel.string.content.length", locale, max);
        //有配置的模版批注则优先使用配置的模版批注
        if (StringUtils.isNotEmpty(annotationStr)){
            title = "";
            content = annotationStr;
        }
        dataValidation.createPromptBox(title, content);
        dataValidation.createErrorBox(title, content);
        dataValidation.setShowErrorBox(true);
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setSuppressDropDownArrow(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }

    public static void excelRuleDateFormat(String locale, Sheet sheet, String start, String end, String format, int firstRow, int lastRow, int firstCol, int lastCol) {
        excelRuleDateFormatWithAnnotation(locale, sheet, start, end, format, firstRow, lastRow, firstCol, lastCol, "");
    }

    /**
     *
     * @param locale
     * @param sheet
     * @param start
     * @param end
     * @param format
     * @param firstRow
     * @param lastRow
     * @param firstCol
     * @param lastCol
     * @param annotationStr
     * 设置时间规则 批注    优先使用配置的模版批注
     */
    public static void excelRuleDateFormatWithAnnotation(String locale, Sheet sheet, String start, String end, String format, int firstRow, int lastRow, int firstCol, int lastCol,String annotationStr) {
        locale = handleLocale(locale);
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        //唯一
        DataValidationConstraint constraint = helper.createDateConstraint(DataValidationConstraint.OperatorType.GREATER_OR_EQUAL, start, end, format);
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        String content = MessageUtil.getMessageByLocale("excel.date.content.enterCorrect", locale, format);
        //有配置的模版批注则优先使用配置的模版批注
        if (StringUtils.isNotEmpty(annotationStr)){
            content = annotationStr;
        }
        dataValidation.createPromptBox("", content);
        dataValidation.createErrorBox(MessageUtil.getMessageByLocale("excel.date.title.enterCorrect", locale), content);
        dataValidation.setShowErrorBox(true);
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setSuppressDropDownArrow(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }

    public static void excelRuleSelect(String locale, Sheet sheet, String[] rule, int firstRow, int lastRow, int firstCol, int lastCol) {
        excelRuleSelectWithAnnotation(locale, sheet, rule, firstRow, lastRow, firstCol, lastCol,"");
    }

    /**
     * 设置规则 批注    优先使用配置的模版批注
     * @param locale
     * @param sheet
     * @param rule
     * @param firstRow
     * @param lastRow
     * @param firstCol
     * @param lastCol
     * @param annotationStr
     */
    public static void excelRuleSelectWithAnnotation(String locale, Sheet sheet, String[] rule, int firstRow, int lastRow, int firstCol, int lastCol,String annotationStr) {
        locale = handleLocale(locale);
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        DataValidationConstraint constraint = helper.createExplicitListConstraint(rule);
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        String itemStr = MessageUtil.getMessageByLocale("excel.select.item.selectCorrect", locale, Arrays.asList(rule).toString());
        String title = MessageUtil.getMessageByLocale("excel.select.title.selectCorrect", locale);
        String content = MessageUtil.getMessageByLocale("excel.select.content.selectCorrect", locale, Arrays.asList(rule).toString());
        //有配置的模版批注则优先使用配置的模版批注
        if (StringUtils.isNotEmpty(annotationStr)){
            itemStr = annotationStr;
            content = annotationStr;
        }
        dataValidation.createPromptBox("", itemStr);
        dataValidation.createErrorBox(title, content);
        if (dataValidation instanceof XSSFDataValidation) {
            dataValidation.setSuppressDropDownArrow(true);
            dataValidation.setShowErrorBox(true);
        } else {
            dataValidation.setSuppressDropDownArrow(false);
        }
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }
    /**
     * #22869
     * 1.hover提示的标题去除
     * 2.hover内容超过255自动截取，如正好截取在一个字段，则整个字段截取
     * 3.下拉框枚举值内容限定在255个字符，自动截取，存在255所在字段值，则整个字段截取
     * 设置批注    优先使用配置的模版批注
     * @param locale   语言别
     * @param sheet    sheet
     * @param rule     枚举值
     * @param firstRow 第一行
     * @param lastRow  最后一行
     * @param firstCol 第一列
     * @param lastCol  最后一列
     */
    public static void excelEnumRuleSelect(String locale, Sheet sheet, String[] rule, int firstRow, int lastRow, int firstCol, int lastCol,String annotationStr) {
        locale = handleLocale(locale);
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        DataValidationConstraint constraint = helper.createExplicitListConstraint(rule);
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        String itemByLocale = Optional.ofNullable(MessageUtil.getMessageByLocale("excel.select.item.selectCorrect", locale, "")).orElse("");
        String itemStr = MessageUtil.getMessageByLocale("excel.select.item.selectCorrect", locale, toStringWithLimit(rule, ENUM_PROMPT_LENGTH_LIMIT - itemByLocale.length()));
        String title = MessageUtil.getMessageByLocale("excel.select.title.selectCorrect", locale);
        String contentByLocale = Optional.ofNullable(MessageUtil.getMessageByLocale("excel.select.content.selectCorrect", locale, "")).orElse("");
        String content = MessageUtil.getMessageByLocale("excel.select.content.selectCorrect", locale, toStringWithLimit(rule, ENUM_PROMPT_LENGTH_LIMIT - contentByLocale.length()));
        //有配置的模版批注则优先使用配置的模版批注
        if (StringUtils.isNotEmpty(annotationStr)){
            itemStr = annotationStr;
            content = annotationStr;
        }
        dataValidation.createPromptBox(null, itemStr);
        dataValidation.createErrorBox(title, content);
        if (dataValidation instanceof XSSFDataValidation) {
            dataValidation.setSuppressDropDownArrow(true);
            dataValidation.setShowErrorBox(true);
        } else {
            dataValidation.setSuppressDropDownArrow(false);
        }
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }

    private static String toStringWithLimit(String[] array, int limit) {
        // 去掉中括号的占用
        limit -= 2;
        // 转换数组为字符串
        String arrayString = Arrays.toString(array);
        // 去掉前后的中括号
        arrayString = arrayString.substring(1, arrayString.length() - 1);
        // 检查字符串长度是否超过limit
        while (arrayString.length() > limit) {
            // 如果超过，删除数组的最后一个元素
            array = Arrays.copyOf(array, array.length - 1);
            // 重新转换剩余的数组为字符串
            arrayString = Arrays.toString(array);
            arrayString = arrayString.substring(1, arrayString.length() - 1);
        }
        // 返回最终的字符串
        return "[" + arrayString + "]";
    }

    public static void excelRuleIsNumber(String locale, Sheet sheet, int firstRow, int lastRow, int firstCol, int lastCol) {
        excelRuleIsNumberWithAnnotation(locale, sheet, firstRow, lastRow, firstCol, lastCol, "");
    }

    /**
     *
     * @param locale
     * @param sheet
     * @param firstRow
     * @param lastRow
     * @param firstCol
     * @param lastCol
     * @param annotationStr
     * 设置数字规则 批注    优先使用配置的模版批注
     */
    public static void excelRuleIsNumberWithAnnotation(String locale, Sheet sheet, int firstRow, int lastRow, int firstCol, int lastCol,String annotationStr) {
        locale = handleLocale(locale);
        String isNumber = "=ISNUMBER(INDIRECT(ADDRESS(ROW(),COLUMN())))";
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        //设置数据
        DataValidationConstraint constraint = helper.createCustomConstraint(isNumber);
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        //有配置的模版批注则优先使用配置的模版批注
        dataValidation.createPromptBox("",StringUtils.isNotEmpty(annotationStr)? annotationStr : MessageUtil.getMessageByLocale("excel.number.content.isNumber", locale));
        dataValidation.createErrorBox(MessageUtil.getMessageByLocale("excel.number.title.isNumber", locale), StringUtils.isNotEmpty(annotationStr)? annotationStr : MessageUtil.getMessageByLocale("excel.number.content.isNumber", locale));
        //处理Excel兼容性问题
        if (dataValidation instanceof XSSFDataValidation) {
            dataValidation.setSuppressDropDownArrow(true);
            dataValidation.setShowErrorBox(true);
        } else {
            dataValidation.setSuppressDropDownArrow(false);
        }
        dataValidation.setShowErrorBox(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setSuppressDropDownArrow(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }
    public static void setDefaultExcelType(ExcelParserBean parserBean) {
        // 获取文件后缀
        final String filePath = parserBean.getFilePath();
        final String fileExtension = ExcelUtil.getFileExtension(filePath);

        ExcelTypeEnum excelType = parserBean.getExcelType();
        // 默认为xls格式excel
        if ((excelType == null) && (fileExtension == null)) {
            excelType = ExcelTypeEnum.XLS;
        } else if ((excelType == null) && (fileExtension != null)) {
            if (fileExtension.equalsIgnoreCase(ExcelTypeEnum.XLS.name())) {
                excelType = ExcelTypeEnum.XLS;
            } else if (fileExtension.equalsIgnoreCase(ExcelTypeEnum.XLSX.name())) {
                excelType = ExcelTypeEnum.XLSX;
            }
        }
        parserBean.setExcelType(excelType);
    }

    public static Workbook createWorkbook(ExcelParserBean parserBean) {
        Workbook wb = null;
        final ExcelTypeEnum excelType = parserBean.getExcelType();
        try {
            if (ExcelTypeEnum.XLS.equals(excelType)) {
                wb = new HSSFWorkbook(parserBean.getInput());
            } else if (ExcelTypeEnum.XLSX.equals(excelType)) {
                wb = new XSSFWorkbook(parserBean.getInput());
            } else {
                wb = new HSSFWorkbook(parserBean.getInput());
            }
        } catch (final IOException e) {
            log.error(e.toString());
        }
        return wb;
    }

    /**
     * 导入时读取sheet页中的数据
     *
     * @param sheet
     * @param apiDataFieldLocaleMetadataDTOS
     * @return
     */
    public static List<Map> readDataForImport(Sheet sheet, List<ApiDataFieldLocaleMetadataDTO> apiDataFieldLocaleMetadataDTOS) {
        //将存在合并单元格的列记录入put进hashmap并返回
        Map<String, Integer[]> map = ExcelUtil.getMergedRegionMap(sheet);
        //去除空行后的sheet页
        Sheet sheetExcepNullRow = ExcelUtil.getAccuracyContextNum(sheet, map);

        if (CollectionUtils.isEmpty(apiDataFieldLocaleMetadataDTOS)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0086.getErrCode(), MessageUtil.getMessage("basedata.apiNoDF"));
        }
        //将参数列表转为map
        Map<String, String> metaMap = apiDataFieldLocaleMetadataDTOS.stream()
                .collect(Collectors.toMap(ApiDataFieldLocaleMetadataDTO::getData_name,
                        ApiDataFieldLocaleMetadataDTO::getData_type, (item1, item2) -> item1));
        List<CellTypeHolder> cellTypeHolders = new LinkedList<>();

        //第一二行为headers
        List<String> headerKeys = new LinkedList<>();
        Row headRow2 = sheetExcepNullRow.getRow(1);
        int cellStartIdx = 0;
        while (null != headRow2.getCell(cellStartIdx)) {
            String key = headRow2.getCell(cellStartIdx).getStringCellValue();
            headerKeys.add(key);
            cellStartIdx++;
        }

        //遍历表头字段名
        for (int i = 0; i < headerKeys.size(); i++) {
            //仅将元数据描述包含的入参字段加入
            if (null != metaMap.get(headerKeys.get(i))) {
                CellTypeHolder cellTypeHolder = new CellTypeHolder();
                cellTypeHolder.setHeaderKey(headerKeys.get(i));
                cellTypeHolder.setIndex(i);
                cellTypeHolder.setMetaDataType(MetaDataType.valueOf(metaMap.get(headerKeys.get(i)).toUpperCase()));
                switch (MetaDataType.valueOf(metaMap.get(headerKeys.get(i)).toUpperCase())) {
                    case NUMERIC:
                    case NUMBER:
                        cellTypeHolder.setCellType(CellType.NUMERIC);
                        cellTypeHolders.add(cellTypeHolder);
                        break;
                    case BOOLEAN:
                        cellTypeHolder.setCellType(CellType.BOOLEAN);
                        cellTypeHolders.add(cellTypeHolder);
                        break;
                    case STRING:
                    default:
                        cellTypeHolder.setCellType(CellType.STRING);
                        cellTypeHolders.add(cellTypeHolder);
                        break;
                }
            }
        }

        // 修复bug:76100
        if (CollectionUtils.isEmpty(cellTypeHolders)) {
            log.error("解析Excel的栏位为空！");
            log.info("messages:{},local:{}", MessageUtil.getMessageByLocale("upload.template.error", LocaleContextHolder.getLocale().toString()), LocaleContextHolder.getLocale().toString());
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0116.getErrCode(), MessageUtil.getMessageByLocale("upload.template.error", LocaleContextHolder.getLocale().toString()));
        }
        List<Map> data = new LinkedList<>();
        int rowStartIdx = 2;
        while (null != sheetExcepNullRow.getRow(rowStartIdx)) {
            Map<String, Object> rowData = new LinkedHashMap<>();
            Row row = sheetExcepNullRow.getRow(rowStartIdx);
            for (int i = 0; i < cellTypeHolders.size(); i++) {
                CellTypeHolder cellTypeHolder = cellTypeHolders.get(i);
                //因为数据和根据index获取对应列
                final Cell cell = row.getCell(cellTypeHolder.getIndex());
                Object cellObj = null;
                if (cell != null) {
                    cellObj = ExcelUtil.parseCell(cell, cellTypeHolder.getCellType(), cellTypeHolder.getMetaDataType());
                }
                rowData.put(cellTypeHolder.getHeaderKey(), cellObj);
            }
            data.add(rowData);
            rowStartIdx++;
        }
        return data;
    }

    /**
     * 将存在合并单元格的列记录入put进hashmap并返回
     *
     * @param sheet
     * @return
     */
    public static Map<String, Integer[]> getMergedRegionMap(Sheet sheet) {

        Map<String, Integer[]> result = new HashMap<String, Integer[]>();

        //获取excel中的所有合并单元格信息
        int sheetMergeCount = sheet.getNumMergedRegions();

        //遍历处理
        for (int i = 0; i < sheetMergeCount; i++) {

            //拿到每个合并单元格，开始行，结束行，开始列，结束列
            CellRangeAddress range = sheet.getMergedRegion(i);
            int firstColumn = range.getFirstColumn();
            int lastColumn = range.getLastColumn();
            int firstRow = range.getFirstRow();
            int lastRow = range.getLastRow();

            //构造一个开始行和开始列组成的数组
            Integer[] firstRowNumberAndCellNumber = new Integer[]{firstRow, firstColumn};

            //遍历，将单元格中的所有行和所有列处理成由行号和下划线和列号组成的key，然后放在hashmap中
            for (int currentRowNumber = firstRow; currentRowNumber <= lastRow; currentRowNumber++) {

                for (int currentCellNumber = firstColumn; currentCellNumber <= lastColumn; currentCellNumber++) {
                    result.put(currentRowNumber + "_" + currentCellNumber, firstRowNumberAndCellNumber);
                }
            }
        }
        return result;
    }

    /**
     * 去除空行后的sheet页
     *
     * @param sheetOld
     * @param mergedRegionMap
     * @return
     */
    public static Sheet getAccuracyContextNum(Sheet sheetOld, Map<String, Integer[]> mergedRegionMap) {
        // 取第一个sheet
        Sheet sheet = sheetOld;
        // 删除空行
        for (int i = 0; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);
            // 处理Excel中的空行，并上移空行之后的所有行以覆盖空行
            if (row == null) {
                // 删除空行
                int lastRowNum = sheet.getLastRowNum();
                if (i >= 0 && i < lastRowNum) {
                    sheet.shiftRows(i + 1, lastRowNum, -1);// 将行号为i+1一直到行号为lastRowNum的单元格全部上移一行，以便删除i行
                }
                i--;
                continue;
            }
            int cellNum = row.getLastCellNum();
            boolean flag = false;
            for (int j = 0; j < cellNum; j++) {
                Integer[] firstRowNumberAndCellNumber = mergedRegionMap.get(i + "_" + j);
                if (firstRowNumberAndCellNumber != null) {
                    flag = true;
                }
            }
            if (isRowEmpty(row) && !flag) {
                // 删除空行
                if (isRowEmpty(row)) {
                    int lastRowNum = sheet.getLastRowNum();
                    if (i >= 0 && i < lastRowNum) {
                        sheet.shiftRows(i + 1, lastRowNum, -1);// 将行号为i+1一直到行号为lastRowNum的单元格全部上移一行，以便删除i行
                    }
                    if (i == lastRowNum) {
                        if (row != null) {
                            sheet.removeRow(row);
                        }
                    }
                    i--;
                }
            }
        }
        return sheet;
    }

    /**
     * 判断是否有空行
     *
     * @param row
     * @return
     */
    protected static boolean isRowEmpty(Row row) {
        //行不存在
        if (row == null) {
            return true;
        }
        //第一个列位置
        int firstCellNum = row.getFirstCellNum();
        //最后一列位置
        int lastCellNum = row.getLastCellNum();
        //空列数量
        int nullCellNum = 0;
        for (int c = firstCellNum; c < lastCellNum; c++) {
            Cell cell = row.getCell(c);
            if (null == cell || cell.getCellType() == CellType.BLANK) {
                nullCellNum++;
                continue;
            }
            if (CellType.STRING.equals(cell.getCellTypeEnum())) {
                String cellValue = cell.getStringCellValue().trim();
                if (StringUtils.isEmpty(cellValue)) {
                    nullCellNum++;
                }
            }
        }
        //所有列都为空
        if (nullCellNum == (lastCellNum - firstCellNum)) {
            return true;
        }
        return false;
    }

    /**
     * @param cellTypeContainerList
     * @return
     */
    public static List<String> getBusinessKeyList(List<CellTypeContainer> cellTypeContainerList) {
        if (CollectionUtils.isEmpty(cellTypeContainerList)) {
            return new ArrayList<>();
        }
        return cellTypeContainerList.stream().filter(cellTypeContainer -> cellTypeContainer.getBusinessKey()).map(CellTypeContainer::getKeyName).collect(Collectors.toList());
    }

    protected static Object parseCell(Cell cell, CellType metaCellType, MetaDataType metaType) {
        Object obj;
        final CellType cellType = cell.getCellTypeEnum();
        switch (cellType) {
            case FORMULA:
                /*
                 * if (cell != null) { cell.setCellType(Cell.CELL_TYPE_FORMULA); obj = cell.getStringCellValue(); }
                 */
                try {
                    /*
                     * 此处判断使用公式生成的字符串有问题，因为HSSFDateUtil.isCellDateFormatted(cell) 判断过程中cell
                     * .getNumericCellValue();方法会抛出java.lang.NumberFormatException异常
                     */
                    if (DateUtil.isCellDateFormatted(cell)) {
                        final Date date = cell.getDateCellValue();
                        obj = (date.getYear() + 1900) + "-" + (date.getMonth() + 1) + "-" + date.getDate();
                        break;
                    } else {
                        obj = String.valueOf(cell.getNumericCellValue());
                    }
                } catch (final IllegalStateException e) {
                    obj = String.valueOf(cell.getRichStringCellValue());
                }
                break;
            case BOOLEAN:
                obj = cell.getBooleanCellValue();
                break;
            case NUMERIC:
                // 处理日期格式、时间格式
                int number58 = 58;
                SimpleDateFormat sdf;
                switch (metaType) {
                    case DATE:
                        sdf = new SimpleDateFormat("yyyy-MM-dd");
                        break;
                    case DATETIME:
                        sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        break;
                    case TIME:
                        sdf = new SimpleDateFormat("HH:mm:ss");
                        break;
                    default:
                        sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        break;
                }

                if (DateUtil.isCellDateFormatted(cell)) {
                    final Date date = cell.getDateCellValue();
                    obj = sdf.format(date);
                } else if (cell.getCellStyle().getDataFormat() == number58) {
                    // 处理自定义日期格式：m月d日(通过判断单元格的格式id解决，id的值是58)
                    final double value = cell.getNumericCellValue();
                    final Date date = DateUtil.getJavaDate(value);
                    obj = sdf.format(date);
                } else {
                    final double dValue = cell.getNumericCellValue();
                    obj = dValue;
                    return obj;
                /*final CellStyle style = cell.getCellStyle();
                final DecimalFormat format = new DecimalFormat();
                final String temp = style.getDataFormatString();
                // 单元格设置成常规
                if ("General".equals(temp)) {
                    format.applyPattern("#");
                }
                obj = format.format(value);*/
                }
                break;
            default:
                obj = cell.getStringCellValue();
                break;
        }
        //return obj;
        if (null == metaCellType) {
            return obj;
        }
        switch (metaCellType) {
            case STRING:
                return null == obj ? "" : obj.toString();
            default:
                return obj;
        }
    }

    private static String getFileExtension(String filePath) {
        if (StringUtils.isEmpty(filePath)) {
            return null;
        }
        final int lastIndexOf = filePath.lastIndexOf(".");
        if (lastIndexOf != -1) {
            final String extension = filePath.substring(lastIndexOf + 1);
            return extension;
        }
        return null;
    }

    public static String getSheetNameInParens(String sheetName) {
        String regex = "[\\(（](.*?)[\\)）]";
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(sheetName);
        if (matcher.find()) {
            // 使用group(1)获取第一个括号中的内容
            return matcher.group(1);
        } else {
            return sheetName;
        }
    }

    /**
     * 校验日期格式
     * @param dateString
     * @param dateFormat
     * @return
     */
    public static boolean isValidDate(String dateString, String dateFormat) {
        SimpleDateFormat formatter = new SimpleDateFormat(dateFormat);
        formatter.setLenient(false);
        try {
            formatter.parse(dateString);
            return true;
        } catch (ParseException e) {
            return false;
        }
    }

}
