package com.digiwin.athena.abt.application.utils;

import com.digiwin.athena.abt.application.dto.request.UploadBatchFileReq;
import com.digiwin.athena.abt.application.dto.request.UploadBatchReq;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import lombok.extern.slf4j.Slf4j;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author wzq
 */
@Slf4j
public class JsonToMd5Converter {

    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper()
            .configure(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS, true)
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);
    private static final char[] HEX_ARRAY = "0123456789abcdef".toCharArray();

    public static Map<Integer, UploadBatchFileReq> getFileUids(UploadBatchReq uploadBatchReq) {
        Map<Integer, UploadBatchFileReq> result = new HashMap<>(uploadBatchReq.getFiles().size());
        try {
            List<UploadBatchFileReq> files = uploadBatchReq.getFiles();
            for (int i = 0; i < files.size(); i++) {
                UploadBatchFileReq currentFile = files.get(i);
                // 构建modifiedReq
                UploadBatchReq modifiedReq = buildModifiedRequest(uploadBatchReq);

                // 序列化为排序后的JSON
                String sortedJson = convertToSortedJson(modifiedReq);

                // 获取当前文件的checkSum并拼接
                String combined = sortedJson + ":" + currentFile.getCheckSum();
                log.info("Filtered JSON: " + combined);

                // 生成MD5校验码
                String fileUid = generateMD5(combined);
                log.info("Generated MD5 ID: " + fileUid);
                currentFile.setFileUid(fileUid);
                result.put(i, currentFile);
            }
        } catch (Exception e) {
            log.error("获取md5失败", e);
        }

        return result;
    }

    /**
     * 创建修改后的文件列表，移除目标文件的checkSum
     */
    private static List<UploadBatchFileReq> createModifiedFiles(List<UploadBatchFileReq> originalFiles) {
        List<UploadBatchFileReq> modifiedFiles = new ArrayList<>();
        for (UploadBatchFileReq file : originalFiles) {
            UploadBatchFileReq modifiedFile = new UploadBatchFileReq();
            modifiedFile.setFileName(null);
            modifiedFile.setFileUrl(null);
            modifiedFile.setCheckSum(null);
            modifiedFiles.add(modifiedFile);
        }
        return modifiedFiles;
    }

    /**
     * 构建修改后的请求对象
     */
    private static UploadBatchReq buildModifiedRequest(UploadBatchReq original) {
        return UploadBatchReq.builder()
                .actionId(original.getActionId())
                .activityId(original.getActivityId())
                .actionInfo(original.getActionInfo())
                .requiredFields(original.getRequiredFields())
                .isReImport(original.getIsReImport())
                .files(null)
                .build();
    }

    /**
     * 将对象转换为排序后的JSON字符串
     */
    private static String convertToSortedJson(UploadBatchReq obj) throws JsonProcessingException {
        return OBJECT_MAPPER.writeValueAsString(obj);
    }

    /**
     * 生成MD5校验码
     */
    private static String generateMD5(String input) throws NoSuchAlgorithmException {
        MessageDigest md = MessageDigest.getInstance("MD5");
        byte[] digest = md.digest(input.getBytes());
        return bytesToHex(digest);
    }

    /**
     * 将字节数组转换为十六进制字符串
     */
    private static String bytesToHex(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; j++) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_ARRAY[v & 0x0F];
        }
        return new String(hexChars);
    }

    public static void main(String[] args) throws Exception {
//        String input = "{\"actionId\":\"esp_dpbas.sspresstest.create\",\"activityId\":\"DE_e74e78421000166a\",\"actionInfo\":\"{\\\"sheetRequiredFiled\\\":{\\\"sspresstesta\\\":[\\\"sspnoa\\\"]}}\",\"requiredFields\":[],\"files\":["
//                + "{\"fileName\":\"ss压测双档_测试作业 - 副本\",\"checkSum\":\"1\",\"fileUrl\":\"1d3fd47a-1215-4ecf-b1e2-d45fbcdb3e16\"},"
//                + "{\"fileName\":\"ss压测双档_测试作业\",\"checkSum\":\"2\",\"fileUrl\":\"1d3fd47a-1215-4ecf-b1e2-d45fbcdb3e22\"}"
//                + "]}";
//
//        UploadBatchReq uploadBatchReq = JSON.parseObject(input, UploadBatchReq.class);
//
//        Map<Integer, String> md5Result = getMd5(uploadBatchReq);
//        System.out.println(md5Result);;
    }
}