package com.digiwin.athena.abt.application.utils;

import com.digiwin.athena.abt.application.configuration.EnvProperties;
import com.digiwin.athena.abt.application.dto.migration.atmc.emc.AppMessageDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.emc.AppMessageReceiverDTO;
import com.digiwin.athena.abt.application.dto.migration.atmc.message.MessageDO;
import com.digiwin.athena.abt.application.dto.migration.atmc.usersetting.PersonalizedDto;
import com.digiwin.athena.abt.application.service.abt.migration.ptm.ShareService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.emc.SendEmailService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.iam.UserService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.iam.impl.UserServiceImpl;
import com.digiwin.athena.abt.infrastructure.mapper.biz.migration.atmc.TaskMapper;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.atmc.Task;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author lzw
 * @date 2024/4/26
 * @description:
 **/
@Service
public class NoticeMessageUtil {

    private static final Logger logger = LoggerFactory.getLogger(NoticeMessageUtil.class);

    private static final String ACTIVITY = "activity";

    private static final String TASK = "task";

    @Autowired
    private UserService userService;

    @Autowired
    private SendEmailService sendEmailService;

    @Autowired
    private ShareService shareService;

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private TaskMapper taskMapper;
    /**
     * 邮件发送的方式，进行强提醒
     * @param appMessageDTO appMessageDTO
     */
    public void sendRemindNotice(AppMessageDTO appMessageDTO)
    {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        if(null == appMessageDTO.getReceivers() || appMessageDTO.getReceivers().isEmpty())
        {
            logger.warn("no member need to send notice to client.");
            return;
        }

        //判断用户对应的配置，如果使用邮件或者im+邮件的，才需要发送邮件
        List<AppMessageReceiverDTO> receiveList = new ArrayList<>();
        //检查用户的配置的类型，如果是是非邮件类型，则收不到邮件提醒
        for(AppMessageReceiverDTO dto : appMessageDTO.getReceivers())
        {
            int result = checkSendType(dto.getTenantSid(), dto.getUserId());
            if(0!= result)
            {
                receiveList.add(dto);
            }
        }
        List<Map<String,Object>> receiverLists = new ArrayList<>();
        //获取用户的邮箱列表，并给对应的用户发送邮件通知
        for(AppMessageReceiverDTO dto : receiveList)
        {
            Map<String,Object> receiverMap = new HashMap<>();
            receiverMap.put("userId",dto.getUserId());
            receiverMap.put("email",userService.queryUserEmail(dto.getUserId()));
            receiverMap.put("templateId", getUserTemplateId(dto.getUserId(),authoredUser.getToken()));
            receiverLists.add(receiverMap);
        }
        MessageDO messageDO = new MessageDO();
        messageDO.setType(appMessageDTO.getType());
        messageDO.setContent(appMessageDTO.getContent());
        messageDO.setTenantId(getTargetTenantId(appMessageDTO.getReceivers()));
        Map<String, Object> mailData = dealNoticeMailMessage(authoredUser, messageDO);
        //获取对应的邮件配置模版
        for(Map<String,Object> temp:receiverLists)
        {
            String email = MapUtils.getString(temp,"email");
            String userId = MapUtils.getString(temp,"userId");
            String templateId = MapUtils.getString(temp,"templateId");
            if(StringUtils.isBlank(email))
            {
                logger.warn("send {} to user:{} with no email",mailData,userId);
                continue;
            }
            sendEmailService.sendEmailWithTemplate(templateId, email, mailData, null);
        }


    }

    /**
     * 判断是否需要发送邮件通知（默认不发送邮件通知)
     *
     * @param tenantSid 租户信息
     * @param userId 用户id
     * @return int 0 :仅im 1：仅mail 2im+邮件
     */
    public int checkSendType(Long tenantSid,String userId) {
        PersonalizedDto userData = userService.queryPersonalizedInfo(tenantSid, userId, UserServiceImpl.MESSAGE_NOTICE_TYPE_KEY, UserServiceImpl.CATALOG_ID);
        if (null == userData || StringUtils.isEmpty(userData.getValue())) {
            return 0;
        }
        switch (userData.getValue()) {
            case "mail":
                return 1;
            case "imAndMail":
                return 2;
            default:
                return 0;
        }
    }
    /**
     * 根据用户信息生成对应的模版id
     * @param userId 用户id
     * @param token token
     * @return string
     */
    private String getUserTemplateId(String userId,String token)
    {

        //查询用户的语言别，发送对应的邮件
        String langName = userService.getUserLangMetadataAllTenant(userId,token);
        if(StringUtils.isBlank(langName))
        {
            langName = "zh_CN";
        }
        return "ATHENA_ATMC_MAIL_NOTICE_" + langName;
    }

    private String getTargetTenantId(List<AppMessageReceiverDTO> receivers) {

        if (CollectionUtils.isEmpty(receivers)){
            return "";
        }
        return receivers.get(0).getTenantId();
    }
    /**
     * 处理邮件发送消息内容
     *
     * @param user    用户信息
     * @param message message
     * @return String
     */
    private Map<String, Object> dealNoticeMailMessage(AuthoredUser user, MessageDO message) {
        Map<String, Object> retMap = new HashMap<>();
        Map<String, Object> map =  JsonUtils.jsonToObject(JsonUtils.objectToString(message.getContent()),Map.class);
        //获取消息的类型
        String content ;
        String title;
        String msg;
        // 4789 任务单
        map.put("targetTenantId", message.getTenantId());
        String linkUrl = generateLinkUrl(user, message.getType(), map);
        if(StringUtils.isNotBlank(linkUrl))
        {
            retMap.put("linkUrl",linkUrl);
        }


        title = MapUtils.getString(map, "title");
        // 需求13551 如果邮件内容不为空，说明单独配置了邮件内容，则取邮件内容
        if(StringUtils.isNotBlank(MapUtils.getString(map, "emailMsg"))){
            msg = MapUtils.getString(map, "emailMsg");
            // 需求13551 文案调整，没有title
            content = msg;
        }else {
            msg = MapUtils.getString(map, "msg");
            content = title + msg;
        }
        retMap.put("title", title);
        retMap.put("content", content);
        if (MapUtils.isNotEmpty(MapUtils.getMap(map, "data"))) {
            retMap.putAll(MapUtils.getMap(map, "data"));
        }
        return retMap;

    }
    public String generateLinkUrl(AuthoredUser user,String type,Map<String, Object> map ) {
        //获取消息的类型
        String urlDetail = "";
        if (ACTIVITY.equals(type) || TASK.equals(type)) {
            String defaultType;
            String id = MapUtils.getString(map, "id");
            String targetTenantId = MapUtils.getString(map,"targetTenantId");
            if (ACTIVITY.equals(type)) {
                defaultType = "task";
                urlDetail = "task/detail/" + id  + "?targetTenantId=" + targetTenantId + "&shareCode=";
            } else  {
                defaultType = "project";
                urlDetail = "project/share/" + id + "/";
            }

            //通过分享任务卡或者项目卡的方式获取分享码
            String shareCode = shareService.createSharingCode(user, defaultType, MapUtils.getLong(map, "id"));
            Task task = taskMapper.selectPartialById(Long.valueOf(id));
            if (task == null){
                //获取分享的url链接，如果task表里面不存在，就是任务引擎
                urlDetail =  envProperties.getWebUri() + urlDetail + shareCode + "?engineType=TaskEngine";
                if(urlDetail.contains("project")){
                    urlDetail = urlDetail + "&targetTenantId=" + targetTenantId;
                }
            }else {
                urlDetail =  envProperties.getWebUri() + urlDetail + shareCode;
                if(urlDetail.contains("project")){
                    urlDetail = urlDetail + "?targetTenantId=" + targetTenantId;
                }
            }

        }
        return urlDetail;
    }

    /**
     * 走邮件发送通知提醒
     *
     * @param user    user信息
     * @param message 消息信息
     */
    public void sendNoticeWithMail(AuthoredUser user, MessageDO message) {
        logger.info("[messeage] : {}",message);
        //根据用户的信息查询对应的联系方式
        String email = userService.queryUserEmail(message.getUserId());
        if (StringUtils.isEmpty(email)) {
            logger.debug("Notice user with no email info.{}", user.getUserId() + user);
            return;
        }

        Map<String, Object> mailData = dealNoticeMailMessage(user, message);

        //根据用户id获取对应的模版信息
        String templateId = getUserTemplateId(message.getUserId(),user.getToken());

        //获取对应的邮件配置模版
        sendEmailService.sendEmailWithTemplate(templateId, email, mailData, null);
    }

}
