package com.digiwin.athena.abt.event.job;

import com.digiwin.athena.abt.application.dto.migration.abt.summary.RefreshCardMessageDTO;
import com.digiwin.athena.abt.application.service.abt.migration.quartz.JobManager;
import com.digiwin.athena.abt.application.service.abt.migration.summary.RefreshCardMessageService;
import com.digiwin.athena.abt.core.meta.constants.JobConstants;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.quartz.*;
import org.springframework.beans.factory.annotation.Autowired;





/**
 * RefreshCardMessageJob Description
 * 刷新任务卡/项目卡的 summary、history_message、search_message信息
 *
 * @author majianfu
 * @date 2023/3/31
 * @since
 */
@Slf4j
public class RefreshCardMessageJob implements Job {
    @Autowired
    private JobManager jobManager;

    @Autowired
    private RefreshCardMessageService refreshMessageService;

    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException {
        RefreshCardMessageDTO refreshCardMessageDTO = parseJobData(context.getJobDetail().getJobDataMap());

        JobKey jobKey = context.getTrigger().getJobKey();
        String jobName = jobKey.getName();
        String jobGroupName = jobKey.getGroup();
        if (null == refreshCardMessageDTO) {
            // 无jobData，移除该定时任务
            log.warn("refresh card message job:{}.{} lack jobData, will delete..", jobName, jobGroupName);
            refreshMessageService.deleteRefreshCardMessageJob(jobName);
            return;
        }

        log.info("fire refresh card message job:{}.{}", jobName, jobGroupName);
        try {
            doRefreshCardMessage(refreshCardMessageDTO);
            // 重置定时任务连续执行失败次数计数器
            resetConFailCount(refreshCardMessageDTO);
        } catch (Exception ex) {
            log.warn("fire fresh card message job:{}.{} error:{}", jobName, jobGroupName, ex);
            handleJobFiredError(jobName, jobGroupName, refreshCardMessageDTO);
        }
    }

    /**
     * 处理job执行失败的情况
     *
     * @param jobName
     * @param jobGroupName
     * @param refreshCardMessageDTO
     */
    private void handleJobFiredError(String jobName, String jobGroupName, RefreshCardMessageDTO refreshCardMessageDTO) {
        /**
         * 记录连续失败的异常次数：
         * 1、超过阈值直接关闭该定时任务;
         * 2、没超过则累加1；并跟新定时任务jobData
         */
        int conFailCount = refreshCardMessageDTO.getConFailCount();
        conFailCount = conFailCount + 1;
        if (conFailCount >= JobConstants.JOB_CONSECUTIVE_FAIL_COUNT_THRESHOLD) {
            log.warn("fire refresh card message job:{}.{} conFailCount({}) ge {}, will delete..", jobName, jobGroupName, conFailCount, JobConstants.JOB_CONSECUTIVE_FAIL_COUNT_THRESHOLD);
            refreshMessageService.deleteRefreshCardMessageJob(jobName);
        } else {
            log.warn("fire refresh card message job:{}.{} conFailCount({}) lt {}, will increase and refresh conFailCount..", jobName, jobGroupName, conFailCount, JobConstants.JOB_CONSECUTIVE_FAIL_COUNT_THRESHOLD);
            refreshCardMessageDTO.setConFailCount(conFailCount);
            // 刷新jobData
            refreshMessageService.updateRefreshCardMessageJobData(refreshCardMessageDTO);
        }
    }

    /**
     * 执行刷新任务卡/项目卡的message信息
     *
     * @param refreshCardMessageDTO
     * @return 该定时任务是否被删除
     */
    private void doRefreshCardMessage(RefreshCardMessageDTO refreshCardMessageDTO) {
        refreshMessageService.refreshCardMessage(refreshCardMessageDTO);
    }

    /**
     * 重置“连续失败次数”
     *
     * @param refreshCardMessageDTO
     */
    private void resetConFailCount(RefreshCardMessageDTO refreshCardMessageDTO) {
        if (refreshCardMessageDTO.getConFailCount() <= 0) {
            return;
        }
        boolean jobExist = jobManager.exist(String.valueOf(refreshCardMessageDTO.getId()), JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
        if (!jobExist) {
            return;
        }

        refreshCardMessageDTO.setConFailCount(0);
        log.info("reset refresh card message conFailCount, job:{}.{}", refreshCardMessageDTO.getId(), JobConstants.REFRESH_MESSAGE_TASK_GROUP_NAME);
        // 刷新jobData
        refreshMessageService.updateRefreshCardMessageJobData(refreshCardMessageDTO);
    }

    /**
     * 解析jobDataMap
     *
     * @param jobDataMap
     * @return
     */
    private RefreshCardMessageDTO parseJobData(JobDataMap jobDataMap) {
        if (null == jobDataMap || jobDataMap.isEmpty()) {
            return null;
        }

        try {
            return JsonUtils.jsonToObject(jobDataMap.getString(JobConstants.JOB_DATA_KEY), RefreshCardMessageDTO.class);
        } catch (Exception ex) {
            return null;
        }
    }
}