package com.digiwin.athena.abt.presentation.mq;

import com.digiwin.athena.abt.application.configuration.DMCConfig;
import com.digiwin.athena.abt.application.configuration.DirectRabbitConfig;
import com.digiwin.athena.abt.application.dto.migration.abt.api.ActionInfoDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.api.UploadParamDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.esp.EspResponse;
import com.digiwin.athena.abt.application.dto.migration.abt.event.ImportSuccessEvent;
import com.digiwin.athena.abt.application.dto.migration.abt.event.ImportSuccessEventFactory;
import com.digiwin.athena.abt.application.dto.migration.abt.excel.ExcelParserBean;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.DMCTokenBean;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ImportBatchRecord;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ImportStatistics;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.ApiDataFieldLocaleMetadataDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.DataEntryTask;
import com.digiwin.athena.abt.application.service.abt.migration.esp.EspService;
import com.digiwin.athena.abt.application.service.abt.migration.event.EventPublisher;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.ExcelHelper;
import com.digiwin.athena.abt.application.service.abt.migration.inout.*;
import com.digiwin.athena.abt.application.service.abt.migration.lock.LockPool;
import com.digiwin.athena.abt.application.utils.ExcelUtil;
import com.digiwin.athena.abt.application.utils.LockPoolDataEntryHelper;
import com.digiwin.athena.abt.core.meta.constants.EntryConstant;
import com.digiwin.athena.abt.core.meta.dto.CellTypeContainer;
import com.digiwin.athena.abt.core.meta.enums.ExcelTypeEnum;
import com.digiwin.athena.abt.core.meta.enums.MetaDataType;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.auth.service.TokenVerifyService;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.knowledgegraph.sdk.manager.KnowledgegraphManager;
import com.digiwin.athena.knowledgegraph.sdk.meta.dto.response.thememap.TmActivityResponseDTO;
import com.digiwin.athena.knowledgegraph.sdk.meta.dto.response.thememap.TmOperationDTO;
import com.digiwin.service.permission.DWSecurityTokenGenerator;
import com.digiwin.service.permission.pojo.DWSecurityContext;
import com.digiwin.service.permission.pojo.DWSecurityToken;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.slf4j.MDC;
import org.springframework.amqp.rabbit.annotation.RabbitHandler;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;

/**
 * @ClassName DataEntryListener
 * @Description TODO
 * @Author zhuangli
 * @Date 2021/4/22 17:39
 * @Version 1.0
 **/
@Slf4j
@Component
public class DataEntryListener {

    @Autowired
    RabbitTemplate rabbitTemplate;
    @Autowired
    ImportStatisticsDomainService importStatisticsDomainService;
    @Autowired
    ImportStatisticsFactory importStatisticsFactory;
    @Autowired
    MetaDataService metaDataService;
    @Autowired
    EspService espService;
    @Autowired
    @Qualifier("rabbitErrorHandlerServiceImpl")
    ErrorHandlerService errorHandlerService;
    @Autowired
    EventPublisher eventPublisher;
    @Autowired
    DirectRabbitConfig directRabbitConfig;
    @Autowired
    LockPool lockPool;

    @Autowired
    private TokenVerifyService tokenVerifyService;


    @Resource
    RestTemplate dmcRestTemplate;

    @Autowired
    DMCConfig dmcConfig;

    @Autowired
    DMCTokenBean dmcTokenBean;


    @Value("${dmc.uri}")
    private String dmcUrl;
    @Value("${athena.auth.appToken}")
    private String appToken;


    @Autowired
    ExcelHelper excelHelper;

    @RabbitListener(
            queues = "#{dataEntryQueueName}",ackMode = "AUTO"
    )
    @RabbitHandler
    public void consumer(String msg) {
        DataEntryTask dataEntryTask = JsonUtils.jsonToObject(msg, DataEntryTask.class);

        ImportStatistics importStatistics = importStatisticsDomainService.getByMasterId(dataEntryTask.getMasterId());
        ImportBatchRecord importBatchRecord = importStatisticsDomainService.getBatchRecordByPIdAndSeq(dataEntryTask.getMasterId(), dataEntryTask.getBatchNum());
        if (null == importStatistics || null == importBatchRecord) {
            //将消息丢弃
            log.error("未找到对应数据masterId:{},batchNum:{},importStatistics:{},importBatchRecord:{}", dataEntryTask.getMasterId(), dataEntryTask.getBatchNum(),importStatistics,importBatchRecord);
            // 抛异常，在catch中统一处理
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0060.getErrCode(),MessageUtil.getMessage("delivery.dataNotFound"));
        }
        // 待导入数据个数
        final Integer processingNum = importStatistics.getProcessingNum();
        if (!StringUtils.isEmpty(importStatistics.getRouterKey())) {
            MDC.put(GlobalConstant.ROUTER_KEY, importStatistics.getRouterKey());
        }

        //防止重复消费
        if (importBatchRecord.getHandleFlag() == 1) {
            log.error("数据已经处理masterId:{},batchNum:{}", dataEntryTask.getMasterId(), dataEntryTask.getBatchNum());
            // 重复处理就不发MQTT消息，直接return
            return;
        }

        int lockId = LockPoolDataEntryHelper.getIdByUUID(dataEntryTask.getMasterId(), lockPool.getPoolSize());
        ReentrantLock lock = lockPool.getLockById(lockId);
        lock.lock();
        try {
            DWSecurityToken dwSecurityToken;
            try {
                dwSecurityToken = DWSecurityTokenGenerator.parseSecurityToken(dataEntryTask.getUserToken());
            } catch (Exception e) {
                log.error("解析token失败:{}", dataEntryTask.getUserToken());
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0106.getErrCode(),MessageUtil.getMessage("delivery.resolveTokenError"), e);
            }
            DWSecurityContext dwSecurityContext = dwSecurityToken.getContext();
            // 查询用户信息，后续调外部接口时需要传routerKey
            AuthoredUser authoredUser = tokenVerifyService.getUserInfo(dwSecurityContext.getUserToken());
            AppAuthContextHolder.getContext().setAuthoredUser(authoredUser);
            GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaDataBySecurityToken(importStatistics.getActionId(), dwSecurityContext.getUserToken(), dataEntryTask.getLocale(), dataEntryTask.getUserToken());
            ApiDataFieldLocaleMetadataDTO mainMetadata = metaDataService.getMainMetadata(metadataDTO, dataEntryTask.getActionInfo());
            List<CellTypeContainer> cellTypes = metaDataService.getCellTypeContainers(mainMetadata.getField(), authoredUser.getToken(), null, dataEntryTask.getLocale(), 1);
            //首次消费创建导入异常数据临时队列
            boolean createErrorQueueFlag = importStatistics.getFailedNum() == 0;
            // 总异常信息
            List<Map> errorList = new ArrayList<>();
            // 校验每行数据
            List<Map> validateErrorList = metaDataService.validateByBatch(dataEntryTask, cellTypes);
            EspResponse response;
            // 如果校验成功则调用导入API
            if(CollectionUtils.isEmpty(validateErrorList)){
                // 1、调用应用导入API
                response = sendByBatch(dataEntryTask);
                //TODO 异常处理
                if (response.isOK()) {
                    //更新对应masterId成功条数
                    importStatisticsFactory.calBySucceededNum(dataEntryTask.getDataList().size(), importStatistics, importBatchRecord);
                    importStatisticsDomainService.updateThisAndBatchRecordsById(importStatistics);
                } else {
                    List<Map<String, Object>> importErrorResult;
                    if (null == response.getData()) {
                        log.error("调用esp服务失败");
                        importErrorResult = batchError(dataEntryTask.getKeyList(), dataEntryTask.getDataList());
                    } else {
                        importErrorResult = importStatisticsFactory.parseResponse(dataEntryTask.getTableKey(), response);
                    }

                    if(Objects.isNull(importErrorResult)){
                        log.error("调用esp服务，解析导入异常数据失败!");
                        // 抛异常，在catch中统一处理
                        throw BusinessException.create(ErrorCodeEnum.NUM_500_0107.getErrCode(),MessageUtil.getMessage("delivery.handleDataError"));
                    }
                    //将失败行存入数据库并更新masterId失败条数
                    importStatisticsFactory.calByTotalAndFailNum(dataEntryTask.getDataList().size(), importErrorResult.size(), importStatistics, importBatchRecord);
                    //更新记录
                    importStatisticsDomainService.updateThisAndBatchRecordsById(importStatistics);
                    //合并错误信息
                    errorList.addAll(importErrorResult);
                }
            } else {
                //将失败行存入数据库并更新masterId失败条数
                importStatisticsFactory.calByTotalAndFailNum(dataEntryTask.getDataList().size(), validateErrorList.size(), importStatistics, importBatchRecord);
                //更新记录
                importStatisticsDomainService.updateThisAndBatchRecordsById(importStatistics);
                //合并错误信息
                errorList.addAll(validateErrorList);
            }

            // 2、处理异常数据，发送异常数据到MQ
            if (!CollectionUtils.isEmpty(errorList)) {
                try {
                    errorHandlerService.handleErrorList(importStatistics, errorList, createErrorQueueFlag);
                } catch (Exception e) {
                    log.error("处理失败数据失败", e);
                    // 抛异常，在catch中统一处理
                    throw BusinessException.create(ErrorCodeEnum.NUM_500_0107.getErrCode(),MessageUtil.getMessage("delivery.handleDataError"));
                }
            }

            // 3、所有数据处理完毕且有导入异常的数据，则从MQ拉取异常数据生成异常文件
            if (importStatistics.getProcessingNum() == 0 && importStatistics.getFailedNum() >0) {
                Map<String, String> metaData = metaDataService.getHeaderMap(mainMetadata);
                Map<String, CellTypeContainer> cellTypeContainerMap = metaDataService.getResponseCellTypeContainersBySecurityToken(
                        metadataDTO, dwSecurityContext.getUserToken(), importStatistics.getUserToken(), dataEntryTask.getLocale());
                // 全部批次数据处理完, 将所有错误数据存入文档中心
                String mainKey = metadataDTO.getResponse().getData().getData_name();
                String mainKeyDescription = metadataDTO.getResponse().getData().getDescription();
                // 生成异常文件
                String failedUrl = errorHandlerService.onImportFinish(ExcelHelper.getSheetName(mainKey,mainKeyDescription,true),dataEntryTask.getKeyList(), cellTypeContainerMap, metaData, importStatistics);
                if (!StringUtils.isEmpty(failedUrl)) {
                    importStatistics.setFailedUrl(failedUrl);
                    importStatisticsDomainService.updateById(importStatistics);
                }else {
                    log.error("failedUrl is null!");
                }
            }
        } catch (Exception e){
            // 未处理前出现异常报错才算导入失败；处理后出现异常报错不算导入失败，防止调用ESP正常，在生成异常文件或上传异常文件出现报错，又将其变更成导入失败。
            if(Objects.equals(importStatistics.getProcessingNum(), processingNum)){
                importStatisticsFactory.calErrorNum(dataEntryTask.getDataList().size(), importStatistics, importBatchRecord);
                importStatisticsDomainService.updateThisAndBatchRecordsById(importStatistics);
            }
            log.error("DataEntryListener consume error:{}", e);
        } finally {
            lock.unlock();
            if (!StringUtils.isEmpty(importStatistics.getRouterKey())) {
                MDC.remove(GlobalConstant.ROUTER_KEY);
            }
        }

        // 4、所有数据处理完毕，发送MQTT消息
        if(importStatistics.getProcessingNum() == 0){
            ImportSuccessEvent importSuccessEvent = ImportSuccessEventFactory.produceByImportStatistics(importStatistics);
            eventPublisher.publish(importSuccessEvent);
        }
    }

    @Autowired
    BaseDataEntryApplicationServiceImpl baseDataEntryApplicationService;

    @RabbitListener(
            queues = "#{dataEntryQueueNameAsync}",ackMode = "AUTO"
    )
    @RabbitHandler
    public void consumerAsync(String msg) {
        UploadParamDTO UploadParam = JsonUtils.jsonToObject(msg, UploadParamDTO.class);
        //构建入参
        ExcelParserBean excelParserBean = new ExcelParserBean();
        InputStream fileInputStream = getFileInputStream(UploadParam.getFileUrl());
        excelParserBean.setInput(fileInputStream);
        excelParserBean.setFilePath(UploadParam.getFileName());
        baseDataEntryApplicationService.upload(UploadParam, excelParserBean);
    }


    /**
     * 根据文件地址，从dmc获取文件流
     * @param fileUrl dmc中的文件地址
     * @return 文件流
     */
    private InputStream getFileInputStream(String fileUrl) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", dmcTokenBean.getToken());
        headers.add("digi-middleware-auth-app", appToken);
        List list = new ArrayList<>();
        list.add(MediaType.parseMediaType(ExcelTypeEnum.XLSX.value()));
        headers.setAccept(list);
        String url = dmcUrl + "/api/dmc/v2/file/" + dmcConfig.getBucket() + "/download/" + fileUrl;
        ResponseEntity<byte[]> restRes = dmcRestTemplate.exchange(
                url,
                HttpMethod.GET,
                new HttpEntity<byte[]>(headers),
                byte[].class);
        return new ByteArrayInputStream(restRes.getBody());
    }

    /**
     * @Author zhuangli
     * @Description 处理死信队列消息:1.将死信中的消息全部标记为失败 2.
     * @Date 10:23 2021/9/23
     * @Param
     * @return
     **/
    @RabbitListener(
            queues = "#{dataEntryDeadLetterQueueName}",ackMode = "AUTO"
    )
    @RabbitHandler
    @Transactional(rollbackFor = Exception.class)
    public void handleDeadLetter(String msg) {
        DataEntryTask dataEntryTask = JsonUtils.jsonToObject(msg, DataEntryTask.class);
        ImportStatistics importStatistics = importStatisticsDomainService.getByMasterId(dataEntryTask.getMasterId());
        ImportBatchRecord importBatchRecord = importStatisticsDomainService.getBatchRecordByPIdAndSeq(dataEntryTask.getMasterId(), dataEntryTask.getBatchNum());
        if (null == importStatistics || null == importBatchRecord) {
            //将消息丢弃
            log.error("未找到对应数据masterId:{},batchNum:{}", dataEntryTask.getMasterId(), dataEntryTask.getBatchNum());
            return;
        }
        if (!StringUtils.isEmpty(importStatistics.getRouterKey())) {
            MDC.put(GlobalConstant.ROUTER_KEY, importStatistics.getRouterKey());
        }
        try {
            //首次消费创建临时队列
            boolean createQueueFlag = importStatistics.getFailedNum() == 0 ? true : false;
            if (importBatchRecord.getHandleFlag() == 1) {
                log.error("数据已经处理masterId:{},batchNum:{}", dataEntryTask.getMasterId(), dataEntryTask.getBatchNum());
                return;
            }
            //将数据全部标记失败
            List<Map<String, Object>> importErrorResult = batchError(dataEntryTask.getKeyList(), dataEntryTask.getDataList());
            //将失败行存入数据库并更新masterId失败条数
            importStatisticsFactory.calByTotalAndFailNum(dataEntryTask.getDataList().size(), importErrorResult.size(),
                    importStatistics, importBatchRecord);
            //更新记录
            importStatisticsDomainService.updateThisAndBatchRecordsById(importStatistics);
            //合并错误信息
            List<Map> errorList = new LinkedList<>();
            errorList.addAll(importErrorResult);

            DWSecurityToken dwSecurityToken;
            try {
                dwSecurityToken = DWSecurityTokenGenerator.parseSecurityToken(dataEntryTask.getUserToken());
            } catch (Exception e) {
                log.error("解析token失败:{}", dataEntryTask.getUserToken());
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0108.getErrCode(),MessageUtil.getMessage("delivery.resolveTokenError"), e);
            }
            DWSecurityContext dwSecurityContext = dwSecurityToken.getContext();
            //处理失败数据
            GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaDataBySecurityToken(importStatistics.getActionId(), dwSecurityContext.getUserToken(), dataEntryTask.getLocale(), dataEntryTask.getUserToken());
            ApiDataFieldLocaleMetadataDTO mainMetadata = metaDataService.getMainMetadata(metadataDTO, dataEntryTask.getActionInfo());
            Map<String, String> metaData = metaDataService.getHeaderMap(mainMetadata);
            if (!CollectionUtils.isEmpty(errorList)) {
                try {
                    errorHandlerService.handleErrorList(importStatistics, errorList, createQueueFlag);
                } catch (IOException e) {
                    log.error("error handleErrorList", e);
                }
            }
            //所有数据处理完毕
            if (importStatistics.getProcessingNum() == 0) {
                //获取responseKey类型
                Map<String, CellTypeContainer> cellTypeContainerMap = metaDataService.getResponseCellTypeContainersBySecurityToken(
                        metadataDTO, dwSecurityContext.getUserToken(), importStatistics.getUserToken(), dataEntryTask.getLocale());
                //触发领域事件[维护信息完毕]
                String mainKey = metadataDTO.getResponse().getData().getData_name();
                String mainKeyDescription = metadataDTO.getResponse().getData().getDescription();
                String failedUrl = errorHandlerService.onImportFinish(ExcelHelper.getSheetName(mainKey,mainKeyDescription,true),dataEntryTask.getKeyList(), cellTypeContainerMap, metaData, importStatistics);
                if (!StringUtils.isEmpty(failedUrl)) {
                    importStatistics.setFailedUrl(failedUrl);
                    importStatisticsDomainService.updateById(importStatistics);
                }
                ImportSuccessEvent importSuccessEvent = ImportSuccessEventFactory.produceByImportStatistics(importStatistics);
                eventPublisher.publish(importSuccessEvent);
            }
        }finally {
            if (!StringUtils.isEmpty(importStatistics.getRouterKey())) {
                MDC.remove(GlobalConstant.ROUTER_KEY);
            }
        }
    }

    private List<Map<String, Object>> batchError(List<String> keyList, List<Map> dataList) {
        List<Map<String, Object>> result = new LinkedList<>();
        dataList.forEach(item -> {
            Map row = new HashMap(item.size() * 2);
            row.putAll(item);
            row.put("batch_error_msg", MessageUtil.getMessage("delivery.batchHandleError"));
            result.add(row);
        });
        return result;
    }

    public EspResponse sendByBatch(DataEntryTask dataEntryTask) {
        final String actionId =dataEntryTask.getActionId();
        DWSecurityToken dwSecurityToken;
        try {
            dwSecurityToken = DWSecurityTokenGenerator.parseSecurityToken(dataEntryTask.getUserToken());
        } catch (Exception e) {
            log.error("解析token失败:{}", dataEntryTask.getUserToken());
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0109.getErrCode(),MessageUtil.getMessage("delivery.resolveTokenError"), e);
        }
        DWSecurityContext dwSecurityContext = dwSecurityToken.getContext();
        GetActionLocaleResponseDTO metaData = metaDataService.getActionMetaDataBySecurityToken(actionId, dwSecurityContext.getUserToken(), dataEntryTask.getUserToken());
        String productName = metaDataService.getProductNameBySecurityToken(dwSecurityContext.getUserToken(), actionId.substring(actionId.contains("esp_") ? 4 : 0), dataEntryTask.getUserToken());
        //组装请求表格
        return espService.sendByBatch(dataEntryTask, metaData, dataEntryTask.getDataList(), productName);
    }

    private List<Map> formTable(List<String> tableFields, List<List<Object>> table) {
        List<Map> result = new LinkedList<>();
        table.forEach(item->{
            Map<String, Object> entity = new HashMap<>();
            AtomicInteger idx = new AtomicInteger();
            item.forEach(inner -> {
                entity.put(tableFields.get(idx.get()), inner);
                idx.getAndIncrement();
            });
            result.add(entity);
        });
        return result;
    }

}