package com.digiwin.athena.abt.presentation.mq;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.athena.abt.application.configuration.DMCConfig;
import com.digiwin.athena.abt.application.dto.migration.abt.event.ExportSuccessEvent;
import com.digiwin.athena.abt.application.dto.migration.abt.event.ExportSuccessEventFactory;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ExportBatchRecord;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ExportStatistics;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.FileInfo;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.ExportFileMsg;
import com.digiwin.athena.abt.application.service.abt.migration.event.EventPublisher;
import com.digiwin.athena.abt.application.service.abt.migration.event.ttl.ExportMQListenerInterface;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.ExcelHelper;
import com.digiwin.athena.abt.application.service.abt.migration.inout.ErrorHandlerService;
import com.digiwin.athena.abt.application.service.abt.migration.inout.ExportStatisticsDomainService;
import com.digiwin.athena.abt.application.service.abt.migration.inout.MetaDataService;
import com.digiwin.athena.abt.application.service.abt.migration.restfull.uibot.UibotService;
import com.digiwin.athena.abt.application.service.atmc.migration.restfull.thememap.ThemeMapService;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import com.digiwin.athena.abt.core.meta.constants.ImportAndExportStatisticsConstants;
import com.digiwin.athena.abt.core.meta.dto.CellTypeContainer;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.abt.core.meta.enums.ExcelTypeEnum;
import com.digiwin.athena.abt.infrastructure.mapper.biz.migration.abt.ExportBatchRecordMapper;
import com.digiwin.athena.abt.infrastructure.pojo.po.migration.abt.ExportBatchRecordPO;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.auth.service.TokenVerifyService;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.TimeUtils;
import com.digiwin.service.permission.DWSecurityTokenGenerator;
import com.digiwin.service.permission.pojo.DWSecurityContext;
import com.digiwin.service.permission.pojo.DWSecurityToken;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.amqp.rabbit.annotation.RabbitHandler;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;

/**
 * 导出历史项目/任务明细
 *
 * @author xuhx
 * @date 2025/5/15
 */
@Slf4j
@Component("exportHistoryProjectTaskDetailMQListener")
public class ExportHistoryProjectTaskDetailMQListener implements ExportMQListenerInterface {

    @Autowired
    ExportStatisticsDomainService exportStatisticsDomainService;
    @Autowired
    MetaDataService metaDataService;
    @Autowired
    EventPublisher eventPublisher;

    @Autowired
    private ExcelHelper excelHelper;

    @Autowired
    private TokenVerifyService tokenVerifyService;

    @Autowired
    private UibotService uibotService;

    @Autowired
    private ExportBatchRecordMapper exportBatchRecordMapper;

    @Autowired
    private DMCConfig dmcConfig;

    @Autowired
    private ThemeMapService themeMapService;

    @Autowired
    @Qualifier("rabbitErrorHandlerServiceImpl")
    private ErrorHandlerService errorHandlerService;

    @RabbitListener(
            queues = "#{exportHistoryProjectTaskDetailQueueName}", ackMode = "AUTO"
    )
    @RabbitHandler
    public void consumer(String msg) {
        ExportFileMsg exportFileMsg = JsonUtils.jsonToObject(msg, ExportFileMsg.class);
        String masterId = exportFileMsg.getMasterId();
        ExportStatistics exportStatistics = exportStatisticsDomainService.getByMasterId(masterId);
        if (exportStatistics == null) {
            log.error("ExportHistoryProjectTaskDetailMQListener未找到对应数据masterId:{}", masterId);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0060.getErrCode(), MessageUtil.getMessage("delivery.dataNotFound"));
        }

        ExportBatchRecord exportBatchRecord = exportFileMsg.getExportBatchRecord();
        Long exportBatchRecordId = exportBatchRecord.getId();
        Integer batchSeq = exportBatchRecord.getBatchSeq();
        Date date = new Date();

        ExportBatchRecordPO exportBatchRecordPO = new ExportBatchRecordPO();
        exportBatchRecordPO.setId(exportBatchRecordId);
        exportBatchRecordPO.setUpdateTime(date);

        try {
            ExportBatchRecordPO existingRecord = exportBatchRecordMapper.queryByMasterIdAndBatchSeq(masterId, batchSeq);
            if (existingRecord != null && existingRecord.getState() == 1) {
                return;
            }

            // Token 解析与上下文设置
            String userToken = exportFileMsg.getUserToken();
            DWSecurityToken token = DWSecurityTokenGenerator.parseSecurityToken(userToken);
            DWSecurityContext context = token.getContext();
            AuthoredUser user = tokenVerifyService.getUserInfo(context.getUserToken());
            AppAuthContextHolder.getContext().setAuthoredUser(user);
            AppAuthContextHolder.getContext().setProxyToken(exportBatchRecord.getProxyToken());

            // 获取数据
            Map<String, Object> paramMap = new HashMap<>();
            paramMap.put("backlogId", exportBatchRecord.getRelate());
            Map<String, Object> dataResult = uibotService.dataByBacklogIdAndState(paramMap);

            List<Map<String, Object>> tableList = Optional.ofNullable(dataResult)
                    .map(data -> (List<Map<String, Object>>) data.get("data"))
                    .map(data -> JSON.parseObject(JSON.toJSONString(data), new TypeReference<List<Map<String, Object>>>() {
                    }))
                    .orElse(Collections.emptyList());

            String actionId = Optional.ofNullable(dataResult.get("actionId")).map(Object::toString).orElse(null);
            if (StringUtils.isBlank(actionId)) {
                throw new BusinessException("actionId为空,exportBatchRecordId:" + exportBatchRecordId);
            }
            // todo 后续可能要支持没有 actionId 的情况，直接查到什么数据，就直接导出，不依赖元数据这种情况
            GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaDataBySecurityToken(actionId, context.getUserToken(), exportFileMsg.getLocale(), userToken);

            String mainKey = metadataDTO.getResponse().getData().getData_name();
            String mainKeyDesc = metadataDTO.getResponse().getData().getDescription();

            Map<String, CellTypeContainer> cellTypes = metaDataService.getResponseCellTypeContainersBySecurityToken(metadataDTO, context.getUserToken(), userToken, exportFileMsg.getLocale());
            List<String> keyList = new ArrayList<>();
            Map<String, String> headers = metaDataService.getResponseHeaders(metadataDTO, keyList);
            // 限制 2000 条数据
            if (tableList.size() > ImportAndExportStatisticsConstants.EXPORT_LIMIT) {
                tableList = tableList.subList(0, ImportAndExportStatisticsConstants.EXPORT_LIMIT);
            }

            String fileId = handleDownloadHistoryProjectTask(exportBatchRecord.getFileName(), exportFileMsg.getLocale(), exportStatistics.getExpireTime(), ExcelHelper.getSheetName(mainKey, mainKeyDesc, true), keyList, headers, cellTypes, tableList);
            if (StringUtils.isNotEmpty(fileId)) {
                exportBatchRecordPO.setFileUrl(fileId);
                exportBatchRecordPO.setTotalSize(tableList.size());
                exportBatchRecordPO.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_SUCCESS);
            } else {
                log.error("ExportHistoryProjectTaskDetailMQListener文件上传失败masterId:{}, exportBatchRecordId:{}", masterId, exportBatchRecordId);
                exportBatchRecordPO.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL);
                exportBatchRecordPO.setRemark("文件上传失败!");
            }

        } catch (Exception e) {
            log.error("ExportHistoryProjectTaskDetailMQListener导出处理异常exportBatchRecordId:" + exportBatchRecordId + ", error:", e);
            exportBatchRecordPO.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL);
            exportBatchRecordPO.setRemark(e.getMessage());
        } finally {
            // 补充 name，多语言
            String type = Optional.ofNullable(exportBatchRecord.getType()).orElse("default");
            String name;
            switch (type) {
                case "2":
                    name = themeMapService.getTaskNamesByCodes(exportBatchRecord.getCode());
                    break;
                case "3":
                    name = themeMapService.getProjectNamesByCodes(exportBatchRecord.getCode());
                    break;
                default:
                    name = metaDataService.getActivityNameByActivityId(exportBatchRecord.getCode(), null);
            }
            exportBatchRecordPO.setName(name);
            exportBatchRecordMapper.updateById(exportBatchRecordPO);
        }

        // 更新 export_statistics 状态
        updateExportStatistics(exportStatistics, exportBatchRecordId);
    }

    private void updateExportStatistics(ExportStatistics exportStatistics, Long exportBatchRecordId) {
        try {
            List<ExportBatchRecordPO> batchList = exportBatchRecordMapper.listByMasterId(exportStatistics.getMasterId(), null);
            long failedCount = batchList.stream().filter(r -> ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL.equals(r.getState())).count();
            boolean hasProcessing = batchList.stream().anyMatch(r -> ImportAndExportStatisticsConstants.EXPROT_STATE_PROCESSING.equals(r.getState()));

            if (!hasProcessing) {
                int newState = ImportAndExportStatisticsConstants.EXPROT_STATE_SUCCESS;
                if (failedCount > 0) {
                    newState = (failedCount == exportStatistics.getTotalSize())
                            ? ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL
                            : ImportAndExportStatisticsConstants.EXPROT_STATE_ABNORMAL;
                }

                exportStatistics.setState(newState);
                exportStatistics.setFailedSize((int) failedCount);
                exportStatistics.setUpdateTime(new Date());
                exportStatisticsDomainService.updateById(exportStatistics);
                // 发送 MQTT
                ExportSuccessEvent event = ExportSuccessEventFactory.produceByExportStatistics(exportStatistics);
                eventPublisher.publish(event);
            }
        } catch (Exception e) {
            log.error("ExportHistoryProjectTaskDetailMQListener,updateExportStatistics更新exportStatistics异常exportBatchRecordId:" + exportBatchRecordId + ", error:", e);
        }
    }

    private String handleDownloadHistoryProjectTask(String fileName,
                                                    String locale,
                                                    Date expireTime,
                                                    String sheetName,
                                                    List<String> keyList,
                                                    Map<String, String> headers,
                                                    Map<String, CellTypeContainer> cellTypeContainerMap,
                                                    List<Map<String, Object>> tableList) {
        MultipartFile multipartFile;
        ByteArrayOutputStream outputStream = null;
        InputStream inputStream = null;
        try {
            XSSFWorkbook xssfwb = new XSSFWorkbook();
            ByteArrayOutputStream bos = null;
            try {
                bos = new ByteArrayOutputStream();
                xssfwb.write(bos);
            } catch (IOException e) {
                log.error("handleDownloadHistoryProjectTask转换multipartfile失败:", e);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0076.getErrCode(), "handleDownloadHistoryProjectTask转换multipartFile失败:", e);
            } finally {
                if (bos != null) {
                    try {
                        bos.close();
                    } catch (IOException e) {
                        log.error("handleDownloadHistoryProjectTask Exception occurred:", e);
                    }
                }
            }

            // 使用SXSSFWorkbook只会保留100条数据在内存中，其它的数据都会写到磁盘里，减少内存占用
            SXSSFWorkbook sxssfwb = new SXSSFWorkbook(xssfwb, ImportAndExportStatisticsConstants.SXSSFWorkbook_MEMORY_SIZE);
            int rowIndex = 2;
            List<CellTypeContainer> businessKeyContainer = excelHelper.getBusinessKeyContainer(new ArrayList<>(cellTypeContainerMap.values()));
            excelHelper.createDataExcel(sheetName, locale, rowIndex, keyList, cellTypeContainerMap, headers, tableList, sxssfwb, businessKeyContainer, 0, null, null);
            outputStream = new ByteArrayOutputStream();
            sxssfwb.write(outputStream);
            byte[] bytes = outputStream.toByteArray();
            inputStream = new ByteArrayInputStream(bytes);
            multipartFile = new MockMultipartFile("file", fileName, ExcelTypeEnum.XLSX.value(), inputStream);
            outputStream.flush();
            // 释放workbook所占用的所有windows资源
            sxssfwb.dispose();
        } catch (Exception e) {
            log.error("handleDownloadHistoryProjectTask fail:", e);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0076.getErrCode(), "handleDownloadHistoryProjectTask fail:", e);
        } finally {
            if (outputStream != null) {
                try {
                    outputStream.close();
                } catch (IOException e) {
                    log.error("handleDownloadHistoryProjectTask close outputStream fail:", e);
                }
            }
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    log.error("handleDownloadHistoryProjectTask close inputStream fail:", e);
                }
            }

        }
        FileInfo fileInfo = new FileInfo();
        fileInfo.setDisplayName(fileName);
        fileInfo.setFileName(fileName + ".xlsx");
        fileInfo.setDirectoryId(dmcConfig.getErrorTableUUID());
        fileInfo.setExpireDate(TimeUtils.format(expireTime, TimeUtils.DEFAULT_FORMAT));
        //上传文件至文件服务器
        return errorHandlerService.upload(multipartFile, fileInfo);
    }

    @RabbitListener(
            queues = "#{exportHistoryProjectTaskDetailDeadLetterQueueName}", ackMode = "AUTO"
    )
    @RabbitHandler
    public void deadLetterConsumer(String msg) {
        ExportFileMsg exportFileMsg = JsonUtils.jsonToObject(msg, ExportFileMsg.class);
        String masterId = exportFileMsg.getMasterId();
        try {
            ExportStatistics exportStatistics = exportStatisticsDomainService.getByMasterId(masterId);
            ExportBatchRecordPO exportBatchRecord = exportBatchRecordMapper.queryByMasterIdAndBatchSeq(masterId, exportFileMsg.getExportBatchRecord().getBatchSeq());
            if (exportStatistics == null || exportBatchRecord == null) {
                log.error("ExportHistoryProjectTaskDetail,deadLetterConsumer未找到对应数据masterId:{}", masterId);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0060.getErrCode(), MessageUtil.getMessage("delivery.dataNotFound"));
            }
            // Token 解析与上下文设置
            String userToken = exportFileMsg.getUserToken();
            DWSecurityToken token = DWSecurityTokenGenerator.parseSecurityToken(userToken);
            DWSecurityContext context = token.getContext();
            AuthoredUser user = tokenVerifyService.getUserInfo(context.getUserToken());
            AppAuthContextHolder.getContext().setAuthoredUser(user);
            AppAuthContextHolder.getContext().setProxyToken(exportBatchRecord.getProxyToken());
            // 补充 name，多语言
            String type = Optional.ofNullable(exportBatchRecord.getType()).orElse("default");
            String name;
            switch (type) {
                case "2":
                    name = themeMapService.getTaskNamesByCodes(exportBatchRecord.getCode());
                    break;
                case "3":
                    name = themeMapService.getProjectNamesByCodes(exportBatchRecord.getCode());
                    break;
                default:
                    name = metaDataService.getActivityNameByActivityId(exportBatchRecord.getCode(), null);
            }
            // 更新子表 export_batch_record 为失败状态
            ExportBatchRecordPO exportBatchRecordPO = new ExportBatchRecordPO();
            exportBatchRecordPO.setId(exportBatchRecord.getId());
            exportBatchRecordPO.setName(name);
            exportBatchRecordPO.setUpdateTime(new Date());
            exportBatchRecordPO.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL);
            exportBatchRecordPO.setRemark("消息消费超时，进入死信队列，直接失败");
            int i = exportBatchRecordMapper.updateById(exportBatchRecordPO);
            // 更新主表 export_statistics，根据子表的情况，统计数据，更新主表状态、失败数量，最后推送MQTT消息
            if (i == 1) {
                updateExportStatistics(exportStatistics, exportBatchRecord.getId());
            }
        } catch (Exception e) {
            log.error("ExportHistoryProjectTaskDetail,deadLetterConsumer error,masterId:" + masterId, e);
        }
    }
}

