package com.digiwin.athena.abt.presentation.mq;

import com.digiwin.athena.abt.application.configuration.DirectRabbitConfig;
import com.digiwin.athena.abt.application.dto.migration.abt.api.DownloadBaseDataParamDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.esp.EspBody;
import com.digiwin.athena.abt.application.dto.migration.abt.esp.EspResponse;
import com.digiwin.athena.abt.application.dto.migration.abt.event.ExportSuccessEvent;
import com.digiwin.athena.abt.application.dto.migration.abt.event.ExportSuccessEventFactory;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ExportStatistics;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.ExportFileMsg;
import com.digiwin.athena.abt.application.service.abt.migration.esp.EspService;
import com.digiwin.athena.abt.application.service.abt.migration.event.EventPublisher;
import com.digiwin.athena.abt.application.service.abt.migration.event.ttl.ExportMQListenerInterface;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.ExcelHelper;
import com.digiwin.athena.abt.application.service.abt.migration.inout.BaseDataEntryApplicationServiceImpl;
import com.digiwin.athena.abt.application.service.abt.migration.inout.ExportStatisticsDomainService;
import com.digiwin.athena.abt.application.service.abt.migration.inout.ImportStatisticsFactory;
import com.digiwin.athena.abt.application.service.abt.migration.inout.MetaDataService;
import com.digiwin.athena.abt.application.service.abt.migration.lock.LockPool;
import com.digiwin.athena.abt.application.utils.LockPoolDataEntryHelper;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import com.digiwin.athena.abt.core.meta.constants.ImportAndExportStatisticsConstants;
import com.digiwin.athena.abt.core.meta.dto.CellTypeContainer;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.auth.service.TokenVerifyService;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.service.permission.DWSecurityTokenGenerator;
import com.digiwin.service.permission.pojo.DWSecurityContext;
import com.digiwin.service.permission.pojo.DWSecurityToken;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.amqp.rabbit.annotation.RabbitHandler;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.locks.ReentrantLock;

/**
 * @ClassName ExportFileListener
 * @Description 异步导出的Listener
 * 导出基础数据为excel, 支持数据量为60000条.
 * 1.根据actionid调用espapi获取需要导出的数据
 * 2.元数据api获得column名称并和返回数据匹配组装excel数据结构
 * 3.将数据写入excel并上传到dmc文档中心
 * @Author lisheng
 * @Date 2023/9/20 17:39
 * @Version 1.0
 **/
@Slf4j
@Component("exportMQListener")
public class ExportMQListener implements ExportMQListenerInterface {

    @Autowired
    RabbitTemplate rabbitTemplate;
    @Autowired
    ExportStatisticsDomainService exportStatisticsDomainService;
    @Autowired
    ImportStatisticsFactory importStatisticsFactory;
    @Autowired
    MetaDataService metaDataService;
    @Autowired
    EspService espService;
    @Autowired
    EventPublisher eventPublisher;
    @Autowired
    DirectRabbitConfig directRabbitConfig;
    @Autowired
    LockPool lockPool;

    @Autowired
    BaseDataEntryApplicationServiceImpl baseDataEntryService;

    @Autowired
    private TokenVerifyService tokenVerifyService;

    @RabbitListener(
            queues = "#{exportFileQueueName}", ackMode = "AUTO"
    )
    @RabbitHandler
    public void consumer(String msg) {
        ExportFileMsg exportFileMsg = JsonUtils.jsonToObject(msg, ExportFileMsg.class);
        ExportStatistics exportStatistics = exportStatisticsDomainService.getByMasterId(exportFileMsg.getMasterId());
        if (Objects.isNull(exportStatistics)) {
            //将消息丢弃
            log.error("未找到对应数据masterId:{},exportStatistics:{}", exportFileMsg.getMasterId(), exportStatistics);
            // 抛异常，在catch中统一处理
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0060.getErrCode(), MessageUtil.getMessage("delivery.dataNotFound"));
        }

//        Date now = new Date();
        int lockId = LockPoolDataEntryHelper.getIdByUUID(exportFileMsg.getMasterId(), lockPool.getPoolSize());
        ReentrantLock lock = lockPool.getLockById(lockId);
        lock.lock();
        try {
            // 1、查询API元数据
            String actionId = exportFileMsg.getActionId();
            String userToken = exportFileMsg.getUserToken();
            String locale = exportFileMsg.getLocale();

            // 获取元数据信息
            DWSecurityToken dwSecurityToken;
            try {
                dwSecurityToken = DWSecurityTokenGenerator.parseSecurityToken(userToken);
            } catch (Exception e) {
                log.error("解析token失败:{}", userToken);
                // 抛异常，在catch中统一处理
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0106.getErrCode(), MessageUtil.getMessage("delivery.resolveTokenError"), e);
            }
            DWSecurityContext dwSecurityContext = dwSecurityToken.getContext();

            // 查询用户信息，后续调外部接口时需要传routerKey
            AuthoredUser authoredUser = tokenVerifyService.getUserInfo(dwSecurityContext.getUserToken());
            AppAuthContextHolder.getContext().setAuthoredUser(authoredUser);

            GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaDataBySecurityToken(actionId, dwSecurityContext.getUserToken(), locale, userToken);
            DownloadBaseDataParamDTO downloadBaseDataParamDTO = exportFileMsg.getDownloadBaseDataParamDTO();
            baseDataEntryService.getActionParas(downloadBaseDataParamDTO);

            exportFileMsg.setApplication(downloadBaseDataParamDTO.getApplication());
            exportFileMsg.setType(exportStatistics.getType());
            // 2、判断是否支持分页,调用应用导出API
            List<Map<String, Object>> tableList = new ArrayList<>();
            String productName = metaDataService.getProductName(dwSecurityContext.getUserToken(), actionId.substring(actionId.contains("esp_") ? 4 : 0));
            //response主key
            String mainKey = metadataDTO.getResponse().getData().getData_name();
            String mainKeyDesc = metadataDTO.getResponse().getData().getDescription();
            if (Boolean.TRUE.equals(metaDataService.useHasNext(metadataDTO))) {
                Integer pageNum = ImportAndExportStatisticsConstants.EXPORT_FILE_DEFAULT_PAGE_NO;
                getBaseDataByPage(downloadBaseDataParamDTO.getActionInfo().getActionParas(), exportFileMsg, productName, mainKey, dwSecurityContext.getUserToken(), pageNum, tableList);
            } else {
                getBaseData(downloadBaseDataParamDTO.getActionInfo().getActionParas(), exportFileMsg, productName, tableList, mainKey, dwSecurityContext.getUserToken());
            }
            //获取responseKey类型
            Map<String, CellTypeContainer> cellTypeContainerMap = metaDataService.getResponseCellTypeContainersBySecurityToken(metadataDTO, dwSecurityContext.getUserToken(), userToken, locale);
            //根据actionId获取excel列名
            List<String> keyList = new ArrayList<>();
            Map<String, String> headers = metaDataService.getResponseHeaders(metadataDTO, keyList);

            // 3、生成导出Excel,并上传至dmc
            // 双档导出数据最多2000条（按单头数量计算）
            if (!CollectionUtils.isEmpty(tableList) && tableList.size() > ImportAndExportStatisticsConstants.EXPORT_LIMIT && ImportAndExportStatisticsConstants.CATEGORY_DOUBLE_DOCUMENT.equals(downloadBaseDataParamDTO.getActionInfo().getCategory())) {
                tableList = tableList.subList(0, ImportAndExportStatisticsConstants.EXPORT_LIMIT);
            }
            // 过滤导出字段
            baseDataEntryService.getExportFileds(downloadBaseDataParamDTO.getActionInfo().getSelectField(), keyList, cellTypeContainerMap);
            String fileUrl = exportStatisticsDomainService.handleDownloadBaseData(exportStatistics, keyList, headers, cellTypeContainerMap, tableList, ExcelHelper.getSheetName(mainKey, mainKeyDesc, true));
            if (!StringUtils.isEmpty(fileUrl)) {
                exportStatistics.setFileUrl(fileUrl);
                exportStatistics.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_SUCCESS);
            } else {
                log.error("fileUrl is null!");
                exportStatistics.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL);
            }
            exportStatistics.setUpdateTime(new Date());
            JaLog.info("has been export records size: {}",tableList.size());
            exportStatistics.setTotalSize(tableList.size());
            exportStatisticsDomainService.updateById(exportStatistics);
        } catch (Exception e) {
            log.error("ExportMQListener consume error:{}", e);
            exportStatistics.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL);
            exportStatistics.setUpdateTime(new Date());
            exportStatistics.setTotalSize(0);
            exportStatisticsDomainService.updateById(exportStatistics);
        } finally {
            lock.unlock();
        }

        // 4、所有数据处理完毕，发送MQTT消息
        ExportSuccessEvent exportSuccessEvent = ExportSuccessEventFactory.produceByExportStatistics(exportStatistics);
        eventPublisher.publish(exportSuccessEvent);
    }

    /**
     * 设置分页参数查询应用接口
     *
     * @param requestParam
     * @param exportFileMsg
     * @param productName
     * @param mainKey
     * @param userToken
     * @return
     */
    private void getBaseDataByPage(Map<String, Object> requestParam, ExportFileMsg exportFileMsg, String productName, String mainKey, String userToken, Integer pageNum, List<Map<String, Object>> tableList) {
        requestParam.put("use_has_next", true);
        requestParam.put("page_size", ImportAndExportStatisticsConstants.EXPORT_FILE_DEFAULT_PAGE_SIZE);
        requestParam.put("page_no", pageNum);

        Boolean hasNext = getBaseData(requestParam, exportFileMsg, productName, tableList, mainKey, userToken);
        if (Boolean.TRUE.equals(hasNext)) {
            pageNum++;
            getBaseDataByPage(requestParam, exportFileMsg, productName, mainKey, userToken, pageNum, tableList);
        }
    }

    /**
     * 查询应用接口
     *
     * @param requestParam
     * @param exportFileMsg
     * @param productName
     * @param tableList
     * @param mainKey
     * @param userToken
     * @return
     */
    private Boolean getBaseData(Map<String, Object> requestParam, ExportFileMsg exportFileMsg, String productName, List<Map<String, Object>> tableList, String mainKey, String userToken) {
        Boolean hasNext;
        EspResponse espResponse = espService.getBaseData(requestParam, exportFileMsg, productName, userToken);
        if (Objects.isNull(espResponse) || !espResponse.isOK()) {
            log.error("getBaseData fail! exportFileMsg:{}", exportFileMsg);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0067.getErrCode(), MessageUtil.getMessage("delivery.exportDataError"));
        }
        EspBody espBody = (EspBody) espResponse.getData();
        if (Objects.isNull(espBody) || Objects.isNull(espBody.getStd_data()) || CollectionUtils.isEmpty(espBody.getStd_data().getParameter())
                || Objects.isNull(espBody.getStd_data().getParameter().get(mainKey))) {
            log.error("getBaseData espResponse is null! exportFileMsg:{}", exportFileMsg);
            return false;
        }
        if (espBody.getStd_data().getParameter().containsKey(ImportAndExportStatisticsConstants.HAS_NEXT)) {
            hasNext = (Boolean) espBody.getStd_data().getParameter().get(ImportAndExportStatisticsConstants.HAS_NEXT);
        } else {
            hasNext = false;
        }
        tableList.addAll((List<Map<String, Object>>) espBody.getStd_data().getParameter().get(mainKey));
        return hasNext;
    }
}

