package com.digiwin.athena.abt.presentation.mq;

import com.digiwin.athena.abt.application.dto.migration.abt.event.ExportSuccessEvent;
import com.digiwin.athena.abt.application.dto.migration.abt.event.ExportSuccessEventFactory;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ExportStatistics;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.ExportFileMsg;
import com.digiwin.athena.abt.application.service.abt.migration.event.EventPublisher;
import com.digiwin.athena.abt.application.service.abt.migration.inout.ExportStatisticsDomainService;
import com.digiwin.athena.abt.core.ie.contants.IEExportBusinessTypeEnum;
import com.digiwin.athena.abt.core.ie.dto.IERedisMainCountInfoTask;
import com.digiwin.athena.abt.core.ie.dto.IERedisSubTask;
import com.digiwin.athena.abt.core.ie.strategy.AbstractIETaskStrategy;
import com.digiwin.athena.abt.core.meta.constants.ImportAndExportStatisticsConstants;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.framework.core.context.ASKContext;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;

@Component
@Slf4j
public class HistoryTaskStrategy extends AbstractIETaskStrategy {
    @Autowired
    private ExportHistoryDataRedisListener exportHistoryDataRedisListener;

    @Autowired
    private ExportStatisticsDomainService exportStatisticsDomainService;

    @Autowired
    private EventPublisher eventPublisher;

    @Override
    public boolean supports(String businessType) {
        return IEExportBusinessTypeEnum.HISTORY.getType().equalsIgnoreCase(businessType);
    }

    @Override
    public void doExecute(IERedisSubTask ieRedisTask) {
        log.info("------------>execute执行历史任务:" + JsonUtils.objectToString(ieRedisTask));
        AppAuthContextHolder.clearContext();
        ASKContext.getInstance().remove();
        LocaleContextHolder.resetLocaleContext();
        exportHistoryDataRedisListener.consumer((ExportFileMsg) ieRedisTask);
    }

    @Override
    public void doSuccess(IERedisSubTask ieRedisTask) {
        ExportStatistics exportStatistics = getExportFileMsg((ExportFileMsg) ieRedisTask);
        if (exportStatistics == null) {
            return;
        }

        publisEvent(exportStatistics);
        log.info("=============>doSuccess,masterId:{}", exportStatistics.getMasterId());
    }

    @Override
    public void doFail(IERedisSubTask ieRedisTask, Throwable throwable) {
        ExportStatistics exportStatistics = getExportFileMsg((ExportFileMsg) ieRedisTask);
        if (exportStatistics == null) {
            return;
        }

        exportStatistics.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL);
        exportStatistics.setUpdateTime(new Date());
        exportStatistics.setTotalSize(0);
        exportStatisticsDomainService.update(exportStatistics);

        publisEvent(exportStatistics);
        log.error("=============>doFail,masterId:{}", exportStatistics.getMasterId(), throwable);
    }



    @Override
    public void doTimeout(IERedisSubTask ieRedisTask) {
        ExportStatistics exportStatistics = getExportFileMsg((ExportFileMsg) ieRedisTask);
        if (exportStatistics == null) {
            return;
        }

        exportStatistics.setState(ImportAndExportStatisticsConstants.EXPROT_STATE_FAIL);
        exportStatistics.setUpdateTime(new Date());
        exportStatistics.setTotalSize(0);
        exportStatisticsDomainService.update(exportStatistics);

        publisEvent(exportStatistics);
        log.info("=============>doTimeout,masterId:{}", exportStatistics.getMasterId());
    }

    @Override
    public void doAllComplete(IERedisMainCountInfoTask mainCoutInfoTask) {
        List<String> list = mainCoutInfoTask.getInfo().getSubTaskIdList();
        log.info("=============>doAllComplete,masterId:{}", CollectionUtils.isEmpty(list) ? "" : String.join(", ", list));

    }

    private ExportStatistics getExportFileMsg(ExportFileMsg exportFileMsg) {
        ExportStatistics exportStatistics = exportStatisticsDomainService.getByMasterId(exportFileMsg.getMasterId());
        if (Objects.isNull(exportStatistics)) {
            //将消息丢弃
            log.error("未找到对应数据masterId:{}", exportFileMsg.getMasterId());
            return null;
        }
        return exportStatistics;
    }

    private void publisEvent(ExportStatistics exportStatistics) {
        //所有数据处理完毕，发送MQTT消息
        ExportSuccessEvent exportSuccessEvent = ExportSuccessEventFactory.produceByExportStatistics(exportStatistics);
        eventPublisher.publish(exportSuccessEvent);
    }
}
