package com.digiwin.athena.abt.presentation.mq;

import cn.hutool.core.map.MapUtil;
import com.digiwin.athena.abt.application.dto.migration.abt.esp.EspBody;
import com.digiwin.athena.abt.application.dto.migration.abt.esp.EspResponse;

import com.digiwin.athena.abt.application.dto.migration.abt.inout.ImportBatchRecord;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ImportStatistics;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.MetadataContext;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.ApiDataFieldLocaleMetadataDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.FileInfo;
import com.digiwin.athena.abt.application.dto.migration.abt.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.DataEntryTask;
import com.digiwin.athena.abt.application.service.abt.migration.esp.EspService;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.ExcelHelper;
import com.digiwin.athena.abt.application.service.abt.migration.helpler.ExcelHelperV2;
import com.digiwin.athena.abt.application.service.abt.migration.inout.*;
import com.digiwin.athena.abt.application.utils.ConcurrentSubmitUtils;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import com.digiwin.athena.abt.core.concurrent.pool.DynamicThreadPool;
import com.digiwin.athena.abt.core.meta.dto.CellTypeContainer;
import com.digiwin.athena.abt.core.meta.dto.ExeclCacheLog;
import com.digiwin.athena.abt.core.meta.dto.ImportCounters;
import com.digiwin.athena.abt.core.meta.dto.IndexOption;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.abt.core.uiils.CounterContext;
import com.digiwin.athena.abt.core.uiils.MongoCacheUtils;
import com.digiwin.athena.appcore.auth.AppAuthContext;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.auth.service.TokenVerifyService;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.service.permission.DWSecurityTokenGenerator;
import com.digiwin.service.permission.pojo.DWSecurityContext;
import com.digiwin.service.permission.pojo.DWSecurityToken;
import com.google.common.base.Stopwatch;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.spring.boot.util.JaI18nUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;

import org.springframework.beans.factory.annotation.Autowired;

import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.digiwin.athena.abt.core.meta.constants.RedisQueueContant.*;

/**
 * @Author wzq
 **/
@Slf4j
@Component
public class ImportBasicDataRedisListener {

    @Autowired
    private MetaDataService metaDataService;
    @Autowired
    private EspService espService;
    @Autowired
    private TokenVerifyService tokenVerifyService;
    @Autowired
    private ExcelHelperV2 excelHelperV2;
    @Autowired
    private ExportStatisticsDomainService exportStatisticsDomainService;
    @Autowired
    private ImportStatisticsDomainService importStatisticsDomainService;
    @Autowired
    private DynamicThreadPool taskExecutor;

    public void consumerBatch(DataEntryTask dataEntryTask) {
        // 获取导入统计信息
        ImportStatistics importStatistics = null;
        // 获取批次记录
        List<ImportBatchRecord> importBatchRecords = null;
        SXSSFWorkbook errorWorkbook = null;
        String masterId = dataEntryTask.getMasterId();
        try {
            importStatistics = importStatisticsDomainService.queryImportStatisticsByMasterId(masterId);

            importBatchRecords = importStatisticsDomainService.queryImportBatchRecordsByMasterId(masterId);

            // 初始化计数器
            CounterContext.put(masterId, createCounters(importStatistics));

            // 处理元数据
            MetadataContext metadataContext = initializeMetadataContext(dataEntryTask, importStatistics);

            // 创建专门用于同步的锁对象
            final Object workbookLock = new Object();
            errorWorkbook = excelHelperV2.createSXSSFWorkbook();
            // 处理批次记录
            processAllBatches(dataEntryTask, importBatchRecords, importStatistics, metadataContext, errorWorkbook, workbookLock);
        } catch (Exception e) {
            log.error("[基础资料导入]初始化数据失败masterId:{}", masterId, e);
            CounterContext.markInitializationError(masterId);
        } finally {
            // 最终处理（更新状态、上传错误文件等）
            finalizeProcessing(importStatistics, importBatchRecords, CounterContext.get(masterId), errorWorkbook);
        }
    }

    private void processAllBatches(DataEntryTask dataEntryTask, List<ImportBatchRecord> importBatchRecords,
                                   ImportStatistics importStatistics, MetadataContext metadataContext,
                                   SXSSFWorkbook errorWorkbook, Object workbookLock) {
        AppAuthContext context = AppAuthContextHolder.getContext();
        Locale locale = LocaleContextHolder.getLocale();

        ConcurrentSubmitUtils task = ConcurrentSubmitUtils.build(taskExecutor);

        importBatchRecords.forEach(importBatchRecord -> task.addCollections(()->{
            AppAuthContextHolder.setContext(context);
            LocaleContextHolder.setLocale(locale);

            processBatchRecord(dataEntryTask, importBatchRecord.getBatchSeq(), importStatistics, metadataContext, errorWorkbook, workbookLock);
        }));

        task.getCollectionsWithTimeout(JaProperty.getLong(BASE_IMPORT_BATCH_ASYNC_TIMEOUT, 60 * 30L), TimeUnit.SECONDS);
    }


    // 辅助方法 - 初始化元数据上下文
    private MetadataContext initializeMetadataContext(DataEntryTask dataEntryTask, ImportStatistics importStatistics) {
        try {
            DWSecurityToken dwSecurityToken = DWSecurityTokenGenerator.parseSecurityToken(dataEntryTask.getUserToken());
            DWSecurityContext dwSecurityContext = dwSecurityToken.getContext();

            // 查询用户信息
            AuthoredUser authoredUser = tokenVerifyService.getUserInfo(dwSecurityContext.getUserToken());
            AppAuthContextHolder.getContext().setAuthoredUser(authoredUser);

            LocaleContextHolder.setLocale(JaI18nUtil.parseLocales(dataEntryTask.getLocale().replace('_', '-')));

            // 获取元数据
            GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaDataBySecurityToken(
                    importStatistics.getActionId(),
                    dwSecurityContext.getUserToken(),
                    dataEntryTask.getLocale(),
                    dataEntryTask.getUserToken()
            );

            ApiDataFieldLocaleMetadataDTO mainMetadata = metaDataService.getMainMetadata(
                    metadataDTO, dataEntryTask.getActionInfo()
            );

            List<CellTypeContainer> cellTypeContainers = metaDataService.getCellTypeContainers(
                    mainMetadata.getField(),
                    authoredUser.getToken(),
                    null,
                    dataEntryTask.getLocale(),
                    1
            );

            Map<String, String> metaData = metaDataService.getHeaderMap(mainMetadata);
            Map<String, CellTypeContainer> cellTypeContainerMap = metaDataService.getResponseCellTypeContainersBySecurityToken(
                    metadataDTO,
                    dwSecurityContext.getUserToken(),
                    importStatistics.getUserToken(),
                    dataEntryTask.getLocale()
            );

            List<CellTypeContainer> businessKeyContainer = excelHelperV2.getBusinessKeyContainer(
                    new ArrayList<>(cellTypeContainerMap.values())
            );

            List<CellTypeContainer> arrayKeyContainer = excelHelperV2.getArrayKeyContainer(
                    new ArrayList<>(cellTypeContainerMap.values())
            );

            Set<String> arrayField = CollectionUtils.isEmpty(arrayKeyContainer) ?
                    new HashSet<>() :
                    arrayKeyContainer.stream()
                            .map(CellTypeContainer::getKeyName)
                            .collect(Collectors.toSet());

            String productName = metaDataService.getProductNameBySecurityToken(dwSecurityContext.getUserToken(),
                    dataEntryTask.getActionId().substring(dataEntryTask.getActionId().contains("esp_") ? 4 : 0), dataEntryTask.getUserToken());

            return new MetadataContext(metadataDTO, metaData, cellTypeContainerMap, businessKeyContainer, arrayField, cellTypeContainers, productName);
        } catch (Exception e) {
            log.error("初始化元数据失败:{}", dataEntryTask.getUserToken(), e);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0106.getErrCode(), MessageUtil.getMessage("delivery.resolveTokenError"), e);
        }
    }

    // 辅助方法 - 处理批次记录
    private void processBatchRecord(DataEntryTask dataEntryTask, int batchNo, ImportStatistics importStatistics,
                                    MetadataContext metadataContext, SXSSFWorkbook existingWorkbook, Object workbookLock) {
        Stopwatch stopwatch = Stopwatch.createStarted();
        int batchSize = 0;
        String masterId = importStatistics.getMasterId();
        List<Map<String, Object>> errorList = null;
        try {
            errorList = new ArrayList<>();

            List<Map> data = getImportDataList(importStatistics.getMasterId(), batchNo, dataEntryTask.getTableKey());

            batchSize = data.size();

            // 校验数据
            List<Map> validateErrorList = metaDataService.validateByBatchV2(dataEntryTask, metadataContext.getCellTypes(), data);

            if (CollectionUtils.isEmpty(validateErrorList)) {
                EspResponse response = sendByBatchWapper(dataEntryTask, metadataContext, data);

                if (response.isOK()) {
                    CounterContext.markSuccess(masterId, batchSize);
                } else {
                    handleApiError(data, response, dataEntryTask, errorList, batchSize, batchNo);
                }
            } else {
                handleValidationError(masterId, errorList, batchSize, validateErrorList);
            }
        } catch (Exception e) {
            log.error("[基础资料导入]处理导入异常", e);
            CounterContext.markError(masterId, batchSize);
        } finally {
            // 处理错误数据
            if (!CollectionUtils.isEmpty(errorList)) {
                synchronized (workbookLock) {
                    createErrorExcel(metadataContext, importStatistics, dataEntryTask, errorList, existingWorkbook);
                }
            }
        }
        log.info("[基础资料导入]批次 {} 处理完成, 耗时: {}ms", batchNo, stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }

    private EspResponse sendByBatchWapper(DataEntryTask dataEntryTask, MetadataContext metadataContext, List<Map> data) {
        // 判断是否开启mock
        if (!JaProperty.getBoolean(ATHENA_IMPORT_REDIS_QUEUE_MOCK_ENABLE, false)) {
            // 调用导入API
            EspResponse response = espService.sendByBatch(dataEntryTask, metadataContext.getMetadataDTO(), data, metadataContext.getProductName());
            if (null == response) {
                response = new EspResponse();
                response.setData(null);
                response.setStatus(EspResponse.Status.FAILED);
            }
            return response;
        } else {
            // 进行模拟耗时
            try {
                Thread.sleep(JaProperty.getLong(ATHENA_IMPORT_REDIS_QUEUE_MOCK_TIMEOUT, 200L));
            } catch (InterruptedException e) {
                log.error("[基础资料导入]InterruptedException error", e);
            }
            EspResponse response = new EspResponse();
            response.setData(null);
            response.setStatus(EspResponse.Status.SUCCESS);
            return response;
        }
    }

    // 辅助方法 - 处理API错误
    private void handleApiError(List<Map> data, EspResponse response, DataEntryTask dataEntryTask,
                                List<Map<String, Object>> errorList, int batchSize, int batchNo) {
        log.error("[基础资料导入]调用esp服务失败，批次 {}", batchNo);
        List<Map<String, Object>> importErrorResult;

        if (response.getData() == null) {
            log.error("[基础资料导入]调用esp服务失败，data数据为空");
            importErrorResult = batchError(data);
        } else {
            importErrorResult = parseResponse(dataEntryTask.getTableKey(), response);
        }

        if (importErrorResult == null) {
            log.error("[基础资料导入]调用esp服务，解析导入异常数据失败,解析错误信息为空!");
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0107.getErrCode(), MessageUtil.getMessage("delivery.handleDataError"));
        }
        CounterContext.markFailure(dataEntryTask.getMasterId(), batchSize - importErrorResult.size(), importErrorResult.size());
        errorList.addAll(importErrorResult);
    }

    // 辅助方法 - 处理验证错误
    private void handleValidationError(String masterId, List<Map<String, Object>> errorList, int batchSize,
                                       List<Map> validateErrorList) {
        CounterContext.markFailure(masterId, batchSize - validateErrorList.size(), validateErrorList.size());
        errorList.addAll(validateErrorList.stream()
                .map(map -> (Map<String, Object>) map)
                .collect(Collectors.toList()));
    }

    // 辅助方法 - 创建错误Excel
    private void createErrorExcel(MetadataContext metadataContext, ImportStatistics importStatistics,
                                  DataEntryTask dataEntryTask, List<Map<String, Object>> errorList,
                                  SXSSFWorkbook existingWorkbook) {
        try {
//            existingWorkbook = existingWorkbook != null ? existingWorkbook : excelHelperV2.createSXSSFWorkbook();

            GetActionLocaleResponseDTO metadataDTO = metadataContext.getMetadataDTO();
            String mainKey = metadataDTO.getResponse().getData().getData_name();
            String mainKeyDescription = metadataDTO.getResponse().getData().getDescription();

            excelHelperV2.createDataExcel(
                    ExcelHelper.getSheetName(mainKey, mainKeyDescription, true),
                    importStatistics.getLocale(),
                    2,
                    dataEntryTask.getKeyList(),
                    metadataContext.getCellTypeContainerMap(),
                    metadataContext.getMetaData(),
                    errorList,
                    existingWorkbook,
                    metadataContext.getBusinessKeyContainer(),
                    0,
                    null,
                    null,
                    metadataContext.getArrayField()
            );
        } catch (Exception e) {
            log.error("[基础资料导入]生成错误文件失败", e);
        }

    }

    // 辅助方法 - 最终处理
    private void finalizeProcessing(ImportStatistics importStatistics, List<ImportBatchRecord> importBatchRecords, ImportCounters counters,
                                    SXSSFWorkbook errorWorkbook) {
        if (null == counters) {
            return;
        }
        // 处理错误文件
        counters.setFailUrl(handleErrorFile(importStatistics, errorWorkbook));

        //更新导入表信息
        importStatisticsDomainService.updateImportDb(importStatistics, importBatchRecords, counters);

    }

    // 辅助方法 - 处理错误文件
    private String handleErrorFile(ImportStatistics importStatistics, SXSSFWorkbook errorWorkbook) {
        if (errorWorkbook == null) {
            return null;
        }

        String failedUrl = null;
        try {
            if (!excelHelperV2.isWorkbookEmpty(errorWorkbook)) {
                FileInfo fileInfo = new FileInfo();
                fileInfo.setFileName(importStatistics.getMasterId());
                failedUrl = exportStatisticsDomainService.handleDownloadBaseDataV2(errorWorkbook, fileInfo);
            } else {
                //如果没有错误数据，释放workbook资源
                errorWorkbook.dispose();
            }
        } catch (Exception e) {
            log.error("[基础资料导入]上传文件报错", e);
        }
        return failedUrl;
    }

    private List<Map<String, Object>> batchError(List<Map> dataList) {
        List<Map<String, Object>> result = new LinkedList<>();
        dataList.forEach(item -> {

            Map row = MapUtil.newHashMap(item.size());
            row.putAll(item);
            row.put("batch_error_msg", MessageUtil.getMessage("delivery.batchHandleError"));
            result.add(row);
        });
        return result;
    }

    public List<Map> getImportDataList(String masterId, Integer batchNo, String dataName) {
        List<Map> dataList = new ArrayList<>();
        List<ExeclCacheLog> mainDataList = MongoCacheUtils.queryMainDataSourceBatch(masterId, batchNo, dataName);
        log.info("[基础资料导入]批次[{}]主数据量: {}", batchNo, mainDataList.size());
        if (!CollectionUtils.isEmpty(mainDataList)) {
            Map<String, List<ExeclCacheLog>> cache = MapUtil.newHashMap(mainDataList.size());
            preloadDataToCache(mainDataList, cache);

            mainDataList.forEach(mainData -> {
                dataList.add(mainData.getData());
                processChildIteratively(mainData, cache);
            });
        }
        return dataList;
    }

    /**
     * 使用迭代代替递归处理子节点
     */
    private void processChildIteratively(ExeclCacheLog root, Map<String, List<ExeclCacheLog>> cache) {
        Deque<ExeclCacheLog> stack = new ArrayDeque<>();
        stack.push(root);

        while (!stack.isEmpty()) {
            ExeclCacheLog current = stack.pop();
            String currentKey = buildKey(current.getBk(), current.getMasterId(), current.getNextDb());
            List<ExeclCacheLog> children = cache.getOrDefault(currentKey, Collections.emptyList());

            if (!CollectionUtils.isEmpty(children)) {
                current.getData().put(current.getNextDb(),
                        children.stream().map(ExeclCacheLog::getData).collect(Collectors.toList()));

                // 将子节点压栈继续处理
                children.forEach(stack::push);
            }
        }
    }


    // 预加载所有可能用到的数据到缓存
    private void preloadDataToCache(List<ExeclCacheLog> mainDataList, Map<String, List<ExeclCacheLog>> cache) {
        List<ExeclCacheLog> result = mainDataList.stream().filter(t -> !Objects.equals(t.getNextDb(), MongoCacheUtils.NULL_DB)).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(result)) {
            return;
        }
        List<List<IndexOption>> indexs = result.stream()
                .filter(log -> log.getBk() != null && !log.getBk().isEmpty())
                .map(ExeclCacheLog::getBk).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(indexs)) {
            return;
        }
        List<ExeclCacheLog> childs = MongoCacheUtils.queryByPbksBatch(indexs, result.get(0).getMasterId(), result.get(0).getNextDb());
        for (ExeclCacheLog child : childs) {
            List<IndexOption> pBk = child.getPBk();
            String masterId = child.getMasterId();
            String db = child.getDb();
            String key = buildKey(pBk, masterId, db);
            if (StringUtils.isEmpty(key))
                continue;
            cache.compute(key, (k, oldValue) -> {
                if (Objects.isNull(oldValue)) {
                    oldValue = new ArrayList<>();
                }
                oldValue.add(child);
                return oldValue;
            });
        }
        preloadDataToCache(childs, cache);
    }

    private String buildKey(List<IndexOption> pBk, String masterId, String db) {
        if (StringUtils.isEmpty(masterId) || StringUtils.isEmpty(db) || CollectionUtils.isEmpty(pBk)) {
            return null;
        }

        return pBk.stream()
                .map(item -> String.valueOf(item.getIndex()))
                .sorted()
                .collect(Collectors.joining("|", masterId + ":" + db + ":", ""));
    }


    private ImportCounters createCounters(ImportStatistics stats) {
        return new ImportCounters(stats.getProcessingNum(), stats.getFailedNum(),
                stats.getErrorNum(), stats.getSucceededNum());
    }

    public List<Map<String, Object>> parseResponse(String tableKey, EspResponse response) {
        EspBody espBody = (EspBody) response.getData();
        if (null == espBody) {
            log.error("[基础资料导入]调用esp服务失败，espBody.data数据为空");
            return Collections.emptyList();
        }
        return (List<Map<String, Object>>) espBody.getStd_data().getParameter().get(tableKey);
    }
}
