package com.digiwin.athena.abt.presentation.mq;

import com.digiwin.athena.abt.application.dto.migration.abt.event.ImportSuccessEvent;
import com.digiwin.athena.abt.application.dto.migration.abt.event.ImportSuccessEventFactory;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ImportBatchRecord;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.ImportStatistics;
import com.digiwin.athena.abt.application.dto.migration.abt.worker.DataEntryTask;
import com.digiwin.athena.abt.application.service.abt.migration.event.EventPublisher;
import com.digiwin.athena.abt.application.service.abt.migration.inout.ImportStatisticsDomainService;
import com.digiwin.athena.abt.core.ie.contants.IEExportBusinessTypeEnum;
import com.digiwin.athena.abt.core.ie.dto.IERedisMainCountInfoTask;
import com.digiwin.athena.abt.core.ie.dto.IERedisSubTask;
import com.digiwin.athena.abt.core.ie.strategy.AbstractIETaskStrategy;
import com.digiwin.athena.abt.core.meta.dto.ImportCounters;
import com.digiwin.athena.abt.core.uiils.CounterContext;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.framework.core.context.ASKContext;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * @Author wzq
 **/
@Component
@Slf4j
public class ImportBasicDataTaskStrategy extends AbstractIETaskStrategy {
    @Autowired
    private ImportBasicDataRedisListener importBasicDataRedisListener;

    @Autowired
    private EventPublisher eventPublisher;

    @Autowired
    private ImportStatisticsDomainService importStatisticsDomainService;

    @Override
    public boolean supports(String businessType) {
        return IEExportBusinessTypeEnum.BASIC_IMPORT.getType().equalsIgnoreCase(businessType);
    }

    @Override
    public void doExecute(IERedisSubTask ieRedisTask) {
        DataEntryTask task = castToDataEntryTask(ieRedisTask);
        log.info("------------>execute执行基础资料导入任务:" + JsonUtils.objectToString(ieRedisTask));

        clearContext(task);

        importBasicDataRedisListener.consumerBatch(task);
    }

    @Override
    public void doSuccess(IERedisSubTask ieRedisTask) {
        DataEntryTask task = castToDataEntryTask(ieRedisTask);
        try {
            publishCompletionEvent(task);
        } finally {
            clearContext(task);
        }
        log.info("=============>doSuccess,masterId:{}", task.getMasterId());
    }

    @Override
    public void doFail(IERedisSubTask ieRedisTask, Throwable throwable) {
        DataEntryTask task = castToDataEntryTask(ieRedisTask);
        try {
            updateTaskStatus(task);

            publishCompletionEvent(task);
        } finally {
            clearContext(task);
        }
        log.error("=============>doFail,masterId:{}", task.getMasterId(), throwable);
    }

    @Override
    public void doTimeout(IERedisSubTask ieRedisTask) {
        DataEntryTask task = castToDataEntryTask(ieRedisTask);
        try {
            updateTaskStatus(task);

            publishCompletionEvent(task);
        } finally {
            clearContext(task);
        }
        log.info("=============>doTimeout,masterId:{}", task.getMasterId());
    }

    @Override
    public void doAllComplete(IERedisMainCountInfoTask mainCoutInfoTask) {
        List<String> list = mainCoutInfoTask.getInfo().getSubTaskIdList();
        log.info("=============>doAllComplete,masterId:{}", CollectionUtils.isEmpty(list) ? "" : String.join(", ", list));
    }

    private static void clearContext(DataEntryTask dataEntryTask) {
        AppAuthContextHolder.clearContext();
        ASKContext.getInstance().remove();
        LocaleContextHolder.resetLocaleContext();
        CounterContext.clear(dataEntryTask.getMasterId());
    }


    /**
     * 安全的类型转换
     */
    private DataEntryTask castToDataEntryTask(IERedisSubTask ieRedisTask) {
        if (!(ieRedisTask instanceof DataEntryTask)) {
            String errorMsg = "任务类型不匹配，期望类型: DataEntryTask, 实际类型: " + ieRedisTask.getClass().getName();
            log.error(errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }
        return (DataEntryTask) ieRedisTask;
    }

    /**
     * 更新任务状态
     */
    private void updateTaskStatus(DataEntryTask task) {
        ImportStatistics importStatistics = importStatisticsDomainService.queryImportStatisticsByMasterId(task.getMasterId());
        List<ImportBatchRecord> importBatchRecords = importStatisticsDomainService.queryImportBatchRecordsByMasterId(task.getMasterId());

        ImportCounters counters = CounterContext.get(task.getMasterId());
        if (counters != null) {
            // 对于失败和超时情况，标记初始化错误
            counters.markInitializationError();
            importStatisticsDomainService.updateImportDb(importStatistics, importBatchRecords, counters);
        }
    }

    /**
     * 发布任务完成事件
     */
    private void publishCompletionEvent(DataEntryTask task) {
        ImportStatistics importStatistics = importStatisticsDomainService.queryImportStatisticsByMasterId(task.getMasterId());
        if (importStatistics != null && importStatistics.getProcessingNum() == 0) {
            ImportSuccessEvent event = ImportSuccessEventFactory.produceByImportStatistics(importStatistics);
            eventPublisher.publish(event);
        }
    }
}
