package com.digiwin.athena.abt.presentation.server.web;

import com.digiwin.athena.abt.application.dto.migration.abt.api.*;
import com.digiwin.athena.abt.application.dto.migration.abt.excel.ExcelParserBean;
import com.digiwin.athena.abt.application.dto.migration.abt.inout.*;
import com.digiwin.athena.abt.application.dto.response.BatchImportListRespDTO;
import com.digiwin.athena.abt.application.dto.response.ExportBasicDataRespDTO;
import com.digiwin.athena.abt.application.service.abt.migration.assembler.ImportStatisticsAssembler;
import com.digiwin.athena.abt.application.service.abt.migration.inout.BaseDataEntryApplicationServiceImpl;
import com.digiwin.athena.abt.application.service.abt.migration.inout.BaseDataEntryRedisService;
import com.digiwin.athena.abt.application.service.abt.migration.inout.ExportStatisticsFactory;
import com.digiwin.athena.abt.application.service.abt.migration.inout.ExportHistoryDataRedisService;
import com.digiwin.athena.abt.application.utils.MessageUtil;
import com.digiwin.athena.abt.core.meta.enums.ErrorCodeEnum;
import com.digiwin.athena.abt.core.meta.constants.RedisQueueContant;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.jugg.agile.framework.core.config.JaProperty;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.digiwin.athena.abt.core.meta.constants.RedisQueueContant.EXPORT_DETAIL_TYPE;

/**
 * @ClassName BaseDataEntryController
 * @Description 基础数据录入接口层
 * @Author zhuangli
 * @Date 2021/4/1 15:23
 * @Version 1.0
 **/
@RestController
@RequestMapping("/api/abt/v1/baseDataEntry")
//@CrossOrigin
@Slf4j
public class BaseDataEntryInterface {

    @Autowired
    BaseDataEntryApplicationServiceImpl baseDataEntryService;

    @Autowired
    ExportStatisticsFactory exportStatisticsFactory;

    @Autowired
    private ExportHistoryDataRedisService exportHistoryDataRedisService;

    @Autowired
    private BaseDataEntryRedisService baseDataEntryRedisService;

    @PostMapping("upload")
    @ResponseBody
    public DapResponse upload(@RequestPart("uploadParam") UploadParamDTO uploadParam, @RequestPart("file") MultipartFile file) {
        if (null == file || file.isEmpty() || StringUtils.isEmpty(file.getOriginalFilename())) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0078.getErrCode(), MessageUtil.getMessage("delivery.fileEmpty"));
        }
        String extension = file.getOriginalFilename().substring(file.getOriginalFilename().lastIndexOf(".") + 1);
        if (!StringUtils.equalsIgnoreCase(extension, "xls") && !StringUtils.equalsIgnoreCase(extension, "xlsx")) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0079.getErrCode(), MessageUtil.getMessage("delivery.fileFormatEmpty"));
        }
        ExcelParserBean parserBean = new ExcelParserBean();
        try {
            parserBean.setInput(file.getInputStream());
        } catch (Exception e) {
            log.error("upload exception", e);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0080.getErrCode(), MessageUtil.getMessage("delivery.uploadError2"));
        }
        parserBean.setFilePath(file.getOriginalFilename());
        baseDataEntryService.upload(uploadParam, parserBean);
        log.info("upload success");
        return DapResponse.ok(MessageUtil.getMessage("delivery.success"));
    }


    @PostMapping("uploadAsync")
    @ResponseBody
    public DapResponse uploadAsync(@RequestBody List<UploadParamDTO> uploadParam) {
        // 不需要传递file
        baseDataEntryService.uploadAsync(uploadParam);
        log.info("upload success");
        return DapResponse.ok(MessageUtil.getMessage("delivery.success"));
    }

    @PostMapping("republish")
    public DapResponse republish(@RequestBody UploadParamDTO uploadParamDTO) {
        baseDataEntryService.republish(uploadParamDTO);
        return DapResponse.ok();
    }

    @PostMapping(value = "republishById")
    public DapResponse republishById(@RequestBody UploadParamDTO uploadParamDTO) {
        baseDataEntryService.republishById(uploadParamDTO);
        return DapResponse.ok();
    }

    @PostMapping("downloadTemplate")
    public DapResponse downloadTemplate(@RequestBody DownloadTemplateParamDTO param) {
        byte[] fileBytes = baseDataEntryService.downloadTemplate(param);
        return DapResponse.ok(fileBytes);
    }


    @GetMapping("getTableHeaders")
    public DapResponse getTableHeaders(@RequestParam(value = "actionId") String actionId) {
        return DapResponse.ok(baseDataEntryService.getTableHeaders(actionId));
    }


    @PostMapping("getErrorTable")
    public DapResponse getErrorTable(@RequestBody GetErrorTableParamDTO param) {
        ErrorTable errorTable = baseDataEntryService.getErrorTableByMasterId(param.getMasterId(), param.getLocale());
        ErrorTableDTO errorTableDTO = ImportStatisticsAssembler.errorTableToDTO(errorTable);
        return DapResponse.ok(errorTableDTO);
    }

    @PostMapping("getActivityStatistics")
    public DapResponse getActivityStatistics(@RequestBody GetActivityStatisticsParamDTO param) {
        ActivityStatistics activityStatistics = baseDataEntryService.getActivityStatistics(param.getActivityCode(), param.getUserId());
        ActivityStatisticsDTO activityStatisticsDTO = ImportStatisticsAssembler.activityStatisticsToDTO(activityStatistics);
        return DapResponse.ok(activityStatisticsDTO);
    }

    @PostMapping("getImportStatistics")
    public DapResponse getImportStatistics(@RequestBody QueryImportStatisticsParamDTO param) {
        ImportStatistics importStatistics = baseDataEntryService.getImportStatistics(param.getMasterId());
        ImportStatisticsDTO importStatisticsDTO = ImportStatisticsAssembler.toDTO(importStatistics);
        return DapResponse.ok(importStatisticsDTO);
    }

    @PostMapping("getProcessingNum")
    public DapResponse getProcessingNum(@RequestBody GetProcessingNumParamDTO param) {
        String userId = param.getUserId();
        UserStatistics userStatistics = baseDataEntryService.getProcessingNum(userId, param.getTenantId());
        UserStatisticsDTO userStatisticsDTO = new UserStatisticsDTO();
        userStatisticsDTO.setUserId(userId);
        userStatisticsDTO.setProcessingNum(userStatistics.getProcessingNum());
        return DapResponse.ok(userStatisticsDTO);
    }

    @PostMapping("getRecords")
    public DapResponse getRecords(@RequestBody GetRecordsParamDTO param) {
        List<ImportStatistics> importStatisticsList = baseDataEntryService.getRecordsByUserId(
                param.getUser(),
                param.getLocale(),
                param.getGetRecordsNum(),
                param.getActivityName(),
                param.getStartTime(),
                param.getEndTime(),
                param.getType(),
                param.getState(),
                param.getOffset());
        List<ImportStatisticsDTO> importStatisticsDTOList = ImportStatisticsAssembler.importStatisticsListToDTO(importStatisticsList);
        return DapResponse.ok(importStatisticsDTOList);
    }

    /**
     * upload entry points for all basic data input operations.
     *
     * @return
     */
    @GetMapping(value = "/import/basicDataList")
    public DapResponse getBasicDataList(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {

        List<BatchImportListRespDTO> basicDataList = baseDataEntryService.getBasicDataList(user);
        return DapResponse.ok(basicDataList);
    }

    @GetMapping(value = "/export/basicDataList")
    public DapResponse exportBasicDataList() {
        List<ExportBasicDataRespDTO> exportBasicDataList = baseDataEntryService.getExportBasicDataList();
        return DapResponse.ok(exportBasicDataList);
    }


    @PostMapping("downloadBaseData")
    public DapResponse downloadBaseData(@RequestBody DownloadBaseDataParamDTO downloadBaseDataParamDTO) {
        byte[] fileBytes = baseDataEntryService.downloadBaseData(downloadBaseDataParamDTO);
        return DapResponse.ok(fileBytes);
    }

    /**
     * 发起下载
     *
     * @param param
     * @return
     */
    @PostMapping("startDownload")
    public DapResponse startDownload(@RequestBody DownloadBaseDataParamDTO param) {
        String type = Optional.ofNullable(param.getType()).orElse("default");
        if (EXPORT_DETAIL_TYPE.contains(type)) {
            return DapResponse.ok(baseDataEntryService.startDownload(param));
        }
        return DapResponse.ok(JaProperty.getBoolean(RedisQueueContant.REDIS_QUEUE_ENABLE, Boolean.TRUE)
                ? baseDataEntryRedisService.startDownload(param)
                : baseDataEntryService.startDownload(param));
    }

    /**
     * 发起下载 历史项目/任务
     *
     * @param paramMap
     * @return
     */
    @PostMapping("historyStartDownload")
    public DapResponse historyStartDownload(@RequestBody Map<String, Object> paramMap) {
        //默认走新的导出方式
        if (JaProperty.getBoolean(RedisQueueContant.REDIS_QUEUE_ENABLE, Boolean.TRUE)) {
            return DapResponse.ok(exportHistoryDataRedisService.historyStartDownload(paramMap));
        }
        return DapResponse.ok(baseDataEntryService.historyStartDownload(paramMap));
    }

    /**
     * 查看导出详情
     *
     * @param param
     */
    @PostMapping("queryExportStatistics")
    public DapResponse queryExportStatistics(@RequestBody QueryExportStatisticsParamDTO param) {
        ExportStatisticsDTO exportStatisticsDTO = baseDataEntryService.queryExportStatistics(param.getMasterId());
        return DapResponse.ok(exportStatisticsDTO);
    }

    /**
     * 查询导出记录
     *
     * @param param
     * @return
     */
    @PostMapping("queryExportRecords")
    public DapResponse queryExportRecords(@RequestBody GetRecordsParamDTO param) {
        List<ExportStatistics> exportStatisticsList = baseDataEntryService.getExportRecordsByUserId(param.getUser(), param.getLocale(), param);
        List<ExportStatisticsDTO> importStatisticsDTOList = ExportStatisticsFactory.exportStatisticsListToDTO(exportStatisticsList);
        return DapResponse.ok(importStatisticsDTOList);
    }

    /**
     * 查询导出中数量
     *
     * @param param
     * @return
     */
    @PostMapping("getExportingNum")
    public DapResponse getExportingNum(@RequestBody GetExportingNumParamDTO param) {
        String userId = param.getUserId();
        UserStatistics userStatistics = baseDataEntryService.getExportingNum(userId, param.getTenantId(), param.getType(), param.getTypes());
        UserStatisticsDTO userStatisticsDTO = new UserStatisticsDTO();
        userStatisticsDTO.setUserId(userId);
        userStatisticsDTO.setProcessingNum(userStatistics.getProcessingNum());
        return DapResponse.ok(userStatisticsDTO);
    }


    @PostMapping("updateRetryDownloadState")
    public DapResponse updateRetryDownloadState(@RequestParam(value = "masterId") String masterId) {
        baseDataEntryService.updateRetryDownloadState(masterId);
        return DapResponse.ok();
    }
}
