package com.digiwin.athena.abt.presentation.server.web;

import com.digiwin.athena.abt.application.dto.migration.abt.api.RefreshCardDto;
import com.digiwin.athena.abt.application.dto.migration.abt.api.RefreshCardMessageCompensateDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.summary.CardJob;
import com.digiwin.athena.abt.application.dto.migration.abt.summary.RefreshCardInfosDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.summary.RefreshCardMessageDTO;
import com.digiwin.athena.abt.application.dto.migration.abt.summary.UpdateTaskWorkItemMessageDTO;
import com.digiwin.athena.abt.application.service.abt.migration.summary.RefreshCardMessageConverter;
import com.digiwin.athena.abt.application.service.abt.migration.summary.RefreshCardMessageService;
import com.digiwin.athena.appcore.auth.AppAuthContext;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import org.apache.commons.lang3.StringUtils;
import org.quartz.SchedulerException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * RefreshCardMessageController Description
 *
 * @author majianfu
 * @date 2023/4/3
 * @since
 */
@RestController
@RequestMapping("/api/abt/v1/refresh/card/message")
public class RefreshCardMessageController {
    @Autowired
    private RefreshCardMessageService refreshCardMessageService;

    @Autowired
    private RefreshCardMessageConverter refreshCardMessageConverter;

    /**
     * 为任务卡/项目卡创建定时刷新 message（summary、history_message、search_message）的定时任务
     *
     * @param refreshCardMessageDTO
     * @return 无返回值
     */
    @PostMapping("/job/createIfAbsent")
    public ResponseEntity<?> createRefreshCardMessageJobIfAbsent(@Valid @RequestBody RefreshCardMessageDTO refreshCardMessageDTO) {
        RefreshCardMessageDTO refreshCardMessageDO = refreshCardMessageConverter.to(refreshCardMessageDTO);
        return ResponseEntityWrapper.wrapperOk(refreshCardMessageService.createRefreshCardMessageJobIfAbsent(refreshCardMessageDO));
    }

    /**
     * 删除为任务卡/项目卡创建定时刷新 message（summary、history_message、search_message）的定时任务
     *
     * @param cardId 定时任务id（任务卡/项目卡）
     * @return 无返回值
     */
    @DeleteMapping("/job")
    public ResponseEntity<?> deleteRefreshCardMessageJob(@NotNull @RequestParam Long cardId) {
        refreshCardMessageService.deleteRefreshCardMessageJob(String.valueOf(cardId));
        return ResponseEntityWrapper.wrapperOk();
    }


    /**
     * 为指定的任务卡/项目卡列表补充定时任务
     *
     * @param refreshCardMessageCompensateList
     * @return
     */
    @PostMapping("/job/compensate")
    public ResponseEntity<?> compensateRefreshCardMessage(@RequestBody List<RefreshCardMessageCompensateDTO> refreshCardMessageCompensateList) {
        refreshCardMessageService.compensateRefreshCardMessage(refreshCardMessageCompensateList);
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 手动触发定时任务
     *
     * @param jobName
     * @return
     */
    @GetMapping("/triggerManual/{jobName}")
    public String triggerManual(@PathVariable String jobName) {
        return refreshCardMessageService.triggerManual(jobName);
    }

    /**
     * 根据code理解刷新标签，分组，筛选，排序
     * @param refreshCardDto
     */
    @PostMapping("/refreshCard")
    public ResponseEntity<?> refreshCard(@RequestBody RefreshCardDto refreshCardDto) {
        refreshCardMessageService.refreshCard(refreshCardDto);
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 设置分区大小
     * @param size
     * @return
     */
    @GetMapping("/refreshCard/settingSize/{size}")
    public ResponseEntity<?> settingSize(@PathVariable Integer size) {
        refreshCardMessageService.settingSize(size);
        return ResponseEntityWrapper.wrapperOk();
    }


    @PostMapping("/refreshCard/createOrUpdateCardJob")
    public ResponseEntity<?> createOrUpdateCardJob(@RequestBody CardJob cardJob) throws SchedulerException {
        if(StringUtils.isBlank(cardJob.getTenantId())){
            AppAuthContext context = AppAuthContextHolder.getContext();
            AuthoredUser authoredUser = context.getAuthoredUser();
            cardJob.setTenantId(authoredUser.getTenantId());
        }
        refreshCardMessageService.createOrUpdateCardJob(cardJob);
        return ResponseEntityWrapper.wrapperOk();
    }



    @PostMapping("/refreshCard/updateCardInfo")
    public ResponseEntity<?> updateCardInfo(@RequestBody CardJob cardJob) {
        if(StringUtils.isBlank(cardJob.getTenantId())){
            AppAuthContext context = AppAuthContextHolder.getContext();
            AuthoredUser authoredUser = context.getAuthoredUser();
            cardJob.setTenantId(authoredUser.getTenantId());
        }
        refreshCardMessageService.updateCardInfo(cardJob);
        return ResponseEntityWrapper.wrapperOk();
    }

    @PostMapping("/open/updateCardByBk")
    public ResponseEntity<?> updateCardByBk(@RequestBody RefreshCardInfosDTO refreshCardInfos) throws Exception {
        if(StringUtils.isBlank(refreshCardInfos.getTenantId())){
            AppAuthContext context = AppAuthContextHolder.getContext();
            AuthoredUser authoredUser = context.getAuthoredUser();
            refreshCardInfos.setTenantId(authoredUser.getTenantId());
        }
        refreshCardMessageService.updateCardByBk(refreshCardInfos);
        return ResponseEntityWrapper.wrapperOk();
    }


    /**
     * 更新任务卡摘要信息和search_message（来源于atmc调用）
     */
    @PostMapping("/updateTaskWorkItemMessage")
    public ResponseEntity<?> updateTaskWorkItemMessage(@RequestBody UpdateTaskWorkItemMessageDTO updateDTO) throws Exception {
        refreshCardMessageService.updateTaskWorkItemMessage(updateDTO);
        return ResponseEntityWrapper.wrapperOk();
    }
}
