package com.digiwin.athena.apimgmt.services;

import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDao;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import jakarta.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 放开原有的限制，版本在原有的基础上+-1
 */
@Service
public class ApiMgmtApiVersionService {

    private final ApiMgmtStandardApiDao apiDao;

    public ApiMgmtApiVersionService(ApiMgmtStandardApiDao apiDao) {
        this.apiDao = apiDao;
    }

    /**
     * 获取上一个版本
     *
     * @param pVersion pVersion
     * @return String
     */
    public String getPreviousVersion(String pVersion) {
        if (pVersion.equals(ApimgmtConstant.API_VER)) {
            return ApimgmtConstant.PER_API_VER;
        }

        String[] versions = pVersion.split("\\.");
        if (judgeNewBranch(pVersion)) { //2.0 3.0
            return ApimgmtConstant.PER_API_VER;
        }

        int version = Integer.parseInt(versions[1]) - 1;
        return versions[0] + "." + version;
    }

    /**
     * @return default 0
     */
    public int getMajorVersion(String version) {
        if (StringUtils.isBlank(version)) {
            return 0;
        }
        try {
            String[] parts = version.split("\\.");
            return parts.length > 0 ? Integer.parseInt(parts[0]) : 0;
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    /**
     * 获取上一个版本分支
     *
     * @return [2.0,2.9]=>1.0；[,1.0]=>null
     */
    public @Nullable String getPrevBranchByVersion(String version) {
        int majorVersion = Integer.parseInt(version.split("\\.")[0]);
        if (majorVersion > 1) {
            return (majorVersion - 1) + ".0";
        }

        return null;
    }

    public String getBranchFromVersion(String pVersion) {
        String[] versions = pVersion.split("\\.");
        return versions[0] + ".0";
    }

    /**
     * 获取下一个版本号
     *
     * @param pVersion pVersion
     * @return String
     */
    public String getNextVersion(String pVersion) {
        String[] versions = pVersion.split("\\.");
        int version = Integer.parseInt(versions[1]) + 1;
        return versions[0] + "." + version;
    }

    /**
     * 获取最新版本
     *
     * @param pStandardApiVersions pStandardApiVersions
     * @return StandardApiVersion
     */
    public StandardApiVersion getLatestVersion(List<StandardApiVersion> pStandardApiVersions) {
        Map<Integer, StandardApiVersion> apiVersionMap
                = pStandardApiVersions.stream().collect(Collectors.toMap(version -> {
            String[] versions = version.getVersion().split("\\.");
            return Integer.parseInt(versions[1]);
        }, data -> data));
        List<Integer> keys = new ArrayList<>(apiVersionMap.keySet());
        keys.sort(Collections.reverseOrder());
        return apiVersionMap.get(keys.get(0));
    }

    /**
     * 获取第一个版本 第一个版本不一定是1.0
     * by songwq 20230625
     *
     * @param pStandardApiVersions pStandardApiVersions
     * @return StandardApiVersion
     */
    public StandardApiVersion getFirstVersion(List<StandardApiVersion> pStandardApiVersions) {
        Map<Integer, StandardApiVersion> apiVersionMap
                = pStandardApiVersions.stream().collect(Collectors.toMap(version -> {
            String[] versions = version.getVersion().split("\\.");
            return Integer.parseInt(versions[1]);
        }, data -> data));
        List<Integer> keys = new ArrayList<>(apiVersionMap.keySet());
        Collections.sort(keys);
        return apiVersionMap.get(keys.get(0));
    }

    /**
     * 判定当前版本是否符合要求
     * 匹配上1.x
     *
     * @param pVersion pVersion
     * @return String
     */
    public Boolean judgeVersion(String pVersion) {
        return pVersion.matches("\\d+\\.\\d+");
    }

    /**
     * 判断是否是起始版本，1.0  2.0  3.0
     * @param pVersion
     * @return
     */
    public Boolean judgeNewBranch(String pVersion) {
        String[] versions = pVersion.split("\\.");
        int minor = Integer.parseInt(versions[1]);
        return minor == 0;
    }

}