package com.digiwin.athena.apimgmt.services;

import com.digiwin.athena.apimgmt.ApiMgmtApplicationParameter;
import com.digiwin.athena.apimgmt.constants.*;
import com.digiwin.athena.apimgmt.dao.*;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.IsRequiredEnum;
import com.digiwin.athena.apimgmt.model.*;
import com.digiwin.athena.apimgmt.util.StandardApiMessageUtil;
import com.google.common.collect.Sets;
import cn.hutool.core.collection.CollUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.DataValidation;
import org.apache.poi.ss.usermodel.DataValidationConstraint;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.xssf.usermodel.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.File;
import java.io.FileInputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiMgmtStandardApiExportWithXssService {

    @Autowired
    ApiMgmtStandardApiDao standardApiDao;

    @Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;

    @Autowired
    ApiMgmtStandardApiDataNameDao standardApiDataNameDao;

    @Autowired
    ApiMgmtStandardDataNameDao standardDataNameDao;

    @Autowired
    ApiMgmtStandardApiPagingDao standardApiPagingDao;

    @Autowired
    ApiMgmtStandardApiCategoryDao standardApiCategoryDao;

    @Autowired
    ApiMgmtProjectDao projectDao;

    @Autowired
    ApiMgmtProjectVersionService projectVersionService;

    /**
     * 创建文件
     */
    public XSSFWorkbook createXSSFWorkbook() {
        try {
            File d = new File(ApiMgmtApplicationParameter._FILE_EXPORT_PATH);
            if (!d.exists()) {
                boolean mk = d.mkdir();
                log.info("创建路径成功" + mk);
            }
            // 取得樣板
            return getExportTemplateFile();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 文件删除
     *
     * @param filePath filePath
     */
    public boolean removeFile(String filePath) {
        try {
            if (StringUtils.isNotBlank(filePath)) {
                return new File(filePath).delete();
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * createSheet
     *
     * @param pXSSFWorkbook       pXSSFWorkbook
     * @param pStandardApiVersion pStandardApiVersion
     */
    public void createXSSFSheet(XSSFWorkbook pXSSFWorkbook, StandardApiVersion pStandardApiVersion, Integer number) {
        StandardApi tStandardApi = pStandardApiVersion.getStandardApi();
        List<StandardApiDataName> tStandardApiDataNames = standardApiDataNameDao.getByApiVerId(pStandardApiVersion.getId(), true);
        tStandardApiDataNames = tStandardApiDataNames.stream().distinct().collect(Collectors.toList());
        Map<Integer, List<StandardApiDataName>> tMsgTypeMap = StandardApiMessageUtil.getMsgTypeMap(tStandardApiDataNames);
        int tRequestCount = tMsgTypeMap.get(1).size();
        int tResponseSuccessCount = tMsgTypeMap.get(2).size();
        int tResponseFalseCount = tMsgTypeMap.get(3).size();
        XSSFSheet tDataSheet = pXSSFWorkbook.getSheetAt(ApimgmtConstant.DATA_SHEET_NUMBER);
        XSSFSheet tTemplateSheet = pXSSFWorkbook.getSheetAt(ApimgmtConstant.TEMPLATE_SHEET_NUMBER);
        XSSFSheet tNewSheet = pXSSFWorkbook.createSheet((number + 1) + "." + tStandardApi.getName());
        // 比照範例模版設定欄寬
        setSheetWidthAndHight(tTemplateSheet, tNewSheet);
        // 生成下拉列表
        createDropDownMenu(tDataSheet, tTemplateSheet, tNewSheet, tRequestCount, tResponseSuccessCount, tResponseFalseCount);
        writeStandardAPI(pXSSFWorkbook, tTemplateSheet, tNewSheet, tStandardApi, pStandardApiVersion);
        writeStandardAPIDataName(pXSSFWorkbook, tTemplateSheet, tNewSheet, tMsgTypeMap, tStandardApiDataNames);
        // [S00-20190802002] 下載的規格檔: (1)使用者開啟下載檔案時，預設顯示sheet為該api的規格sheet
        // (2)拿掉範例的sheet.
        int tNewSheetIdx = pXSSFWorkbook.getSheetIndex(tNewSheet.getSheetName());
        // 焦點focus到 New Sheet
        pXSSFWorkbook.setActiveSheet(tNewSheetIdx);
        // 隱藏template
        pXSSFWorkbook.setSheetHidden(ApimgmtConstant.TEMPLATE_SHEET_NUMBER, true);
    }

    /**
     * 生成下拉列表
     *
     * @param pDataSheet            pDataSheet
     * @param pTemplateSheet        pTemplateSheet
     * @param pNewSheet             pNewSheet
     * @param pRequestCount         pRequestCount
     * @param pResponseSuccessCount pResponseSuccessCount
     * @param pResponseFalseCount   pResponseFalseCount
     */
    private void createDropDownMenu(XSSFSheet pDataSheet, XSSFSheet pTemplateSheet, XSSFSheet pNewSheet, int pRequestCount, int pResponseSuccessCount, int pResponseFalseCount) {
        // (24)API类型 (25)请求路径 (26)请求方法 (27)关联产品名称 (28)关联产品版本
        int tApiTypeCol = 24; // (24)API类型
        int tRequestMethodCol = 26; // 请求方法字段
        int tSyncTypeCol = 6; // 調用模式欄位.
        int tPagingCol = 7; // 分頁模式欄位.
        int tIsBtachCol = 10; // 須分包欄位.
        int tGroupCol = 11; // 信息大類欄位.
        int tCategoryCol = 12; // 信息類別欄位.
        int tMsgFormatCol = 13; // 信息格式欄位.
        // 計算下拉選單開始與結束列(Request、ResponseSuccess與ResponseFalse)
        int tRequestFirstRow = 6;
        int tRequestLastRow = pRequestCount == 0 ? tRequestFirstRow + pRequestCount : tRequestFirstRow + pRequestCount - 1;
        int tResponseSuccessFirstRow = tRequestLastRow + 4;
        int tResponseSuccessLastRow = pResponseSuccessCount == 0 ? tResponseSuccessFirstRow + pResponseSuccessCount : tResponseSuccessFirstRow + pResponseSuccessCount - 1;
        int tResponseFalseFirstRow = tResponseSuccessLastRow + 4;
        int tResponseFalseLastRow = pResponseFalseCount == 0 ? tResponseFalseFirstRow + pResponseFalseCount : tResponseFalseFirstRow + pResponseFalseCount - 1;
        int tColumnTypeCol = 1; // 類型欄位.
        int tDataTypeCol = 5; // 辭彙型態欄位.
        int tIsRequiredCol = 6; // 必要欄位.
        int tIsDatakeyCol = 7; // 為Datakey欄位.
        int tIsArrayCol = 8; // 為數組is_array欄位.
        int tCFCol = 4;
        // 設定下拉框區域
        CellRangeAddressList tApiTypeRegions = new CellRangeAddressList(ApimgmtConstant.STANDARD_API_ROW, ApimgmtConstant.STANDARD_API_ROW, tApiTypeCol, tApiTypeCol); // API类型
        CellRangeAddressList tRequestMethodRegions = new CellRangeAddressList(ApimgmtConstant.STANDARD_API_ROW, ApimgmtConstant.STANDARD_API_ROW, tRequestMethodCol, tRequestMethodCol); // 请求方法
        CellRangeAddressList tSyncTypeRegions = new CellRangeAddressList(ApimgmtConstant.STANDARD_API_ROW, ApimgmtConstant.STANDARD_API_ROW, tSyncTypeCol, tSyncTypeCol); // 調用模式.
        CellRangeAddressList tPagingRegions = new CellRangeAddressList(ApimgmtConstant.STANDARD_API_ROW, ApimgmtConstant.STANDARD_API_ROW, tPagingCol, tPagingCol);
        CellRangeAddressList tIsBatchRegions = new CellRangeAddressList(ApimgmtConstant.STANDARD_API_ROW, ApimgmtConstant.STANDARD_API_ROW, tIsBtachCol, tIsBtachCol);
        CellRangeAddressList tGroupRegions = new CellRangeAddressList(ApimgmtConstant.STANDARD_API_ROW, ApimgmtConstant.STANDARD_API_ROW, tGroupCol, tGroupCol); // 信息大類.
        CellRangeAddressList tCategoryRegions = new CellRangeAddressList(ApimgmtConstant.STANDARD_API_ROW, ApimgmtConstant.STANDARD_API_ROW, tCategoryCol, tCategoryCol);
        CellRangeAddressList tMsgFormatRegions = new CellRangeAddressList(ApimgmtConstant.STANDARD_API_ROW, ApimgmtConstant.STANDARD_API_ROW, tMsgFormatCol, tMsgFormatCol);
        CellRangeAddressList tRequestColumnTypeRegions = new CellRangeAddressList(tRequestFirstRow, tRequestLastRow, tColumnTypeCol, tColumnTypeCol);
        CellRangeAddressList tResponseSuccessColumnTypeRegions = new CellRangeAddressList(tResponseSuccessFirstRow, tResponseSuccessLastRow, tColumnTypeCol, tColumnTypeCol);
        CellRangeAddressList tResponseFalseColumnTypeRegions = new CellRangeAddressList(tResponseFalseFirstRow, tResponseFalseLastRow, tColumnTypeCol, tColumnTypeCol);
        CellRangeAddressList tRequestDataTypeRegions = new CellRangeAddressList(tRequestFirstRow, tRequestLastRow, tDataTypeCol, tDataTypeCol);
        CellRangeAddressList tResponseSuccessDataTypeRegions = new CellRangeAddressList(tResponseSuccessFirstRow, tResponseSuccessLastRow, tDataTypeCol, tDataTypeCol);
        CellRangeAddressList tResponseFalseDataTypeRegions = new CellRangeAddressList(tResponseFalseFirstRow, tResponseFalseLastRow, tDataTypeCol, tDataTypeCol);
        CellRangeAddressList tRequestIsRequiredRegions = new CellRangeAddressList(tRequestFirstRow, tRequestLastRow, tIsRequiredCol, tIsRequiredCol);
        CellRangeAddressList tResponseSuccessIsRequiredRegions = new CellRangeAddressList(tResponseSuccessFirstRow, tResponseSuccessLastRow, tIsRequiredCol, tIsRequiredCol);
        CellRangeAddressList tResponseFalseIsRequiredRegions = new CellRangeAddressList(tResponseFalseFirstRow, tResponseFalseLastRow, tIsRequiredCol, tIsRequiredCol);
        // Datakey.
        CellRangeAddressList tRequestIsDatakeyRegions = new CellRangeAddressList(tRequestFirstRow, tRequestLastRow, tIsDatakeyCol, tIsDatakeyCol);
        CellRangeAddressList tResponseSuccessIsDatakeyRegions = new CellRangeAddressList(tResponseSuccessFirstRow, tResponseSuccessLastRow, tIsDatakeyCol, tIsDatakeyCol);
        CellRangeAddressList tResponseFalseIsDatakeyRegions = new CellRangeAddressList(tResponseFalseFirstRow, tResponseFalseLastRow, tIsDatakeyCol, tIsDatakeyCol);
        // 為數組is_array.
        CellRangeAddressList tRequestIsArrayRegions = new CellRangeAddressList(tRequestFirstRow, tRequestLastRow, tIsArrayCol, tIsArrayCol);
        CellRangeAddressList tResponseSuccessIsArrayRegions = new CellRangeAddressList(tResponseSuccessFirstRow, tResponseSuccessLastRow, tIsArrayCol, tIsArrayCol);
        CellRangeAddressList tResponseFalseIsArrayRegions = new CellRangeAddressList(tResponseFalseFirstRow, tResponseFalseLastRow, tIsArrayCol, tIsArrayCol);
        CellRangeAddressList tCFRegions = new CellRangeAddressList();
        tCFRegions.addCellRangeAddress(tRequestFirstRow, tCFCol, tRequestLastRow, tCFCol);
        tCFRegions.addCellRangeAddress(tResponseSuccessFirstRow, tCFCol, tResponseSuccessLastRow, tCFCol);
        tCFRegions.addCellRangeAddress(tResponseFalseFirstRow, tCFCol, tResponseFalseLastRow, tCFCol);
        /* 生成下拉框内容 */
        XSSFDataValidationHelper helper = new XSSFDataValidationHelper(pNewSheet);
        // API类型
        String[] tApiTypeList = ApiTypeConstants.API_TYPE_ARRAY;
        DataValidationConstraint tApiTypeConstraint = helper.createExplicitListConstraint(tApiTypeList);
        // 请求方法
        String[] tRequestMethodList = RequestMethodConstants.REQUEST_METHOD_ARRAY;
        DataValidationConstraint tRequestMethodConstraint = helper.createExplicitListConstraint(tRequestMethodList);
        // 調用模式.
        String[] tIsSyncTypeList = SyncTypeConstants.SYNC_TYPE_ARRAY;
        DataValidationConstraint tSyncTypeConstraint = helper.createExplicitListConstraint(tIsSyncTypeList);
        // 分頁模式.
        List<StandardApiPaging> tStandardApiPagingList = standardApiPagingDao.getAll();
        String[] tPagingList = new String[tStandardApiPagingList.size()];
        for (int i = 0; i < tStandardApiPagingList.size(); i++) {
            tPagingList[i] = tStandardApiPagingList.get(i).getId() + "." + tStandardApiPagingList.get(i).getNameZhTw();
        }
        DataValidationConstraint tPagingConstraint = helper.createExplicitListConstraint(tPagingList);
        // 須分包.
        String[] tIsBatchList = {"Y", "N"};
        DataValidationConstraint tIsBatchConstraint = helper.createExplicitListConstraint(tIsBatchList);
        // 信息大類.
        String[] tGroupList = getDropDownData(pDataSheet, 0);
        DataValidationConstraint tGroupConstraint = helper.createExplicitListConstraint(tGroupList);
        // 信息類別.
        List<StandardApiCategory> tStandardApiCategoryList = standardApiCategoryDao.getAll();
        String[] tCategoryList = new String[tStandardApiCategoryList.size()];
        for (int i = 0; i < tStandardApiCategoryList.size(); i++) {
            tCategoryList[i] = tStandardApiCategoryList.get(i).getId() + "." + tStandardApiCategoryList.get(i).getNameZhTw();
        }
        DataValidationConstraint tCategoryConstraint = helper.createExplicitListConstraint(tCategoryList);

        String[] tMsgFormatlist = getDropDownData(pDataSheet, 2);
        DataValidationConstraint tMsgFormatConstraint = helper.createExplicitListConstraint(tMsgFormatlist);

        String[] tColumnTypelist = getDropDownData(pDataSheet, 3);
        DataValidationConstraint tColumnTypeConstraint = helper.createExplicitListConstraint(tColumnTypelist);

        String[] tDataTypelist = getDropDownData(pDataSheet, 4);
        DataValidationConstraint tDataTypeConstraint = helper.createExplicitListConstraint(tDataTypelist);

        String[] tIsRequiredlist = getDropDownData(pDataSheet, 5);
        DataValidationConstraint tIsRequiredConstraint = helper.createExplicitListConstraint(tIsRequiredlist);

        // Datakey.
        String[] tIsDatakeyList = {"Y"};
        DataValidationConstraint tIsDatakeyConstraint = helper.createExplicitListConstraint(tIsDatakeyList);

        // 為數組is_array.
        String[] tIsArrayList = {"Y", "N"};
        DataValidationConstraint tIsArrayConstraint = helper.createExplicitListConstraint(tIsArrayList);

        // 綁定下拉框和作用區域
        DataValidation tApiTypeValidation = helper.createValidation(tApiTypeConstraint, tApiTypeRegions); // API类型
        DataValidation tRequestMethodValidation = helper.createValidation(tRequestMethodConstraint, tRequestMethodRegions); // 请求方法
        DataValidation tSyncTypeValidation = helper.createValidation(tSyncTypeConstraint, tSyncTypeRegions);

        DataValidation tPagingValidation = helper.createValidation(tPagingConstraint, tPagingRegions);
        DataValidation tIsBatchValidation = helper.createValidation(tIsBatchConstraint, tIsBatchRegions);
        DataValidation tGroupValidation = helper.createValidation(tGroupConstraint, tGroupRegions);
        DataValidation tCategoryValidation = helper.createValidation(tCategoryConstraint, tCategoryRegions);
        DataValidation tMsgFormatValidation = helper.createValidation(tMsgFormatConstraint, tMsgFormatRegions);
        DataValidation tRequestColumnTypeValidation = helper.createValidation(tColumnTypeConstraint, tRequestColumnTypeRegions);
        DataValidation tResponseSuccessColumnTypeValidation = helper.createValidation(tColumnTypeConstraint, tResponseSuccessColumnTypeRegions);
        DataValidation tResponseFalseColumnTypeValidation = helper.createValidation(tColumnTypeConstraint, tResponseFalseColumnTypeRegions);
        DataValidation tRequestDataTypeValidation = helper.createValidation(tDataTypeConstraint, tRequestDataTypeRegions);
        DataValidation tResponseSuccessDataTypeValidation = helper.createValidation(tDataTypeConstraint, tResponseSuccessDataTypeRegions);
        DataValidation tResponseFalseDataTypeValidation = helper.createValidation(tDataTypeConstraint, tResponseFalseDataTypeRegions);
        DataValidation tRequestIsRequiredValidation = helper.createValidation(tIsRequiredConstraint, tRequestIsRequiredRegions);
        DataValidation tResponseSuccessIsRequiredValidation = helper.createValidation(tIsRequiredConstraint, tResponseSuccessIsRequiredRegions);
        DataValidation tResponseFalseIsRequiredValidation = helper.createValidation(tIsRequiredConstraint, tResponseFalseIsRequiredRegions);
        // Datakey.
        DataValidation tRequestIsDatakeyValidation = helper.createValidation(tIsDatakeyConstraint, tRequestIsDatakeyRegions);
        DataValidation tResponseSuccessIsDatakeyValidation = helper.createValidation(tIsDatakeyConstraint, tResponseSuccessIsDatakeyRegions);
        DataValidation tResponseFalseDatakeyValidation = helper.createValidation(tIsDatakeyConstraint, tResponseFalseIsDatakeyRegions);
        // 為數組is_array.
        DataValidation tRequestIsArrayValidation = helper.createValidation(tIsArrayConstraint, tRequestIsArrayRegions);
        DataValidation tResponseSuccessIsArrayValidation = helper.createValidation(tIsArrayConstraint, tResponseSuccessIsArrayRegions);
        DataValidation tResponseFalseArrayValidation = helper.createValidation(tIsArrayConstraint, tResponseFalseIsArrayRegions);
        /* 對sheet頁生效 */
        pNewSheet.addValidationData(tApiTypeValidation); // API类型
        pNewSheet.addValidationData(tRequestMethodValidation); // 请求方法
        pNewSheet.addValidationData(tSyncTypeValidation);
        pNewSheet.addValidationData(tPagingValidation);
        pNewSheet.addValidationData(tIsBatchValidation);
        pNewSheet.addValidationData(tGroupValidation);
        pNewSheet.addValidationData(tCategoryValidation);
        pNewSheet.addValidationData(tMsgFormatValidation);
        pNewSheet.addValidationData(tRequestColumnTypeValidation);
        pNewSheet.addValidationData(tResponseSuccessColumnTypeValidation);
        pNewSheet.addValidationData(tResponseFalseColumnTypeValidation);
        pNewSheet.addValidationData(tRequestDataTypeValidation);
        pNewSheet.addValidationData(tResponseSuccessDataTypeValidation);
        pNewSheet.addValidationData(tResponseFalseDataTypeValidation);
        pNewSheet.addValidationData(tRequestIsRequiredValidation);
        pNewSheet.addValidationData(tResponseSuccessIsRequiredValidation);
        pNewSheet.addValidationData(tResponseFalseIsRequiredValidation);
        pNewSheet.addValidationData(tRequestIsDatakeyValidation);
        pNewSheet.addValidationData(tResponseSuccessIsDatakeyValidation);
        pNewSheet.addValidationData(tResponseFalseDatakeyValidation);
        // 為數組is_array.
        pNewSheet.addValidationData(tRequestIsArrayValidation);
        pNewSheet.addValidationData(tResponseSuccessIsArrayValidation);
        pNewSheet.addValidationData(tResponseFalseArrayValidation);
        pNewSheet.getSheetConditionalFormatting().addConditionalFormatting(tCFRegions.getCellRangeAddresses(), pTemplateSheet.getSheetConditionalFormatting().getConditionalFormattingAt(0).getRule(0));
    }

    /**
     * 取得下拉選單資料 writeStandardAPI
     *
     * @param pDataSheet pDataSheet
     * @param pRowNumber pRowNumber
     */
    private String[] getDropDownData(XSSFSheet pDataSheet, int pRowNumber) {
        XSSFRow Row = pDataSheet.getRow(pRowNumber);
        int tCellCount = Row.getPhysicalNumberOfCells();
        String[] tCellList = new String[tCellCount - 1];

        for (int tCellNumber = 1; tCellNumber < tCellCount; tCellNumber++) {
            tCellList[tCellNumber - 1] = Row.getCell(tCellNumber).getStringCellValue();
        }
        return tCellList;
    }

    public XSSFWorkbook getExportTemplateFile() {
        XSSFWorkbook tXSSFWorkbook;
        try {
            String tTemplateFilePath = ApiMgmtApplicationParameter._FILE_TEMPLATE_PATH + "/" + ApiMgmtApplicationParameter._EXPORT_TEMPLATE_FILE_NAME;
            log.info("file path: " + tTemplateFilePath);
            tXSSFWorkbook = new XSSFWorkbook(new FileInputStream(tTemplateFilePath));
        } catch (Exception e) {
            log.error("Catch exception when export Get Template file.");
            log.error(e.getMessage());
            return null;
        }
        return tXSSFWorkbook;
    }

    public void setEasyApiInfo(XSSFSheet pDataNameSheet, Map<String, Object> pObjectMap) {
        XSSFCell tCell;
        // 從第1列開始寫
        XSSFRow tRow = pDataNameSheet.createRow(Integer.parseInt(pObjectMap.get(ApimgmtConstant.NUMERIC).toString()) + 1);
        // (0)服务名称
        int tColumnNumber = 0;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.NAME)));
        // (1)說服务说明
        tColumnNumber = 1;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.DESCRIPTION)));
        // (2)服务版本
        tColumnNumber = 2;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.VERSION)));
        // (3)API类型
        tColumnNumber = 3;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(ApiTypeConstants.getNameByCode(String.valueOf(pObjectMap.get(ApimgmtConstant.API_TYPE))));
        // (4)产品名称
        tColumnNumber = 4;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.PROJECT_NAME)));
        // (5)产品版本名称
        tColumnNumber = 5;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.PROJECT_VERSION_NAME_LIST_STR)));
        // (6)服务状态
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.APPROVED_STATUS)));
        // (7)定版日期
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApiAttributeEnum.approvedTime.toString())));
        // (8)备注
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.REMARK)));
        // (9)调用模式
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.SYNC_TYPE)));
        // (10)信息格式
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.MSG_FORMAT)));
        // (11)信息类别
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.CATEGORY_NAME)));
        // (12)发起方方产品
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.REQUESTER)));
        // (13)接收方产品
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.PROVIDER)));
        // (14)第一版申请人
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.FIRST_APPLICANT)));
        // (15)最新版本申请人
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.LAST_APPLICANT)));
        // (16)api租户Id
        tColumnNumber++;
        tCell = tRow.createCell(tColumnNumber);
        tCell.setCellValue(String.valueOf(pObjectMap.get(ApimgmtConstant.TENANT_ID)));
    }

    /**
     * writeStandardAPI
     *
     * @param pXSSFWorkbook       pXSSFWorkbook
     * @param pTemplateSheet      pTemplateSheet
     * @param pNewSheet           pNewSheet
     * @param pStandardApi        pStandardApi
     * @param pStandardApiVersion pStandardApiVersion
     */
    private void writeStandardAPI(XSSFWorkbook pXSSFWorkbook, XSSFSheet pTemplateSheet, XSSFSheet pNewSheet, StandardApi pStandardApi, StandardApiVersion pStandardApiVersion) {
        XSSFRow tNewRow;
        // copy title
        for (int tRowNum = 0; tRowNum < ApimgmtConstant.STANDARD_API_ROW; tRowNum++) {
            tNewRow = pNewSheet.createRow(tRowNum);
            XSSFRow tTemplateStandardAPI = pTemplateSheet.getRow(tRowNum);
            copyRow(pXSSFWorkbook, tTemplateStandardAPI, tNewRow, null, true);
        }
        tNewRow = pNewSheet.createRow(ApimgmtConstant.STANDARD_API_ROW);
        copyRow(pXSSFWorkbook, pTemplateSheet.getRow(ApimgmtConstant.STANDARD_API_ROW), tNewRow, null, false);
        // (0)空欄
        // (1)服務名稱 (2)服務說明(繁體) (3)服務說明(簡體) (4)服務說明(英文)
        // (5)服務版本 (6)調用模式 (7)分頁模式 (8)服務叫用方 (9)服務接收方 (10)須分包
        // (11)信息大類 (12)信息類別 (13)信息格式
        // (14)信息標籤(繁體) (15)信息標籤(簡體) (16)信息標籤(英文)
        // (17)備註(繁體) (18)備註(簡體) (19)備註(英文)
        // (20)租戶ID (21)修改自API (22)是否符合冪等性 (23)修改紀錄

        tNewRow.getCell(1).setCellValue(pStandardApi.getName());
        tNewRow.getCell(2).setCellValue(pStandardApi.getDescriptionZhTw());
        tNewRow.getCell(3).setCellValue(pStandardApi.getDescriptionZhCn());
        tNewRow.getCell(4).setCellValue(pStandardApi.getDescriptionEnUs());
        tNewRow.getCell(5).setCellValue(pStandardApiVersion.getVersion());
        tNewRow.getCell(6).setCellValue(SyncTypeConstants.getNameByCode(pStandardApi.getStandardApiSyncType().getId()));
        tNewRow.getCell(7).setCellValue(pStandardApi.getStandardApiPaging().getId() + ApimgmtConstant.DOT + pStandardApi.getStandardApiPaging().getNameZhTw());
        tNewRow.getCell(8).setCellValue(pStandardApi.getRequester());
        tNewRow.getCell(9).setCellValue(pStandardApi.getProvider());
        if (pStandardApi.getIsBatch()) {
            tNewRow.getCell(10).setCellValue("Y");
        } else {
            tNewRow.getCell(10).setCellValue("N");
        }
        tNewRow.getCell(11).setCellValue(pStandardApi.getStandardApiGroup().getId() + ApimgmtConstant.DOT + pStandardApi.getStandardApiGroup().getNameZhTw()); // 信息大類.
        tNewRow.getCell(12).setCellValue(pStandardApi.getStandardApiCategory().getId() + ApimgmtConstant.DOT + pStandardApi.getStandardApiCategory().getNameZhTw());
        tNewRow.getCell(13).setCellValue(pStandardApi.getMsgFormat());
        tNewRow.getCell(14).setCellValue(pStandardApi.getTagZhTw());
        tNewRow.getCell(15).setCellValue(pStandardApi.getTagZhCn());
        tNewRow.getCell(16).setCellValue(pStandardApi.getTagEnUs());
        tNewRow.getCell(17).setCellValue(pStandardApi.getRemarkZhTw());
        tNewRow.getCell(18).setCellValue(pStandardApi.getRemarkZhCn());
        tNewRow.getCell(19).setCellValue(pStandardApi.getRemarkEnUs());
        tNewRow.getCell(20).setCellValue(pStandardApi.getTenantId());
        tNewRow.getCell(21).setCellValue(pStandardApi.getPackageName());
        if (pStandardApiVersion.getIdempotency() != null) {
            if (pStandardApiVersion.getIdempotency()) {
                tNewRow.getCell(22).setCellValue("Y");
            } else {
                tNewRow.getCell(22).setCellValue("N");
            }
        } else {
            tNewRow.getCell(22).setCellValue("N");
        }
        if (pStandardApiVersion.getUpdateHistory() == null) {
            tNewRow.getCell(23).setCellValue("");
        } else {
            tNewRow.getCell(23).setCellValue(pStandardApiVersion.getUpdateHistory());
        }
        // (24)API类型
        int tColumnNumber = 24;
        tNewRow.getCell(tColumnNumber).setCellValue(ApiTypeConstants.getNameByCode(pStandardApi.getApiType()));
        if (ApimgmtConstant.API_TYPE_OPENAPI.equals(pStandardApi.getApiType())) {
            // (25)请求路径
            tColumnNumber ++;
            tNewRow.getCell(tColumnNumber).setCellValue(pStandardApi.getApiPath());
            // (26)请求方法
            tColumnNumber ++;
            tNewRow.getCell(tColumnNumber).setCellValue(RequestMethodConstants.getNameByCode(pStandardApi.getRequestMethod()));
        }
        // (27)关联产品名称
        tColumnNumber = 27;
        if (CollUtil.isEmpty(pStandardApi.getProjectVersionRelations())) {
            tNewRow.getCell(tColumnNumber).setCellValue(ApimgmtConstant.DEFAULT_PROJECT_NAME);
            tNewRow.getCell(tColumnNumber+1).setCellValue("");
        } else {
            Long projectId = pStandardApi.getProjectVersionRelations().get(0).getProjectId();
            Project project = projectDao.get(projectId);
            tNewRow.getCell(tColumnNumber).setCellValue(project.getProjectName());

            List<Long> idList = pStandardApi.getProjectVersionRelations().stream()
                    .map(ProjectVersionRelation::getProjectVersionId)
                    .distinct().collect(Collectors.toList());
            List<ProjectVersion> projectVersionList = projectVersionService.queryProjectVersionByIdList(idList);
            List<String> nameList = projectVersionList.stream()
                    .map(ProjectVersion::getProjectVersionName)
                    .collect(Collectors.toList());
            String joinedString = String.join(ApimgmtConstant.SPLITER, nameList);
            tNewRow.getCell(tColumnNumber+1).setCellValue(joinedString);
        }

    }

    /**
     * writeStandardAPIDataName
     *
     * @param pXSSFWorkbook         pXSSFWorkbook
     * @param pTemplateSheet        pTemplateSheet
     * @param pNewSheet             pNewSheet
     * @param pMsgTypeMap           pMsgTypeMap
     * @param pStandardApiDataNames pStandardApiDataNames
     */
    private void writeStandardAPIDataName(XSSFWorkbook pXSSFWorkbook, XSSFSheet pTemplateSheet, XSSFSheet pNewSheet, Map<Integer, List<StandardApiDataName>> pMsgTypeMap, List<StandardApiDataName> pStandardApiDataNames) {
        // title是用複製的 計算各個title的開始與結束列(Request、ResponseSuccess與ResponseFalse)
        final int tRequestTitleRow = 3;
        final int tRequestRow = tRequestTitleRow + 3;
        final int tResponseSuccessTitleRow = tRequestRow + pMsgTypeMap.get(1).size();
        final int tResponseSuccessRow = tResponseSuccessTitleRow + 3;
        final int tResponseFalseTitleRow = tResponseSuccessRow + pMsgTypeMap.get(2).size();
        final int tResponseFalseRow = tResponseFalseTitleRow + 3;
        XSSFRow tNewRow;
        // 樣版sheet的title列
        int tTemplateRequestTitleRow = 3;
        int tTemplateResponseSuccessTitleRow = 36;
        int tTemplateResponseFalseTitleRow = 54;
        int tTemplateStandardApiDataNameRow = 6;
        TreeSet<String> tTreeSet = collectPrdNames(pStandardApiDataNames);
        Object[] tArray = tTreeSet.toArray();
        int tStartTitleRow = 0;
        int tStartContentRow = 0;
        int tStartRow;
        int tTemplateTitleRow = 0;
        for (Integer tInteger : pMsgTypeMap.keySet()) {
            List<StandardApiDataName> tStandardApiDataNames = pMsgTypeMap.get(tInteger);
            switch (tInteger) {
                case StandardApiMessageTypeConstants.REQUEST:
                    tStartTitleRow = tRequestTitleRow;
                    tStartContentRow = tRequestRow;
                    tTemplateTitleRow = tTemplateRequestTitleRow;
                    break;
                case StandardApiMessageTypeConstants.RESPONSE_SUCCESS:
                    tStartTitleRow = tResponseSuccessTitleRow;
                    tStartContentRow = tResponseSuccessRow;
                    tTemplateTitleRow = tTemplateResponseSuccessTitleRow;
                    break;
                case StandardApiMessageTypeConstants.RESPONSE_FAILED:
                    tStartTitleRow = tResponseFalseTitleRow;
                    tStartContentRow = tResponseFalseRow;
                    tTemplateTitleRow = tTemplateResponseFalseTitleRow;
                    break;
                default:
                    break;
            }
            // copy title
            for (tStartRow = tStartTitleRow; tStartRow < tStartContentRow; tStartRow++, tTemplateTitleRow++) {
                tNewRow = pNewSheet.createRow(tStartRow);
                XSSFRow tTemplateStandardAPI = pTemplateSheet.getRow(tTemplateTitleRow);
                copyRow(pXSSFWorkbook, tTemplateStandardAPI, tNewRow, null, true);
                if (tStartRow + 1 == tStartContentRow) {
                    int tSize = tArray.length;
                    for (int ii = 0; ii < tSize; ii++) {
                        if (null == tNewRow.getCell(22 + ii)) {
                            tNewRow.createCell(22 + ii);
                        }
                        tNewRow.getCell(22 + ii).setCellValue(((String) tArray[ii]));
                    }
                }
            }
            pNewSheet.addMergedRegion(new CellRangeAddress(tStartTitleRow, tStartTitleRow, 1, 2));
            pNewSheet.addMergedRegion(new CellRangeAddress(tStartTitleRow + 1, tStartTitleRow + 1, 1, 3));
            // APIDataName只建一次樣式重複使用
            XSSFCellStyle tAPIDataNameStyle = newAPIDataNameStyle(pXSSFWorkbook, pTemplateSheet.getRow(tTemplateStandardApiDataNameRow));
            int tIndex;
            for (tStartRow = tStartContentRow, tIndex = 0; tIndex < tStandardApiDataNames.size(); tStartRow++, tIndex++) {
                tNewRow = pNewSheet.createRow(tStartRow);
                copyRow(pXSSFWorkbook, pTemplateSheet.getRow(tTemplateStandardApiDataNameRow), tNewRow, tAPIDataNameStyle, false);
                StandardApiDataName tStandardApiDataName = tStandardApiDataNames.get(tIndex);
                // (0)(原空欄改成)is_businesskey
                // (1)類型 (2)所屬單頭/身 (3)辭彙檢查 (4)辭彙代號 (5)辭彙型態 (6)可為空
                // (7)為Datakey (8)為數組(array) (9)可排序
                // (10)說明(繁體) (11)說明(簡體) (12)說明(英文) (13)範例值
                // (14)備註(繁體) (15)備註(簡體) (16)備註(英文)
                // (17)列舉值(繁體) (18)列舉值(簡體) (19)列舉值(英文)
                // (20)修改紀錄 (21)長度精度

                // (0)(原空欄改成)is_businesskey
                if (tStandardApiDataName.getIsBusinesskey() != null && tStandardApiDataName.getIsBusinesskey()) {
                    tNewRow.getCell(0).setCellValue("Y");
                }
                // (1)類型.
                tNewRow.getCell(1).setCellValue(tStandardApiDataName.getColumnType());
                // (2)所屬單頭/身.
                if (tStandardApiDataName.getParent() != null) {
                    tNewRow.getCell(2).setCellValue(tStandardApiDataName.getParent().getStandardDataName());
                }
                // (3)辭彙檢查(複製公式)
                String tDataNameSheetName = pXSSFWorkbook.getSheetAt(ApimgmtConstant.AUDIT_USE_DATA_NAME_SHEET_NUMBER).getSheetName();
                int tCheckRow = tStartRow + 1;
                /*
                 * IF( OR(
                 * COUNTIF(辭彙清單_審核!A:A,E7),AND(OR(E7="EntId",E7="CompanyId",E7=
                 * "PlantId"),B7= "datakey"),B7="H" ),"","NEW" )
                 **/
                String tFormulaFormat = "IF(ISBLANK(E%2$d),\"NEW\",IF(OR(SUMPRODUCT(--(EXACT(%1$s!A:A,E%2$d))),AND(OR(E%2$d=\"EntId\",E%2$d=\"CompanyId\",E%2$d=\"PlantId\"),B%2$d=\"datakey\"),B%2$d=\"H\"),\"\",\"NEW\"))";
                String tCheckDataNameFormula = String.format(tFormulaFormat, tDataNameSheetName, tCheckRow);
                tNewRow.getCell(3).setCellFormula(tCheckDataNameFormula);
                // (4)辭彙代號.
                tNewRow.getCell(4).setCellValue(tStandardApiDataName.getStandardDataName());
                // (5)辭彙型態.
                tNewRow.getCell(5).setCellValue(tStandardApiDataName.getDataType());

                // (6) 必要.
                if (tStandardApiDataName.getIsRequired()) {
                    tNewRow.getCell(6).setCellValue(IsRequiredEnum.Y.toString());
                } else {
                    tNewRow.getCell(6).setCellValue(IsRequiredEnum.N.toString());
                }

                // (7)為Datakey
                if (tStandardApiDataName.getIsDatakey() != null && tStandardApiDataName.getIsDatakey()) {
                    tNewRow.getCell(7).setCellValue("Y");
                }
                // (8)為數組.
                if (tStandardApiDataName.getIsArray() != null && tStandardApiDataName.getIsArray()) {
                    tNewRow.getCell(8).setCellValue("Y");
                }
                // (9)可排序.
                if (tStandardApiDataName.getCanSort() != null && tStandardApiDataName.getCanSort()) {
                    tNewRow.getCell(9).setCellValue("Y");
                }
                // (10)可筛选.
                if (tStandardApiDataName.getCanFilter() != null && tStandardApiDataName.getCanFilter()) {
                    tNewRow.getCell(10).setCellValue("Y");
                }

                // (10)說明(繁體).
                tNewRow.getCell(11).setCellValue(tStandardApiDataName.getDescriptionZhTw());
                // (11)說明(簡體).
                tNewRow.getCell(12).setCellValue(tStandardApiDataName.getDescriptionZhCn());
                // (12)說明(英文).
                tNewRow.getCell(13).setCellValue(tStandardApiDataName.getDescriptionEnUs());
                // (13)範例值.
                tNewRow.getCell(14).setCellValue(tStandardApiDataName.getValue());
                // (14)備註(繁體).
                if (tStandardApiDataName.getRemarkZhTw() != null) {
                    tNewRow.getCell(15).setCellValue(tStandardApiDataName.getRemarkZhTw());
                }
                // (15)備註(簡體).
                if (tStandardApiDataName.getRemarkZhCn() != null) {
                    tNewRow.getCell(16).setCellValue(tStandardApiDataName.getRemarkZhCn());
                }
                // (16)備註(英文).
                if (tStandardApiDataName.getRemarkEnUs() != null) {
                    tNewRow.getCell(17).setCellValue(tStandardApiDataName.getRemarkEnUs());
                }
                // (17)列舉值(繁體).
                if (tStandardApiDataName.getListEnumZhTw() != null) {
                    tNewRow.getCell(18).setCellValue(tStandardApiDataName.getListEnumZhTw());
                }
                // (18)列舉值(簡體).
                if (tStandardApiDataName.getListEnumZhTw() != null) {
                    tNewRow.getCell(19).setCellValue(tStandardApiDataName.getListEnumZhCn());
                }
                // (19)列舉值(英文).
                if (tStandardApiDataName.getListEnumZhTw() != null) {
                    tNewRow.getCell(20).setCellValue(tStandardApiDataName.getListEnumEn());
                }
                // (20)修改紀錄
                if (tStandardApiDataName.getUpdateHistory() == null) {
                    tNewRow.getCell(21).setCellValue("");
                } else {
                    tNewRow.getCell(21).setCellValue(tStandardApiDataName.getUpdateHistory());
                }
                // (21)後為長度精度
                for (int ii = 0; ii < tArray.length; ii++) {
                    ProductApiDataName[] tProductApiDataNames = filterProductApiDataName(tStandardApiDataName.getProductApiDataNames(), (String) tArray[ii]);
                    if (0 < tProductApiDataNames.length && null != tProductApiDataNames[0].getLength()) {
                        String tStrLi = "";
                        if ("string".equals(tStandardApiDataName.getDataType())) {
                            tStrLi = tProductApiDataNames[0].getLength().toString();
                        } else if ("numeric".equals(tStandardApiDataName.getDataType())) {
                            tStrLi = tProductApiDataNames[0].getLength() + "," + tProductApiDataNames[0].getPrecision();
                        }
                        if (null == tNewRow.getCell(22 + ii)) {
                            tNewRow.createCell(22 + ii);
                        }
                        tNewRow.getCell(22 + ii).setCellValue(tStrLi);
                    }
                }
            }
        }
    }

    /**
     * 設定欄寬
     *
     * @param pTemplateSheet pTemplateSheet
     * @param pNewSheet      pNewSheet
     */
    private void setSheetWidthAndHight(XSSFSheet pTemplateSheet, XSSFSheet pNewSheet) {
        for (int tColumnNumber = 0; tColumnNumber <= ApimgmtConstant.TOTAL_COLUMN_NUMBER; tColumnNumber++) {
            pNewSheet.setColumnWidth(tColumnNumber, pTemplateSheet.getColumnWidth(tColumnNumber));
        }
    }

    /**
     * 複製行
     *
     * @param pWorkbook      pWorkbook
     * @param pTemplateRow   pTemplateRow
     * @param pNewRow        pNewRow
     * @param pCopyValueFlag pCopyValueFlag
     */
    private void copyRow(XSSFWorkbook pWorkbook, XSSFRow pTemplateRow, XSSFRow pNewRow, XSSFCellStyle pNewstyle, boolean pCopyValueFlag) {
        for (Iterator<Cell> tCellIt = pTemplateRow.cellIterator(); tCellIt.hasNext(); ) {
            XSSFCell tTemplateCell = (XSSFCell) tCellIt.next();
            XSSFCell tNewCell = pNewRow.createCell(tTemplateCell.getColumnIndex());
            copyCell(pWorkbook, tTemplateCell, tNewCell, pNewstyle, pCopyValueFlag);
        }
    }

    /**
     * @param pXSSFWorkbook                            pXSSFWorkbook
     * @param pTemplateCell                            pTemplateCell
     * @param pNewCell                                 pNewCell
     * @param pNewstyle(有傳值是StandardAPIDataName的單元格樣式) pNewstyle
     * @param copyValueFlag(true連同cell的内容一起複製)         copyValueFlag
     */
    private void copyCell(XSSFWorkbook pXSSFWorkbook, XSSFCell pTemplateCell, XSSFCell pNewCell, XSSFCellStyle pNewstyle, boolean copyValueFlag) {
        if (pNewstyle != null) {
            if (pNewCell.getColumnIndex() < 22) { // 長度精度不需套用樣式
                pNewCell.setCellStyle(pNewstyle);
            }
        } else {
            XSSFCellStyle tNewstyle = pXSSFWorkbook.createCellStyle();
            tNewstyle.cloneStyleFrom(pTemplateCell.getCellStyle());
            tNewstyle.setWrapText(true);
            pNewCell.setCellStyle(tNewstyle);
        }
        // 超連結
        if (pTemplateCell.getHyperlink() != null) {
            pNewCell.setHyperlink(pTemplateCell.getHyperlink());
        }
        // 是否複製值
        if (copyValueFlag) {
            pNewCell.setCellValue(pTemplateCell.getRichStringCellValue());
        }
    }

    private TreeSet<String> collectPrdNames(List<StandardApiDataName> pStandardApiDataNames) {
        TreeSet<String> tPrdNms = Sets.newTreeSet();
        for (StandardApiDataName tStandardApiDataName : pStandardApiDataNames) {
            List<ProductApiDataName> tProductApiDataNames = tStandardApiDataName.getProductApiDataNames();
            for (ProductApiDataName tProductApiDataName : tProductApiDataNames) {
                String tPrdNm = tProductApiDataName.getProduct().getName();
                tPrdNms.add(tPrdNm);
            }
        }
        return tPrdNms;
    }

    private ProductApiDataName[] filterProductApiDataName(List<ProductApiDataName> pList, final String pPrdName) {
        return CollUtil.emptyIfNull(pList).stream().filter(tProductApiDataName -> pPrdName.equals(tProductApiDataName.getProduct().getName())).toArray(ProductApiDataName[]::new);
    }

    private XSSFCellStyle newAPIDataNameStyle(XSSFWorkbook pXSSFWorkbook, XSSFRow pTemplateRow) {
        XSSFCellStyle tNewstyle = pXSSFWorkbook.createCellStyle();
        XSSFCell tCell = pTemplateRow.getCell(0);
        tNewstyle.cloneStyleFrom(tCell.getCellStyle());
        tNewstyle.setWrapText(true);
        return tNewstyle;
    }

}
