package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtProjectDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtProjectVersionDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtTenantConfigDao;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.LocaleEnum;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.model.*;
import com.digiwin.athena.apimgmt.service.model.Page;
import com.digiwin.athena.apimgmt.util.StandardApiMessageUtil;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.github.fge.jackson.JsonLoader;
import cn.hutool.core.collection.CollUtil;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 服务标准API查询,设计API查询-用于进阶查询、列表查询
 * 4合一接口
 * /restful/standard/apimgmt/AdvanceSearchApiListOther/Get
 */
@Slf4j
@Service
public class AdvanceSearchApiListGetOtherService {

    @Autowired
    private ApiMgmtStandardApiDao standardApiDao;

    @Autowired
    private ApiMgmtTenantConfigDao tenantConfigDao;

    ObjectMapper mapper = new ObjectMapper();
    @Autowired
    private ApiMgmtProjectDao projectDao;

    @Autowired
    private ApiMgmtProjectVersionDao projectVersionDao;

    public AdvanceSearchApiListGetOtherService() {
        super();
    }

    /**
     * 进阶查询
     *
     * @param validatorResult validatorResult
     * @return arraylist
     * @throws Exception Exception
     */
    public Map<String, Object> execute(String validatorResult) throws Exception {
        log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass() + ", 原始訊息: " + validatorResult);
        // 取得語系
        String tLocale = ApiMgmtServiceContextHolder.getLocale();
        // header沒傳語系的話，默認回傳英文
        tLocale = tLocale == null ? LocaleEnum.EN_US.getType() : tLocale;
        ObjectNode tResponse = mapper.createObjectNode();
        JsonNode tRequestJsonNode = JsonLoader.fromString(validatorResult);
        // 进阶查询检索条件
        ArrayNode tConditionArrayNode = (ArrayNode) tRequestJsonNode.get(ApimgmtConstant.CONDITION_LIST);
        ArrayNode simpleConditionArrayNode = (ArrayNode) tRequestJsonNode.get(ApimgmtConstant.SIMPLE_CONDITION_LIST);
        // 标准API列表查询入参
        JsonNode tCategoryId = tRequestJsonNode.get(ApimgmtConstant.CATEGORY_ID);
        // 产品ID
        JsonNode projectId = tRequestJsonNode.get(ApimgmtConstant.PROJECT_ID);
        //产品版本ID
        JsonNode projectVersionId = tRequestJsonNode.get(ApimgmtConstant.PROJECT_VERSION_ID);
        // 设计API列表查询入参
        JsonNode tStatusId = tRequestJsonNode.get(ApimgmtConstant.STATUS_ID);
        // 查询类型: true:设计api查询、false:标准api查询
        JsonNode tType = tRequestJsonNode.get(ApimgmtConstant.TYPE_STRING);
        // 分页讯息
        JsonNode pageNum = tRequestJsonNode.get(ApimgmtConstant.PAGE_NUM);
        int pPageNum = null != pageNum ? pageNum.asInt() : 1;
        JsonNode pageSize = tRequestJsonNode.get(ApimgmtConstant.PAGE_SIZE);
        int pPageSize = null != pageSize ? pageSize.asInt() : 10;
        if (null == tType || StringUtil.isEmptyOrSpace(tType.asBoolean())) {
            throw new Exception("请选择查询类型!");
        }
        String tToken = ApiMgmtServiceContextHolder.getToken();
        // 设计API查询-必须登录
        if (tType.asBoolean() && null == tToken) {
            throw new Exception("当前用户未登录,请登录!");
        }
        String tTenantId = ApiMgmtServiceContextHolder.getTenantId() ;
        String teamType = ApiMgmtServiceContextHolder.getTeamType();
        String tUserId = ApiMgmtServiceContextHolder.getUserId();
        // 标准API查询-判断租戶是否可查看全部API
        boolean tViewAllApi = false;
        if (!tType.asBoolean() && !StringUtil.isEmptyOrSpace(tTenantId)) {
            tViewAllApi = tenantConfigDao.viewAllApiTenantExist(tTenantId);
        }
        // 查询条件
        List<List<AdvanceSearchValue>> arrayList = StandardApiMessageUtil.generateQuery(tConditionArrayNode, tCategoryId, tStatusId, tType, projectId, projectVersionId, tUserId);
        // 查询版本信息
        Long startTime = System.currentTimeMillis();
        Page<StandardApiVersion> standardApiVersionPage = standardApiDao.advanceApiSearch(arrayList, teamType, tTenantId, tType.asBoolean(), tViewAllApi, pPageNum, pPageSize);
        Long idsTime = System.currentTimeMillis();
        log.info("分页查询数据库请求处理时间" + (idsTime - startTime));
        setAdvanceSearchApiResponse(standardApiVersionPage, tLocale, tResponse, tType.asBoolean());
        Long endTime = System.currentTimeMillis();
        log.info("分页查询数据拼接请求处理时间" + (endTime - idsTime));
        return mapper.convertValue(tResponse, new TypeReference<Map<String, Object>>() {
        });
    }

    public void setProjectAndVersion(List<StandardApi> apiList){
        //获取项目
        List<Long> projectIds = apiList.stream()
                .flatMap(api -> api.getProjectVersionRelations().stream())
                .map(ProjectVersionRelation::getProjectId)
                .distinct()
                .collect(Collectors.toList());
        //获取版本
        List<Long> projectVersionIds = apiList.stream()
                .flatMap(api -> api.getProjectVersionRelations().stream())
                .map(ProjectVersionRelation::getProjectVersionId)
                .distinct()
                .collect(Collectors.toList());
        //填充属性
        Map<Long, Project> projectMap = projectDao.queryByIdList(projectIds).stream()
                .collect(Collectors.toMap(Project::getProjectId, project -> project));
        Map<Long, ProjectVersion> projectVersionMap = projectVersionDao.queryProjectVersionByIdList(projectVersionIds).stream()
                .collect(Collectors.toMap(ProjectVersion::getProjectVersionId, projectVersion -> projectVersion));

        Set<Long> existVersionsProjectIds = projectVersionDao.selectExistProjectId(projectIds);
        apiList.forEach(api -> {
            if (CollUtil.isNotEmpty(api.getProjectVersionRelations())) {
                ProjectVersionRelation versionRelation = api.getProjectVersionRelations().get(0);
                api.setProjectName(projectMap.get(versionRelation.getProjectId()).getProjectName());

                List<String> projectVersiontNameList = new ArrayList<>();
                api.getProjectVersionRelations().forEach(projectVersionRelation -> {
                    if (ObjectUtil.isNotNull(projectVersionRelation.getProjectVersionId())) {
                        ProjectVersion projectVersion = projectVersionMap.get(projectVersionRelation.getProjectVersionId());
                        projectVersiontNameList.add(projectVersion.getProjectVersionName());
                    }
                });

                api.setProjectVersionNameList(projectVersiontNameList);
                api.setHasVersions(existVersionsProjectIds.contains(versionRelation.getProjectId()));
            }
        });
    }

    private void setAdvanceSearchApiResponse(Page<StandardApiVersion> pStandardApiVersionPage, String tLocale, ObjectNode tResponse, boolean tType) {
        ArrayNode arrayNode = mapper.createArrayNode();
        if (CollUtil.isNotEmpty(pStandardApiVersionPage.getList())) {
            List<StandardApi> standardApiList = pStandardApiVersionPage.getList().stream().map(StandardApiVersion::getStandardApi).collect(Collectors.toList());
            setProjectAndVersion(standardApiList);
            for (StandardApiVersion tStandardApiVersion : pStandardApiVersionPage.getList()) {
                StandardApi tStandardApi = tStandardApiVersion.getStandardApi();
                ObjectNode tApiMap = mapper.createObjectNode();
                // 检视所有版本
                if (!tType) {
                    // 最大1.0版本说明只有1.0，最大不是1.0则查询1.0的版本
                    if (tStandardApiVersion.getVersion().equals("1.0")) {
                        tApiMap.put(ApimgmtConstant.FIRST_APPLICANT, tStandardApiVersion.getApplicant());
                        tApiMap.put(ApimgmtConstant.LAST_APPLICANT, "");
                    } else {
                        tApiMap.put(ApimgmtConstant.FIRST_APPLICANT, tStandardApi.getFirstApplicant());
                        tApiMap.put(ApimgmtConstant.LAST_APPLICANT, tStandardApiVersion.getApplicant());
                    }
                }
                String tApprovedStatus;
                long tApprovedStatusCode;
                String tDescription;
                String tRemark;
                String tSyncType;
                String tCategory;
                switch (tLocale) {
                    case ApimgmtConstant.ZH_CN:
                        tApprovedStatus = tStandardApiVersion.getApprovedStatus().getNameZhCn();
                        tApprovedStatusCode = tStandardApiVersion.getApprovedStatus().getId();
                        tDescription = tStandardApi.getDescriptionZhCn();
                        tRemark = tStandardApi.getRemarkZhCn();
                        tSyncType = tStandardApi.getStandardApiSyncType().getNameZhCn();
                        tCategory = tStandardApi.getStandardApiCategory().getNameZhCn();
                        break;
                    case ApimgmtConstant.ZH_TW:
                        tApprovedStatus = tStandardApiVersion.getApprovedStatus().getNameZhTw();
                        tApprovedStatusCode = tStandardApiVersion.getApprovedStatus().getId();
                        tDescription = tStandardApi.getDescriptionZhTw();
                        tRemark = tStandardApi.getRemarkZhTw();
                        tSyncType = tStandardApi.getStandardApiSyncType().getNameZhTw();
                        tCategory = tStandardApi.getStandardApiCategory().getNameZhTw();
                        break;
                    case ApimgmtConstant.EN_US:
                    default:
                        tApprovedStatus = tStandardApiVersion.getApprovedStatus().getNameEnUs();
                        tApprovedStatusCode = tStandardApiVersion.getApprovedStatus().getId();
                        tDescription = tStandardApi.getDescriptionEnUs();
                        tRemark = tStandardApi.getRemarkEnUs();
                        tSyncType = tStandardApi.getStandardApiSyncType().getNameEnUs();
                        tCategory = tStandardApi.getStandardApiCategory().getNameEnUs();
                        break;
                }
                // 組成回傳訊息
                tApiMap.put(ApimgmtConstant.API_ID, tStandardApi.getId());
                tApiMap.put(ApimgmtConstant.NAME, tStandardApi.getName());
                tApiMap.put(ApimgmtConstant.VERSION, tStandardApiVersion.getVersion());
                tApiMap.put(ApimgmtConstant.APPROVED_STATUS, tApprovedStatus);
                tApiMap.put(ApiAttributeEnum.approvedTime.toString(), DateUtil.formatLocalDateTime(tStandardApiVersion.getApprovedTime()));
                tApiMap.put(ApimgmtConstant.APPROVED_STATUS_CODE, tApprovedStatusCode);
                tApiMap.put(ApimgmtConstant.DESCRIPTION, tDescription);
                tApiMap.put(ApimgmtConstant.REMARK, tRemark);
                tApiMap.put(ApimgmtConstant.SYNC_TYPE, tSyncType);
                tApiMap.put(ApimgmtConstant.CATEGORY_ID, tStandardApi.getStandardApiCategory().getId());
                tApiMap.put(ApimgmtConstant.CATEGORY_NAME, tCategory);
                tApiMap.put(ApimgmtConstant.MSG_FORMAT, tStandardApi.getMsgFormat());
                tApiMap.put(ApimgmtConstant.REQUESTER, tStandardApi.getRequester());
                tApiMap.put(ApimgmtConstant.PROVIDER, tStandardApi.getProvider());
                tApiMap.put(ApimgmtConstant.TENANT_ID, tStandardApi.getTenantId());
                tApiMap.put(ApimgmtConstant.API_BUILD_ACCT, tStandardApi.getBuildAcct());
                tApiMap.put(ApimgmtConstant.API_BUILD_TIME, DateUtil.formatLocalDateTime(tStandardApi.getBuildTime()));
                tApiMap.put(ApimgmtConstant.DESIGN_TENANT_ID, tStandardApiVersion.getStandardApi().getDesignTenantId());
                tApiMap.put(ApimgmtConstant.TEAM_ID, tStandardApiVersion.getStandardApi().getTeamId());
                tApiMap.put(ApimgmtConstant.STATUS_ID, tStandardApiVersion.getApprovedStatus().getId());
                tApiMap.put(ApimgmtConstant.API_TYPE, tStandardApiVersion.getStandardApi().getApiType());
                tApiMap.put(ApimgmtConstant.PROJECT_NAME, tStandardApiVersion.getStandardApi().getProjectName());
                tApiMap.put(ApimgmtConstant.PROJECT_HAS_VERSIONS,tStandardApiVersion.getStandardApi().getHasVersions());

                if (tStandardApiVersion.getStandardApi().getProjectVersionNameList() != null) {
                    ArrayNode projectVersionNameListNode = mapper.createArrayNode();
                    tStandardApiVersion.getStandardApi().getProjectVersionNameList().forEach(name -> projectVersionNameListNode.add(name));
                    tApiMap.put(ApimgmtConstant.PROJECT_VERSION_NAME_LIST, projectVersionNameListNode);
                    tApiMap.put(ApimgmtConstant.PROJECT_VERSION_NAME_LIST_STR, String.join(ApimgmtConstant.SPLITER, tStandardApiVersion.getStandardApi().getProjectVersionNameList()));
                }
                arrayNode.add(tApiMap);
            }
            tResponse.put(ApimgmtConstant.PAGE_NUM, pStandardApiVersionPage.getPageNum());
            tResponse.put(ApimgmtConstant.PAGE_SIZE, pStandardApiVersionPage.getPageSize());
            tResponse.put(ApimgmtConstant.PAGE_TOTAL, pStandardApiVersionPage.getPageTotal());
            tResponse.set(ApimgmtConstant.VALUE, arrayNode);
        }
    }
}
