package com.digiwin.athena.apimgmt.apiservice;

import com.digiwin.athena.apimgmt.ApiMgmtApplicationParameter;
import com.digiwin.athena.apimgmt.annotate.OperateAuthorityVerify;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import com.digiwin.athena.apimgmt.dao.*;
import com.digiwin.athena.apimgmt.enums.APIExportImportEnums;
import com.digiwin.athena.apimgmt.enums.LocaleEnum;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.exception.TeamIdNotFindValidationException;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.model.*;
import com.digiwin.athena.apimgmt.service.util.*;
import com.digiwin.athena.apimgmt.services.ApiMgmtApiVersionService;
import com.digiwin.athena.apimgmt.services.ApiMgmtProductCategoryService;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import cn.hutool.core.collection.CollUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Sheet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.FileOutputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 保存API规格
 * /restful/standard/apimgmt/ApiSpec/Save
 */
@Slf4j
@Service
@OperateAuthorityVerify
@Transactional
public class ApiSpecSaveService extends AbstractApiService {

    private static final Lock LOCK = new ReentrantLock(true);
    @Autowired
    private ApiMgmtProjectVersionRelationDao projectVersionRelationDao;

    @Autowired
    private ApiMgmtApiVersionService apiVersionService;

    @Autowired
    private ApiMgmtStandardApiDao standardApiDao;

    @Autowired
    private ApiMgmtStandardApiVersionDao standardApiVersionDao;

    @Autowired
    private ApiMgmtStandardDataNameDao standardDataNameDao;

    @Autowired
    private ApiMgmtTenantConfigDao tenantConfigDao;

    @Autowired
    private ApiMgmtProductCategoryService productCategoryService;

    public ApiSpecSaveService() {
        super();
        jsonSchemaFileName = ApimgmtSchemaConstant.API_SPEC_SAVE_SCHEMA;
    }

    /**
     * 当前接口用于新增跟编辑后得保存按钮-需要区分开
     *
     * @param validatorResult validatorResult
     * @return map
     */
    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) {
        // 取得語系
        String tLocale = ApiMgmtServiceContextHolder.getLocale();
        // header沒傳語系的話，默認回傳英文
        tLocale = tLocale == null ? LocaleEnum.EN_US.getType() : tLocale;
        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = tStateCode.getDescription();
        StringBuilder tDescriptionBuilder = new StringBuilder();
        // 建立response node
        ObjectNode tResponseNode = mapper.createObjectNode();
        // 全部API的錯誤總Map
        Map<String, List<Map<String, String>>> tTotalApiErrorMap = new HashMap<>();
        try {
            String tTeamId = ApiMgmtServiceContextHolder.getTeamId();
            String teamType = ApiMgmtServiceContextHolder.getTeamType();
            String tTenantId = ApiMgmtServiceContextHolder.getTenantId();
            String tUserId = ApiMgmtServiceContextHolder.getUserId();
            if (StringUtil.isEmpty(teamType)) {
                throw new TeamIdNotFindValidationException(this.getClass());
            }
            JsonNode tRequestJsonNode = validatorResult.getJsonContent();
            // 如果Tenant被設定為略過不檢查字段，則不進記憶體or資料庫查Data Name清單
            boolean tSkipReview = tenantConfigDao.getSkipReviewDataNameTenant(tTenantId);
            List<String> tDataNameList;
            LOCK.lock();
            try {
                if (!tSkipReview) {
                    tDataNameList = ApiDataNameServiceUtil.getApiDataNameList();
                    Long tDataNameCount = standardDataNameDao.getDataNameCount();
                    if (tDataNameList.size() != tDataNameCount) {
                        tDataNameList = standardDataNameDao.getAllDataName();
                        ApiDataNameServiceUtil.setApiDataNameList(tDataNameList);
                    }
                }
            } finally {
                LOCK.unlock();
            }
            // 讀取StandardAPI資訊
            List<Map<String, String>> tSetStdApiResult;
            StandardApi tStandardApi = new StandardApi();
            tSetStdApiResult = StandardApiSpecAnalyzeServiceUtil.setStandardApi(tStandardApi, tRequestJsonNode, tLocale);
            String tApiName = tStandardApi.getName();
            String tApiTenantId = tStandardApi.getTenantId();
            String tVersion = tStandardApi.getStandardApiVersions().get(0).getVersion();
            tStandardApi.setTeamId(tTeamId);
            tStandardApi.setTeamType(teamType);
            /*
             *  API 已存在的情况下
             *  1、区分是保存 还是 编辑
             *  2、编辑区分是API创建人、非创建人
             */
            List<StandardApiVersion> tStandardApiVersionList = standardApiVersionDao.getExistApiVersion(tApiName, tApiTenantId, tVersion);
            if (CollUtil.isNotEmpty(tStandardApiVersionList)) {
                StandardApiVersion tStandardApiVersion = tStandardApiVersionList.get(0);
                if (StringUtils.isBlank(tStandardApi.getFirstApplicant())) {
                    tStandardApi.setFirstApplicant(tStandardApiVersion.getStandardApi().getFirstApplicant());
                }
                //处于非草稿状态下或者编辑人与创建人不一致或者同一人不同租户抛出异常
                if (tStandardApiVersion.getApprovedStatus().getId() != ApimgmtConstant.APPROVED_STATUS_DRAFT
                        || !tStandardApiVersion.getUserId().equals(tUserId)
                        || (!StringUtil.isEmptyOrSpace(tStandardApiVersion.getStandardApi().getDesignTenantId())
                        && !tStandardApiVersion.getStandardApi().getDesignTenantId().equals(tTenantId))) {
                    StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.API_VERSION_IS_EXIST.toString(), null, null, 0, 0, tLocale);
                    String tApiError = tApiName + "/" + tVersion;
                    tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                    return result(tStateCode, tDescription, tTotalApiErrorMap);
                }
            }
            // 取得前一版StandardApiVersion
            String tPreviousVersion = ApiVersionServiceUtil.getPreviousVersion(tVersion);
            // 檢查前一版是否匯入，除了1.0不檢查 todo
            //if (Float.parseFloat(tVersion) > Float.parseFloat(ApimgmtConstant.API_VER)) {
            if (!apiVersionService.judgeNewBranch(tVersion)) {
                List<StandardApiVersion> tPreviStandardApiVersions = standardApiVersionDao.getExistApiVersion(tApiName, tApiTenantId, tPreviousVersion);
                if (tPreviStandardApiVersions.isEmpty()) {
                    StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.PREVI_API_VERSION_IS_EXIST.toString(), null, null, 0, 0, tLocale);
                } else {
                    StandardApi previousStandardApi = tPreviStandardApiVersions.get(0).getStandardApi();
                    ApprovedStatus previousApprovedStatus = tPreviStandardApiVersions.get(0).getApprovedStatus();
                    if (previousApprovedStatus.getId() != ApimgmtConstant.APPROVED_STATUS_COMFIRMED) {
                        StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.PREVI_IS_DEVELOPING.toString(), null, null, 0, 0, tLocale);
                    }
                    /*
                     * 判定父类api是否一致，不一致则抛出异常
                     * 1.0父类不存在 1.1父类不存在 ok、1.1父类存在 则异常
                     * 1.0父类存在 1.1父类不存在 ok、1.1父类存在且一致 ok、1.1父类存在且不一致 则异常
                     * 增加父类api租户的判定
                     */
                    if ((null != tStandardApi.getParentApiName()
                            && (null == previousStandardApi.getParentApiName()
                            || (null != previousStandardApi.getParentApiName()
                            && (!tStandardApi.getParentApiName().equals(previousStandardApi.getParentApiName())
                            || !tStandardApi.getParentBranch().equals(previousStandardApi.getParentBranch())
                            || !tStandardApi.getParentApiTenantId().equals(previousStandardApi.getParentApiTenantId())))))) {
                        StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.PARENT_NAME_INCONSISTENT.toString(), null, null, 0, 0, tLocale);
                    }
                    // 判定当前版本的调用模式是否与上一版一致
                    if (!previousStandardApi.getStandardApiSyncType().getId().equals(tStandardApi.getStandardApiSyncType().getId())) {
                        StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.SYNC_TYPE_CAN_NOT_CHANGE.toString(), null, null, 0, 0, tLocale);
                    }
                    // 判定当前版本的分包属性是否与上一版一致
                    if (!previousStandardApi.getIsBatch().equals(tStandardApi.getIsBatch())) {
                        StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.IS_BATCH_CAN_NOT_CHANGE.toString(), null, null, 0, 0, tLocale);
                    }
                    // 判定上一版本与当前token中的TenantId不一致并且teamType = 1 鼎捷租户
                    String designTenantId = previousStandardApi.getDesignTenantId();
                    if (!tTenantId.equals(designTenantId) && !"1".equals(teamType)) {
                        Map<String, String> errorMap = new HashMap<>();
                        errorMap.put(ApimgmtConstant.DESCRIPTION, "当前租户无权限执行当前操作");
                        errorMap.put(ApimgmtConstant.SOLUTION, "请使用租户ID：" + designTenantId + "执行当前操作");
                        tSetStdApiResult.add(errorMap);
                    }
                }
            }
            // API不存在，但是有發生規格異常，將異常寫到excel檔案
            if (tSetStdApiResult.size() != 0) {
                String tApiError = tApiName + "/" + tVersion;
                tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                return result(tStateCode, tDescription, tTotalApiErrorMap);
            }
            Map<String, StandardApiDataName> tAddDataName = new HashMap<>();
            List<Map<String, String>> tSetStdApiDataNameResult = StandardApiSpecAnalyzeServiceUtil.setStandardApiDataName(tStandardApi, tRequestJsonNode, tAddDataName, tLocale);
            if (!tSetStdApiDataNameResult.isEmpty()) {
                String tApiError = tApiName + "/" + tVersion;
                tTotalApiErrorMap.put(tApiError, tSetStdApiDataNameResult);
                return result(tStateCode, tDescription, tTotalApiErrorMap);
            }
            if (CollUtil.isNotEmpty(tStandardApiVersionList)) {
                for (StandardApiVersion tStandardApiVersion : tStandardApiVersionList) {
                    standardApiVersionDao.delete(tStandardApiVersion.getId(), false);
                }
            }

            //保存API
            standardApiDao.save(tStandardApi, true);
            //保存接收者信息
            productCategoryService.saveProductCategory(tStandardApi.getProvider());
        } catch (BaseException e) {
            log.error(e.getMessage(), e);
            tStateCode = getStateCode(e.getStateEnum().getCode());
            tDescriptionBuilder.append(tStateCode.getDescription());
            if (!StringUtil.isEmptyOrSpace(e.getMessage())) {
                tDescriptionBuilder.append(ApimgmtConstant.BLANK).append(e.getMessage());
            }
            tDescription = tDescriptionBuilder.toString();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            tStateCode = getStateCode(ValidateStateEnum.UNEXPECTED_ERROR.getCode());
            tDescriptionBuilder.append(tStateCode.getDescription()).append(ApimgmtConstant.BLANK).append(e.getClass().toString()).append(":").append(e.getMessage());
            tDescription = tDescriptionBuilder.toString();
        }
        tResponseNode.put(ApimgmtConstant.CODE, tStateCode.getCode());
        tResponseNode.put(ApimgmtConstant.DESCRIPTION, tDescription);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tResponseNode);
        return converJsonNodeToMap(tResponseJsonNode);
    }

    private Map<String, Object> result(StateCode tStateCode, String tDescription,
                                       Map<String, List<Map<String, String>>> pTotalApiErrorMap) throws Exception {
        ObjectNode tResponseNode = mapper.createObjectNode();
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tResponseNode);
        tStateCode = getStateCode(ValidateStateEnum.IMPORT_SPEC_FORMAT_ERROR.getCode());
        tDescription = tStateCode.getDescription();
        // 取得樣板
        HSSFWorkbook tHSSFWorkbook = StandardApiExportServiceUtil.getTemplateFile(false);
        Sheet tErrorSheet = tHSSFWorkbook.createSheet("錯誤明細");
        StandardApiFileReadServiceUtil.writeImportErrorFile(tErrorSheet, pTotalApiErrorMap);
        String tFileName = ApiMgmtApplicationParameter._TEMPLATE_FILE_NAME;
        String tFileExportPath = ApiMgmtApplicationParameter._FILE_EXPORT_PATH + "/" + tFileName;
        FileOutputStream fileOut = new FileOutputStream(tFileExportPath);
        tHSSFWorkbook.write(fileOut);
        fileOut.close();
        int pFailedNum = pTotalApiErrorMap.keySet().size();
        // 上傳檔案並提供下載連結
        DmcFileServiceUtil.uploadFileAndGenerateDownloadLink(tResponseNode, tFileExportPath, tFileName, true, pFailedNum);
        tResponseNode.put(ApimgmtConstant.CODE, tStateCode.getCode());
        tResponseNode.put(ApimgmtConstant.DESCRIPTION, tDescription);
        return converJsonNodeToMap(tResponseJsonNode);
    }
}
