package com.digiwin.athena.apimgmt.dao.impl.mybatis;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtProjectVersionRelationDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDao;
import com.digiwin.athena.apimgmt.dao.impl.mybatis.mapper.ApiMgmtStandardApiMapper;
import com.digiwin.athena.apimgmt.model.AdvanceSearchValue;
import com.digiwin.athena.apimgmt.model.StandardApi;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import com.digiwin.athena.apimgmt.service.model.Page;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Primary
@Slf4j
@Repository
public class ApiMgmtStandardApiDaoMybatisImpl extends GenericDaoMybatisImpl<ApiMgmtStandardApiMapper, StandardApi, Long>
        implements ApiMgmtStandardApiDao {
    private ApiMgmtProjectVersionRelationDao projectVersionRelationDao;

    public ApiMgmtStandardApiDaoMybatisImpl() {
        super();
    }

    @Override
    public List<StandardApi> advanceSearch(ArrayList<Map<String, String>> pConditionList, String teamType, boolean searchType, boolean tViewAllApi) {
        List<WhereCondition> conditionList = DaoConverter.INSTANCE.toWhereCondition(pConditionList);

        String tenantId = pConditionList.stream()
                .map(v -> v.get(ApimgmtConstant.TENANT))
                .filter(Objects::nonNull)
                .findFirst()
                .orElse(null);

        return baseMapper.advanceSearch(conditionList, tenantId, teamType, searchType, tViewAllApi);
    }

    @Override
    public List<StandardApi> getByNameAndTenantId(String pName, String pTenantId) {
        LambdaQueryWrapper<StandardApi> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(StandardApi::getName, pName)
                .eq(StandardApi::getTenantId, pTenantId);
        return baseMapper.selectList(wrapper);
    }

    @Override
    public List<StandardApi> getListByNameAndBranchAndTenantId(String pName, String branch, String pTenantId) {
        LambdaQueryWrapper<StandardApi> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(StandardApi::getName, pName)
                .eq(StandardApi::getBranch, branch)
                .eq(StandardApi::getTenantId, pTenantId);
        return baseMapper.selectList(wrapper);
    }

    @Override
    public StandardApi getByNameAndBranchAndTenantId(String pName, String branch, String pTenantId) {
        LambdaQueryWrapper<StandardApi> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(StandardApi::getName, pName)
                .eq(StandardApi::getBranch, branch)
                .eq(StandardApi::getTenantId, pTenantId)
                .last("LIMIT 1");
        return baseMapper.selectOne(wrapper);
    }

    @Override
    public Page<StandardApiVersion> advanceApiSearch(List<List<AdvanceSearchValue>> pList, String teamType,
                                                     String tenantId, boolean searchType, boolean tViewAllApi,
                                                     Integer pageNum, Integer pageSize) {
        Page<StandardApiVersion> page = new Page<>();
        page.setPageNum(pageNum);
        page.setPageSize(pageSize);

        List<WhereCondition> conditions = DaoConverter.INSTANCE.convertToWhereCondition(pList);

        Long count = baseMapper.advanceApiSearchCount(conditions, teamType, tenantId, searchType, tViewAllApi);
        page.setPageTotal(count);

        if (count > 0) {
            List<Long> versionIds = baseMapper.advanceApiSearchVersionIds(conditions, teamType,
                    tenantId, searchType, tViewAllApi, new MpPage<>(pageNum, pageSize));
            if (versionIds != null && !versionIds.isEmpty()) {
                List<StandardApiVersion> versions = baseMapper.selectVersionsByIds(versionIds);
                fillApi(versions);
                page.setList(versions);
            }
        }

        return page;
    }

    @Override
    public List<Long> advanceStandardApiCountSearch(List<List<AdvanceSearchValue>> pList, String teamType,
                                                    String pTenantId, boolean tViewAllApi) {
        List<WhereCondition> conditions = DaoConverter.INSTANCE.convertToWhereCondition(pList);
        return baseMapper.advanceStandardApiCountSearch(conditions, teamType, pTenantId, false, tViewAllApi);
    }

    @Override
    @Transactional
    public StandardApi save(StandardApi pStandardApi, boolean cleanProjectRelation) {
        String tApiName = pStandardApi.getName();
        String tTenantId = pStandardApi.getTenantId();
        String branch = pStandardApi.getBranch();
        StandardApi existingApi = getByNameAndBranchAndTenantId(tApiName, branch, tTenantId);

        if (existingApi != null) {
            pStandardApi.setId(existingApi.getId());
            pStandardApi.setMsgMdata(existingApi.getMsgMdata());

            if (cleanProjectRelation) {
                projectVersionRelationDao.deleteByApiId(pStandardApi.getId());
            }
        }

        for (StandardApiVersion tVersion : pStandardApi.getStandardApiVersions()) {
            tVersion.setApprovedTime(LocalDateTime.now());
        }
        pStandardApi.setBuildTime(LocalDateTime.now());

        if (pStandardApi.getId() == null) {
            baseMapper.insert(pStandardApi);
        } else {
            baseMapper.updateById(pStandardApi);
        }

        log.info("儲存API. API名稱: " + pStandardApi.getName());
        return pStandardApi;
    }

    @Override
    public void delete(Long pId) {
        baseMapper.deleteById(pId);
    }


    @Override
    public void fillApi(List<StandardApiVersion> versions) {
        if (CollUtil.isEmpty(versions)) {
            return;
        }

        List<StandardApi> standardApis = baseMapper.selectBatchIds(
                versions.stream().map(StandardApiVersion::getApiId).collect(Collectors.toList())
        );

        Map<Long, StandardApi> apiIdMap = CollUtil.toMap(standardApis, MapUtil.newHashMap(standardApis.size()), StandardApi::getId);
        versions.forEach(ver -> ver.setStandardApi(apiIdMap.get(ver.getApiId())));
    }

    @Autowired
    public void setProjectVersionRelationDao(ApiMgmtProjectVersionRelationDao projectVersionRelationDao) {
        this.projectVersionRelationDao = projectVersionRelationDao;
    }


    public static class DaoConverter {
        public static final Map<String, String> fieldMapping = new HashMap<>();
        public static final Map<String, WhereCondition.Type> typeMapping = new HashMap<>();
        public static DaoConverter INSTANCE = new DaoConverter();

        static {
            fieldMapping.put("apiName", "stdapi.name");
            fieldMapping.put("apiType", "stdapi.api_type");
            fieldMapping.put("projectId", "pvr.project_id");
            fieldMapping.put("projectVersionId", "pvr.project_version_id");
            fieldMapping.put("apiStatus", "astat.id");
            fieldMapping.put("apiTag", "stdapi.tag");
            typeMapping.put("apiTag", WhereCondition.Type.MULTI_LANG);
            fieldMapping.put("apiDataName", "stdapidataname.api_ver_id");
            fieldMapping.put("apiDescription", "stdapi.description");
            typeMapping.put("apiDescription", WhereCondition.Type.MULTI_LANG);
            fieldMapping.put("apiRemark", "stdapi.remark");
            typeMapping.put("apiRemark", WhereCondition.Type.MULTI_LANG);
            fieldMapping.put("apiSyncType", "stdapist.id");
            fieldMapping.put("apiCategory", "stdapicat.id");
            fieldMapping.put("provider", "stdapi.provider");
            fieldMapping.put("requester", "stdapi.requester");
            fieldMapping.put("tenantId", "stdapi.tenant_id");
            fieldMapping.put("apiBuildTime", "stdapi.build_time");
            typeMapping.put("apiBuildTime", WhereCondition.Type.DATE);
            fieldMapping.put("userId", "stdapiver.user_id");
        }

        public List<WhereCondition> convertToWhereCondition(List<List<AdvanceSearchValue>> pList) {
            return pList.stream()
                    .flatMap(list -> {
                        List<WhereCondition> child = list.stream()
                                .map(v -> WhereCondition.from(
                                        fieldMapping.get(v.getField()),
                                        typeMapping.getOrDefault(v.getField(), WhereCondition.Type.STRING),
                                        v))
                                .collect(Collectors.toList());

                        if (child.isEmpty()) {
                            return Stream.empty();
                        }

                        return Stream.of(WhereCondition.segment(child.get(0).isAnd(), child));
                    })
                    .collect(Collectors.toList());
        }

        public List<WhereCondition> toWhereCondition(ArrayList<Map<String, String>> list) {
            return list.stream()
                    .flatMap(inMap -> fieldMapping.keySet().stream()
                            .map(field -> {
                                String val = inMap.get(field);
                                if (val == null) {
                                    return null;
                                }

                                String ops = inMap.get(field + "_ops");
                                if ("provider".equals(field)) {
                                    ops = ops == null ? WhereCondition.Operator.LIKE.getOp() : ops;
                                }

                                AdvanceSearchValue searchValue = new AdvanceSearchValue(val,
                                        ops,
                                        ApimgmtConstant.MARK_AND,
                                        field,
                                        0);
                                return WhereCondition.from(
                                        fieldMapping.get(field),
                                        typeMapping.getOrDefault(field, WhereCondition.Type.STRING),
                                        searchValue);
                            })
                            .filter(Objects::nonNull)
                    )
                    .collect(Collectors.toList());
        }

    }
}
