package com.digiwin.athena.apimgmt.apiservice;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.model.StateCode;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 
 * JSON智能識別導入
 * /restful/standard/apimgmt/JsonReqImport/Get
 * 
 */
@Slf4j
@Service
public class ApiMgmtJsonReqImportGetService extends AbstractApiService {

	private static final Lock LOCK = new ReentrantLock(true);

	Long tId = 1L;

	public ApiMgmtJsonReqImportGetService() {
		super();
		jsonSchemaFileName = ApimgmtSchemaConstant.JSON_REQ_IMPORT_GET_SCHEMA;
	}

	/**
	 * 数据组装
	 * @param validatorResult validatorResult
	 * @return Map
	 * @throws BaseException BaseException
	 * @throws InterruptedException InterruptedException
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, InterruptedException {
		// 取得request node
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		// 建立response node
		ObjectNode tResponseNode = mapper.createObjectNode();
		// 建立response array node
		ArrayNode tResponseArrayNode = mapper.createArrayNode();
		StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
		String tDescription = tStateCode.getDescription();
		JsonNode tParamJsonNode;
		JsonNode tExecutionJsonNode = null;
		if (tRequestJsonNode.has(ApimgmtConstant.STD_DATA)) {
			tParamJsonNode = tRequestJsonNode.get(ApimgmtConstant.STD_DATA).get(ApimgmtConstant.PARAMETER);
			tExecutionJsonNode = tRequestJsonNode.get(ApimgmtConstant.STD_DATA).get(ApimgmtConstant.EXECUTION);
		} else {
			tParamJsonNode = tRequestJsonNode;
		}
		StringBuilder tErrorString = new StringBuilder();
		LOCK.lock();
		try {
			resetId();
			if (tExecutionJsonNode != null) {
				analyzeJsonNode(tResponseArrayNode, tExecutionJsonNode);
			}
			analyzeJsonNode(tResponseArrayNode, tParamJsonNode, null, null, tErrorString);
		} finally {
			LOCK.unlock();
		}
		if(!StringUtil.isEmptyOrSpace(tErrorString)){
			tStateCode = getStateCode(ValidateStateEnum.JSON_CANNOT_HAS_NULL_OR_EMPTY.getCode());
			tDescription = tStateCode.getDescription();
		} else {
			tResponseNode.set(ApimgmtConstant.MESSAGE_SPEC, tResponseArrayNode);
		}
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tResponseNode);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	private void resetId() {
		tId = 1L;
	}

	private void analyzeJsonNode(ArrayNode pResponseArrayNode, JsonNode pParamJsonNode) {
		List<String> tDataNameList = getJsonNodeKey(pParamJsonNode);
		for (String tDataName : tDataNameList) {
			ObjectNode tResponseObjectNode = mapper.createObjectNode();
			tResponseObjectNode.put(ApimgmtConstant.COLUMN_TYPE, ApimgmtConstant.COLUMN_E);
			tResponseObjectNode.put(ApimgmtConstant.ID, tId);
			tResponseObjectNode.put(ApimgmtConstant.DATA_NAME, tDataName);
			tResponseObjectNode.put(ApimgmtConstant.DATA_TYPE, ApimgmtConstant.STRING);
			tResponseObjectNode.put(ApimgmtConstant.IS_ARRAY, ApimgmtConstant.NO);
			tResponseObjectNode.put(ApimgmtConstant.IS_REQUIRED, ApimgmtConstant.NO);
			tResponseObjectNode.put(ApimgmtConstant.CAN_SORT, ApimgmtConstant.NO);
			tResponseObjectNode.put(ApimgmtConstant.IS_BUSINESS_KEY, ApimgmtConstant.NO);
			tResponseObjectNode.put(ApimgmtConstant.IS_DATAKEY, ApimgmtConstant.NO);
			tResponseObjectNode.put(ApimgmtConstant.CAN_FILTER, ApimgmtConstant.NO);
			pResponseArrayNode.add(tResponseObjectNode);
			tId++;
		}
	}

	/**
	 * 解析json node
	 */
	private void analyzeJsonNode(ArrayNode pResponseArrayNode, JsonNode pParamJsonNode, Long pFatherId, String pFatherColumn, StringBuilder tErrorString) {
		List<String> tDataNameList = getJsonNodeKey(pParamJsonNode);
		Long tFatherId = pFatherId;
		boolean tHasChildrens = false;
		String tColumnType = "P";
		String tIsArray = null;
		String tDataType = null;
		if (pFatherColumn != null) {
			switch (pFatherColumn) {
			case "M":
				tColumnType = "MF";
				break;
			case "D":
				tColumnType = "DF";
				break;
			case "SD":
				tColumnType = "SDF";
				break;
			case "4D":
				tColumnType = "4DF";
				break;
			case "5D":
				tColumnType = "5DF";
				break;
			case "6D":
				tColumnType = "6DF";
				break;
			case "7D":
				tColumnType = "7DF";
				break;
			case "8D":
				tColumnType = "8DF";
				break;
			case "9D":
				tColumnType = "9DF";
				break;
			}
		}
		int tColumnIndex = Arrays.asList(ApimgmtConstant.COLUMN_TYPE_ARRAY_OLD).indexOf(tColumnType);
		String tNextColumn = ApimgmtConstant.COLUMN_TYPE_ARRAY_OLD[tColumnIndex + 1];
		for (String tDataName : tDataNameList) {
			ObjectNode tResponseObjectNode = mapper.createObjectNode();
			JsonNode tJsonNode = pParamJsonNode.get(tDataName);
			JsonNode tArrayColumnJsonNode = null;
			if(tJsonNode.asText().equals("null")){
				tErrorString.append("ERROR");
				continue;
			}
			if (tJsonNode.isValueNode()) {
				tResponseObjectNode.put(ApimgmtConstant.COLUMN_TYPE, tColumnType);
				tIsArray = ApimgmtConstant.NO;
				tHasChildrens = false;
				tDataType = jsonNodeType(tJsonNode);
			} else if (tJsonNode.isArray()) {
				ArrayNode tArrayNode = (ArrayNode) tJsonNode;
				tArrayColumnJsonNode = tArrayNode.get(0);
				if(tArrayColumnJsonNode != null){
					if (tArrayColumnJsonNode.isValueNode()) {
						tResponseObjectNode.put(ApimgmtConstant.COLUMN_TYPE, tColumnType);
						tIsArray = ApimgmtConstant.NO;
						tHasChildrens = false;
						tDataType = jsonNodeType(tJsonNode);
						
					} else if (tArrayColumnJsonNode.isObject()) {
						tResponseObjectNode.put(ApimgmtConstant.COLUMN_TYPE, tNextColumn);
						tIsArray = ApimgmtConstant.YES;
						tHasChildrens = true;
						tDataType = jsonNodeType(tJsonNode);
					}
				} else {
					tErrorString.append("ERROR");
					continue;
				}
			} else if (tJsonNode.isObject()) {
				tResponseObjectNode.put(ApimgmtConstant.COLUMN_TYPE, tNextColumn);
				tArrayColumnJsonNode = tJsonNode;
				tIsArray = ApimgmtConstant.NO;
				tHasChildrens = true;
				tDataType = jsonNodeType(tJsonNode);
			}
			String tColumn = tResponseObjectNode.get(ApimgmtConstant.COLUMN_TYPE).asText();
			if (!Arrays.asList(ApimgmtConstant.DATANAME_FATHER_UNREQUIRED_OLD).contains(tColumn)) {
				tResponseObjectNode.put(ApimgmtConstant.PARENT, tFatherId);
			}
			if (Arrays.asList(ApimgmtConstant.ALLOW_ARRAY_COLUMN_TYPE_OLD).contains(tColumn)) {
				pFatherId = tId;
			}
			tResponseObjectNode.put(ApimgmtConstant.ID, tId);
			tResponseObjectNode.put(ApimgmtConstant.DATA_NAME, tDataName);
			tResponseObjectNode.put(ApimgmtConstant.DATA_TYPE, tDataType);
			tResponseObjectNode.put(ApimgmtConstant.IS_ARRAY, tIsArray);
			tResponseObjectNode.put(ApimgmtConstant.IS_REQUIRED, ApimgmtConstant.NO);
			tResponseObjectNode.put(ApimgmtConstant.CAN_SORT, ApimgmtConstant.NO);
			tResponseObjectNode.put(ApimgmtConstant.IS_BUSINESS_KEY, ApimgmtConstant.NO);
			tResponseObjectNode.put(ApimgmtConstant.IS_DATAKEY, ApimgmtConstant.NO);
			tResponseObjectNode.put(ApimgmtConstant.CAN_FILTER, ApimgmtConstant.NO);
			pResponseArrayNode.add(tResponseObjectNode);
			tId++;
			if (tHasChildrens) {
				analyzeJsonNode(pResponseArrayNode, tArrayColumnJsonNode, pFatherId, tNextColumn, tErrorString);
			}
		}
	}

	private String jsonNodeType(JsonNode pJsonNode) {
		String tDataType;
		if(pJsonNode.isBoolean()) {
			//ex:true、false
			tDataType = ApimgmtConstant.BOOLEAN;
		} else if(pJsonNode.isInt()) {
			//ex:1、0.5
			tDataType = ApimgmtConstant.NUMERIC;
		} else if(pJsonNode.isNumber()) {
			//ex:2E+1
			tDataType = ApimgmtConstant.NUMERIC;
		} else {
			tDataType = ApimgmtConstant.STRING;
		}
		return tDataType;
	}

	private List<String> getJsonNodeKey(JsonNode pParamJsonNode) {
		Iterator<Entry<String, JsonNode>> nodes = pParamJsonNode.fields();
		List<String> tDataNameList = new ArrayList<>();
		while (nodes.hasNext()) {
			Map.Entry<String, JsonNode> entry = nodes.next();
			tDataNameList.add(entry.getKey());
		}
		return tDataNameList;
	}
}
