package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.util.ObjectUtil;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiVersionDao;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.model.StandardApi;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import com.digiwin.athena.apimgmt.model.StateCode;
import com.digiwin.athena.apimgmt.service.util.ApiVersionServiceUtil;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import cn.hutool.core.collection.CollUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * API 查询新增|进版的版本号
 * /restful/standard/apimgmt/ApiVersion/Get
 */
@Slf4j
@Service
public class ApiSearchVersionService extends AbstractApiService {

	@Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;
    @Autowired
    private ApiMgmtStandardApiDao standardApiDao;

	public ApiSearchVersionService() {
		super();
		jsonSchemaFileName = ApimgmtSchemaConstant.API_SEARCH_VERSION_SCHEMA;
	}

	/**
	 * 查询API版本号
	 * @param validatorResult validatorResult
	 * @return map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		// 通过api名称+租户获取对应的版本
		Map<String, Object> tResponse = new HashMap<>();
		StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
		String description = tStateCode.getDescription();
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		String tApiName = tRequestJsonNode.get(ApiAttributeEnum.apiName.toString()).asText();
		String tenantId = tRequestJsonNode.get(ApiAttributeEnum.tenantId.toString()).asText();

        // 是否新增
        Boolean isNew = Optional.ofNullable(tRequestJsonNode.get(ApiAttributeEnum.isNew.toString()))
                .map(JsonNode::asBoolean)
                .orElse(false);
		//兼容，非必传
		String branch = ObjectUtil.isNotNull(tRequestJsonNode.get(ApiAttributeEnum.branch.toString())) ?
				tRequestJsonNode.get(ApiAttributeEnum.branch.toString()).asText() : ApimgmtConstant.DEFAULT_BRANCH;

		String tVersion = ApimgmtConstant.API_VER;
		String parentApiName = null;
		String parentBranch = null;
		String parentApiTenantId = null;
		//先查API表
		StandardApi standardApi = standardApiDao.getByNameAndBranchAndTenantId(tApiName, branch, tenantId);
		if (ObjectUtil.isNull(standardApi)) {
			tVersion = branch;
        } else if (isNew) {
            // 新增api时，如果已存在则报错
            tStateCode = getStateCode(ValidateStateEnum.API_NAME_EXISTS.getCode());
            description = String.format(tStateCode.getDescription(), tApiName);
            tVersion = null;
        } else {
			//再查版本
			List<StandardApiVersion> standardApiVersionList = standardApiVersionDao.getVersionListByApiId(standardApi.getId());
			if (CollUtil.isNotEmpty(standardApiVersionList)) {
				standardApiVersionList.sort((o1, o2) -> o2.getApprovedTime().compareTo(o1.getApprovedTime()));
				StandardApiVersion standardApiVersion = standardApiVersionList.get(0);
				if (standardApiVersion.getApprovedStatus().getId() != ApimgmtConstant.APPROVED_STATUS_COMFIRMED) {
					tStateCode = getStateCode(ValidateStateEnum.API_PERV_VER_NOT_FIX.getCode());
					description = String.format(tStateCode.getDescription(),tApiName,standardApiVersion.getVersion());
					tVersion = null;
				}else {
					try {
						tVersion = ApiVersionServiceUtil.getNextVersion(standardApiVersion.getVersion());
						parentApiName = standardApiVersion.getStandardApi().getParentApiName();
						parentBranch = standardApiVersion.getStandardApi().getParentBranch();
						parentApiTenantId = standardApiVersion.getStandardApi().getParentApiTenantId();
					} catch (Exception e) {
						log.error(e.getMessage(), e);
						tStateCode = getStateCode(ValidateStateEnum.API_VERSION_EXCEEDS_LIMIT.getCode());
						description = tStateCode.getDescription();
						tVersion = null;
					}
				}
			}
		}
		tResponse.put(ApimgmtConstant.CODE, tStateCode.getCode());
		tResponse.put(ApimgmtConstant.DESCRIPTION, description);
		tResponse.put(ApiAttributeEnum.version.toString(), tVersion);
		tResponse.put(ApiAttributeEnum.parentApiName.toString(), parentApiName);
		tResponse.put(ApiAttributeEnum.parentBranch.toString(), parentBranch);
		tResponse.put(ApiAttributeEnum.parentApiTenantId.toString(), parentApiTenantId);
		return tResponse;
	}
}
