package com.digiwin.athena.apimgmt.services;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.URLUtil;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtInterfaceConstant;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.infra.http.HttpRequest;
import com.digiwin.athena.apimgmt.infra.http.MultipartFile;
import com.digiwin.athena.apimgmt.infra.util.HttpUtil;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtProp;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtDmcProp;
import com.digiwin.athena.apimgmt.service.util.DmcFileServiceUtil;
import com.digiwin.athena.apimgmt.util.Base64;
import com.digiwin.athena.apimgmt.util.Base64Util;
import com.digiwin.athena.apimgmt.util.DateUtil;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.gson.JsonObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

@Slf4j
@Service
public class ApiMgmtDmcFileService {


    @Autowired
    private ApiMgmtDmcProp apiMgmtDmcProp;
    @Autowired
    private ApiMgmtProp prop;

    ObjectMapper mapper = new ObjectMapper();

    /**
     * 上傳檔案並提供下載連結
     **/
    public void uploadFileAndGenerateDownloadLink(ObjectNode pResponseNode, String pFilePath, String pFileName, boolean pIsDetail, int pFailedNum) throws Exception {
        String tFileId;
        String tSharedDownloadLink = null;
        // 上傳規格檔
        tFileId = DmcFileServiceUtil.uploadFile(new File(pFilePath), pFileName);
        // 分享規格檔
        if (!StringUtil.isEmptyOrSpace(tFileId)) {
            log.info("上傳API規格檔至dmc成功，檔案名稱:" + pFileName + " 檔案ID:" + tFileId);
            tSharedDownloadLink = generateShareDownloadLink(tFileId);
        }
        if (!StringUtil.isEmptyOrSpace(tSharedDownloadLink)) {
            // 提供API規格檔下載連結
            log.info("分享API規格檔至dmc成功，檔案名稱:" + pFileName + " 檔案ID:" + tFileId + " 下載連結:" + tSharedDownloadLink);
            if (pIsDetail) {
                pResponseNode.put(ApiAttributeEnum.detailFileLink.toString(), tSharedDownloadLink);
                if (pFailedNum != 0) {
                    pResponseNode.put(ApiAttributeEnum.failedNum.toString(), pFailedNum);
                }
            } else {
                pResponseNode.put(ApiAttributeEnum.fileLink.toString(), tSharedDownloadLink);
                pResponseNode.put(ApiAttributeEnum.fileName.toString(), pFileName);
            }
        }
    }

    /**
     * 上傳檔案到dmc
     **/
    public String uploadFile(File pFile, String pFileName) throws Exception {
        String tUserToken = login();
        String tFileId = "";
        if (!StringUtil.isEmptyOrSpace(tUserToken)) {
            String tDmcHost = apiMgmtDmcProp.getUrl();
            String tBucket = apiMgmtDmcProp.getBucket();
            String tDmcDirectoryId = apiMgmtDmcProp.getDirectoryId();
            String tUriStr = URLUtil.completeUrl(tDmcHost,
                    String.format(ApimgmtInterfaceConstant.DMC_UPLOAD_FILE_PATH, tBucket) + "?bucket=" + tBucket);
            HttpRequest tPost = HttpRequest.ofPost(tUriStr);
            // http header
            tPost.setHeader(ApimgmtConstant.DIGI_MIDDLEWARE_AUTH_USER, tUserToken);
            tPost.setHeader(ApimgmtConstant.DIGI_MIDDLEWARE_AUTH_APP, prop.getIamToken());
            tPost.setHeader(ApimgmtConstant.TOKEN, tUserToken);
            if (null != ApiMgmtServiceContextHolder.getRouterKey()) {
                tPost.setHeader(ApimgmtConstant.ROUTER_KEY, ApiMgmtServiceContextHolder.getRouterKey());
            }
            // http body
            MultipartFile file = new MultipartFile("file", pFile);
            JsonObject tFileInfo = new JsonObject();
            tFileInfo.addProperty("directoryId", tDmcDirectoryId);
            tFileInfo.addProperty("displayName", pFileName);
            tFileInfo.addProperty("fileName", pFileName);
            tFileInfo.addProperty("expireDate", DateUtil.formatToSecond(new Date(System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(5))));
            tPost.setEntity(MapUtil.of("fileInfo", tFileInfo));
            String tResponse = HttpUtil.uploadFile(tPost, file, String.class);
            JsonNode tDataNode;
            JsonNode tResponseNode = mapper.readTree(tResponse);
            if (tResponseNode.get("data") != null) {
                tDataNode = tResponseNode.get("data");
                if (tDataNode.get("id") != null) {
                    tFileId = tDataNode.get("id").asText();
                }
            }
        }
        return tFileId;
    }

    /**
     * 產生分享下載連結
     **/
    public String generateShareDownloadLink(String pFileId) throws Exception {
        String tUserToken = login();
        String tFileId = "";
        if (!StringUtil.isEmptyOrSpace(tUserToken)) {
            String tDmcHost = apiMgmtDmcProp.getUrl();
            String tBucket = apiMgmtDmcProp.getBucket();
            String tUriStr = URLUtil.completeUrl(tDmcHost,
                    String.format(ApimgmtInterfaceConstant.DMC_SHARED_FILE_LINK_PATH, tBucket) + "?bucket=" + tBucket);
            HttpRequest tPost = HttpRequest.ofPost(tUriStr);
            // http header
            tPost.setHeader(ApimgmtConstant.DIGI_MIDDLEWARE_AUTH_USER, tUserToken);
            tPost.setHeader(ApimgmtConstant.DIGI_MIDDLEWARE_AUTH_APP, prop.getIamToken());
            tPost.setHeader("Content-Type", "application/json");
            tPost.setHeader(ApimgmtConstant.TOKEN, tUserToken);
            if (null != ApiMgmtServiceContextHolder.getRouterKey()) {
                tPost.setHeader(ApimgmtConstant.ROUTER_KEY, ApiMgmtServiceContextHolder.getRouterKey());
            }
            // http body
            Map<String, Object> tRequestBody = new HashMap<>();
            tRequestBody.put("fileId", pFileId);
            tRequestBody.put("expireDate", DateUtil.formatToSecond(new Date(System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(5))));
            tPost.setEntity(tRequestBody);
            String tResponse = HttpUtil.execute(tPost, String.class);
            JsonNode tDataNode;
            JsonNode tResponseNode = mapper.readTree(tResponse);
            if (tResponseNode.get("data") != null) {
                tDataNode = tResponseNode.get("data");
                if (tDataNode.get("url") != null) {
                    tFileId = tDataNode.get("url").asText();
                }
            }
        }
        return tFileId;
    }

    /**
     * dmc登入
     **/
    public String login() throws Exception {
        String tToken = null;
        // DAP呼叫法
        HttpRequest tPost = HttpRequest.ofPost(URLUtil.completeUrl(apiMgmtDmcProp.getUrl(), ApimgmtInterfaceConstant.DMC_LOGIN_PATH));
        Map<String, Object> tRequestBody = new HashMap<>();
        tRequestBody.put(ApimgmtConstant.USER_NAME, apiMgmtDmcProp.getUser());
        tRequestBody.put(ApimgmtConstant.PWD_HASH, passwordHash(Base64Util.decodeString(apiMgmtDmcProp.getPassword())));
        tPost.setHeader("Content-Type", "application/json");
        tPost.setHeader(ApimgmtConstant.DIGI_MIDDLEWARE_AUTH_APP, prop.getIamToken());
        if (null != ApiMgmtServiceContextHolder.getRouterKey()) {
            tPost.setHeader(ApimgmtConstant.ROUTER_KEY, ApiMgmtServiceContextHolder.getRouterKey());
        }
        tPost.setEntity(tRequestBody);
        Map<String, String> tResponse = HttpUtil.execute(tPost, HashMap.class);
        if (null != tResponse && tResponse.get(ApimgmtConstant.USERTOKEN) != null) {
            tToken = tResponse.get(ApimgmtConstant.USERTOKEN);
        }
        return tToken;
    }

    private String passwordHash(String pPassword) {
        String tPasswordHash = null;
        try {
            byte[] hash;
            hash = MessageDigest.getInstance("SHA-256").digest(pPassword.getBytes(StandardCharsets.UTF_8));
            hash = MessageDigest.getInstance("SHA-256").digest(hash);
            tPasswordHash = Base64.encode(hash);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return tPasswordHash;
    }

}
