package com.digiwin.athena.apimgmt.dao.impl.mybatis;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.model.AdvanceSearchValue;
import lombok.Data;
import lombok.Getter;

import java.time.LocalDateTime;
import java.util.List;

@Data
public class WhereCondition {
    private Type type;
    private Operator operator;
    private String column;
    private Object val;

    private boolean isAnd = true;
    private List<WhereCondition> child;

    public static WhereCondition from(String column, Type type, AdvanceSearchValue item) {
        WhereCondition condition = new WhereCondition();

        condition.setColumn(column);
        condition.setType(type);
        condition.setVal(item.getContent());
        condition.setAnd(!ApimgmtConstant.MARK_OR.equalsIgnoreCase(item.getMark()));

        String operationStr = StrUtil.trimToEmpty(item.getEquation());
        condition.setOperator(Operator.ofOps(operationStr));
        if (condition.getOperator() == null) {
            condition.setOperator(Operator.EQ);
        }

        if (type == Type.MULTI_LANG) {
            return buildMultiLang(condition);
        } else if (type == Type.DATE) {
            return buildDate(condition);
        }

        return condition;
    }

    public static WhereCondition buildDate(WhereCondition condition) {
        LocalDateTime datetime = Convert.convert(LocalDateTime.class, condition.getVal());
        if(datetime == null){
            return condition;
        }

        if (condition.getOperator() == Operator.LE) {
            condition.setOperator(Operator.LT);
            condition.setVal(datetime.plusSeconds(1));
        } else if (condition.getOperator() == Operator.EQ) {
            WhereCondition begin = BeanUtil.copyProperties(condition, WhereCondition.class);
            begin.setAnd(true);
            begin.setOperator(Operator.GE);

            WhereCondition end = BeanUtil.copyProperties(begin, WhereCondition.class);
            end.setOperator(Operator.LT);
            end.setVal(datetime.plusSeconds(1));

            condition.setType(Type.SEGMENT);
            condition.setChild(ListUtil.of(begin, end));
        } else if (condition.getOperator() == Operator.GT) {
            condition.setOperator(Operator.GE);
            condition.setVal(datetime.plusSeconds(1));
        }

        return condition;
    }

    public static WhereCondition buildMultiLang(WhereCondition condition) {
        WhereCondition zh = BeanUtil.copyProperties(condition, WhereCondition.class);
        zh.setAnd(false);
        zh.setColumn(zh.getColumn() + "_zh_cn");

        WhereCondition tw = BeanUtil.copyProperties(zh, WhereCondition.class);
        tw.setColumn(zh.getColumn() + "_zh_tw");

        WhereCondition en = BeanUtil.copyProperties(zh, WhereCondition.class);
        en.setColumn(zh.getColumn() + "_en_us");

        return segment(condition.isAnd(), ListUtil.of(zh, tw, en));
    }

    public static WhereCondition segment(boolean isAnd, List<WhereCondition> child) {
        WhereCondition condition = new WhereCondition();

        condition.setAnd(isAnd);
        condition.setType(WhereCondition.Type.SEGMENT);
        condition.setChild(child);

        return condition;
    }

    public void setVal(Object val) {
        if (type == Type.DATE && val instanceof String) {
            this.val = DateUtil.parseLocalDateTime(val.toString());
        } else if (operator == Operator.LIKE) {
            this.val = "%" + val + "%";
        } else {
            this.val = val;
        }
    }

    public boolean notEmpty() {
        return ObjUtil.isNotEmpty(val);
    }


    public enum Operator {
        EQ("=", true, "EQ", "equal"),
        NE("!=", true, "NE"),
        GT(">", true, "GT"),
        LT("<", true, "LT"),
        GE(">=", true, "GE"),
        LE("<=", true, "LE"),
        LIKE("LIKE", true),
        NOT_LIKE("NOT LIKE", true),
        IN("IN", true),
        NOT_IN("NOT IN", true),
        IS_NULL("IS NULL", false, "is"),
        IS_NOT_NULL("IS NOT NULL", false),
        ;
        private final String[] expr;
        @Getter
        private final String op;
        @Getter
        private final boolean hasVal;

        Operator(String op, boolean hasVal, String... expr) {
            this.op = op;
            this.hasVal = hasVal;
            this.expr = expr;
        }

        public static Operator ofOps(String operationStr) {
            for (Operator op : values()) {
                if (op.op.equalsIgnoreCase(operationStr)){
                    return op;
                }

                for (String expr : op.expr) {
                    if (expr.equalsIgnoreCase(operationStr)) {
                        return op;
                    }
                }
            }

            return null;
        }
    }

    public enum Type {
        IGNORE,
        SEGMENT,
        STRING,
        DATE,
        MULTI_LANG
    }
}
