package com.digiwin.athena.apimgmt.services;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.digiwin.athena.apimgmt.constants.*;
import com.digiwin.athena.apimgmt.dao.*;
import com.digiwin.athena.apimgmt.enums.*;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtProp;
import com.digiwin.athena.apimgmt.model.*;
import com.digiwin.athena.apimgmt.service.model.ParseExcelHelper;
import com.digiwin.athena.apimgmt.service.util.ApiVersionServiceUtil;
import com.digiwin.athena.apimgmt.service.util.ProjectVersionServiceUtil;
import com.digiwin.athena.apimgmt.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor.HSSFColorPredefined;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiMgmtStandardApiFileReadService extends AbstractService {

    @Autowired
    ApiMgmtStandardApiCategoryDao standarAPICategoryDao;

    @Autowired
    ApiMgmtStandardApiGroupDao standardApiGroupDao;

    @Autowired
    ApiMgmtStandardApiSyncTypeDao standardApiSyncTypeDao;

    @Autowired
    ApiMgmtStandardApiPagingDao standardApiPagingDao;

    @Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;

    @Autowired
    ApimgmtProductDao productDao;

    @Autowired
    ApiMgmtStandardDataNameDao standardDataNameDao;

    @Autowired
    ApiMgmtApprovedStatusDao approvedStatusDao;

    @Autowired
    ApiMgmtWrongDescriptionDao wrongDescriptionDao;

    @Autowired
    private ApiMgmtApiVersionService apiVersionService;

    @Autowired
    private ApiMgmtProjectDao projectDao;

    @Autowired
    private ApiMgmtProjectVersionDao projectVersionDao;

    @Autowired
    private ApiMgmtProp prop;

    public Workbook getWorkBook(String pFileType, InputStream pFileIn) throws IOException {
        POIFSFileSystem tPOIFSFileSystem;
        Workbook tWorkbook = null;
        // 不同檔案類型載入不同POI(Excel套件)
        if (pFileType.equals("xls")) {
            tPOIFSFileSystem = new POIFSFileSystem(pFileIn);
            tWorkbook = new HSSFWorkbook(tPOIFSFileSystem);
        } else if (pFileType.equals("xlsx")) {
            tWorkbook = new XSSFWorkbook(pFileIn);
        }
        return tWorkbook;
    }

    /**
     * setStandardApi
     */
    public List<Map<String, String>> setStandardApi(StandardApi pStandardApi, String pApplicant, Sheet pSheet, String pLocale) throws Exception {
        Row tRow;
        Cell tCell;
        int tRowNumber = 0;
        int tColumnNumber = 1;
        StandardApiGroup tStandardApiGroup = null;
        StandardApiCategory tStandardApiCategory = null;
        StandardApiPaging tStandardApiPaging = null;
        String tTagZhTW = null, tTagZhCN, tTagEnUS;
        String tMsgFormat;
        String tApiName, tVersion;
        String tDescriptionZhTW = null, tDescriptionZhCN = null, tDescriptionEnUS;
        boolean tIsBatch, IsIdempotency;
        String tRemarkZhTW = null, tRemarkZhCN = null, tRemarkEnUS;
        String tRequester, tProvider;
        StandardApiVersion tStandardApiVersion = new StandardApiVersion();
        List<Map<String, String>> tResult = new ArrayList<>();
        int offset = 1;
        // 讀取StandardAPI資訊(第0列開始)
        tRow = pSheet.getRow(tRowNumber);
        if (tRow == null || tRow.getCell(1, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL) == null || !tRow.getCell(1).getStringCellValue().equals(ApimgmtConstant.STANDARD_API)) {
            // 沒有讀取到StandardAPI就直接結束
            uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_SHEET_FORMAT.toString(), null, null, tRowNumber, tColumnNumber, pLocale);
            return tResult;
        }
        tRowNumber = 2;
        tRow = pSheet.getRow(tRowNumber);
        // (0)空欄
        // (1)服務名稱 (2)服務說明(繁體) (3)服務說明(簡體) (4)服務說明(英文)
        // (5)服務版本 (6)調用模式 (7)分頁模式 (8)服務叫用方 (9)服務接收方 (10)須分包
        // (11)信息大類 (12)信息類別 (13)信息格式
        // (14)信息標籤(繁體) (15)信息標籤(簡體) (16)信息標籤(英文)
        // (17)備註(繁體) (18)備註(簡體) (19)備註(英文)
        // (20)租戶ID (21)修改自API (22)是否符合冪等性 (23)修改紀錄
        // (1)服務名稱
        // (24)API类型
        tColumnNumber = 24;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            pStandardApi.setApiType(ApimgmtConstant.DEFAULT_API_TYPE);
        } else {
            pStandardApi.setApiType(ApiTypeConstants.getCodeByName(tCell.toString()));
        }
        tColumnNumber = 1;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            // 沒有填服務名稱就直接結束
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 1L, null, tRowNumber, tColumnNumber, pLocale);
            return tResult;
        } else {
            tApiName = tCell.toString().trim();
            pStandardApi.setName(tApiName);
        }
        // (2)服務說明(繁體)
        tColumnNumber = 2;
        boolean tDescriptionZhTwEmpty = false;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tDescriptionZhTwEmpty = true;
        } else {
            tDescriptionZhTW = tCell.toString().trim();
            pStandardApi.setDescriptionZhTw(tDescriptionZhTW);
        }
        // (3)服務說明(簡體)
        tColumnNumber = 3;
        boolean tDescriptionZhCnEmpty = false;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tDescriptionZhCnEmpty = true;
        } else {
            tDescriptionZhCN = tCell.toString().trim();
            pStandardApi.setDescriptionZhCn(tDescriptionZhCN);
        }
        // 如果服務說明(繁體)、服務說明(簡體)兩個都沒有填寫，才會顯示必要欄位沒有填寫
        if (tDescriptionZhTwEmpty && tDescriptionZhCnEmpty) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 3L, null, tRowNumber, tColumnNumber, pLocale);
            // 如果服務說明(繁體)沒有填寫，用服務說明(簡體)翻譯的結果填入
        } else if (tDescriptionZhTwEmpty) {
            tDescriptionZhTW = localeTranslate(tDescriptionZhCN, LocaleEnum.ZH2HANT.getType());
            pStandardApi.setDescriptionZhTw(tDescriptionZhTW);
            // 如果服務說明(簡體)沒有填寫，用服務說明(繁體)翻譯的結果填入
        } else if (tDescriptionZhCnEmpty) {
            tDescriptionZhCN = localeTranslate(tDescriptionZhTW, LocaleEnum.ZH2HANS.getType());
            pStandardApi.setDescriptionZhCn(tDescriptionZhCN);
        }
        // (4)服務說明(英文)
        tColumnNumber = 4;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            pStandardApi.setDescriptionEnUs(tDescriptionZhTW);// 若無英文先放中文
        } else {
            tDescriptionEnUS = tCell.toString().trim();
            pStandardApi.setDescriptionEnUs(tDescriptionEnUS);
        }
        // (5)服務版本
        tColumnNumber = 5;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 5L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            tVersion = tCell.toString().trim();
            if (!tVersion.matches(ApimgmtConstant.REGEX_VERSION)) {
                uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 6L, null, tRowNumber, tColumnNumber, pLocale);
            }else {
                pStandardApi.addStandardApiVersion(tStandardApiVersion);
                pStandardApi.setBranch(ApiVersionServiceUtil.getBranchFromVersion(tVersion));
                tStandardApiVersion.setStandardApi(pStandardApi);
                tStandardApiVersion.setVersion(tVersion);
                tStandardApiVersion.setApplicant(pApplicant);
                ApprovedStatus approvedStatus = approvedStatusDao.get(ApimgmtConstant.APPROVED_STATUS_DRAFT);
                tStandardApiVersion.setApprovedStatus(approvedStatus);
            }
        }
        // (6)調用模式.
        tColumnNumber = 6;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 7L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            String tCellValue = tCell.toString().trim();
            Long tSyncTypeId = SyncTypeConstants.getCodeByName(tCellValue);
            if (-1L == tSyncTypeId) {
                uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 8L, null, tRowNumber, tColumnNumber, pLocale);
            } else {
                StandardApiSyncType tStandardApiSyncType = standardApiSyncTypeDao.get(tSyncTypeId);
                pStandardApi.setStandardApiSyncType(tStandardApiSyncType);
            }
        }
        // (7)分頁模式
        tColumnNumber = 7;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 9L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            String tPagingValue = tCell.toString().trim();
            if (!tPagingValue.matches(ApimgmtConstant.REGEX_NUMBER)) {
                uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 10L, null, tRowNumber, tColumnNumber, pLocale);
            } else {
                try {
                    tStandardApiPaging = standardApiPagingDao.get(Long.valueOf(tPagingValue.substring(0, tPagingValue.indexOf("."))));
                    Long tPaging = tStandardApiPaging.getId();
                    // 「.get」才能使用「2.僅支持分頁」及「3.同時支持」
                    if (pStandardApi.getApiType().equals(ApimgmtConstant.DEFAULT_API_TYPE) && !tApiName.endsWith(".get") && tPaging != 1) {
                        uploadCheckResult(tResult, APIExportImportEnums.CAN_NOT_PAGING.toString(), 11L, null, tRowNumber, tColumnNumber, pLocale);
                    }
                } catch (Exception e) {
                    uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 10L, null, tRowNumber, tColumnNumber, pLocale);
                }
                pStandardApi.setStandardApiPaging(tStandardApiPaging);
            }
        }
        // (8)服務叫用方
        tColumnNumber = 8;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 12L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            tRequester = tCell.toString().replaceAll(ApimgmtConstant.SPACE, "");
            pStandardApi.setRequester(tRequester);
        }
        // (9)服務接收方
        tColumnNumber = 9;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 13L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            tProvider = tCell.toString().replaceAll(ApimgmtConstant.SPACE, "");
            pStandardApi.setProvider(tProvider);
        }
        // (10)須分包
        tColumnNumber = 10;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 14L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            tIsBatch = ApimgmtConstant.YES.equals(tCell.toString());
            if (!tApiName.endsWith(".get") && tIsBatch) {
                uploadCheckResult(tResult, APIExportImportEnums.CAN_NOT_BATCH.toString(), 15L, null, tRowNumber, tColumnNumber, pLocale);
            }
            pStandardApi.setIsBatch(tIsBatch);
        }
        // (11)信息大類
        tColumnNumber = 11;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 16L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            String tGroupValue = tCell.toString().trim();
            if (!tGroupValue.matches(ApimgmtConstant.REGEX_NUMBER)) {
                uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 17L, null, tRowNumber, tColumnNumber, pLocale);
            } else {
                try {
                    tStandardApiGroup = standardApiGroupDao.get(Long.valueOf(tGroupValue.substring(0, tGroupValue.indexOf("."))));
                } catch (Exception e) {
                    uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 17L, null, tRowNumber, tColumnNumber, pLocale);
                }
                pStandardApi.setStandardApiGroup(tStandardApiGroup);
            }
        }
        // (12)信息類別
        tColumnNumber = 12;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 18L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            String tCategoryValue = tCell.toString().trim();
            if (!tCategoryValue.matches(ApimgmtConstant.REGEX_NUMBER)) {
                uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 19L, null, tRowNumber, tColumnNumber, pLocale);
            } else {
                try {
                    tStandardApiCategory = standarAPICategoryDao.get(Long.valueOf(tCategoryValue.substring(0, tCategoryValue.indexOf("."))));
                } catch (Exception e) {
                    uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 19L, null, tRowNumber, tColumnNumber, pLocale);
                }
                pStandardApi.setStandardApiCategory(tStandardApiCategory);
            }
        }
        // (13)信息格式
        tColumnNumber = 13;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 20L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            tMsgFormat = tCell.toString().trim();
            try {
                MessageFormatEnum.valueOf(tMsgFormat.toUpperCase());
            } catch (IllegalArgumentException e) {
                // 為匯入JSON 1.0特別設定
                if (tMsgFormat.equals(MessageFormatEnum.JSON10.getMsgFormat()) || tMsgFormat.equals(MessageFormatEnum.JSON_XML.getMsgFormat())) {
                    pStandardApi.setMsgFormat(tMsgFormat);
                } else {
                    uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 21L, null, tRowNumber, tColumnNumber, pLocale);
                }
            }
            pStandardApi.setMsgFormat(tMsgFormat);
        }
        // (14)信息標籤(繁體)
        tColumnNumber = 14;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            tTagZhTW = tCell.toString().trim();
            pStandardApi.setTagZhTw(tTagZhTW);
        }
        // (15)信息標籤(簡體)
        tColumnNumber = 15;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            tTagZhCN = tCell.toString().trim();
            pStandardApi.setTagZhCn(tTagZhCN);
        }
        // (16)信息標籤(英文)
        tColumnNumber = 16;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            tTagEnUS = tCell.toString().trim();
            pStandardApi.setTagEnUs(tTagEnUS);
        } else {
            // 若無英文先放繁中
            pStandardApi.setTagEnUs(tTagZhTW);
        }
        // (17)備註(繁體)
        tColumnNumber = 17;
        boolean tRemarkZhTWEmpty = false;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tRemarkZhTWEmpty = true;
        } else {
            tRemarkZhTW = tCell.toString().trim();
            pStandardApi.setRemarkZhTw(tRemarkZhTW);
        }

        // (18)備註(簡體)
        tColumnNumber = 18;
        boolean tRemarkZhCNEmpty = false;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tRemarkZhCNEmpty = true;
        } else {
            tRemarkZhCN = tCell.toString().trim();
            pStandardApi.setRemarkZhCn(tRemarkZhCN);
        }
        // 如果備註(繁體)、備註(簡體)兩個都沒有填寫，才會顯示必要欄位沒有填寫
        if (tRemarkZhTWEmpty && tRemarkZhCNEmpty) {
            uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 23L, null, tRowNumber, tColumnNumber, pLocale);
            // 如果備註(繁體)沒有填寫，用備註(簡體)翻譯的結果填入
        } else if (tRemarkZhTWEmpty) {
            tRemarkZhTW = localeTranslate(tRemarkZhCN, LocaleEnum.ZH2HANT.getType());
            pStandardApi.setRemarkZhTw(tRemarkZhTW);
            // 如果備註(簡體)沒有填寫，用備註(繁體)翻譯的結果填入
        } else if (tRemarkZhCNEmpty) {
            tRemarkZhCN = localeTranslate(tRemarkZhTW, LocaleEnum.ZH2HANS.getType());
            pStandardApi.setRemarkZhCn(tRemarkZhCN);
        }
        // (19)備註(英文)
        tColumnNumber = 19;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            tRemarkEnUS = tCell.toString().trim();
            pStandardApi.setRemarkEnUs(tRemarkEnUS);
        } else {
            pStandardApi.setRemarkEnUs(tRemarkZhTW);
        }
        // (20)租戶ID
        tColumnNumber = 20;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            tCell.setCellType(CellType.STRING);
            pStandardApi.setTenantId(tCell.toString().trim());
        } else {
            pStandardApi.setTenantId("");
        }
        // (21)修改自API
        tColumnNumber = 21;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            pStandardApi.setPackageName(tCell.toString().trim());
        }
        // (22)是否符合冪等性
        tColumnNumber = 22;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            IsIdempotency = false;
        } else {
            IsIdempotency = ApimgmtConstant.YES.equals(tCell.toString());
        }
        tStandardApiVersion.setIdempotency(IsIdempotency);
        // (23)修改紀錄
        tColumnNumber = 23;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            tStandardApiVersion.setUpdateHistory(tCell.toString().trim());
        }
//        // (25)分支
//        tColumnNumber = 25;
//        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
//        if (StringUtil.isEmptyOrSpace(tCell)) {
//            pStandardApi.setBranch(ApimgmtConstant.DEFAULT_BRANCH);
//        } else {
//            pStandardApi.setBranch(tCell.toString());
//        }
        if (ApimgmtConstant.API_TYPE_OPENAPI.equals(pStandardApi.getApiType())) {
//            // (25)协议
//            tColumnNumber = 25;
//            tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
//            if (StringUtil.isEmptyOrSpace(tCell)) {
//                pStandardApi.setProtocol(ApimgmtConstant.DEFAULT_PROTOCOL);
//            } else {
//                pStandardApi.setProtocol(tCell.toString());
//            }
            // (25)请求路径
            tColumnNumber = 25;
            tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            if (StringUtil.isEmptyOrSpace(tCell)) {
                uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), 39L,
                        null, tRowNumber, tColumnNumber, pLocale);
            } else {
                pStandardApi.setApiPath(tCell.toString());
            }
            // (26)请求方法
            tColumnNumber = 26;
            tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            if (StringUtil.isEmptyOrSpace(tCell)) {
                pStandardApi.setRequestMethod(ApimgmtConstant.DEFAULT_REQUEST_METHOD);
            } else {
                pStandardApi.setRequestMethod(tCell.toString());
            }
        } else {
            //esp默认赋值为POST
            pStandardApi.setRequestMethod(ApimgmtConstant.DEFAULT_REQUEST_METHOD);
        }
        // (29)关联产品名称
        tColumnNumber = 27;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        Long projectId = ApimgmtConstant.DEFAULT_PROJECT;
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            //校验产品名称
            List<Project> projectList = projectDao.getByName(tCell.toString().trim());
            if (CollUtil.isEmpty(projectList)) {
                uploadCheckResult(tResult, APIExportImportEnums.PROJECT_NOT_EXIST.toString(), null, null, 2, tColumnNumber + offset, null);
            } else {
                projectId = projectList.get(0).getProjectId();
            }
        }
        pStandardApi.setProjectId(projectId);
        // (30)关联产品版本
        List<ProjectVersionRelation> projectVersionRelationList = Collections.emptyList();
        tColumnNumber = 28;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell) && ObjectUtil.isNotNull(pStandardApi.getProjectId())) {
            //分隔数据
            String[] split = tCell.toString().trim().split(ApimgmtConstant.SPLITER);
            List<String> projectVersionNames = Arrays.stream(split).distinct().collect(Collectors.toList());
            //校验数据
            List<ProjectVersion> projectVersionList = ProjectVersionServiceUtil.queryProjectVersionList(projectId, projectVersionNames);
            if (projectVersionNames.size() != projectVersionList.size()) {
                uploadCheckResult(tResult, APIExportImportEnums.PROJECT_VERSION_NOT_EXIST.toString(), null, null, 2, tColumnNumber + offset, null);
            } else {
                //处理成对象
                Long projectIdTemp = projectId;
                projectVersionRelationList = projectVersionList.stream().map(projectVersion -> {
                    ProjectVersionRelation projectVersionRelation = new ProjectVersionRelation();
                    projectVersionRelation.setProjectId(projectIdTemp);
                    projectVersionRelation.setProjectVersionId(projectVersion.getProjectVersionId());
                    // 设置关联的StandardApi对象，这样在级联保存时能正确获取api_id
                    projectVersionRelation.setStandardApi(pStandardApi);
                    return projectVersionRelation;
                }).collect(Collectors.toList());
            }
        } else {
            projectVersionRelationList = new ArrayList<>();
            ProjectVersionRelation projectRelation = new ProjectVersionRelation();
            projectRelation.setProjectId(projectId);
            // 设置关联的StandardApi对象，这样在级联保存时能正确获取api_id
            projectRelation.setStandardApi(pStandardApi);
            projectVersionRelationList.add(projectRelation);
        }
        // 设置新的关联关系
        pStandardApi.setProjectVersionRelations(projectVersionRelationList);

        // 校验字段
        standardApiPostProcess(null, tResult, pStandardApi, pLocale);

        return tResult;
    }


    protected void standardApiPostProcess(Map<String, String> checkResult, List<Map<String, String>> tResult,
                                          StandardApi pStandardApi, String pLocale) {
        if (pStandardApi.validateName()) {
            if (prop.isEnvType()) {
                pStandardApi.setName(ApimgmtConstant.API_PAAS_REGEX + pStandardApi.getName());
            }
        } else {
            uploadCheckResult(tResult,
                    APIExportImportEnums.apiNameWrong(pStandardApi.getApiTypeOrDefault()),
                    APIExportImportEnums.apiNameWrongDescId(pStandardApi.getApiTypeOrDefault()),
                    null, 2, 1, pLocale);
            if (checkResult != null) {
                checkResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), tResult.get(0).get(ApimgmtConstant.DESCRIPTION));
            }
        }
    }

    /**
     * setStandardApiDataName
     */
    public List<Map<String, String>> setStandardApiDataName(StandardApi pStandardApi, Map<String, StandardApiDataName> pDataNameMap, Sheet pSheet, Boolean pCheckDataName, String pLocale, boolean pSkipReview, String pLoginAcct) throws Exception {
        List<Map<String, String>> tResult = new ArrayList<>();
        // 2022-06-02 Lillian 規格增加可排序欄位，過渡時期由程式自動判斷規格為新or舊
        Row tRow2 = pSheet.getRow(5);
        Cell tCell = tRow2.getCell(18, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        Cell tCellFilter = tRow2.getCell(10, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        boolean isNew = tCell.getStringCellValue().trim().equals("列舉值(繁體)") || tCell.getStringCellValue().trim().equals("列举值(繁体)");
        boolean isFilter = tCellFilter.getStringCellValue().trim().equals("可筛选");
        if (!isFilter) {
            uploadCheckResult(tResult, APIExportImportEnums.OLD_SHEET_FORMAT.toString(), null, null, 5, 10, pLocale);
            return tResult;
        }
        Map<String, StandardApiDataName> tFatherMap = new HashMap<>();
        ParseExcelHelper tParseExcelHelper = new ParseExcelHelper();
        Row tRow;
        int tMsgType = 0;
        Boolean[] tOnlyAddOnce = new Boolean[]{true};
        // if(isNew){
        // 讀取Request、Response Success與Response False資訊，第3列之前為StandardAPI資訊
        for (int tRowNumber = 3; tRowNumber < pSheet.getPhysicalNumberOfRows(); tRowNumber++) {
            tRow = pSheet.getRow(tRowNumber);
            // Cell(1)類型與Cell(4)辭彙代號必填欄位，若都沒有值判定為空列並略過.
            if ((tRow != null && tRow.getCell(1, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL) != null) || (tRow != null && tRow.getCell(4, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL) != null)) {
                if (!StringUtil.isEmptyOrSpace(tRow.getCell(1))) {
                    switch (tRow.getCell(1).getStringCellValue()) {
                        case ApimgmtConstant.REQUEST:
                            tMsgType = StandardApiMessageTypeConstants.REQUEST;
                            tRowNumber = tRowNumber + 2;
                            tParseExcelHelper.setTitleRowNum(tRowNumber);
                            tParseExcelHelper.setPhysicalNumberOfRequestTitleCells(pSheet.getRow(tRowNumber).getPhysicalNumberOfCells());
                            tFatherMap.clear();
                            break;
                        case ApimgmtConstant.REPONSE_SUCCESS:
                            tMsgType = StandardApiMessageTypeConstants.RESPONSE_SUCCESS;
                            tRowNumber = tRowNumber + 2;
                            tParseExcelHelper.setPhysicalNumberOfReponseSuccessTitleCells(pSheet.getRow(tRowNumber).getPhysicalNumberOfCells());
                            tFatherMap.clear();
                            break;
                        case ApimgmtConstant.REPONSE_FAILED:
                            tMsgType = StandardApiMessageTypeConstants.RESPONSE_FAILED;
                            tRowNumber = tRowNumber + 2;
                            tParseExcelHelper.setPhysicalNumberOfReponseFailedTitleCells(pSheet.getRow(tRowNumber).getPhysicalNumberOfCells());
                            tFatherMap.clear();
                            break;
                        default:
                            if (tMsgType == 0) {
                                // 如果讀取不到Request、Response success、Response
                                // failed就直接結束
                                uploadCheckResult(tResult, APIExportImportEnums.INCORRECT_SHEET_FORMAT.toString(), null, null, tRow.getRowNum(), ApimgmtConstant.EMPTY_NUMBER, pLocale);
                                return tResult;
                            } else {
                                // 讀取StandardAPIDataName資訊
                                setStandardApiDataNameByRow(pStandardApi, pSheet, tRow, tMsgType, tFatherMap, tParseExcelHelper, pDataNameMap, isNew, tResult, tOnlyAddOnce, pLocale);
                            }
                            break;
                    }

                } else {
                    uploadCheckResult(tResult, APIExportImportEnums.REQUIRED.toString(), null, null, tRow.getRowNum(), 1, pLocale);
                }
            }
        }
        return tResult;
    }

    public void addDataName(String pDataName, String pDescriptionZhTW, String pDescriptionZhCN, String pDescriptionEnUs,
                            long pApprovedStatus, String tBuildAcct, String pLoginAcct, LocalDateTime pBuildTime) {
        StandardDataName tStandardDataName = new StandardDataName();
        tStandardDataName.setDataName(pDataName);
        tStandardDataName.setDescriptionZhTw(pDescriptionZhTW);
        tStandardDataName.setDescriptionZhCn(pDescriptionZhCN);
        tStandardDataName.setDescriptionEnUs(pDescriptionEnUs);
        // 狀態:開發中
        tStandardDataName.setApprovedStatus(pApprovedStatus);
        // 核可人:登入帳號
        tStandardDataName.setApprovedAcct(pLoginAcct);
        // 核可日期
        tStandardDataName.setApprovedTime(LocalDateTime.now());
        tStandardDataName.setBuildTime(pBuildTime);
        tStandardDataName.setBuildAcct(tBuildAcct);
        tStandardDataName.setDirty(false);
        standardDataNameDao.saveOrupdate(tStandardDataName);
    }

    private String getExcelColumnName(int colNum) {
        StringBuilder colName = new StringBuilder();
        while (colNum > 0) {
            char tColumnValue;
            if (colNum > 26) {
                tColumnValue = (char) (colNum / 26 + 64);
                colNum = colNum%26;
            } else {
                tColumnValue = (char) ((colNum % 26 + 1) + 64);
                colNum -= 26;
            }
            colName.append(tColumnValue);
        }

        return colName.toString();
    }

    /**
     * uploadCheckResult
     */
    public void uploadCheckResult(List<Map<String, String>> pResult, String pReason, Long pSolutionId, String pApiName, int pRowNumber, int pColumnNumber, String pLocale) {
        int tRowValue = pRowNumber + 1;
        String tRow = String.valueOf(tRowValue);
        String tColumn = getExcelColumnName(pColumnNumber);
        Map<String, String> tErrorLog = new HashMap<>();
        if (pApiName == null) {
            tErrorLog.put("data_name", "");
        } else {
            tErrorLog.put("data_name", pApiName);
        }
        if (pRowNumber == 0 && pColumnNumber == 0) {
            tErrorLog.put("position", "");
        } else {
            tErrorLog.put("position", tColumn + ApimgmtConstant.SPLITER + tRow);
        }
        WrongDescription tWrongDescription;
        if (pSolutionId != null) {
            tWrongDescription = wrongDescriptionDao.get(pSolutionId);
            switch (pLocale) {
                case ApimgmtConstant.ZH_CN:
                    tErrorLog.put("description", tWrongDescription.getWrongDescChs());
                    tErrorLog.put("solution", tWrongDescription.getSolutionChs());
                    break;
                case ApimgmtConstant.EN_US:
                    tErrorLog.put("description", tWrongDescription.getWrongDescEn());
                    tErrorLog.put("solution", tWrongDescription.getSolutionEn());
                    break;
                case ApimgmtConstant.ZH_TW:
                default:
                    tErrorLog.put("description", tWrongDescription.getWrongDescCht());
                    tErrorLog.put("solution", tWrongDescription.getSolutionCht());
                    break;
            }
        } else {
            if (pReason.equals(APIExportImportEnums.INCORRECT_SHEET_FORMAT.toString())) {
                tErrorLog.put("description", "模版格式有誤，請參考範例模版");
            } else if (pReason.equals(APIExportImportEnums.APINAME_WRONG.toString())) {
                tErrorLog.put("description", "API名稱不正確，須由「小寫英文」、「數字」、「_」和「.」組成，不可有其它符號或空格");
            } else if (pReason.equals(APIExportImportEnums.OPEN_API_NAME_WRONG.toString())) {
                tErrorLog.put("description", "API名稱不正確，須由「大小寫英文」、「數字」、「_」和「.」組成，不可有其它符號或空格");
            } else if (pReason.equals(APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString())) {
                tErrorLog.put("description", "資料格式/內容有誤，請參考範例模版");
            } else if (pReason.equals(APIExportImportEnums.CAN_NOT_PAGING.toString())) {
                tErrorLog.put("description", "非get類型的API，不可使用分頁機制「2.僅支持分頁」、「3.同時支持」");
            } else if (pReason.equals(APIExportImportEnums.CAN_NOT_BATCH.toString())) {
                tErrorLog.put("description", "非get類型的API，不可使用分包機制");
            } else if (pReason.equals(APIExportImportEnums.REQUIRED.toString())) {
                tErrorLog.put("description", "必要欄位未填寫");
            } else if (pReason.equals(APIExportImportEnums.FATHER_REQUIRED.toString())) {
                tErrorLog.put("description", "欄位「類型」為HF、MF、D、DF、SD或SDF一定要有「所屬單頭/身」");
            } else if (pReason.equals(APIExportImportEnums.DATANAME_NOT_EXIST.toString())) {
                tErrorLog.put("description", "「辭彙代號」不存在");
            } else if (pReason.equals(APIExportImportEnums.LIST_ENUM_FORMAT_ERROR.toString())) {
                tErrorLog.put("description", "列舉值欄位格式錯誤");
            } else if (pReason.equals(APIExportImportEnums.NO_INPUT_PROD_NAME.toString())) {
                tErrorLog.put("description", "長度精度資訊未輸入產品名稱");
            } else if (pReason.equals(APIExportImportEnums.INCORRECT_FORMAT_OF_PRECISION.toString())) {
                tErrorLog.put("description", "長度精度格式錯誤");
            } else if (pReason.equals(APIExportImportEnums.INVALID_PROD_NAME.toString())) {
                tErrorLog.put("description", "無效的產品名稱");
            } else if (pReason.equals(APIExportImportEnums.CAN_NOT_FIND_FATHER.toString())) {
                tErrorLog.put("description", "未找到對應的「所屬單頭/身」");
            } else if (pReason.equals(APIExportImportEnums.DUPLICATE_DATANAME.toString())) {
                tErrorLog.put("description", "「辭彙代號」已存在於相同「所屬單頭/身」下");
            } else if (pReason.equals(APIExportImportEnums.API_VERSION_IS_EXIST.toString())) {
                tErrorLog.put("description", "当前API名稱与版本已被他人创建");
            } else if (pReason.equals(APIExportImportEnums.PREVI_API_VERSION_IS_EXIST.toString())) {
                tErrorLog.put("description", "API上一版本不存在，請確認上一版是否已匯入");
            } else if (pReason.equals(APIExportImportEnums.PREV_API_BRANCH_NOT_EXIST.toString())) {
                tErrorLog.put("description", "API上一分支不存在，請確認上一版是否已匯入");
            } else if (pReason.equals(APIExportImportEnums.PREVI_IS_DEVELOPING.toString())) {
                tErrorLog.put("description", "API上一版本為「草稿」或「開發中」，請定版後再進行匯入");
            } else if (pReason.equals(APIExportImportEnums.IS_BATCH_CAN_NOT_CHANGE.toString())) {
                tErrorLog.put("description", "「須分包」與上一版不一致，「須分包」不可以異動");
            } else if (pReason.equals(APIExportImportEnums.SYNC_TYPE_CAN_NOT_CHANGE.toString())) {
                tErrorLog.put("description", " 「調用模式」與上一版不一致，「調用模式」不可以異動");
            } else if (pReason.equals(APIExportImportEnums.DATA_TYPE_WRONG.toString())) {
                tErrorLog.put("description", "該字段類型不可為E");
            } else if (pReason.equals(APIExportImportEnums.CAN_NOT_INPUT_PRECISIONS.toString())) {
                tErrorLog.put("description", "該字段的辭彙型態不可填寫長度和精度，只有辭彙型態string和numeric字段才可填寫產品的對應欄位的長度和精度");
            } else if (pReason.equals(APIExportImportEnums.OLD_SHEET_FORMAT.toString())) {
                tErrorLog.put("description", "当前为旧模板，请使用新模板");
            } else if (pReason.equals(APIExportImportEnums.API_VERSION_NOT_COMPLIANCE.toString())) {
                tErrorLog.put("description", "当前版本号错误，请重新输入");
            } else if (pReason.equals(APIExportImportEnums.PARENT_NAME_INCONSISTENT.toString())) {
                tErrorLog.put("description", "父类API不一致，建议不更改父类");
            } else if (pReason.equals(APIExportImportEnums.PARENT_API_NOT_CONFIRMED.toString())) {
                tErrorLog.put("description", "父类API不存在|父类API不存在已定版的版本");
            } else if (pReason.equals(APIExportImportEnums.PROJECT_NOT_EXIST.toString())) {
                tErrorLog.put("description", "关联的产品不存在");
            } else if (pReason.equals(APIExportImportEnums.PROJECT_VERSION_NOT_EXIST.toString())) {
                tErrorLog.put("description", "关联的产品版本不存在");
            }
        }
        pResult.add(tErrorLog);
    }

    /**
     * setStandardApiDataNameByRow
     */
    public void setStandardApiDataNameByRow(StandardApi pStandardApi, Sheet pSheet, Row pRow, Integer pMsgType, Map<String, StandardApiDataName> tFatherMap, ParseExcelHelper tParseExcelHelper, Map<String, StandardApiDataName> pDataNameMap, boolean pIsNew, List<Map<String, String>> pResult, Boolean[] tOnlyAddOnce, String pLocale) throws Exception {
        int tRowNumber = pRow.getRowNum();
        int tColumnNumber;
        String tColumnType, tFatherDataName = null, tDataName = null, tDataType;
        String tIsBusinesskey;
        String tIsRequired;
        boolean tIsDatakey, tIsArray, tCanSort, tCanFilter;
        String tDescriptionZhTW = null, tDescriptionZhCN = null, tDescriptionEnUS, tValue;
        String tRemarkZhTW = null, tRemarkZhCN = null, tRemarkEnUS;
        boolean tFatherDataNameRequired = false;
        String tFatherColumnType = null;
        String tListEnumZhTw, tListEnumZhCN, tListEnumEnUS;
        String tUpdateHistoryString;
        StandardApiDataName tStandardApiDataName = new StandardApiDataName();
        Cell tCell;
        // (0)(原空欄改成)is_businesskey
        // (1)類型 (2)所屬單頭/身 (3)辭彙檢查 (4)辭彙代號 (5)辭彙型態 (6)可為空
        // (7)為Datakey (8)為數組(array) (9)可排序
        // (10)說明(繁體) (11)說明(簡體) (12)說明(英文) (13)範例值
        // (14)備註(繁體) (15)備註(簡體) (16)備註(英文)
        // (17)列舉值(繁體) (18)列舉值(簡體) (19)列舉值(英文)
        // (20)修改紀錄 (21)長度精度
        // (0)(原空欄改成)is_businesskey
        tColumnNumber = 0;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tStandardApiDataName.setIsBusinesskey(Boolean.FALSE);
        } else {
            tIsBusinesskey = tCell.toString().trim();
            switch (tIsBusinesskey.toUpperCase()) {
                case ApimgmtConstant.YES:
                    tStandardApiDataName.setIsBusinesskey(Boolean.TRUE);
                    break;
                case ApimgmtConstant.NO:
                    tStandardApiDataName.setIsBusinesskey(Boolean.FALSE);
                    break;
                default:
                    uploadCheckResult(pResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), null, null, tRowNumber, tColumnNumber, pLocale);
            }
        }
        tStandardApiDataName.setNullable(false);
        // (1)類型
        tColumnNumber = 1;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(pResult, APIExportImportEnums.REQUIRED.toString(), 24L, null, tRowNumber, tColumnNumber, pLocale);
        }
        tColumnType = tCell.toString().trim();

        try {
            ColumnTypeEnum.valueOf(tColumnType.toUpperCase());
        } catch (IllegalArgumentException e) {
            if (!tColumnType.toUpperCase().matches(ApimgmtConstant.REGEX_COLUMN_TYPE_DETAIL_ALL)) {
                uploadCheckResult(pResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 24L, null, tRowNumber, tColumnNumber, pLocale);
            }
        }
        tStandardApiDataName.setColumnType(tColumnType);
        // (2)所屬單頭/身
        tColumnNumber = 2;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            if (tFatherDataNameRequired) {
                uploadCheckResult(pResult, APIExportImportEnums.FATHER_REQUIRED.toString(), null, null, tRowNumber, tColumnNumber, pLocale);
            }
        } else {
            tFatherDataName = tCell.toString().trim();
        }

        // (3)辭彙檢查.
        tColumnNumber = 3;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);

        // (4)辭彙代號
        tColumnNumber = 4;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(pResult, APIExportImportEnums.REQUIRED.toString(), 27L, null, tRowNumber, tColumnNumber, pLocale);
        } else {
            tDataName = tCell.toString().replaceAll(ApimgmtConstant.SPACE, "");
            // 檢查辭彙代號命名
            if (tDataName.matches(ApimgmtConstant.REGEX_API_DATA_NAME) || Arrays.asList(ApimgmtConstant.ALLOW_API_DATA_NAME).contains(tDataName)) {
                tStandardApiDataName.setStandardDataName(tDataName);
            } else {
                uploadCheckResult(pResult, null, 37L, tDataName, tRowNumber, tColumnNumber, pLocale);
            }
        }

        // (5)辭彙型態
        tColumnNumber = 5;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!Arrays.asList(ApimgmtConstant.ALLOW_ARRAY_COLUMN_TYPE).contains(tColumnType.toUpperCase()) && !tColumnType.toUpperCase().matches(ApimgmtConstant.REGEX_COLUMN_TYPE_DETAIL_HEADER)) {
            if (StringUtil.isEmptyOrSpace(tCell)) {
                uploadCheckResult(pResult, APIExportImportEnums.REQUIRED.toString(), 29L, tDataName, tRowNumber, tColumnNumber, pLocale);
            } else {
                tDataType = tCell.toString().trim();
                try {
                    DataTypeEnum.valueOf(tDataType.toUpperCase());
                } catch (IllegalArgumentException e) {
                    uploadCheckResult(pResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 29L, tDataName, tRowNumber, tColumnNumber, pLocale);
                }
                tStandardApiDataName.setDataType(tDataType.toLowerCase());
            }
        }

        // (6)必要
        tColumnNumber = 6;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        // 沒填, 則預設為FALSE
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tStandardApiDataName.setIsRequired(Boolean.FALSE);
        } else {
            tIsRequired = tCell.toString().trim();
            switch (tIsRequired.toUpperCase()) {
                case ApimgmtConstant.YES:
                    tStandardApiDataName.setIsRequired(Boolean.TRUE);
                    break;
                case ApimgmtConstant.NO:
                    tStandardApiDataName.setIsRequired(Boolean.FALSE);
                    break;
                default:
                    uploadCheckResult(pResult, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), 30L, tDataName, tRowNumber, tColumnNumber, pLocale);
            }
        }

        // (7)為Datakey
        tColumnNumber = 7;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tIsDatakey = false;
        } else {
            tIsDatakey = ApimgmtConstant.YES.equals(tCell.toString());
        }
        tStandardApiDataName.setIsDatakey(tIsDatakey);

        // (8)為Array，只有M,D,SD,4D,5D...允許為array，若空值=false
        tColumnNumber = 8;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tIsArray = false;
        } else {
            tIsArray = ApimgmtConstant.YES.equals(tCell.toString());
        }
        tStandardApiDataName.setIsArray(tIsArray);

        // (9)可排序，只有P和F結尾才允許可排序，若空值=false
        tColumnNumber = 9;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tCanSort = false;
        } else {
            tCanSort = ApimgmtConstant.YES.equals(tCell.toString());
        }
        tStandardApiDataName.setCanSort(tCanSort);

        // (10)可过滤，只有P和F結尾才允許可过滤，若空值=false
        tColumnNumber = 10;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tCanFilter = false;
        } else {
            tCanFilter = ApimgmtConstant.YES.equals(tCell.toString());
        }
        tStandardApiDataName.setCanFilter(tCanFilter);

        // (11)說明(繁體)
        tColumnNumber = 11;
        boolean tDescriptionZhTwEmpty = false;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tDescriptionZhTwEmpty = true;
        } else {
            tDescriptionZhTW = tCell.toString().trim();
            tStandardApiDataName.setDescriptionZhTw(tDescriptionZhTW);
        }

        // (12)說明(簡體)
        tColumnNumber = 12;
        boolean tDescriptionZhCnEmpty = false;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tDescriptionZhCnEmpty = true;
        } else {
            tDescriptionZhCN = tCell.toString().trim();
            tStandardApiDataName.setDescriptionZhCn(tDescriptionZhCN);
        }

        // 如果服務說明(繁體)、服務說明(簡體)兩個都沒有填寫，才會顯示必要欄位沒有填寫
        if (tDescriptionZhTwEmpty && tDescriptionZhCnEmpty) {
            uploadCheckResult(pResult, APIExportImportEnums.REQUIRED.toString(), 31L, tDataName, tRowNumber, tColumnNumber, pLocale);
            // 如果服務說明(繁體)沒有填寫，用服務說明(簡體)翻譯的結果填入
        } else if (tDescriptionZhTwEmpty) {
            tDescriptionZhTW = localeTranslate(tDescriptionZhCN, LocaleEnum.ZH2HANT.getType());
            tStandardApiDataName.setDescriptionZhTw(tDescriptionZhTW);
            // 如果服務說明(簡體)沒有填寫，用服務說明(繁體)翻譯的結果填入
        } else if (tDescriptionZhCnEmpty) {
            tDescriptionZhCN = localeTranslate(tDescriptionZhTW, LocaleEnum.ZH2HANS.getType());
            tStandardApiDataName.setDescriptionZhCn(tDescriptionZhCN);
        }

        // (13)說明(英文)
        tColumnNumber = 13;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            tDescriptionEnUS = tCell.toString().trim();
            tStandardApiDataName.setDescriptionEnUs(tDescriptionEnUS);
        } else {
            if (null != tDataName) {
                tStandardApiDataName.setDescriptionEnUs(tDataName.replace("_", " "));
            }
        }

        // (14)範例值
        tColumnNumber = 14;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            tValue = tCell.toString().trim();
            tStandardApiDataName.setValue(tValue);
        }

        // (15)備註(繁體)
        tColumnNumber = 15;
        boolean tRemarkZhTwEmpty = false;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tRemarkZhTwEmpty = true;
        } else {
            tRemarkZhTW = tCell.toString().trim();
            tStandardApiDataName.setRemarkZhTw(tRemarkZhTW);
        }

        // (16)備註(簡體)
        tColumnNumber = 16;
        boolean tRemarkZhCnEmpty = false;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tRemarkZhCnEmpty = true;
        } else {
            tRemarkZhCN = tCell.toString().trim();
            tStandardApiDataName.setRemarkZhCn(tRemarkZhCN);
        }

        // 如果備註(繁體)、備註(簡體)兩個都沒有填寫就不管
        if (tRemarkZhTwEmpty && tRemarkZhCnEmpty) {

            // 如果備註(繁體)沒有填寫，用備註(簡體)翻譯的結果填入
        } else if (tRemarkZhTwEmpty) {
            tRemarkZhTW = localeTranslate(tRemarkZhCN, LocaleEnum.ZH2HANT.getType());
            tStandardApiDataName.setRemarkZhTw(tRemarkZhTW);
            // 如果備註(簡體)沒有填寫，用備註(繁體)翻譯的結果填入
        } else if (tRemarkZhCnEmpty) {
            tRemarkZhCN = localeTranslate(tRemarkZhTW, LocaleEnum.ZH2HANS.getType());
            tStandardApiDataName.setRemarkZhCn(tRemarkZhCN);
        }

        // (17)備註(英文)
        tColumnNumber = 17;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tStandardApiDataName.setRemarkEnUs(tRemarkZhTW);// 若無英文先放中文
        } else {
            tRemarkEnUS = tCell.toString().trim();
            tStandardApiDataName.setRemarkEnUs(tRemarkEnUS);
        }

        if (!pIsNew) {
            // (18)列舉值
            tColumnNumber = 18;
            tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            String tListEnumString;
            if (!StringUtil.isEmptyOrSpace(tCell)) {
                tListEnumString = tCell.toString().trim();
                // 檢查格式，正確則解析出三個語系的內容
                if (tListEnumString.matches(ApimgmtConstant.REGEX_LIST_ENUM_MULT)) {
                    // zh_TW
                    tStandardApiDataName.setListEnumZhTw(tListEnumString.split("zh_TW:")[1].split("\\[")[1].split("\\]")[0]);
                    tStandardApiDataName.setListEnumZhCn(tListEnumString.split("zh_CN:")[1].split("\\[")[1].split("\\]")[0]);
                    tStandardApiDataName.setListEnumEn(tListEnumString.split("en_US:")[1].split("\\[")[1].split("\\]")[0]);
                } else {
                    uploadCheckResult(pResult, APIExportImportEnums.LIST_ENUM_FORMAT_ERROR.toString(), 32L, tDataName, tRowNumber, tColumnNumber, pLocale);
                }
            }

        } else {
            // (18)列舉值(繁體)
            tColumnNumber = 18;
            tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            if (!StringUtil.isEmptyOrSpace(tCell)) {
                tListEnumZhTw = tCell.toString().trim();
                tStandardApiDataName.setListEnumZhTw(tListEnumZhTw);
            }

            // (19)列舉值(簡體)
            tColumnNumber = 19;
            tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            if (!StringUtil.isEmptyOrSpace(tCell)) {
                tListEnumZhCN = tCell.toString().trim();
                tStandardApiDataName.setListEnumZhCn(tListEnumZhCN);
            }

            // (20)列舉值(英文)
            tColumnNumber = 20;
            tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            if (!StringUtil.isEmptyOrSpace(tCell)) {
                tListEnumEnUS = tCell.toString().trim();
                tStandardApiDataName.setListEnumEn(tListEnumEnUS);
            }
        }

        // (21)修改紀錄
        tColumnNumber = pIsNew ? 21 : 19;
        tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            tUpdateHistoryString = null;
        } else {
            tUpdateHistoryString = tCell.toString().trim();
        }
        tStandardApiDataName.setUpdateHistory(tUpdateHistoryString);

        // (22)之後的為長度精度資訊
        tColumnNumber = pIsNew ? 22 : 20;
        int tMaxNum = -1;
        if (pMsgType == StandardApiMessageTypeConstants.REQUEST) {
            tMaxNum = tParseExcelHelper.getPhysicalNumberOfRequestTitleCells();
        }
        if (pMsgType == StandardApiMessageTypeConstants.RESPONSE_SUCCESS) {
            tMaxNum = tParseExcelHelper.getPhysicalNumberOfReponseSuccessTitleCells();
        }

        if (pMsgType == StandardApiMessageTypeConstants.RESPONSE_FAILED) {
            tMaxNum = tParseExcelHelper.getPhysicalNumberOfReponseFailedTitleCells();
        }
        for (; tColumnNumber < tMaxNum; tColumnNumber++) {
            ProductApiDataName tProductApiDataName = new ProductApiDataName();
            //1、有产品 但是产品不存在
            //2、配置对应的数据，但是产品没有配置
            //查询产品存不存在
            Cell tCellPro = pSheet.getRow(tParseExcelHelper.getTitleRowNum()).getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            if (!StringUtil.isEmptyOrSpace(tCellPro)) {
                String tProdName = tCellPro.toString().trim();
                if (!productDao.exists(tProdName)) {
                    if (tOnlyAddOnce[0]) {
                        uploadCheckResult(pResult, APIExportImportEnums.INVALID_PROD_NAME.toString(), 33L, tDataName, tCellPro.getRowIndex(), tCellPro.getColumnIndex(), pLocale);
                        tOnlyAddOnce[0] = false;
                    }
                } else {
                    Product tProduct = productDao.get(tProdName);
                    tProductApiDataName.setProduct(tProduct);
                    tProductApiDataName.setApiDataName(tStandardApiDataName);
                    tStandardApiDataName.addProductApiDataName(tProductApiDataName);
                }
            }
            tCell = pRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            if (!StringUtil.isEmptyOrSpace(tCell)) {
                tCell.setCellType(CellType.STRING);
                try {
                    if ("string".equals(tStandardApiDataName.getDataType())) {
                        tProductApiDataName.setLength(Integer.parseInt(tCell.toString().trim()));
                    } else if ("numeric".equals(tStandardApiDataName.getDataType())) {
                        tProductApiDataName.setLength(Integer.parseInt(tCell.toString().trim().split(",")[0]));
                        tProductApiDataName.setPrecision(Integer.parseInt(tCell.toString().trim().split(",")[1]));
                    } else {
                        uploadCheckResult(pResult, APIExportImportEnums.CAN_NOT_INPUT_PRECISIONS.toString(), 36L, tDataName, tRowNumber, tColumnNumber, pLocale);
                    }
                } catch (NumberFormatException | ArrayIndexOutOfBoundsException ex) {
                    uploadCheckResult(pResult, APIExportImportEnums.INCORRECT_FORMAT_OF_PRECISION.toString(), 34L, tDataName, tRowNumber, tColumnNumber, pLocale);
                }
                if (StringUtil.isEmptyOrSpace(tCellPro) && tOnlyAddOnce[0]) {
                    // 沒有提供長度精度的產品名稱
                    uploadCheckResult(pResult, APIExportImportEnums.NO_INPUT_PROD_NAME.toString(), 33L, tDataName, tRowNumber, tColumnNumber, pLocale);
                    tOnlyAddOnce[0] = false;
                }
            }
        }

        // 沒有提供API版號就返回
        if (pStandardApi.getStandardApiVersions().size() == 0) {
            return;
        }

        tStandardApiDataName.setStandardApiVersion(pStandardApi.getStandardApiVersions().get(0));
        tStandardApiDataName.setMsgType(pMsgType);
        tStandardApiDataName.setApiVer(ApimgmtConstant.API_VERSION);

        // 「類型」為E時須檢查字段名稱是否為code、sql_code、description
        if (tColumnType.equals(ColumnTypeEnum.E.toString())) {
            if (!ArrayUtils.contains(ApimgmtConstant.COLUMN_TYPE_E_DATA_NAME, tDataName)) {
                uploadCheckResult(pResult, APIExportImportEnums.DATA_TYPE_WRONG.toString(), 35L, tDataName, tRowNumber, 1, pLocale);
            }
        }

        // 「類型」非P、M、E、H、Datakey時須檢查「所屬單頭/身」
        if (!ArrayUtils.contains(ApimgmtConstant.DATANAME_FATHER_UNREQUIRED, tColumnType.toUpperCase())) {
            tFatherDataNameRequired = true;

            switch (tColumnType) {
                case StandardApiColumnTypeConstants.HEADER_FIELD_CONSTANTS:
                    tFatherColumnType = ColumnTypeEnum.H.toString();
                    break;
                case StandardApiColumnTypeConstants.MASTER_FIELD_CONSTANTS:
                case StandardApiColumnTypeConstants.DETAIL_CONSTANTS:
                    tFatherColumnType = ColumnTypeEnum.M.toString();
                    break;
                case StandardApiColumnTypeConstants.DETAIL_FIELD_CONSTANTS:
                case StandardApiColumnTypeConstants.SMALL_DETAIL_CONSTANTS:
                    tFatherColumnType = ColumnTypeEnum.D.toString();
                    break;
                case StandardApiColumnTypeConstants.SMALL_DETAIL_FIELD_CONSTANTS:
                case StandardApiColumnTypeConstants.LEVEL4_DETAIL_CONSTANTS:
                    tFatherColumnType = ColumnTypeEnum.SD.toString();
                    break;
                default:
                    if (tColumnType.endsWith("DF")) {
                        tFatherColumnType = tColumnType.replace("F", "");
                    } else if (tColumnType.endsWith("D")) {
                        int level = Integer.parseInt(tColumnType.toUpperCase().replace("D", ""));
                        tFatherColumnType = (level - 1) + "D";
                    }
                    break;
            }

            StandardApiDataName tFatherStandardApiDataName = tFatherMap.get(tFatherColumnType + tFatherDataName + pMsgType);
            if (tFatherStandardApiDataName == null) {
                uploadCheckResult(pResult, APIExportImportEnums.CAN_NOT_FIND_FATHER.toString(), 25L, tDataName, tRowNumber, 2, pLocale);
            } else {
                for (StandardApiDataName tExistDataName : tFatherStandardApiDataName.getChildrens()) {
                    if (null != tDataName && tDataName.equals(tExistDataName.getStandardDataName())) {
                        uploadCheckResult(pResult, APIExportImportEnums.DUPLICATE_DATANAME.toString(), 28L, tDataName, tRowNumber, 4, pLocale);
                    }
                }
                tFatherStandardApiDataName.addChildren(tStandardApiDataName);
                tStandardApiDataName.setParent(tFatherStandardApiDataName);
                pDataNameMap.put(tColumnType + tDataName + pMsgType + tFatherDataName, tStandardApiDataName);
            }
            if (tColumnType.endsWith("D")) {
                tFatherMap.put(tColumnType + tDataName + pMsgType, tStandardApiDataName);
            }
        } else {
            // 檢查「類型」為Datakey、P、E、M、H，辭彙代號是否重複
            if (tColumnType.toUpperCase().equals(ColumnTypeEnum.DATAKEY.toString()) || tColumnType.toUpperCase().equals(ColumnTypeEnum.P.toString()) || tColumnType.toUpperCase().equals(ColumnTypeEnum.E.toString()) || tColumnType.toUpperCase().equals(ColumnTypeEnum.M.toString()) || tColumnType.toUpperCase().equals(ColumnTypeEnum.H.toString())) {
                StandardApiDataName tDataName2 = pDataNameMap.get(tColumnType + tDataName + pMsgType);
                if (tDataName2 != null) {
                    uploadCheckResult(pResult, APIExportImportEnums.DUPLICATE_DATANAME.toString(), 28L, tDataName, tRowNumber, 4, pLocale);
                }
            }
            // 「類型」為H或M
            if (tColumnType.toUpperCase().equals(ColumnTypeEnum.H.toString()) || tColumnType.toUpperCase().equals(ColumnTypeEnum.M.toString())) {
                tFatherMap.put(tColumnType + tDataName + pMsgType, tStandardApiDataName);
            }
            // 「類型」為Datakey、P、E
            pStandardApi.getStandardApiVersions().get(0).addStandardApiDataName(tStandardApiDataName);
            pDataNameMap.put(tColumnType + tDataName + pMsgType, tStandardApiDataName);
        }
    }

    public void addDuplicateDescription(Workbook pWorkbook, Sheet pSheet, String pDuplicateType, String pOtherUser) {
        int firstRowNumber = 0;
        int firstCellNumber = 0;
        Row row = pSheet.getRow(firstRowNumber);
        Cell cell = row.getCell(firstCellNumber);
        String tDuplicateDescription = "";
        switch (String.valueOf(pDuplicateType)) {
            // 已存在于草稿中
            case "1":
                tDuplicateDescription = "導入失敗原因：對應API正在草稿中";
                break;
            // 正在审核中
            case "2":
                tDuplicateDescription = "導入失敗原因：對應API正在审核中";
                break;
            // 正在开发中
            case "3":
                tDuplicateDescription = "導入失敗原因：對應API正在开发中";
                break;// 正在开发中
            case "4":
                tDuplicateDescription = "導入失敗原因：對應API已定版";
                break;
            // 其他使用者已經匯入過此API版本號
            case "5":
                tDuplicateDescription = "導入失敗原因：其他使用者(" + pOtherUser + ")已經匯入過此API版本";
                break;

            default:
                break;
        }
        CellStyle style = pWorkbook.createCellStyle();
        Font font = pWorkbook.createFont();
        font.setColor(HSSFColorPredefined.RED.getIndex());
        style.setWrapText(true);
        style.setFont(font);
        cell.setCellStyle(style);
        cell.setCellValue(tDuplicateDescription);
    }

    /**
     * 匯出錯誤明細
     */
    public void writeImportErrorFile(Sheet pSheet, Map<String, List<Map<String, String>>> pTotalApiErrorMap) {
        // 標題row
        Row row = pSheet.createRow(0);
        row.createCell(0).setCellValue("服務名稱");
        row.createCell(1).setCellValue("版本");
        row.createCell(2).setCellValue("位置");
        row.createCell(3).setCellValue("辭彙代號");
        row.createCell(4).setCellValue("問題說明");
        row.createCell(5).setCellValue("解決方案");
        int rownum = 1;
        for (String tKey : pTotalApiErrorMap.keySet()) {
            for (Map<String, String> tError : pTotalApiErrorMap.get(tKey)) {
                String[] tApiInfo = tKey.split("/");
                Row tRow = pSheet.createRow(rownum);
                tRow.createCell(0).setCellValue(tApiInfo[0]);
                tRow.createCell(1).setCellValue(tApiInfo.length > 1 ? tApiInfo[1] : "");
                if (tError.get("position") != null) {
                    tRow.createCell(2).setCellValue(tError.get("position"));
                }
                if (tError.get("data_name") != null) {
                    tRow.createCell(3).setCellValue(tError.get("data_name"));
                }
                if (tError.get("description") != null) {
                    tRow.createCell(4).setCellValue(tError.get("description"));
                }
                if (tError.get("solution") != null) {
                    tRow.createCell(5).setCellValue(tError.get("solution"));
                }
                rownum++;
            }
        }
    }

    public Map<String, String> setStandardApiBasicInfo(StandardApi pStandardApi, String pApplicant, Sheet pSheet) {
        Row tRow;
        Cell tCell;
        int tRowNumber = 0;
        int tColumnNumber = 1;
        String tApiName, tVersion;
        String tRequester;
        StandardApiVersion tStandardApiVersion = new StandardApiVersion();
        List<Map<String, String>> result = new ArrayList<>();
        Map<String, String> tResult = new HashMap<>();
        // 讀取StandardAPI資訊(第0列開始)
        tRow = pSheet.getRow(tRowNumber);
        if (tRow == null || tRow.getCell(1, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL) == null || !tRow.getCell(1).getStringCellValue().equals(ApimgmtConstant.STANDARD_API)) {
            // 沒有讀取到StandardAPI就直接結束
            uploadCheckResult(result, APIExportImportEnums.INCORRECT_SHEET_FORMAT.toString(), null, null, tRowNumber, tColumnNumber, null);
            tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
            return tResult;
        }
        tRowNumber = 2;
        tRow = pSheet.getRow(tRowNumber);
        // (0)空欄
        // (1)服務名稱 (2)服務說明(繁體) (3)服務說明(簡體) (4)服務說明(英文)
        // (5)服務版本 (6)調用模式 (7)分頁模式 (8)服務叫用方 (9)服務接收方 (10)須分包
        // (11)信息大類 (12)信息類別 (13)信息格式
        // (14)信息標籤(繁體) (15)信息標籤(簡體) (16)信息標籤(英文)
        // (17)備註(繁體) (18)備註(簡體) (19)備註(英文)
        // (20)租戶ID (21)修改自API (22)是否符合冪等性 (23)修改紀錄
        // (24)API类型  (25)请求路径 (26)请求方法 (27)关联产品名称 (28)关联产品版本
        // (1)服務名稱
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            // 沒有填服務名稱就直接結束
            uploadCheckResult(result, APIExportImportEnums.REQUIRED.toString(), null, null, tRowNumber, tColumnNumber, null);
            tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
            return tResult;
        } else {
            tApiName = tCell.toString().trim();
            pStandardApi.setName(tApiName);
        }
        // (5)服務版本
        tColumnNumber = 5;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(result, APIExportImportEnums.REQUIRED.toString(), null, null, tRowNumber, tColumnNumber, null);
            tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
            return tResult;
        } else {
            tVersion = tCell.toString().trim();
            if (!tVersion.matches(ApimgmtConstant.REGEX_VERSION)) {
                uploadCheckResult(result, APIExportImportEnums.INCORRECT_FORMAT_OR_CONTENT.toString(), null, null, tRowNumber, tColumnNumber, null);
                tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
                return tResult;
            }
            pStandardApi.addStandardApiVersion(tStandardApiVersion);
            //设置分支
            pStandardApi.setBranch(ApiVersionServiceUtil.getBranchFromVersion(tVersion));
            tStandardApiVersion.setStandardApi(pStandardApi);
            tStandardApiVersion.setVersion(tVersion);
            tStandardApiVersion.setApplicant(pApplicant);
            ApprovedStatus approvedStatus = approvedStatusDao.get(ApimgmtConstant.APPROVED_STATUS_DRAFT);
            tStandardApiVersion.setApprovedStatus(approvedStatus);
        }
        // (8)服務叫用方
        tColumnNumber = 8;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(result, APIExportImportEnums.REQUIRED.toString(), null, null, tRowNumber, tColumnNumber, null);
            tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
            return tResult;
        } else {
            tRequester = tCell.toString().replaceAll(ApimgmtConstant.SPACE, "");
            pStandardApi.setRequester(tRequester);
        }
        // (20)租戶ID
        tColumnNumber = 20;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            tCell.setCellType(CellType.STRING);
            pStandardApi.setTenantId(tCell.toString().trim());
        } else {
            pStandardApi.setTenantId("");
        }
        // (21)修改自API
        tColumnNumber = 21;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell)) {
            pStandardApi.setPackageName(tCell.toString().trim());
        }
        // (8)服務叫用方
        tColumnNumber = 8;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            uploadCheckResult(result, APIExportImportEnums.REQUIRED.toString(), null, null, tRowNumber, tColumnNumber, null);
            tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
            return tResult;
        } else {
            tRequester = tCell.toString().replaceAll(ApimgmtConstant.SPACE, "");
            pStandardApi.setRequester(tRequester);
        }
        // (24)API类型
        tColumnNumber = 24;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            pStandardApi.setApiType(ApimgmtConstant.DEFAULT_API_TYPE);
        } else {
            pStandardApi.setApiType(ApiTypeConstants.getCodeByName(tCell.toString()));
        }
        if (ApimgmtConstant.API_TYPE_OPENAPI.equals(pStandardApi.getApiType())) {
//            // (26)协议
//            tColumnNumber = 26;
//            tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
//            if (StringUtil.isEmptyOrSpace(tCell)) {
//                pStandardApi.setProtocol(ApimgmtConstant.DEFAULT_PROTOCOL);
//            } else {
//                pStandardApi.setProtocol(tCell.toString());
//            }
            // (25)请求路径
            tColumnNumber = 25;
            tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            if (!StringUtil.isEmptyOrSpace(tCell)) {
                pStandardApi.setApiPath(tCell.toString());
            }
            // (26)请求方法
            tColumnNumber = 26;
            tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
            if (StringUtil.isEmptyOrSpace(tCell)) {
                pStandardApi.setRequestMethod(ApimgmtConstant.DEFAULT_REQUEST_METHOD);
            } else {
                pStandardApi.setRequestMethod(tCell.toString());
            }
        }
        // (29)关联产品名称
        tColumnNumber = 27;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (StringUtil.isEmptyOrSpace(tCell)) {
            pStandardApi.setProjectId(ApimgmtConstant.DEFAULT_PROJECT);
        } else {
            //校验产品名称
            List<Project> projectList = projectDao.getByName(tCell.toString().trim());
            if (CollUtil.isEmpty(projectList)) {
                uploadCheckResult(result, APIExportImportEnums.PROJECT_NOT_EXIST.toString(), null, null, 2, 29, null);
                tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
            } else {
                pStandardApi.setProjectId(projectList.get(0).getProjectId());
            }
        }
        // (30)关联产品版本
        tColumnNumber = 28;
        tCell = tRow.getCell(tColumnNumber, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
        if (!StringUtil.isEmptyOrSpace(tCell) && ObjectUtil.isNotNull(pStandardApi.getProjectId())) {
            //分隔数据
            String[] split = tCell.toString().trim().split(ApimgmtConstant.SPLITER);
            List<String> projectVersionNames = Arrays.stream(split).distinct().collect(Collectors.toList());
            //校验数据
            if (!ProjectVersionServiceUtil.judgeProjectVersionNames(pStandardApi.getProjectId(), projectVersionNames)) {
                uploadCheckResult(result, APIExportImportEnums.PROJECT_VERSION_NOT_EXIST.toString(), null, null, 2, 30, null);
                tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
            }
        }

        standardApiPostProcess(tResult, result, pStandardApi, null);

        // 新增版本校验 (判定当前输入的版本号是否正确) 通过API名称+租户ID获取最新的版本号进行比对
        //if (Float.parseFloat(tVersion) > 1.0) {
        if (!apiVersionService.judgeNewBranch(tVersion)) {
            // 判定之前的版本是否存在且已经定版
            // 取得前一版StandardApiVersion
            if (ApiVersionServiceUtil.judgeVersion(tVersion)) {
                String tPreviousVersion = ApiVersionServiceUtil.getPreviousVersion(tVersion);
                List<StandardApiVersion> tPreviousStandardApiVersions = standardApiVersionDao.getExistApiVersion(tApiName, pStandardApi.getTenantId(), tPreviousVersion);
                if (tPreviousStandardApiVersions.isEmpty()) {
                    uploadCheckResult(result, APIExportImportEnums.PREVI_API_VERSION_IS_EXIST.toString(), null, null, 2, 5, null);
                    tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
                    return tResult;
                }
                if (CollUtil.isNotEmpty(tPreviousStandardApiVersions)
                        && tPreviousStandardApiVersions.get(0).getApprovedStatus().getId() != 4) {
                    uploadCheckResult(result, APIExportImportEnums.PREVI_IS_DEVELOPING.toString(), null, null, 2, 5, null);
                    tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
                    return tResult;
                }
            } else {
                uploadCheckResult(result, APIExportImportEnums.API_VERSION_NOT_COMPLIANCE.toString(), null, null, 2, 5, null);
                tResult.put(APIExportImportEnums.RESULT_MESSAGE.toString(), result.get(0).get(ApimgmtConstant.DESCRIPTION));
                return tResult;
            }
        }
        return tResult;
    }
}
