package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import com.digiwin.athena.apimgmt.dao.ApiMgmtProjectDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtProjectVersionDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtProjectVersionRelationDao;
import com.digiwin.athena.apimgmt.dto.ApiBindProjectVersionReqDto;
import com.digiwin.athena.apimgmt.dto.PageReqCondition;
import com.digiwin.athena.apimgmt.dto.Pagination;
import com.digiwin.athena.apimgmt.dto.ProjectDto;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.model.Project;
import com.digiwin.athena.apimgmt.model.ProjectVersion;
import com.digiwin.athena.apimgmt.model.ProjectVersionRelation;
import com.digiwin.athena.apimgmt.model.StandardApi;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

@Service
public class ApiMgmtProjectService {

    @Autowired
    private ApiMgmtProjectDao projectDao;

    @Autowired
    private ApiMgmtProjectVersionDao projectVersionDao;

    @Autowired
    private ApiMgmtProjectVersionRelationDao projectVersionRelationDao;

    public Pagination<ProjectDto> queryProjectPage(PageReqCondition<String> pageReqCondition) {
        Long count = projectDao.pageCount(pageReqCondition);

        List<Project> projects = projectDao.selectPageable(pageReqCondition);
        List<ProjectDto> res = projects.stream().map(ProjectDto::create).collect(Collectors.toList());

        List<Long> projectIds = projects.stream().map(Project::getProjectId).collect(Collectors.toList());
        List<ProjectVersion> pvs = projectVersionDao.selectByProjectIds(projectIds);

        Map<Long, List<ProjectVersion>> versionMap = pvs.stream().collect(Collectors.groupingBy(p -> p.getProjectId()));

        res.forEach(r->r.setProjectVersionList(versionMap.get(r.getProjectId())));
        Pagination<ProjectDto> projectDtoPagination = Pagination.buildPagination(pageReqCondition, res, count);
        return projectDtoPagination;
    }

    public List<ProjectDto> queryProject(){
        List<Project> projects = projectDao.selectAll();
        List<ProjectDto> res = new ArrayList<>();

        Iterator<Project> iterator = projects.iterator();
        while(iterator.hasNext()){
            Project next = iterator.next();
            if ("Athena".equals(next.getProjectName())){
                res.add(ProjectDto.create(next));
                iterator.remove();
                break;
            }
        }

        res.addAll(projects.stream().map(ProjectDto::create).collect(Collectors.toList()));
        List<Long> projectIds = res.stream().map(ProjectDto::getProjectId).collect(Collectors.toList());

        if (!projectIds.isEmpty()){
            Set<Long> existProjectIds = projectVersionDao.selectExistProjectId(projectIds);

            for (ProjectDto re : res) {
                if (existProjectIds.contains(re.getProjectId())){
                    re.setHasVersions(true);
                }
            }
        }
        return res;
    }

    public List<ProjectVersion> queryProjectVersion(Long projectId){
        List<ProjectVersion> projectVersions = projectVersionDao.selectByProjectIds(Arrays.asList(projectId));
        return projectVersions;
    }

    @Transactional(rollbackFor = Exception.class)
    public Boolean addProject(ProjectDto projectDto) throws Exception {
        Long count = projectDao.countByName(projectDto.getProjectName());
        Assert.isTrue(count.equals(0L),"已经存在同名的产品");

        Project project = projectDto.createProject();
        String userId = ApiMgmtServiceContextHolder.getUserId();

        project.setCreateBy(userId);
        project.setEditBy(userId);

        Project newProject = projectDao.save(project);
        List<ProjectVersion> projectVersionList = projectDto.getProjectVersionList();
        if (CollUtil.isNotEmpty(projectVersionList)){
            Set<String> versionNameSet = projectVersionList.stream().map(ProjectVersion::getProjectVersionName).collect(Collectors.toSet());
            Assert.isTrue(versionNameSet.size()==projectVersionList.size(),"版本名称存在重复");

            projectVersionList.forEach(p->p.setProjectId(newProject.getProjectId()));
            projectVersionDao.batchSave(projectVersionList);
        }
        return true;
    }

    @Transactional(rollbackFor = Exception.class)
    public Boolean deleteProject(Long projectId){
        Long count = projectVersionRelationDao.countByProjectId(projectId);
        Assert.isTrue(count.equals(0L),"该产品有关联的api不能删除");

        projectDao.deleteById(projectId);
        projectVersionDao.deleteByProjectId(projectId);

        return true;
    }

    @Transactional(rollbackFor = Exception.class)
    public Boolean updateProject(ProjectDto projectDto) throws Exception {
        Project project = projectDao.selectById(projectDto.getProjectId());
        Assert.notNull(project,"产品不存在");
        if (!project.getProjectName().equals(projectDto.getProjectName())){
            Long count = projectDao.countByName(projectDto.getProjectName());
            Assert.isTrue(count.equals(0L),"已经存在同名的产品");
            project.setProjectName(projectDto.getProjectName());
        }

        project.setProjectRemark(projectDto.getProjectRemark());
        project.setProjectDesc(projectDto.getProjectDesc());

        project.setEditTime(new Date());
        project.setEditBy(ApiMgmtServiceContextHolder.getUserId());

        projectDao.save(project);

        List<ProjectVersion> oldProjectVersions = projectVersionDao.selectByProjectIds(Arrays.asList(project.getProjectId()));

        List<ProjectVersion> newVersionList = new ArrayList<>();
        List<ProjectVersion> deleteVersionList = new ArrayList<>();
        List<ProjectVersion> updateVersionList = new ArrayList<>();

        List<ProjectVersion> projectVersionList = projectDto.getProjectVersionList();
        //请求中没有版本列表，就将数据库中的数据都删除
        if (CollUtil.isEmpty(projectVersionList)){
            deleteVersionList.addAll(oldProjectVersions);
        }else{
            Set<String> versionNameSet = projectVersionList.stream().map(ProjectVersion::getProjectVersionName).collect(Collectors.toSet());
            Assert.isTrue(versionNameSet.size()==projectVersionList.size(),"版本名称存在重复");

            //数据库中没有版本记录，那么全部新增
            if (oldProjectVersions.isEmpty()){
                newVersionList.addAll(projectVersionList);
            }else{

                Set<Long> reqVersionIds = projectVersionList.stream().map(ProjectVersion::getProjectVersionId).collect(Collectors.toSet());

                Map<Long,ProjectVersion> oldVersionMap = new HashMap<>();
                for (ProjectVersion oldProjectVersion : oldProjectVersions) {
                    oldVersionMap.put(oldProjectVersion.getProjectVersionId(),oldProjectVersion);
                }

                for (ProjectVersion projectVersion : oldProjectVersions) {
                    if (!reqVersionIds.contains(projectVersion.getProjectVersionId())){
                        deleteVersionList.add(projectVersion);
                    }
                }

                for (ProjectVersion projectVersion : projectVersionList) {
                    if (oldVersionMap.containsKey(projectVersion.getProjectVersionId())){
                        ProjectVersion pv = oldVersionMap.get(projectVersion.getProjectVersionId());
                        pv.setProjectVersionName(projectVersion.getProjectVersionName());
                        pv.setRemark(projectVersion.getRemark());

                        updateVersionList.add(pv);
                    }else{
                        newVersionList.add(projectVersion);
                    }
                }
            }
        }

        if (!deleteVersionList.isEmpty()){
            List<Long> projectVersionIds = deleteVersionList.stream().map(ProjectVersion::getProjectVersionId).collect(Collectors.toList());
            Long count = projectVersionRelationDao.countByProjectVersionId(projectVersionIds);
            Assert.isTrue(count.equals(0L),"删除的版本有关联的api，无法删除");
            projectVersionDao.deleteByIds(projectVersionIds);
        }

        if (!updateVersionList.isEmpty()){
            projectVersionDao.batchUpdate(updateVersionList);
        }

        if (!newVersionList.isEmpty()){
            projectVersionDao.batchSave(newVersionList);
        }
        return true;
    }

    @Transactional(rollbackFor = Exception.class)
    public Boolean apiBindProjectVersion(ApiBindProjectVersionReqDto apiBindProjectVersionReqDto) throws Exception {
        List<Long> apiIds = apiBindProjectVersionReqDto.getApiIds();
        Long projectId = apiBindProjectVersionReqDto.getProjectId();
        List<Long> projectVersionIds = apiBindProjectVersionReqDto.getProjectVersionIds();

        for (Long apiId : apiIds) {
            List<ProjectVersionRelation> relations = projectVersionRelationDao.selectByApiId(apiId);

            if (!relations.isEmpty()) {
                projectVersionRelationDao.deleteByApiId(apiId);
            }
            String userId = ApiMgmtServiceContextHolder.getUserId();

            if (CollUtil.isEmpty(projectVersionIds)) {
                ProjectVersionRelation projectVersionRelation = new ProjectVersionRelation();
                projectVersionRelation.setProjectId(projectId);
                StandardApi standardApi = new StandardApi();
                standardApi.setId(apiId);
                projectVersionRelation.setStandardApi(standardApi);
                projectVersionRelation.setCreatedTime(new Date());
                projectVersionRelation.setCreatedBy(userId);

                projectVersionRelationDao.save(projectVersionRelation);
            } else {
                List<ProjectVersionRelation> newProjectVersionList = projectVersionIds.stream().map(p -> {
                    ProjectVersionRelation projectVersionRelation = new ProjectVersionRelation();
                    projectVersionRelation.setProjectVersionId(p);
                    projectVersionRelation.setProjectId(projectId);
                    projectVersionRelation.setCreatedTime(new Date());
                    projectVersionRelation.setCreatedBy(userId);

                    StandardApi standardApi = new StandardApi();
                    standardApi.setId(apiId);
                    projectVersionRelation.setStandardApi(standardApi);
                    return projectVersionRelation;
                }).collect(Collectors.toList());

                projectVersionRelationDao.batchSave(newProjectVersionList);
            }
        }

        return true;
    }

    public ProjectDto getProjectDetail(Long projectId) {
        Project project = projectDao.selectById(projectId);
        Assert.notNull(project,"未查询到产品");

        ProjectDto projectDto = ProjectDto.create(project);
        List<ProjectVersion> projectVersions = projectVersionDao.getByProjectId(projectId);
        projectDto.setProjectVersionList(projectVersions);
        return projectDto;
    }
}
