package com.digiwin.athena.apimgmt.apiservice;

import cn.afterturn.easypoi.excel.entity.ImportParams;
import cn.afterturn.easypoi.excel.entity.result.ExcelImportResult;
import cn.hutool.core.collection.CollUtil;
import com.digiwin.athena.apimgmt.common.model.dto.FileDto;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardDataNameDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStateCodeDao;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.model.StandardDataName;
import com.digiwin.athena.apimgmt.model.StateCode;
import com.digiwin.athena.apimgmt.util.ExcelUtil;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.Map;

/**
 * 词典审核导入(用于词典审核，无页面展示用处)
 * /restful/standard/apimgmt/StandardDate/Import
 */
@Slf4j
@Service
@Transactional(rollbackFor = Exception.class)
public class ApiMgmtStandardDataImportService {


    ObjectMapper mapper = new ObjectMapper();

    @Autowired
    ApiMgmtStateCodeDao stateCodeDao;

    @Autowired
    ApiMgmtStandardDataNameDao standardDataNameDao;

    public ApiMgmtStandardDataImportService() {
        super();
    }

    public synchronized Map<String, Object> execute(FileDto file) throws BaseException {
        // 建立response node
        ObjectNode tResponseNode = mapper.createObjectNode();
        StateCode tStateCode = stateCodeDao.getStateCodeByCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = tStateCode.getDescription();
        StringBuilder tDescriptionBuilder = new StringBuilder();
        try {
            // 取得token
            String tToken = ApiMgmtServiceContextHolder.getToken();
            if (null == tToken) {
                throw new Exception("token未填写！");
            }
            // 取得用戶資訊
            String tUserId = ApiMgmtServiceContextHolder.getUserId();
            if (null == tUserId) {
                throw new Exception("token解析失败！");
            }
            ExcelImportResult<StandardDataName> result
                    = importExcel(file, 0, 0, 1, StandardDataName.class);
            if (null != result && CollUtil.isNotEmpty(result.getList())) {
                result.getList().stream().filter(y -> null != y.getDataName()).forEach(x -> {
                    StandardDataName standardDataName = standardDataNameDao.get(x.getDataName());
                    if (null != standardDataName) {
                        standardDataName.setDescriptionEnUs(StringUtil.isEmptyOrSpace(x.getDescriptionEnUs())
                                ? x.getDescriptionZhTw() : x.getDescriptionEnUs());
                        standardDataName.setDescriptionZhCn(x.getDescriptionZhCn());
                        standardDataName.setDescriptionZhTw(x.getDescriptionZhTw());
                        standardDataName.setTypeEnUs(x.getTypeEnUs());
                        standardDataName.setTypeZhCn(x.getTypeZhCn());
                        standardDataName.setTypeZhTw(x.getTypeZhTw());
                        standardDataName.setUnitOrDefaultZhTw(x.getUnitOrDefaultZhTw());
                        standardDataName.setUnitOrDefaultZhCn(x.getUnitOrDefaultZhTw());
                        standardDataName.setUnitOrDefaultEnUs(x.getUnitOrDefaultZhTw());
                        standardDataName.setRemarkZhTw(x.getRemarkZhTw());
                        standardDataName.setRemarkZhCn(x.getRemarkZhTw());
                        standardDataName.setRemarkEnUs(x.getRemarkZhTw());
                        standardDataName.setApprovedStatus(ApimgmtConstant.APPROVED_STATUS_COMFIRMED);
                        standardDataName.setApprovedAcct(tUserId);
                        standardDataName.setApprovedTime(LocalDateTime.now());
                        standardDataName.setLastUpdateTime(LocalDateTime.now());
                        standardDataNameDao.update(standardDataName);
                    }
                });
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            tStateCode = stateCodeDao.getStateCodeByCode(ValidateStateEnum.UNEXPECTED_ERROR.getCode());
            tDescriptionBuilder.append(tStateCode.getDescription()).append(ApimgmtConstant.BLANK).append(e.getClass().toString()).append(":").append(e.getMessage());
            tDescription = tDescriptionBuilder.toString();
        }
        tResponseNode.put(ApimgmtConstant.CODE, tStateCode.getCode());
        tResponseNode.put(ApimgmtConstant.DESCRIPTION, tDescription);
        log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口：" + this.getClass() + "，回傳訊息：" + tResponseNode.toString());
        return mapper.convertValue(tResponseNode, new TypeReference<Map<String, Object>>() {
        });
    }

    /**
     * @param file      文件
     * @param startRows 开始读取行
     * @param index     sheet页号
     * @param headRows  头部信息
     * @param tClass    识别类
     * @param <T>       识别类
     * @return ExcelImportResult
     */
    public <T> ExcelImportResult<T> importExcel(FileDto file, int startRows, int index, int headRows, Class<T> tClass) {
        ImportParams params = new ImportParams();
        params.setStartRows(startRows);
        params.setStartSheetIndex(index);
        params.setTitleRows(0);
        params.setHeadRows(headRows);
        params.setNeedVerify(false);
        return ExcelUtil.importExcel(file, params, tClass);
    }
}
