package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.collection.CollUtil;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiVersionDao;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.model.StandardApi;
import com.digiwin.athena.apimgmt.model.StateCode;
import com.digiwin.athena.apimgmt.service.util.ApiVersionServiceUtil;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * API 查询新增|进版的版本号
 * /restful/standard/apimgmt/ApiBranch/Get
 */
@Slf4j
@Service
public class ApiBranchGetService extends AbstractApiService {

	@Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;
    @Autowired
    private ApiMgmtStandardApiDao standardApiDao;

	public ApiBranchGetService() {
		super();
		jsonSchemaFileName = ApimgmtSchemaConstant.API_BRANCH_GET_SCHEMA;
	}

	/**
	 * 查询API版本号
	 * @param validatorResult validatorResult
	 * @return map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		// 通过api名称+租户获取对应的版本
		Map<String, Object> tResponse = new HashMap<>();
		StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
		String description = tStateCode.getDescription();
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		String tApiName = tRequestJsonNode.get(ApiAttributeEnum.apiName.toString()).asText();
        String tenantId = Optional.ofNullable(tRequestJsonNode.get(ApiAttributeEnum.tenantId.toString()))
                .map(JsonNode::asText).orElse(null);

		//先查API表
		List<StandardApi> standardApiList = standardApiDao.getByNameAndTenantId(tApiName, tenantId);

        // 获取最大的分支
        String maxBranch = getNextBranch(standardApiList);
		tResponse.put(ApimgmtConstant.CODE, tStateCode.getCode());
		tResponse.put(ApimgmtConstant.DESCRIPTION, description);
        tResponse.put(ApiAttributeEnum.branch.toString(), maxBranch);
		return tResponse;
	}

    public static String getNextBranch(List<StandardApi> standardApiList) {
        int maxBranch = CollUtil.emptyIfNull(standardApiList).stream()
                .mapToInt(ApiBranchGetService::getBranchOrder)
                .max()
                .orElse(0);
        return (maxBranch + 1) + ".0";
	}

    private static int getBranchOrder(StandardApi api) {
        return ApiVersionServiceUtil.getMajorVersion(api.getBranch());
    }

}
