package com.digiwin.athena.apimgmt.services;

import com.digiwin.athena.apimgmt.constants.*;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDataNameDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiVersionDao;
import com.digiwin.athena.apimgmt.enums.*;
import com.digiwin.athena.apimgmt.model.StandardApi;
import com.digiwin.athena.apimgmt.model.StandardApiDataName;
import com.digiwin.athena.apimgmt.util.DigestUtil;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.XMLWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 標準API訊息產生工具(XML or JSON)
 */
@Slf4j
@Service
public class ApiMgmtStandardApiMessageService {

    @Autowired
    private ApiMgmtStandardApiVersionDao apiVersionDao;

    @Autowired
    private ApiMgmtStandardApiDataNameDao apiDataNameDao;

    public final Gson GSON = new GsonBuilder().setPrettyPrinting().create();

    public Map<Integer, List<StandardApiDataName>> getMsgTypeMap(Map<String, StandardApiDataName> pAddDataName) {
        List<StandardApiDataName> tAddDataNameList = new ArrayList<>();
        for (String tKey : pAddDataName.keySet()) {
            tAddDataNameList.add(pAddDataName.get(tKey));
        }
        return getMsgTypeMap(tAddDataNameList);
    }

    public Map<Integer, List<StandardApiDataName>> getMsgTypeMap(List<StandardApiDataName> plistOfNodes) {
        Map<Integer, List<StandardApiDataName>> tMap = new HashMap<>();
        List<StandardApiDataName> tlistOfHierarchyNodesRequest = new ArrayList<>();
        List<StandardApiDataName> tlistOfHierarchyNodesResponseSuccess = new ArrayList<>();
        List<StandardApiDataName> tlistOfHierarchyNodesResponseFailed = new ArrayList<>();
        for (StandardApiDataName tApiDataName : plistOfNodes) {
            switch (tApiDataName.getMsgType()) {
                case StandardApiMessageTypeConstants.REQUEST:
                    tlistOfHierarchyNodesRequest.add(tApiDataName);
                    break;
                case StandardApiMessageTypeConstants.RESPONSE_SUCCESS:
                    tlistOfHierarchyNodesResponseSuccess.add(tApiDataName);
                    break;
                case StandardApiMessageTypeConstants.RESPONSE_FAILED:
                    tlistOfHierarchyNodesResponseFailed.add(tApiDataName);
                    break;
                default:
                    break;
            }
        }
        tMap.put(1, tlistOfHierarchyNodesRequest);
        tMap.put(2, tlistOfHierarchyNodesResponseSuccess);
        tMap.put(3, tlistOfHierarchyNodesResponseFailed);
        return tMap;
    }

    /**
     * 判斷StandardApiDataName是否有父階層
     */
    public List<StandardApiDataName> generateListOfHierarchyNodes(List<StandardApiDataName> pListOfNodes,
                                                                  Long pApVerid) {
        List<StandardApiDataName> listOfHierarchyNodes = new ArrayList<>();
        for (StandardApiDataName tApiDataName : pListOfNodes) {
            if (tApiDataName.getParentId() == null) {
                if (null != pApVerid) {
                    addAllNodes(tApiDataName, pListOfNodes);
                }
                listOfHierarchyNodes.add(tApiDataName);
            }
        }
        return listOfHierarchyNodes;
    }

    /**
     * 根據標準API id、訊息格式(MsgType)取出標準API資料
     */
    public Map<String, String> generateStandardApiMessage(Long pApVerid, StandardApi pStandardApi, String pMsgFormat,
                                                          String pHostProd, String pServiceProd, String tSampleType, String pDeveloperRole, String pCustomUUID, String tLocale) {
        Integer tMessageType = MessageTextSampleTypeEnum.toStdApiDataNameMsgType(tSampleType);
        MessageTextSampleTypeEnum tMessageTextSampleTypeEnum = MessageTextSampleTypeEnum.toEnum(tSampleType);
        StandardApi tStandardApi;
        List<StandardApiDataName> listOfNodes = new ArrayList<>();
        String tApiVersionNumber;
        if (pApVerid == null && pStandardApi != null) {
            for (StandardApiDataName tStandardApiDataName : pStandardApi.getStandardApiVersions().get(0)
                    .getStandardApiDataNames()) {
                if (tStandardApiDataName.getMsgType().equals(tMessageType)) {
                    listOfNodes.add(tStandardApiDataName);
                }
            }
            tStandardApi = pStandardApi;
            tApiVersionNumber = tStandardApi.getStandardApiVersions().get(0).getVersion();
        } else {
            tStandardApi = apiVersionDao.get(pApVerid).getStandardApi();
            tApiVersionNumber = apiVersionDao.get(pApVerid).getVersion();
            listOfNodes = apiDataNameDao.getByApiVerIdAndMsgType(pApVerid, tMessageType, true);
        }
        String tApiName = tStandardApi.getName();
        List<StandardApiDataName> listOfHierarchyNodes = generateListOfHierarchyNodes(listOfNodes, pApVerid);
        String tMessageGenerated = null;
        if (pMsgFormat.equals(MessageFormatEnum.XML.getMsgFormat())) {
            tMessageGenerated = transTreeNodeToXMLString(tStandardApi, tMessageTextSampleTypeEnum, listOfHierarchyNodes, tApiName,
                    pHostProd, pServiceProd, pDeveloperRole, tStandardApi.getIsBatch(), tStandardApi.getMsgMdata(),
                    pCustomUUID, tLocale);
        } else if (pMsgFormat.equals(MessageFormatEnum.JSON.getMsgFormat())) {
            tMessageGenerated = transTreeNodeToJSONString(tStandardApi, tMessageTextSampleTypeEnum, listOfHierarchyNodes, tApiName,
                    pHostProd, pServiceProd, pDeveloperRole, tStandardApi.getIsBatch(), pCustomUUID,
                    GeneratedApiMessageComponentEnum.ALL, GeneratedApiMessageMockValEnum.DESCRIPTION,
                    null, tLocale);
        } else if (pMsgFormat.equals(MessageFormatEnum.JSON10.getMsgFormat())) {
            tMessageGenerated = transTreeNodeToV1JSONString(tStandardApi, tMessageTextSampleTypeEnum, listOfHierarchyNodes, tApiName,
                    pHostProd, pServiceProd, GeneratedApiMessageMockValEnum.DESCRIPTION,
                    null, tLocale);
        }
        Map<String, String> tMessageMap = new HashMap<>();
        tMessageMap.put("MESSAGE_GENERATED", tMessageGenerated);
        tMessageMap.put("API_NAME", tApiName);
        tMessageMap.put("API_VERSION_NUMBER", tApiVersionNumber);
        return tMessageMap;

    }

    public void addAllNodes(StandardApiDataName pParent, List<StandardApiDataName> pListOfNodes) {
        List<StandardApiDataName> tChildrens = new ArrayList<>();
        for (StandardApiDataName tApiDataName : pListOfNodes) {
            // 每一次檢查是子node就先放到這個暫存的List<StandardApiDataName>
            if (tApiDataName.getParentId() != null) {
                if (pParent.getId().equals(tApiDataName.getParentId())) {
                    tChildrens.add(tApiDataName);
                    addAllNodes(tApiDataName, pListOfNodes);
                }
            }
        }
        pParent.setChildrens(tChildrens);
    }

    // 產生標準API XML訊息
    public String transTreeNodeToXMLString(StandardApi standardApi, MessageTextSampleTypeEnum tMessageTextSampleTypeEnum,
                                           List<StandardApiDataName> pStandardApiDataNames, String pApiName, String pHostProd, String pServiceProd,
                                           String pDeveloperRole, boolean pIsBatch, boolean pMsgMdata, String pCustomUUID, String tLocale) {
        Document tDocument = DocumentHelper.createDocument();
        // 產生標準API request訊息
        if (MessageTextSampleTypeEnum.SYNC_REQUEST.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_REQUEST.equals(tMessageTextSampleTypeEnum)) {
            // 產生固定標頭訊息(request標籤、host標籤、service標籤)
            Element tRootElement = tDocument.addElement(ApimgmtConstant.REQUEST_STRING);
            if (DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole)) {
                // 這裡先產生key屬性
                // 值的部分之後再回頭給
                tRootElement.addAttribute(ApimgmtConstant.KEY_STRING, "");
                if (MessageTextSampleTypeEnum.SYNC_REQUEST.equals(tMessageTextSampleTypeEnum)) {
                    tRootElement.addAttribute(ApimgmtConstant.TYPE_STRING, ApimgmtConstant.SYNC);
                } else {
                    tRootElement.addAttribute(ApimgmtConstant.TYPE_STRING, ApimgmtConstant.FASYNC);
                }
            } else if (DeveloperRoleEnum.PROVIDER.toString().equals(pDeveloperRole)) {
                tRootElement.addAttribute(ApimgmtConstant.ID, pCustomUUID);
            }
            Element hostElement = tRootElement.addElement(ApimgmtConstant.HOST_STRING);
            hostElement.addAttribute(ApimgmtConstant.PROD_STRING, pHostProd);
            hostElement.addAttribute(ApimgmtConstant.VER_STRING, "1.0");
            hostElement.addAttribute(ApimgmtConstant.IP_STRING, "10.40.71.91");
            hostElement.addAttribute(ApimgmtConstant.ACCT_STRING, "user01");
            hostElement.addAttribute(ApimgmtConstant.TIMESTAMP_STRING, "20151211123204361");
            Element serviceElement = tRootElement.addElement(ApimgmtConstant.SERVICE_STRING);
            serviceElement.addAttribute(ApimgmtConstant.PROD_STRING, pServiceProd);
            serviceElement.addAttribute(ApimgmtConstant.NAME, pApiName);
            Element datakeyElement = tRootElement.addElement(ApimgmtConstant.DATAKEY_STRING);
            Element payloadElement = tRootElement.addElement(ApimgmtConstant.PAYLOAD_STRING);
            if (DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole)) {
                // 這裡把正確的key值用上去
                String tSecurityKey = ServiceEngineConstant.SERVICE_ENGINE_KEY;
                Pattern tRegexP = Pattern.compile(ServiceEngineConstant.REGEX_CMP_HOST_SRV);
                Matcher tMatch = tRegexP.matcher(tDocument.asXML());
                String tSrcStr = "X";
                if (tMatch.find()) {
                    tSrcStr = tMatch.group();
                }
                String tHashCode = DigestUtil.getDigestCode("MD5", tSrcStr + tSecurityKey);
                tRootElement.addAttribute(ApimgmtConstant.KEY_STRING, tHashCode);
            }
            // 這裡是要有datakey或是有其他標準API資料才會產生的
            Element paramElement;
            Element dataRequestElement;
            Element dataInfoElement;
            // 如果有任何標準API資料且類型不為datakey，則產生param標籤、data_request標籤、datainfo標籤，且僅產生一次。
            paramElement = payloadElement.addElement(ApimgmtConstant.PARAM_STRING);
            paramElement.addAttribute(ApimgmtConstant.KEY_STRING, ApimgmtConstant.STD_DATA);
            paramElement.addAttribute(ApimgmtConstant.TYPE_STRING, ApimgmtConstant.XML_STRING);
            dataRequestElement = paramElement.addElement(ApimgmtConstant.DATAREQUEST_STRING);
            dataInfoElement = dataRequestElement.addElement(ApimgmtConstant.DATAINFO_STRING);
            for (StandardApiDataName tStandardApiDataName : pStandardApiDataNames) {
                // 標準API資料類型為M(單頭)
                if (tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.MASTER_CONSTANTS)) {
                    Element parameterElement = dataInfoElement.addElement(ApimgmtConstant.PARAMETER);
                    if (pMsgMdata) {
                        parameterElement.addAttribute(ApimgmtConstant.KEY_STRING,
                                tStandardApiDataName.getStandardDataName() + ApimgmtConstant.UNDERSCORE_DATA_STRING);
                    } else {
                        parameterElement.addAttribute(ApimgmtConstant.KEY_STRING, tStandardApiDataName.getStandardDataName());
                    }
                    parameterElement.addAttribute(ApimgmtConstant.TYPE_STRING, ApimgmtConstant.DATA_STRING);
                    Element dataElement = parameterElement.addElement(ApimgmtConstant.DATA_STRING);
                    dataElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                    if (tStandardApiDataName.getIsArray()) {
                        Element rowElement = dataElement.addElement(ApimgmtConstant.ROW_STRING);
                        rowElement.addAttribute(ApimgmtConstant.SEQ_STRING, "1");
                        // 因為M底下還有機會會有子標準API資料，故呼叫GenerateAllSubNodeXmlString方法產生所有子標準API資料的XML訊息
                        generateAllSubNodeXmlString(tStandardApiDataName, rowElement, datakeyElement, tLocale);
                    } else {
                        // 因為M底下還有機會會有子標準API資料，故呼叫GenerateAllSubNodeXmlString方法產生所有子標準API資料的XML訊息
                        generateAllSubNodeXmlString(tStandardApiDataName, dataElement, datakeyElement, tLocale);
                    }
                    // 標準API資料類型為P(簡單參數)
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.PARAMETER_CONSTANTS)) {
                    Element parameterElement = dataInfoElement.addElement(ApimgmtConstant.PARAMETER);
                    parameterElement.addAttribute(ApimgmtConstant.KEY_STRING, tStandardApiDataName.getStandardDataName());
                    parameterElement.addAttribute(ApimgmtConstant.TYPE_STRING, tStandardApiDataName.getDataType());
                    setValue(tStandardApiDataName, tLocale, parameterElement);
                    if (tStandardApiDataName.getIsDatakey()) {
                        Element keyElement = datakeyElement.addElement(ApimgmtConstant.KEY_STRING);
                        keyElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                        setValue(tStandardApiDataName, tLocale, keyElement);
                    }
                    // 標準API資料類型為datakey
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.DATAKEY_CONSTANTS)) {
                    Element keyElement = datakeyElement.addElement(ApimgmtConstant.KEY_STRING);
                    keyElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                    setValue(tStandardApiDataName, tLocale, keyElement);
                }
            }
            if (datakeyElement.elements().size() == 0) {
                tRootElement.remove(datakeyElement);
            }
            // 產生標準API RES成功或RES失敗訊息
        } else if (MessageTextSampleTypeEnum.SYNC_RESPONSE_SUCCESS.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.SYNC_RESPONSE_FAILED.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_CALLBACK_SUCCESS.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_CALLBACK_FAILED.equals(tMessageTextSampleTypeEnum)) {
            int paramResExist = 0, datainfoExit = 0;
            // 產生固定標頭訊息(response標籤、reqid標籤、srver標籤、srvcode標籤)
            Element tResponseElement = tDocument.addElement(ApimgmtConstant.RESPONSE_STRING);
            if (DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole)) {
                if (MessageTextSampleTypeEnum.SYNC_RESPONSE_SUCCESS.equals(tMessageTextSampleTypeEnum)
                        || MessageTextSampleTypeEnum.SYNC_RESPONSE_FAILED.equals(tMessageTextSampleTypeEnum)) {
                    Element codeElement = tResponseElement.addElement(ApimgmtConstant.CODE);
                    codeElement.addText("019");
                    Element messageElement = tResponseElement.addElement(ApimgmtConstant.MESSAGE_STRING);
                    messageElement.setText("EAI Response:平台回覆同步結果");
                }
                Element reqIdElement = tResponseElement.addElement(ApimgmtConstant.RES_REQID_STRING);
                reqIdElement.addText(pCustomUUID);
            }
            Element srvverElement = tResponseElement.addElement(ApimgmtConstant.SRVVER_STRING);
            srvverElement.addText("1.0");
            Element srvcodeElement = tResponseElement.addElement(ApimgmtConstant.SRVCODE_STRING);
            Element datakeyElement = tResponseElement.addElement(ApimgmtConstant.DATAKEY_STRING);
            if (MessageTextSampleTypeEnum.SYNC_RESPONSE_SUCCESS.equals(tMessageTextSampleTypeEnum)
                    || MessageTextSampleTypeEnum.FASYNC_CALLBACK_SUCCESS.equals(tMessageTextSampleTypeEnum)) {
                srvcodeElement.addText(ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING);
            } else {
                srvcodeElement.addText(ApimgmtConstant.PROD_RESPONSE_FAILED_STRING);
            }
            Element payloadElement = tResponseElement.addElement(ApimgmtConstant.PAYLOAD_STRING);
            Element paramElement;
            Element dataResponseElement = null;
            Element dataInfoResElement = null;
            Element exeElement;
            Element statusElement = null;
            for (StandardApiDataName tStandardApiDataName : pStandardApiDataNames) {
                // 如果有任何標準API資料，則產生param標籤、data_response標籤、execution標籤，且僅產生一次。
                if (paramResExist == 0) {
                    paramElement = payloadElement.addElement(ApimgmtConstant.PARAM_STRING);
                    paramElement.addAttribute(ApimgmtConstant.KEY_STRING, ApimgmtConstant.STD_DATA);
                    paramElement.addAttribute(ApimgmtConstant.TYPE_STRING, ApimgmtConstant.XML_STRING);
                    dataResponseElement = paramElement.addElement(ApimgmtConstant.DATA_RESPONSE_STRING);
                    exeElement = dataResponseElement.addElement(ApimgmtConstant.EXECUTION);
                    statusElement = exeElement.addElement(ApimgmtConstant.STATUS);
                    paramResExist = 1;
                }
                // 僅產生一次datainfo標籤
                if (!tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.EXECUTION_CONSTANTS)
                        && datainfoExit == 0) {
                    dataInfoResElement = dataResponseElement.addElement(ApimgmtConstant.DATAINFO_STRING);
                    datainfoExit = 1;
                }
                // 標準API資料類型為M(單頭)
                if (tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.MASTER_CONSTANTS)) {
                    Element parameterElement = dataInfoResElement.addElement(ApimgmtConstant.PARAMETER);
                    if (pMsgMdata) {
                        parameterElement.addAttribute(ApimgmtConstant.KEY_STRING,
                                tStandardApiDataName.getStandardDataName() + ApimgmtConstant.UNDERSCORE_DATA_STRING);
                    } else {
                        parameterElement.addAttribute(ApimgmtConstant.KEY_STRING, tStandardApiDataName.getStandardDataName());
                    }
                    parameterElement.addAttribute(ApimgmtConstant.TYPE_STRING, ApimgmtConstant.DATA_STRING);

                    Element dataElement = parameterElement.addElement(ApimgmtConstant.DATA_STRING);
                    dataElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                    if (pIsBatch) {
                        dataElement.addAttribute(ApimgmtConstant.CNT_STRING, ApimgmtConstant.CNT_VALUE);
                        dataElement.addAttribute(ApimgmtConstant.TT_CNT_STRING, ApimgmtConstant.TT_CNT_VALUE);
                    }
                    if (tStandardApiDataName.getIsArray()) {
                        Element rowElement = dataElement.addElement(ApimgmtConstant.ROW_STRING);
                        rowElement.addAttribute(ApimgmtConstant.SEQ_STRING, "1");
                        // 因為M底下還有機會會有子標準API資料，故呼叫GenerateAllSubNodeXmlString方法產生所有子標準API資料的XML訊息
                        generateAllSubNodeXmlString(tStandardApiDataName, rowElement, datakeyElement, tLocale);
                    } else {
                        // 因為M底下還有機會會有子標準API資料，故呼叫GenerateAllSubNodeXmlString方法產生所有子標準API資料的XML訊息
                        generateAllSubNodeXmlString(tStandardApiDataName, dataElement, datakeyElement, tLocale);
                    }
                    // 標準API資料類型為P(簡單參數)
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.PARAMETER_CONSTANTS)) {
                    Element parameterElement = dataInfoResElement.addElement(ApimgmtConstant.PARAMETER);
                    parameterElement.addAttribute(ApimgmtConstant.KEY_STRING, tStandardApiDataName.getStandardDataName());
                    parameterElement.addAttribute(ApimgmtConstant.TYPE_STRING, tStandardApiDataName.getDataType());
                    setValue(tStandardApiDataName, tLocale, parameterElement);
                    if (tStandardApiDataName.getIsDatakey()) {
                        Element keyElement = datakeyElement.addElement(ApimgmtConstant.KEY_STRING);
                        keyElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                        setValue(tStandardApiDataName, tLocale, keyElement);
                    }
                    // 如果標準API資料類型為E
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.EXECUTION_CONSTANTS)) {
                    statusElement.addAttribute(tStandardApiDataName.getStandardDataName(),
                            getValue(tStandardApiDataName, tLocale));
                    // 如果標準API資料類型為datakey
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.DATAKEY_CONSTANTS)) {
                    Element keyElement = datakeyElement.addElement(ApimgmtConstant.KEY_STRING);
                    keyElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                    setValue(tStandardApiDataName, tLocale, keyElement);
                }
            }
            if (datakeyElement.elements().size() == 0) {
                tResponseElement.remove(datakeyElement);
            }
        }
        // 這裡專印訊息
        Writer tOsw = new StringWriter();
        org.dom4j.io.OutputFormat tOpf = new org.dom4j.io.OutputFormat("    ", true, "UTF-8");
        tOpf.setNewLineAfterDeclaration(false);
        XMLWriter writer = new XMLWriter(tOsw, tOpf);
        try {
            writer.write(tDocument.getRootElement());
            writer.close();
            String tXML = tOsw.toString();
            while (tXML.startsWith("\n") || tXML.startsWith("\r")) {
                tXML = tXML.replaceFirst("[\n\r]", "");
            }
            return tXML;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    public String transTreeNodeToV1JSONString(StandardApi standardApi, MessageTextSampleTypeEnum tMessageTextSampleTypeEnum,
                                              List<StandardApiDataName> pStandardApiDataNames, String pApiName, String pHostProd, String pServiceProd,
                                              GeneratedApiMessageMockValEnum pMockValEnum, Map<String, String> pMapMockVal, String tLocale) {
        // 產生標準API request JSON 1.0訊息
        String tMessageReturned = null;
        if (MessageTextSampleTypeEnum.SYNC_REQUEST.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_REQUEST.equals(tMessageTextSampleTypeEnum)) {
            int payloadExist = 0;
            JsonObject tMainObject = new JsonObject();
            JsonObject tHostObject = new JsonObject();
            JsonObject tDatakeyObject = new JsonObject();
            JsonObject tPayloadObject = new JsonObject();
            JsonObject tStdDataObject = new JsonObject();
            JsonObject tParameterObject = new JsonObject();
            tHostObject.addProperty(ApimgmtConstant.PROD_STRING, pHostProd);
            tHostObject.addProperty(ApimgmtConstant.IP_STRING, "10.20.88.30");
            tHostObject.addProperty(ApimgmtConstant.LANG_STRING, ApimgmtConstant.ZH_TW);
            tHostObject.addProperty(ApimgmtConstant.TIMESTAMP_STRING, "20161018132542116");
            tHostObject.addProperty(ApimgmtConstant.ACCT_STRING, "tiptop");
            JsonObject tServiceObject = new JsonObject();
            tServiceObject.addProperty(ApimgmtConstant.PROD_STRING, pServiceProd);
            tServiceObject.addProperty(ApimgmtConstant.NAME, pApiName);
            tServiceObject.addProperty(ApimgmtConstant.IP_STRING, "10.20.88.75");
            tServiceObject.addProperty(ApimgmtConstant.ID, "topprod");
            // 這裡把正確的key值用上去
            String tSecurityKey = ServiceEngineConstant.SERVICE_ENGINE_KEY;
            String tSrcStr = tHostObject.toString() + tServiceObject.toString();
            String tHashCode = DigestUtil.getDigestCode("MD5", tSrcStr + tSecurityKey);
            tMainObject.addProperty(ApimgmtConstant.KEY_STRING, tHashCode);
            if (MessageTextSampleTypeEnum.SYNC_REQUEST.equals(tMessageTextSampleTypeEnum)) {
                tMainObject.addProperty(ApimgmtConstant.TYPE_STRING, ApimgmtConstant.SYNC);
            } else {
                tMainObject.addProperty(ApimgmtConstant.TYPE_STRING, ApimgmtConstant.FASYNC);
            }
            tMainObject.add(ApimgmtConstant.HOST_STRING, tHostObject);
            tMainObject.add(ApimgmtConstant.SERVICE_STRING, tServiceObject);
            // 取得所有最外層節點，最外層節點可為欄位類型: M、P、datakey
            for (StandardApiDataName tStandardApiDataName : pStandardApiDataNames) {
                // 如果有任何標準API資料且類型不為datakey，則產生payload、std_data、parameter，且僅產生一次。
                if (!(tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.DATAKEY_CONSTANTS))
                        && payloadExist == 0) {
                    tPayloadObject.add(ApimgmtConstant.STD_DATA, tStdDataObject);
                    tStdDataObject.add(ApimgmtConstant.PARAMETER, tParameterObject);
                    payloadExist = 1;
                }
                // 標準API資料類型為M(單頭)
                if (tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.MASTER_CONSTANTS)) {
                    JsonArray tArray = new JsonArray();
                    JsonObject tObject = new JsonObject();
                    generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDatakeyObject, pMockValEnum,
                            pMapMockVal, tLocale);
                    tArray.add(tObject);
                    tParameterObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                    // 標準API資料類型為P(簡單參數)
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.PARAMETER_CONSTANTS)) {
                    tParameterObject.addProperty(tStandardApiDataName.getStandardDataName(),
                            getValue(tStandardApiDataName, tLocale));

                    if (tStandardApiDataName.getIsDatakey()) {
                        tDatakeyObject.addProperty(tStandardApiDataName.getStandardDataName(),
                                getValue(tStandardApiDataName, tLocale));
                    }
                    // 標準API資料類型為datakey
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.DATAKEY_CONSTANTS)) {
                    tDatakeyObject.addProperty(tStandardApiDataName.getStandardDataName(),
                            getValue(tStandardApiDataName, tLocale));
                }
            }
            if (tDatakeyObject.entrySet().size() != 0) {
                tMainObject.add(ApimgmtConstant.DATAKEY_STRING, tDatakeyObject);
            }
            if (payloadExist != 0) {
                tMainObject.add(ApimgmtConstant.PAYLOAD_STRING, tPayloadObject);
            }
            tMessageReturned = GSON.toJson(tMainObject);
        } else if (MessageTextSampleTypeEnum.SYNC_RESPONSE_SUCCESS.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.SYNC_RESPONSE_FAILED.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_CALLBACK_SUCCESS.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_CALLBACK_FAILED.equals(tMessageTextSampleTypeEnum)) {
            int parameterExist = 0;
            int stdDataExist = 0;
            int executionExist = 0;
            JsonObject tMainObject = new JsonObject();
            JsonObject tStdDataObject = new JsonObject();
            JsonObject tDatakeyObject = new JsonObject();
            JsonObject tParameterObject = new JsonObject();
            JsonObject tExecutionObject = new JsonObject();
            JsonObject tPayloadObject = new JsonObject();
            if (MessageTextSampleTypeEnum.SYNC_RESPONSE_SUCCESS.equals(tMessageTextSampleTypeEnum)
                    || MessageTextSampleTypeEnum.FASYNC_CALLBACK_SUCCESS.equals(tMessageTextSampleTypeEnum)) {
                tMainObject.addProperty(ApimgmtConstant.SRVVER_STRING, "1.0");
                tMainObject.addProperty(ApimgmtConstant.SRVCODE_STRING, ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING);
            } else if (MessageTextSampleTypeEnum.SYNC_RESPONSE_FAILED.equals(tMessageTextSampleTypeEnum)) {
                tMainObject.addProperty(ApimgmtConstant.SRVVER_STRING, "1.0");
                tMainObject.addProperty(ApimgmtConstant.SRVCODE_STRING, ApimgmtConstant.PROD_RESPONSE_FAILED_STRING);
            }
            // 先產生execution，避免parameter可能會在execution上方的情況
            tStdDataObject.add(ApimgmtConstant.EXECUTION, tExecutionObject);
            for (StandardApiDataName tStandardApiDataName : pStandardApiDataNames) {
                // 如果有任何標準API資料，則產生payload、std_data，且僅產生一次。
                if (stdDataExist == 0) {
                    tMainObject.add(ApimgmtConstant.PAYLOAD_STRING, tPayloadObject);
                    tPayloadObject.add(ApimgmtConstant.STD_DATA, tStdDataObject);
                    stdDataExist = 1;
                }
                // 如果有任何標準API資料且類型不為E，則產生parameter，且僅產生一次。
                if (!(tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.EXECUTION_CONSTANTS))
                        && parameterExist == 0) {
                    tStdDataObject.add(ApimgmtConstant.PARAMETER, tParameterObject);
                    parameterExist = 1;
                }
                // 標準API資料類型為M(單頭)
                if (tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.MASTER_CONSTANTS)) {
                    JsonArray tArray = new JsonArray();
                    JsonObject tObject = new JsonObject();
                    generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDatakeyObject, pMockValEnum,
                            pMapMockVal, tLocale);
                    tArray.add(tObject);
                    tParameterObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                    // 標準API資料類型為P(簡單參數)
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.PARAMETER_CONSTANTS)) {
                    tParameterObject.addProperty(tStandardApiDataName.getStandardDataName(),
                            getValue(tStandardApiDataName, tLocale));
                    if (tStandardApiDataName.getIsDatakey()) {
                        tDatakeyObject.addProperty(tStandardApiDataName.getStandardDataName(),
                                getValue(tStandardApiDataName, tLocale));
                    }
                    // 標準API資料類型為E(exeution)
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.EXECUTION_CONSTANTS)) {
                    tExecutionObject.addProperty(tStandardApiDataName.getStandardDataName(),
                            getValue(tStandardApiDataName, tLocale));
                    executionExist = 1;
                }
            }
            // 如果此標準API沒有columnType=E的標準API資料，則不應該有execution，故把前面產生的execution拿掉
            if (executionExist == 0) {
                tStdDataObject.remove(ApimgmtConstant.EXECUTION);
            }
            if (tDatakeyObject.entrySet().size() != 0) {
                tMainObject.add(ApimgmtConstant.DATAKEY_STRING, tDatakeyObject);
            }
            tMessageReturned = GSON.toJson(tMainObject);
        }
        return tMessageReturned;

    }

    /**
     * 產生標準API request JSON訊息
     */
    public String transTreeNodeToJSONString(StandardApi standardApi, MessageTextSampleTypeEnum tMessageTextSampleTypeEnum,
                                            List<StandardApiDataName> pStandardApiDataNames, String pApiName, String pHostProd, String pServiceProd,
                                            String pDeveloperRole, boolean pIsBatch, String pCustomUUID,
                                            GeneratedApiMessageComponentEnum pComponentEnum, GeneratedApiMessageMockValEnum pMockValEnum,
                                            Map<String, String> pMapMockVal, String tLocale) {
        String tMessageReturned = null;
        if (StringUtils.isNoneBlank(standardApi.getApiType()) && !ApimgmtConstant.DEFAULT_API_TYPE.equals(standardApi.getApiType())) {
            return transTreeNodeToJSONString4OpenAPI(standardApi, tMessageTextSampleTypeEnum, pStandardApiDataNames, pApiName, pHostProd, pServiceProd, pDeveloperRole, pIsBatch, pCustomUUID, pComponentEnum, pMockValEnum, pMapMockVal, tLocale);
        }

        if (MessageTextSampleTypeEnum.SYNC_REQUEST.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_REQUEST.equals(tMessageTextSampleTypeEnum)) {
            StringBuilder tBuilder = new StringBuilder();
            tBuilder.append(ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            tBuilder.append(ApimgmtConstant.DIGI_PROTOCOL_STRING + ApimgmtSymbolConstant.COLON_STRING + "raw" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            JsonObject tHostObject = new JsonObject();
            tHostObject.addProperty(ApimgmtConstant.PROD_STRING, pHostProd);
            tHostObject.addProperty(ApimgmtConstant.IP_STRING, "10.20.88.30");
            tHostObject.addProperty(ApimgmtConstant.LANG_STRING, ApimgmtConstant.ZH_TW);
            tHostObject.addProperty(ApimgmtConstant.TIMESTAMP_STRING, "20161018132542116");
            tHostObject.addProperty(ApimgmtConstant.ACCT_STRING, "tiptop");
            JsonObject tServiceObject = new JsonObject();
            tServiceObject.addProperty(ApimgmtConstant.PROD_STRING, pServiceProd);
            tServiceObject.addProperty(ApimgmtConstant.NAME, pApiName);
            tServiceObject.addProperty(ApimgmtConstant.IP_STRING, "10.20.88.75");
            tServiceObject.addProperty(ApimgmtConstant.ID, "topprod");
            if (DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole)) {
                if (MessageTextSampleTypeEnum.SYNC_REQUEST.equals(tMessageTextSampleTypeEnum)) {
                    tBuilder.append(ApimgmtConstant.DIGI_TYPE_STRING + ApimgmtSymbolConstant.COLON_STRING + ApimgmtConstant.SYNC + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                } else {
                    tBuilder.append(ApimgmtConstant.DIGI_TYPE_STRING + ApimgmtSymbolConstant.COLON_STRING + ApimgmtConstant.FASYNC + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                }
                // 這裡把正確的key值用上去
                String tSecurityKey = ServiceEngineConstant.SERVICE_ENGINE_KEY;
                String tSrcStr = tHostObject.toString() + tServiceObject.toString();
                String tHashCode = DigestUtil.getDigestCode("MD5", tSrcStr + tSecurityKey);
                tBuilder.append(ApimgmtConstant.DIGI_KEY_STRING + ApimgmtSymbolConstant.COLON_STRING).append(tHashCode).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            } else if (DeveloperRoleEnum.PROVIDER.toString().equals(pDeveloperRole)) {
                tBuilder.append(ApimgmtConstant.DIGI_REQID_STRING + ApimgmtSymbolConstant.COLON_STRING).append(pCustomUUID).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            }
            tBuilder.append(ApimgmtConstant.DIGI_HOST_STRING + ApimgmtSymbolConstant.COLON_STRING).append(tHostObject.toString()).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            tBuilder.append(ApimgmtConstant.DIGI_SERVICE_STRING + ApimgmtSymbolConstant.COLON_STRING).append(tServiceObject.toString()).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            JsonObject tBodyObject = new JsonObject();
            JsonObject tStdDataObject = new JsonObject();
            JsonObject tParameterObject = new JsonObject();
            JsonObject tDataKeyObject = new JsonObject();
            Map<String, String> tNonStandardHeader = new HashMap<>();
            tBodyObject.add(ApimgmtConstant.STD_DATA, tStdDataObject);
            tStdDataObject.add(ApimgmtConstant.PARAMETER, tParameterObject);
            // 取得所有最外層節點，最外層節點可為欄位類型: M、P、H、datakey
            for (StandardApiDataName tStandardApiDataName : pStandardApiDataNames) {
                // 標準API資料類型為M(單頭)
                if (tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.MASTER_CONSTANTS)) {
                    JsonObject tObject = new JsonObject();
                    generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDataKeyObject, pMockValEnum,
                            pMapMockVal, tLocale);
                    if (tStandardApiDataName.getIsArray()) {
                        JsonArray tArray = new JsonArray();
                        tArray.add(tObject);
                        tParameterObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                    } else {
                        tParameterObject.add(tStandardApiDataName.getStandardDataName(), tObject);
                    }
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.PARAMETER_CONSTANTS)) {
                    addJsonObjectProperty(tParameterObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    if (tStandardApiDataName.getIsDatakey()) {
                        addJsonObjectProperty(tDataKeyObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    }
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.HEADER_CONSTANTS)) {
                    String tNonStandardHeaderName;
                    String tNonStandardHeaderValue;
                    if (tStandardApiDataName.getValue() != null) {
                        tNonStandardHeaderName = tStandardApiDataName.getStandardDataName();
                        tNonStandardHeaderValue = tStandardApiDataName.getValue();
                    } else {
                        JsonObject tObject = new JsonObject();
                        generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDataKeyObject, pMockValEnum,
                                pMapMockVal, tLocale);
                        tNonStandardHeaderName = tStandardApiDataName.getStandardDataName();
                        if (tObject.size() == 0) {
                            tNonStandardHeaderValue = "";
                        } else {
                            tNonStandardHeaderValue = tObject.toString();
                        }
                    }
                    tNonStandardHeader.put(tNonStandardHeaderName, tNonStandardHeaderValue);
                    // 標準API資料類型為datakey
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.DATAKEY_CONSTANTS)) {
                    addJsonObjectProperty(tDataKeyObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                }
            }
            if (tDataKeyObject.entrySet().size() != 0) {
                tBuilder.append(ApimgmtConstant.DIGI_DATAKEY_STRING + ApimgmtSymbolConstant.COLON_STRING).append(tDataKeyObject.toString()).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            }
            if (!tNonStandardHeader.isEmpty()) {
                for (String tString : tNonStandardHeader.keySet()) {
                    tBuilder.append(tString).append(ApimgmtSymbolConstant.COLON_STRING).append(tNonStandardHeader.get(tString)).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                }
            }
            if (GeneratedApiMessageComponentEnum.ALL.equals(pComponentEnum)) {
                tMessageReturned = tBuilder.append(GSON.toJson(tBodyObject)).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING).toString();
            } else if (GeneratedApiMessageComponentEnum.BODY.equals(pComponentEnum)) {
                tMessageReturned = GSON.toJson(tBodyObject);
            } else if (GeneratedApiMessageComponentEnum.DATAKEY.equals(pComponentEnum)) {
                if (tDataKeyObject.entrySet().size() != 0) {
                    tMessageReturned = tDataKeyObject.toString();
                } else {
                    tMessageReturned = "";
                }
            }
        } else if (MessageTextSampleTypeEnum.SYNC_RESPONSE_SUCCESS.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.SYNC_RESPONSE_FAILED.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_CALLBACK_SUCCESS.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_CALLBACK_FAILED.equals(tMessageTextSampleTypeEnum)) {
            int parameterExist = 0;
            int stdDataExist = 0;
            int executionExist = 0;
            StringBuilder tBuilder = new StringBuilder();
            tBuilder.append(ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            if ((DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole))
                    && (MessageTextSampleTypeEnum.SYNC_RESPONSE_SUCCESS.equals(tMessageTextSampleTypeEnum)
                    || MessageTextSampleTypeEnum.SYNC_RESPONSE_FAILED.equals(tMessageTextSampleTypeEnum))) {
                tBuilder.append(ApimgmtConstant.DIGI_CODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " 019" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                tBuilder.append(ApimgmtConstant.DIGI_MESSAGE_STRING + ApimgmtSymbolConstant.COLON_STRING
                        + " RUFJIFJlc3BvbnNlOuW5s+WPsOWbnuimhuWQjOatpee1kOaenA==" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                tBuilder.append(ApimgmtConstant.DIGI_REQID_STRING + ApimgmtSymbolConstant.COLON_STRING + " ").append(pCustomUUID).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            } else if ((DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole))
                    && (MessageTextSampleTypeEnum.FASYNC_CALLBACK_SUCCESS.equals(tMessageTextSampleTypeEnum)
                    || MessageTextSampleTypeEnum.FASYNC_CALLBACK_FAILED.equals(tMessageTextSampleTypeEnum))) {
                tBuilder.append(ApimgmtConstant.DIGI_REQID_STRING + ApimgmtSymbolConstant.COLON_STRING + " ").append(pCustomUUID).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            } else if ((DeveloperRoleEnum.PROVIDER.toString().equals(pDeveloperRole))
                    && (MessageTextSampleTypeEnum.FASYNC_CALLBACK_SUCCESS.equals(tMessageTextSampleTypeEnum)
                    || MessageTextSampleTypeEnum.FASYNC_CALLBACK_FAILED.equals(tMessageTextSampleTypeEnum))) { // 0EA,
                // 0EB
                tBuilder.append(ApimgmtConstant.DIGI_REQID_STRING + ApimgmtSymbolConstant.COLON_STRING + " ").append(pCustomUUID).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            }
            tBuilder.append(ApimgmtConstant.DIGI_PROTOCOL_STRING + ApimgmtSymbolConstant.COLON_STRING + "raw" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            tBuilder.append(ApimgmtConstant.DIGI_SRVVER_STRING + ApimgmtSymbolConstant.COLON_STRING + " 1.0" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            if (MessageTextSampleTypeEnum.SYNC_RESPONSE_SUCCESS.equals(tMessageTextSampleTypeEnum)
                    || MessageTextSampleTypeEnum.FASYNC_CALLBACK_SUCCESS.equals(tMessageTextSampleTypeEnum)) {
                tBuilder.append(ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING
                        + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            } else {
                tBuilder.append(
                        ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_FAILED_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            }
            JsonObject tBodyObject = new JsonObject();
            JsonObject tStdDataObject = new JsonObject();
            JsonObject tParameterObject = new JsonObject();
            JsonObject tExecutionObject = new JsonObject();
            JsonObject tDataKeyObject = new JsonObject();
            // 先產生execution，避免parameter可能會在execution上方的情況
            tStdDataObject.add(ApimgmtConstant.EXECUTION, tExecutionObject);
            Map<String, String> tNonStandardHeader = new HashMap<>();
            // 取得所有最外層節點，最外層節點可為欄位類型: M、P、H、E
            for (StandardApiDataName tStandardApiDataName : pStandardApiDataNames) {
                // 如果有任何標準API資料且類型不為H，則產生std_data，且僅產生一次。
                if (!tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.HEADER_CONSTANTS)
                        && stdDataExist == 0) {
                    tBodyObject.add(ApimgmtConstant.STD_DATA, tStdDataObject);
                    stdDataExist = 1;
                }
                // 如果有任何標準API資料且類型不為H也不為E，則產生parameter，且僅產生一次。
                if (!(tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.HEADER_CONSTANTS))
                        && !(tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.EXECUTION_CONSTANTS))
                        && parameterExist == 0) {
                    tStdDataObject.add(ApimgmtConstant.PARAMETER, tParameterObject);
                    parameterExist = 1;
                }
                // 標準API資料類型為M(單頭)
                if (tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.MASTER_CONSTANTS)) {
                    if (pIsBatch) {
                        JsonObject tAttrObject = new JsonObject();
                        tAttrObject.addProperty(ApimgmtConstant.CNT_STRING, ApimgmtConstant.CNT_VALUE);
                        tAttrObject.addProperty(ApimgmtConstant.TT_CNT_STRING, ApimgmtConstant.TT_CNT_VALUE);
                        tParameterObject.add(tStandardApiDataName.getStandardDataName() + ApimgmtConstant.UNDERSCORE_ATTR_STRING,
                                tAttrObject);
                    }
                    JsonObject tObject = new JsonObject();
                    generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDataKeyObject, pMockValEnum,
                            pMapMockVal, tLocale);
                    if (tStandardApiDataName.getIsArray()) {
                        JsonArray tArray = new JsonArray();
                        tArray.add(tObject);
                        tParameterObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                    } else {
                        tParameterObject.add(tStandardApiDataName.getStandardDataName(), tObject);
                    }
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.PARAMETER_CONSTANTS)) {
                    addJsonObjectProperty(tParameterObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    if (tStandardApiDataName.getIsDatakey()) {
                        addJsonObjectProperty(tDataKeyObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    }
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.HEADER_CONSTANTS)) {
                    String tNonStandardHeaderName;
                    String tNonStandardHeaderValue;
                    if (tStandardApiDataName.getValue() != null) {
                        tNonStandardHeaderName = tStandardApiDataName.getStandardDataName();
                        tNonStandardHeaderValue = tStandardApiDataName.getValue();
                    } else {
                        JsonObject tObject = new JsonObject();
                        generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDataKeyObject, pMockValEnum,
                                pMapMockVal, tLocale);
                        tNonStandardHeaderName = tStandardApiDataName.getStandardDataName();
                        if (tObject.size() == 0) {
                            tNonStandardHeaderValue = "";
                        } else {
                            tNonStandardHeaderValue = tObject.toString();
                        }
                    }
                    tNonStandardHeader.put(tNonStandardHeaderName, tNonStandardHeaderValue);
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.EXECUTION_CONSTANTS)) {
                    addJsonObjectProperty(tExecutionObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    executionExist = 1;
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.DATAKEY_CONSTANTS)) {
                    addJsonObjectProperty(tDataKeyObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                }
            }
            if (tDataKeyObject.entrySet().size() != 0) {
                tBuilder.append(ApimgmtConstant.DIGI_DATAKEY_STRING + ApimgmtSymbolConstant.COLON_STRING).append(tDataKeyObject.toString()).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            }
            // 如果此標準API沒有columnType=E的標準API資料，則不應該有execution，故把前面產生的execution拿掉
            if (executionExist == 0) {
                tStdDataObject.remove(ApimgmtConstant.EXECUTION);
            }
            if (!tNonStandardHeader.isEmpty()) {
                for (String tString : tNonStandardHeader.keySet()) {
                    tBuilder.append(tString).append(ApimgmtSymbolConstant.COLON_STRING).append(tNonStandardHeader.get(tString)).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                }
            }
            tMessageReturned = tBuilder.append(GSON.toJson(tBodyObject)).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING).toString();
        } else if (MessageTextSampleTypeEnum.FASYNC_RESPONSE.equals(tMessageTextSampleTypeEnum)
                && DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole)) {
            tMessageReturned = ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_CODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " 0E9" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVVER_STRING + ApimgmtSymbolConstant.COLON_STRING + " 1.0" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_REQID_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + pCustomUUID + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_MESSAGE_STRING + ApimgmtSymbolConstant.COLON_STRING
                    + " RUFJIFJlc3BvbnNlOuW5s+WPsOWbnuimhuWFqOeVsOatpeiri+axguaOpeaUtueLgOaFiw=="
                    + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING;
        } else if (MessageTextSampleTypeEnum.FASYNC_CALLBACK_RESPONSE.equals(tMessageTextSampleTypeEnum)
                && DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole)) {
            tMessageReturned = ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING;
        } else if (MessageTextSampleTypeEnum.FASYNC_RESPONSE.equals(tMessageTextSampleTypeEnum)
                && DeveloperRoleEnum.PROVIDER.toString().equals(pDeveloperRole)) { // 0E5
            tMessageReturned = ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVVER_STRING + ApimgmtSymbolConstant.COLON_STRING + " 1.0" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING;
        } else if (MessageTextSampleTypeEnum.FASYNC_CALLBACK_RESPONSE.equals(tMessageTextSampleTypeEnum)
                && DeveloperRoleEnum.PROVIDER.toString().equals(pDeveloperRole)) {
            tMessageReturned = ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_CODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " 0EG" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_MESSAGE_STRING + ApimgmtSymbolConstant.COLON_STRING
                    + " RUFJIFJlc3BvbnNlOuW5s+WPsOWbnuimhuWFqOeVsOatpee1kOaenOaOpeaUtueLgOaFiw=="
                    + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING;
        }
        return tMessageReturned;
    }

    /**
     * Openai单独写一个方法，后续重构
     * @param tMessageTextSampleTypeEnum
     * @param pStandardApiDataNames
     * @param pApiName
     * @param pHostProd
     * @param pServiceProd
     * @param pDeveloperRole
     * @param pIsBatch
     * @param pCustomUUID
     * @param pComponentEnum
     * @param pMockValEnum
     * @param pMapMockVal
     * @param tLocale
     * @return
     */
    private String transTreeNodeToJSONString4OpenAPI(StandardApi standardApi, MessageTextSampleTypeEnum tMessageTextSampleTypeEnum,
                                            List<StandardApiDataName> pStandardApiDataNames, String pApiName, String pHostProd, String pServiceProd,
                                            String pDeveloperRole, boolean pIsBatch, String pCustomUUID,
                                            GeneratedApiMessageComponentEnum pComponentEnum, GeneratedApiMessageMockValEnum pMockValEnum,
                                            Map<String, String> pMapMockVal, String tLocale) {
        String tMessageReturned = null;
        if (MessageTextSampleTypeEnum.SYNC_REQUEST.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_REQUEST.equals(tMessageTextSampleTypeEnum)) {
            StringBuilder tBuilder = new StringBuilder();
            //{@request_method} {@protocol}://IP:PORT/{@api_path}
            String message = String.format("%s %S://IP:PORT%S", standardApi.getRequestMethod(), standardApi.getProtocol(), standardApi.getApiPath());
            tBuilder.append(message);
            tBuilder.append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
//            tBuilder.append(ApimgmtConstant.DIGI_PROTOCOL_STRING + ApimgmtSymbolConstant.COLON_STRING + "raw" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
//            JsonObject tHostObject = new JsonObject();
//            tHostObject.addProperty(ApimgmtConstant.PROD_STRING, pHostProd);
//            tHostObject.addProperty(ApimgmtConstant.IP_STRING, "10.20.88.30");
//            tHostObject.addProperty(ApimgmtConstant.LANG_STRING, ApimgmtConstant.ZH_TW);
//            tHostObject.addProperty(ApimgmtConstant.TIMESTAMP_STRING, "20161018132542116");
//            tHostObject.addProperty(ApimgmtConstant.ACCT_STRING, "tiptop");
//            JsonObject tServiceObject = new JsonObject();
//            tServiceObject.addProperty(ApimgmtConstant.PROD_STRING, pServiceProd);
//            tServiceObject.addProperty(ApimgmtConstant.NAME, pApiName);
//            tServiceObject.addProperty(ApimgmtConstant.IP_STRING, "10.20.88.75");
//            tServiceObject.addProperty(ApimgmtConstant.ID, "topprod");
            if (DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole)) {
                if (MessageTextSampleTypeEnum.SYNC_REQUEST.equals(tMessageTextSampleTypeEnum)) {
                    tBuilder.append(ApimgmtConstant.DIGI_TYPE_STRING + ApimgmtSymbolConstant.COLON_STRING + ApimgmtConstant.SYNC + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                } else {
                    tBuilder.append(ApimgmtConstant.DIGI_TYPE_STRING + ApimgmtSymbolConstant.COLON_STRING + ApimgmtConstant.FASYNC + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                }
            } else if (DeveloperRoleEnum.PROVIDER.toString().equals(pDeveloperRole)) {
                tBuilder.append(ApimgmtConstant.DIGI_REQID_STRING + ApimgmtSymbolConstant.COLON_STRING).append(pCustomUUID).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            }

            //JsonObject tBodyObject = new JsonObject();
            //JsonObject tStdDataObject = new JsonObject();
            JsonObject tParameterObject = new JsonObject();
            JsonObject tDataKeyObject = new JsonObject();
            Map<String, String> tNonStandardHeader = new HashMap<>();
            //tBodyObject.add(ApimgmtConstant.STD_DATA, tStdDataObject);
            //tStdDataObject.add(ApimgmtConstant.PARAMETER, tParameterObject);
            // 取得所有最外層節點，最外層節點可為欄位類型: M、P、H、datakey
            for (StandardApiDataName tStandardApiDataName : pStandardApiDataNames) {
                // 標準API資料類型為M(單頭)
                if (tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.MASTER_CONSTANTS)) {
                    JsonObject tObject = new JsonObject();
                    generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDataKeyObject, pMockValEnum,
                            pMapMockVal, tLocale);
                    if (tStandardApiDataName.getIsArray()) {
                        JsonArray tArray = new JsonArray();
                        tArray.add(tObject);
                        tParameterObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                    } else {
                        tParameterObject.add(tStandardApiDataName.getStandardDataName(), tObject);
                    }
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.PARAMETER_CONSTANTS)) {
                    addJsonObjectProperty(tParameterObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    if (tStandardApiDataName.getIsDatakey()) {
                        addJsonObjectProperty(tDataKeyObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    }
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.HEADER_CONSTANTS)) {
                    String tNonStandardHeaderName;
                    String tNonStandardHeaderValue;
                    if (tStandardApiDataName.getValue() != null) {
                        tNonStandardHeaderName = tStandardApiDataName.getStandardDataName();
                        tNonStandardHeaderValue = tStandardApiDataName.getValue();
                    } else {
                        JsonObject tObject = new JsonObject();
                        generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDataKeyObject, pMockValEnum,
                                pMapMockVal, tLocale);
                        tNonStandardHeaderName = tStandardApiDataName.getStandardDataName();
                        if (tObject.size() == 0) {
                            tNonStandardHeaderValue = "";
                        } else {
                            tNonStandardHeaderValue = tObject.toString();
                        }
                    }
                    tNonStandardHeader.put(tNonStandardHeaderName, tNonStandardHeaderValue);
                    // 標準API資料類型為datakey
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.DATAKEY_CONSTANTS)) {
                    addJsonObjectProperty(tDataKeyObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                }
            }
            if (tDataKeyObject.entrySet().size() != 0) {
                tBuilder.append(ApimgmtConstant.DIGI_DATAKEY_STRING + ApimgmtSymbolConstant.COLON_STRING).append(tDataKeyObject.toString()).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            }
            if (!tNonStandardHeader.isEmpty()) {
                for (String tString : tNonStandardHeader.keySet()) {
                    tBuilder.append(tString).append(ApimgmtSymbolConstant.COLON_STRING).append(tNonStandardHeader.get(tString)).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                }
            }
            if (GeneratedApiMessageComponentEnum.ALL.equals(pComponentEnum)) {
                tMessageReturned = tBuilder.append(GSON.toJson(tParameterObject)).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING).toString();
            } else if (GeneratedApiMessageComponentEnum.BODY.equals(pComponentEnum)) {
                tMessageReturned = GSON.toJson(tParameterObject);
            } else if (GeneratedApiMessageComponentEnum.DATAKEY.equals(pComponentEnum)) {
                if (tDataKeyObject.entrySet().size() != 0) {
                    tMessageReturned = tDataKeyObject.toString();
                } else {
                    tMessageReturned = "";
                }
            }
        } else if (MessageTextSampleTypeEnum.SYNC_RESPONSE_SUCCESS.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.SYNC_RESPONSE_FAILED.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_CALLBACK_SUCCESS.equals(tMessageTextSampleTypeEnum)
                || MessageTextSampleTypeEnum.FASYNC_CALLBACK_FAILED.equals(tMessageTextSampleTypeEnum)) {
            StringBuilder tBuilder = new StringBuilder();
            tBuilder.append(ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            tBuilder.append(ApimgmtConstant.DIGI_PROTOCOL_STRING + ApimgmtSymbolConstant.COLON_STRING + "raw" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            tBuilder.append(ApimgmtConstant.DIGI_SRVVER_STRING + ApimgmtSymbolConstant.COLON_STRING + " 1.0" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            JsonObject tParameterObject = new JsonObject();
            JsonObject tExecutionObject = new JsonObject();
            JsonObject tDataKeyObject = new JsonObject();
            Map<String, String> tNonStandardHeader = new HashMap<>();
            // 取得所有最外層節點，最外層節點可為欄位類型: M、P、H、E
            for (StandardApiDataName tStandardApiDataName : pStandardApiDataNames) {
                // 標準API資料類型為M(單頭)
                if (tStandardApiDataName.getColumnType().equals(StandardApiColumnTypeConstants.MASTER_CONSTANTS)) {
                    if (pIsBatch) {
                        JsonObject tAttrObject = new JsonObject();
                        tAttrObject.addProperty(ApimgmtConstant.CNT_STRING, ApimgmtConstant.CNT_VALUE);
                        tAttrObject.addProperty(ApimgmtConstant.TT_CNT_STRING, ApimgmtConstant.TT_CNT_VALUE);
                        tParameterObject.add(tStandardApiDataName.getStandardDataName() + ApimgmtConstant.UNDERSCORE_ATTR_STRING,
                                tAttrObject);
                    }
                    JsonObject tObject = new JsonObject();
                    generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDataKeyObject, pMockValEnum,
                            pMapMockVal, tLocale);
                    if (tStandardApiDataName.getIsArray()) {
                        JsonArray tArray = new JsonArray();
                        tArray.add(tObject);
                        tParameterObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                    } else {
                        tParameterObject.add(tStandardApiDataName.getStandardDataName(), tObject);
                    }
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.PARAMETER_CONSTANTS)) {
                    addJsonObjectProperty(tParameterObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    if (tStandardApiDataName.getIsDatakey()) {
                        addJsonObjectProperty(tDataKeyObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    }
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.HEADER_CONSTANTS)) {
                    String tNonStandardHeaderName;
                    String tNonStandardHeaderValue;
                    if (tStandardApiDataName.getValue() != null) {
                        tNonStandardHeaderName = tStandardApiDataName.getStandardDataName();
                        tNonStandardHeaderValue = tStandardApiDataName.getValue();
                    } else {
                        JsonObject tObject = new JsonObject();
                        generateAllSubNodeJSONString(tStandardApiDataName, tObject, tDataKeyObject, pMockValEnum,
                                pMapMockVal, tLocale);
                        tNonStandardHeaderName = tStandardApiDataName.getStandardDataName();
                        if (tObject.size() == 0) {
                            tNonStandardHeaderValue = "";
                        } else {
                            tNonStandardHeaderValue = tObject.toString();
                        }
                    }
                    tNonStandardHeader.put(tNonStandardHeaderName, tNonStandardHeaderValue);
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.EXECUTION_CONSTANTS)) {
                    addJsonObjectProperty(tExecutionObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                    addJsonObjectProperty(tParameterObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                } else if (tStandardApiDataName.getColumnType()
                        .equals(StandardApiColumnTypeConstants.DATAKEY_CONSTANTS)) {
                    addJsonObjectProperty(tDataKeyObject, tStandardApiDataName, pMockValEnum, pMapMockVal, tLocale);
                }
            }
            if (tDataKeyObject.entrySet().size() != 0) {
                tBuilder.append(ApimgmtConstant.DIGI_DATAKEY_STRING + ApimgmtSymbolConstant.COLON_STRING).append(tDataKeyObject.toString()).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
            }

            if (!tNonStandardHeader.isEmpty()) {
                for (String tString : tNonStandardHeader.keySet()) {
                    tBuilder.append(tString).append(ApimgmtSymbolConstant.COLON_STRING).append(tNonStandardHeader.get(tString)).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING);
                }
            }
            tMessageReturned = tBuilder.append(GSON.toJson(tParameterObject)).append(ApimgmtSymbolConstant.CARRIGE_RETURN_STRING).toString();
        } else if (MessageTextSampleTypeEnum.FASYNC_RESPONSE.equals(tMessageTextSampleTypeEnum)
                && DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole)) {
            tMessageReturned = ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_CODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " 0E9" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVVER_STRING + ApimgmtSymbolConstant.COLON_STRING + " 1.0" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_REQID_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + pCustomUUID + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_MESSAGE_STRING + ApimgmtSymbolConstant.COLON_STRING
                    + " RUFJIFJlc3BvbnNlOuW5s+WPsOWbnuimhuWFqOeVsOatpeiri+axguaOpeaUtueLgOaFiw=="
                    + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING;
        } else if (MessageTextSampleTypeEnum.FASYNC_CALLBACK_RESPONSE.equals(tMessageTextSampleTypeEnum)
                && DeveloperRoleEnum.REQUESTER.toString().equals(pDeveloperRole)) {
            tMessageReturned = ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING;
        } else if (MessageTextSampleTypeEnum.FASYNC_RESPONSE.equals(tMessageTextSampleTypeEnum)
                && DeveloperRoleEnum.PROVIDER.toString().equals(pDeveloperRole)) { // 0E5
            tMessageReturned = ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVVER_STRING + ApimgmtSymbolConstant.COLON_STRING + " 1.0" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING;
        } else if (MessageTextSampleTypeEnum.FASYNC_CALLBACK_RESPONSE.equals(tMessageTextSampleTypeEnum)
                && DeveloperRoleEnum.PROVIDER.toString().equals(pDeveloperRole)) {
            tMessageReturned = ApimgmtConstant.CONTENT_ENCODE_HEADER_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_CODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " 0EG" + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_MESSAGE_STRING + ApimgmtSymbolConstant.COLON_STRING
                    + " RUFJIFJlc3BvbnNlOuW5s+WPsOWbnuimhuWFqOeVsOatpee1kOaenOaOpeaUtueLgOaFiw=="
                    + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING +
                    ApimgmtConstant.DIGI_SRVCODE_STRING + ApimgmtSymbolConstant.COLON_STRING + " " + ApimgmtConstant.PROD_RESPONSE_SUCCESS_STRING + ApimgmtSymbolConstant.CARRIGE_RETURN_STRING;
        }
        return tMessageReturned;
    }

    public void generateAllSubNodeXmlString(StandardApiDataName pStandardApiDataName, Element pParentElement,
                                            Element pDatakeyElement, String tLocale) {
        // 這邊是確定有子node的才會被叫到這
        for (StandardApiDataName tStandardApiDataName : pStandardApiDataName.getChildrens()) {
            if (tStandardApiDataName.getColumnType().equals("MF")) {
                Element MFElement = pParentElement.addElement(ApimgmtConstant.FIELD_STRING);
                MFElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                MFElement.addAttribute(ApimgmtConstant.TYPE_STRING, tStandardApiDataName.getDataType());
                setValue(tStandardApiDataName, tLocale, MFElement);
                if (tStandardApiDataName.getIsDatakey()) {
                    Element keyElement = pDatakeyElement.addElement(ApimgmtConstant.KEY_STRING);
                    keyElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                    setValue(tStandardApiDataName, tLocale, keyElement);
                }
            } else if (tStandardApiDataName.getColumnType().endsWith("D")) {
                Element detailElement = pParentElement.addElement(ApimgmtConstant.DETAIL_STRING);
                detailElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                if (tStandardApiDataName.getIsArray()) {
                    Element rowElement = detailElement.addElement(ApimgmtConstant.ROW_STRING);
                    generateAllSubNodeXmlString(tStandardApiDataName, rowElement, pDatakeyElement, tLocale);
                } else {
                    generateAllSubNodeXmlString(tStandardApiDataName, detailElement, pDatakeyElement, tLocale);
                }
            } else if (tStandardApiDataName.getColumnType().endsWith("DF")) {
                Element DFElement = pParentElement.addElement(ApimgmtConstant.FIELD_STRING);
                DFElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                DFElement.addAttribute(ApimgmtConstant.TYPE_STRING, tStandardApiDataName.getDataType());
                setValue(tStandardApiDataName, tLocale, DFElement);
                if (tStandardApiDataName.getIsDatakey()) {
                    Element keyElement = pDatakeyElement.addElement(ApimgmtConstant.KEY_STRING);
                    keyElement.addAttribute(ApimgmtConstant.NAME, tStandardApiDataName.getStandardDataName());
                    setValue(tStandardApiDataName, tLocale, keyElement);
                }
            }
        }
    }

    /**
     * // 這邊是確定有子node的才會被叫到這
     */
    public void generateAllSubNodeJSONString(StandardApiDataName pStandardApiDataName, JsonObject pParentElement,
                                             JsonObject pDatakeyObject, GeneratedApiMessageMockValEnum pMockValEnum,
                                             Map<String, String> pMapMockVal, String pLocal) {
        for (StandardApiDataName tStandardApiDataName : pStandardApiDataName.getChildrens()) {
            if (tStandardApiDataName.getColumnType().endsWith("D")) {
                JsonObject tDetailObject = new JsonObject();
                generateAllSubNodeJSONString(tStandardApiDataName, tDetailObject, pDatakeyObject, pMockValEnum,
                        pMapMockVal, pLocal);
                if (tStandardApiDataName.getIsArray()) {
                    JsonArray tDetailArray = new JsonArray();
                    tDetailArray.add(tDetailObject);
                    pParentElement.add(tStandardApiDataName.getStandardDataName(), tDetailArray);
                } else {
                    pParentElement.add(tStandardApiDataName.getStandardDataName(), tDetailObject);
                }
            } else if (tStandardApiDataName.getColumnType().equals("MF")
                    || tStandardApiDataName.getColumnType().endsWith("DF")) {
                addJsonObjectProperty(pParentElement, tStandardApiDataName, pMockValEnum, pMapMockVal, pLocal);
                if (tStandardApiDataName.getIsDatakey()) {
                    addJsonObjectProperty(pDatakeyObject, tStandardApiDataName, pMockValEnum, pMapMockVal, pLocal);
                }
            } else if (tStandardApiDataName.getColumnType().equals("HF")) {
                addJsonObjectProperty(pParentElement, tStandardApiDataName, pMockValEnum, pMapMockVal, pLocal);
            }
        }
    }

    public void addJsonObjectProperty(JsonObject tJsonObject, StandardApiDataName tStandardApiDataName,
                                      GeneratedApiMessageMockValEnum tMockValEnum, Map<String, String> pMapMockVal, String pLocal) {
        if (GeneratedApiMessageMockValEnum.MOCK.equals(tMockValEnum)) {
            String tKey = tStandardApiDataName.getId().toString();
            String tVal = pMapMockVal.get(tKey).trim();
            if (!StringUtils.isBlank(tVal)) {
                String tDataType = tStandardApiDataName.getDataType();
                if ("numeric".equals(tDataType)) {
                    double tDouble = Double.parseDouble(tVal);
                    if (tDouble == Math.floor(tDouble)) {
                        if (tStandardApiDataName.getIsArray()) {
                            JsonArray tArray = new JsonArray();
                            tArray.add(Integer.parseInt(tVal));
                            tJsonObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                        } else {
                            tJsonObject.addProperty(tStandardApiDataName.getStandardDataName(), Integer.parseInt(tVal));
                        }
                    } else {
                        if (tStandardApiDataName.getIsArray()) {
                            JsonArray tArray = new JsonArray();
                            tArray.add(Double.parseDouble(tVal));
                            tJsonObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                        } else {
                            tJsonObject.addProperty(tStandardApiDataName.getStandardDataName(),
                                    Double.parseDouble(tVal));
                        }
                    }
                } else if ("boolean".equals(tDataType)) {
                    if (tStandardApiDataName.getIsArray()) {
                        JsonArray tArray = new JsonArray();
                        tArray.add(Boolean.parseBoolean(tVal));
                        tJsonObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                    } else {
                        tJsonObject.addProperty(tStandardApiDataName.getStandardDataName(), Boolean.parseBoolean(tVal));
                    }
                } else {
                    if (tStandardApiDataName.getIsArray()) {
                        JsonArray tArray = new JsonArray();
                        tArray.add(getValue(tStandardApiDataName, pLocal));
                        tJsonObject.add(tStandardApiDataName.getStandardDataName(), tArray);
                    } else {
                        tJsonObject.addProperty(tStandardApiDataName.getStandardDataName(),
                                getValue(tStandardApiDataName, pLocal));
                    }
                }
            }
        } else {
            if (tStandardApiDataName.getIsArray()) {
                JsonArray tArray = new JsonArray();
                tArray.add(getValue(tStandardApiDataName, pLocal));
                tJsonObject.add(tStandardApiDataName.getStandardDataName(), tArray);
            } else {
                tJsonObject.addProperty(tStandardApiDataName.getStandardDataName(),
                        getValue(tStandardApiDataName, pLocal));
            }
        }
    }

    private void setValue(StandardApiDataName tStandardApiDataName, String tLocale, Element element) {
        if (tStandardApiDataName.getValue() != null) {
            element.addText(tStandardApiDataName.getValue());
        } else if (ApimgmtConstant.ZH_TW.equals(tLocale) && tStandardApiDataName.getDescriptionZhTw() != null) {
            element.addText(tStandardApiDataName.getDescriptionZhTw());
        } else if (ApimgmtConstant.ZH_CN.equals(tLocale) && tStandardApiDataName.getDescriptionZhCn() != null) {
            element.addText(tStandardApiDataName.getDescriptionZhCn());
        } else if (ApimgmtConstant.EN_US.equals(tLocale) && tStandardApiDataName.getDescriptionEnUs() != null) {
            element.addText(tStandardApiDataName.getDescriptionEnUs());
        }
    }

    private String getValue(StandardApiDataName tStandardApiDataName, String tLocale) {
        String value = "";
        if (ApimgmtConstant.ZH_TW.equals(tLocale) && tStandardApiDataName.getDescriptionZhTw() != null) {
            value = tStandardApiDataName.getDescriptionZhTw();
        } else if (ApimgmtConstant.ZH_CN.equals(tLocale) && tStandardApiDataName.getDescriptionZhCn() != null) {
            value = tStandardApiDataName.getDescriptionZhCn();
        } else if (ApimgmtConstant.EN_US.equals(tLocale) && tStandardApiDataName.getDescriptionEnUs() != null) {
            value = tStandardApiDataName.getDescriptionEnUs();
        } else if (tStandardApiDataName.getValue() != null) {
            value = tStandardApiDataName.getValue();
        }
        return value;
    }
}
